/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.util;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.xmi.XMIResource;

/**
 * Contains utility methods to manipulate the Eclipse resources.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ResourceUtil
{
	public static final Map RESOURCE_OPTIONS = new HashMap();
	{
		RESOURCE_OPTIONS.put(XMIResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMIResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
	}
	
	/**
	 * Interface to define a visitor for the <code>getFile(...)</code> methods.
	 */
	public static interface IGetFilesVisitor
	{
		/**
		 * This method is invoked before adding a file to a map.
		 * 
		 * <p>The object returned by this mapping is added to the map.  If the
		 * map returns <code>null</code> then the map is not changed.
		 * 
		 * @param file
		 * @return Object
		 */
		public Object visit(IFile file);
	}
	
	/**
	 * Creates the given container and it's parent if necessary.
	 * @param container
	 * @param progressMonitor
	 */
	public static void createContainer(IContainer container, IProgressMonitor progressMonitor)
	throws CoreException
	{
		if((container == null) || container.exists())
			return;	
		createContainer(container.getParent(), progressMonitor);
		
		if(container instanceof IProject)
			((IProject)container).create(progressMonitor);
			
		else if(container instanceof IFolder)
			((IFolder)container).create(true, true, progressMonitor);		
	}
	
	/**
	 * Returns a map with all the "files" from all the projects in a given 
	 * workspace root that have their file extension listed in the file extensions 
	 * array.
	 * 
	 * <p>This method looks for files recursively if the resource is a project 
	 * or a folder.
	 * 
	 * <p>In order to be added to the map the file must exists and must belong to
	 * an open project.
	 *  
	 * <p>Clients can specify a visitor to be used by this method.  The visitor
	 * is able to defined the object that added to the map - that's the reason
	 * the word file is between quotes in the paragraphs above. If the visitor 
	 * is <code>null</code> then the file is added to the map
	 *  
	 * @param workspaceRoot
	 * @param fileExtensions. 
	 * @param visitor
	 * @return a Map with a <code>List</code> of instances of 
	 * {@link org.eclipse.core.resources.IFile} or with the values defined by the
	 * visitor for each file extension.
	 */
	public static Map getFiles(IWorkspaceRoot workspaceRoot, String[] fileExtensions, IGetFilesVisitor visitor)
	{
		Map filesByFileExtension = new HashMap(fileExtensions.length, 1.1f);
		for(int i = 0, maxi = fileExtensions.length; i < maxi; i++)
			filesByFileExtension.put(fileExtensions[i], new ArrayList());
			
		if(visitor == null)
		{
			visitor = new IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					return file;
				}
			};
		}
			
		IProject[] projects = workspaceRoot.getProjects();
		for(int i=0; i<projects.length; i++)
			getFiles(projects[i], filesByFileExtension, visitor);
			
		return filesByFileExtension;
	}

	/**
	 * Returns a map with all the "files" that are "related" to a given resource
	 * and that have their file extension listed in the file extensions array.
	 * 
	 * <p>This method looks for files recursively if the resource is a project 
	 * or a folder.
	 * 
	 * <p>In order to be added to the map the file must exists and must belong to
	 * an open project.
	 * 
	 * <p>Clients can specify a visitor to be used by this method.  The visitor
	 * is able to defined the object that added to the map - that's the reason
	 * the word file is between quotes in the paragraphs above. If the visitor 
	 * is <code>null</code> then the file is added to the map
	 *  
	 * @param resource
	 * @param fileExtensions. 
	 * @return a Map with a <code>List</code> of instances of 
	 * {@link org.eclipse.core.resources.IFile} or with the values defined by the
	 * visitor for each file extension.
	 */
	public static Map getFiles(IResource resource, String[] fileExtensions, IGetFilesVisitor visitor)
	{
		if((resource == null) || (!resource.exists()))
			return Collections.EMPTY_MAP;
		
		if(visitor == null)
		{
			visitor = new IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					return file;
				}
			};
		}

		Map filesByFileExtension = new HashMap(fileExtensions.length, 1.1f);
		for(int i = 0, maxi = fileExtensions.length; i < maxi; i++)
			filesByFileExtension.put(fileExtensions[i], new ArrayList());
		getFiles(resource, filesByFileExtension, visitor);
		
		return filesByFileExtension;
	}

	/**
	 * Internal implementation of {@link #getFiles(IResource, String[]).  The map
	 * is supposed to have a list for all the required file extensions.
	 * @param resource
	 * @param filesByFileExtension
	 * @param a not <code>null</code> visitor
	 */
	private static void getFiles(IResource resource, Map filesByFileExtension, IGetFilesVisitor visitor)
	{
		if((resource == null) || (!resource.exists()))
			return;
			
		switch(resource.getType())
		{
			case IResource.FILE:
				List files = (List)filesByFileExtension.get(resource.getFileExtension());
				if(files != null)
				{
					Object object = visitor.visit((IFile)resource);
					if(object != null)
						files.add(object);
				}
				break;
				
			case IResource.PROJECT:
				IProject project = (IProject)resource;
				if(!project.isOpen())
					return;
					
			case IResource.FOLDER:
				try
				{
					IResource[] members = ((IContainer)resource).members();
					for(int i=0; i< members.length; i++)
						getFiles(members[i], filesByFileExtension, visitor);
				}
				catch (CoreException e)
				{
					logError(e);
				}
				break;
		}
	}
	
	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> then this method tries
	 * to find a registered resource factory.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param file
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet resourceSet, IFile file)
	{
		if(file == null)
			return new EObject[0];

		return load(resourceSet, file.getFullPath().toString());
	}
	
	/**
	 * Loads the specified file returning all the root EObjects.  If the
	 * resource set argument is <code>null</code> then this method tries
	 * to find a registered resource factory.
	 * 
	 * <p>If the load attempt fails this method adds an entry to the
	 * log and returns an empty array.
	 * 
	 * @param resourceSet 
	 * @param fullPath
	 * @return EObject[]
	 */
	public static EObject[] load(ResourceSet resourceSet, String fullPath)
	{
		if(fullPath == null)
			return new EObject[0];
			
		URI uri = URI.createPlatformResourceURI(fullPath);
		Resource resource = null;
		try
		{
			if(resourceSet != null)
			{
				resource = resourceSet.getResource(uri, true);
			}
			else
			{
				IPath path = new Path(fullPath);				
				Resource.Factory factory = getResourceFactory(path.getFileExtension());
				if(factory != null)
				{
					resource = factory.createResource(uri);
					resource.load(RESOURCE_OPTIONS);
				}
			}
		}
		catch(Exception e)
		{
			logError(e);
		}
		
		if(resource == null)
			return new EObject[0];
			
		return (EObject[])resource.getContents().toArray(new EObject[resource.getContents().size()]);			
	}
	
	/**
	 * Returns the resource factory associated with a given file extension.
	 * If the file extension is <code>null</code> or if there isn't a specifc
	 * factory for the file extension, the this method returns the generic
	 * factory.
	 * @param fileExtension
	 * @return the Resource.Factory or <code>null</code> if none. 
	 */
	public static Resource.Factory getResourceFactory(String fileExtension)
	{
		if(fileExtension != null)
		{
			Resource.Factory factory = (Resource.Factory)Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(fileExtension);
			if(factory != null)
				return factory;
		}
			
		return (Resource.Factory)Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(fileExtension); 
	}
	

	/**
	 * 
	 * @param e
	 */
	public static final void logError(Throwable e)
	{
		IStatus status = new Status(IStatus.ERROR, ResourcesPlugin.getPlugin().getDescriptor().getUniqueIdentifier(), 0, e.getMessage(), e);
		ResourcesPlugin.getPlugin().getLog().log(status);
		e.printStackTrace();
		
	}
}
