/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.ui.internal.views;

import java.io.File;

import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.model.internal.sensor.*;
import org.eclipse.hyades.logging.adapter.ui.*;
import org.eclipse.hyades.logging.adapter.ui.internal.actions.ActionHandlerListener;
import org.eclipse.hyades.logging.adapter.ui.internal.util.*;
import org.eclipse.hyades.logging.events.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.part.ViewPart;


/**
 */
public class LogView extends ViewPart 
{
	private final String LOG_TITLE = AcadEditorPlugin.getPlugin().getString("TEMPLATE_LOG");

	TextViewer eventsViewer;		
	
	Action createRuleAction;
	Action guessRuleAction;
	Action selectAllAction;
	Action nextEventAction;
	Action previousEventAction;
	Action firstEventAction;
	Action lastEventAction;	
	Action refreshAction;
	Action showLog;
	
	/**
	 * Constructor
	 */
	public LogView() {
		super();
		
	}

	/**
	 * @see IViewPart.init(IViewSite)
	 */
	public void init(IViewSite site) throws PartInitException {
		super.init(site);
	}

	/**
	 * Initializes this view with the given view site.  A memento is passed to
 	 * the view which contains a snapshot of the views state from a previous
	 * session.  	
	 */
	public void init(IViewSite site,IMemento memento) throws PartInitException {
		init(site);
		
		enableEditActions();
		
	}
	
	/**
	 * @see IWorkbenchPart#createPartControl(Composite)
	 */
	public void createPartControl(Composite parent) {
		
		SashForm sashForm = new SashForm(parent, SWT.HORIZONTAL);
		sashForm.setLayout(new FillLayout());
		
		// Create viewer.
		eventsViewer = new TextViewer(sashForm, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
		eventsViewer.setEditable(false);
		
		
		// Create menu and toolbars.
		createActions();
		createToolbar();
		createContextMenu();

		set_Title();
		
		WorkbenchHelp.setHelp(parent, ContextIds.ACAD_EDITOR_LOGVIEW);
						
	}
	
	/**
	 * @see WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		if(eventsViewer != null && eventsViewer.getControl().isDisposed())
			eventsViewer.getControl().setFocus();
	}

	/**
	 * Create the actions.
	 */
	public void createActions() {
		
		showLog = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LOG_ACTION")) {
		public void run() {
							showLog();
						}
					};
		showLog.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LOG_ACTION")); 		
		AcadEditorPluginImages.setImageDescriptors(showLog,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_SHOW_LOG);	
		
		nextEventAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_NEXTE_ACTION")) {
			public void run() { 
								showNextEvent();
							}
						};
		nextEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_NEXTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(nextEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_NEXT_EVET);	
		
		previousEventAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_PREVE_ACTION")) {
			public void run() {
								showPreviousEvent();
							}
						};
		previousEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_PREVE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(previousEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_PREVIOUS_EVENT);	
		
		firstEventAction= new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_FIRSTE_ACTION")) {
		public void run() {
							showFirstEvent();
						}
					};
		firstEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_FIRSTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(firstEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_FIRST_EVENT);	
		
		lastEventAction= new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LASTE_ACTION")) {
		public void run() {
							showLastEvent();
						}
					};
		lastEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LASTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(lastEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_LAST_EVENT);	
			
		refreshAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_RERUN_ACTION")) {
			public void run() { 
				refreshData();
			}
		};
		refreshAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_RERUN_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(refreshAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_REAPPLY);	
		
		createRuleAction = new Action(AcadEditorPlugin.getPlugin().getString("ADD_HASH_ACTION")) {
			public void run() { 
				createRule();
			}
		};
		createRuleAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("ADD_HASH_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(createRuleAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_CREATE_RULE);	
		
		guessRuleAction = new Action(AcadEditorPlugin.getPlugin().getString("GUESS_RULE_ACTION")) {
			public void run() {
				guessRule();
			}
		};
		guessRuleAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("GUESS_RULE_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(guessRuleAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_GUESS_RULE);	
		
		
		// Add selection listener.
		eventsViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateActionEnablement();
			}
		});
	}

	/**
	 * Create toolbar.
	 */
	private void createToolbar() {
		IToolBarManager mgr = getViewSite().getActionBars().getToolBarManager();
		mgr.add(refreshAction);
		mgr.add(new Separator());
		mgr.add(firstEventAction);		
		mgr.add(previousEventAction);
		mgr.add(nextEventAction);
		mgr.add(lastEventAction);
		mgr.add(new Separator());
		mgr.add(showLog);
	}
		
	/**
	 * Create context menu.
	 */
	private void createContextMenu() {
		// Create menu manager.
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});
		
		// Create menu.
		Menu menu = menuMgr.createContextMenu(eventsViewer.getControl());
		eventsViewer.getControl().setMenu(menu);
		
		// Register menu for extension.
		getSite().registerContextMenu(menuMgr, eventsViewer);
	}

	/**
	 * Hook global actions
	 */
	private void hookGlobalActions() {
		IActionBars bars = getViewSite().getActionBars();
		bars.setGlobalActionHandler(IWorkbenchActionConstants.SELECT_ALL, selectAllAction);
		bars.setGlobalActionHandler(IWorkbenchActionConstants.DELETE, guessRuleAction);
		eventsViewer.getControl().addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				if (event.character == SWT.DEL && 
					event.stateMask == 0 && 
				guessRuleAction.isEnabled()) 
				{
					guessRuleAction.run();
				}
			}
		});
	}
		
	private void fillContextMenu(IMenuManager mgr) {
		mgr.add(createRuleAction);
		mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		mgr.add(guessRuleAction);

	}

	private void updateActionEnablement() {
		TextSelection sel = (TextSelection)eventsViewer.getSelection();
		createRuleAction.setEnabled(!sel.isEmpty());
		guessRuleAction.setEnabled(!sel.isEmpty());
	}
					
	private void showNextEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;

		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getNextEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		set_Title();
		enableActions();
		
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)		
		   contentView.showLog();
		
	}
	
	private void showPreviousEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getPreviousEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		
		set_Title();
		enableActions();
		
	}
	
	protected void refreshData()
	{
		BusyIndicator.showWhile(Display.getCurrent(), new Runnable() {
			public void run() {
				AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
				
				if(model != null) {
					model.loadTemplateFile();
				}				
		
			}
		});
		
	}
	
	
	public void set_Title()
	{
		AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
		
		StringBuffer title = (new StringBuffer(LOG_TITLE)).append(" ");
		
		if(model != null)
		{
			long currentEvent = model.getCurrentCount();
			if(currentEvent >=0 ) 
			  currentEvent++;
			else 
			  currentEvent = 0;  
			
			if(model.getContextInstance() == null || model.getContextInstance().eContainer() == null)
			{  
				title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_EXTRACTOR"));
			}
			else
			{
				SensorConfigType sensor = AcadGuiModelManager.getLogForConfiguration(model);
				if(sensor != null)
				{  
					SensorType type = sensor.getType();
				
					switch(type.getValue())
					{
						case SensorType.ADAPTER_CBE_SENSOR:
						  title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
						  break;
						case SensorType.SINGLE_FILE_SENSOR:
						  SingleFileSensorType fileSensor = sensor.getSingleFileSensor();
						  if(fileSensor == null)
							title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
						  else
							title.append(fileSensor.getDirectory()).append(File.separator)
								  .append(fileSensor.getFileName());	
						  break;
					}
					String msg = AcadEditorPlugin.getPlugin().getString("STR_LOG_TITLE", new Object[] {String.valueOf(currentEvent), String.valueOf(model.getEventCount())});
					title.append(" ").append(msg);						
				}
				else
				{
					title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
				}
				
			}
				 				 			
		}
		
		super.setTitle(title.toString());
		
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)		
		   contentView.set_Title();
		ResultView resultView = ResultView.getResultView(false);
		if(resultView != null)		
		   resultView.set_Title();
		
		
	}
	
	/**
	 * 
	 *
	 */
	public void showFirstEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		  
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getFirstEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		set_Title();
		enableActions();
	}
	
	/**
	 * 
	 *
	 */
	private void showLastEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getLastEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
	
		}
		
		set_Title();
		enableActions();
	}
	
	private void createRule() 
	{
		MessageDialog.openInformation(Display.getCurrent().getActiveShell()
		   , "Generic Adaptor Message"
		   , "Not implemented");
	}
	
	
	private void guessRule() {
		MessageDialog.openInformation(Display.getCurrent().getActiveShell()
		   , "Generic Adaptor Message"
		   , "Not implemented");
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose() 
	{
		// Auto-generated method stub
		super.dispose();
	}
	
	public void setContent(String content)
	{
		if(eventsViewer != null)		
			eventsViewer.setDocument(new Document(content));
	}
	
	public String getText()
	{
		return ((StyledText)eventsViewer.getControl()).getText();
	}
	/**
	 * 
	 * @return
	 */
	public static final LogView getLogView(boolean show)
	{
		try {
			IWorkbenchPage page = AcadEditorPlugin.getDefault().getActivePage();
			if(page == null)
			  return null;
			
			IViewPart view =
				page.findView(AcadPerspective.ID_LOGVIEW);
				
			if (view != null)
			{
				if(show)
					page.showView(AcadPerspective.ID_LOGVIEW);
				return (LogView) view;	
				
			}
			else if(show)
			{
				page.showView(AcadPerspective.ID_LOGVIEW);
				return (LogView) view;					
			}
		}
		catch(Exception exc)
		{
			exc.printStackTrace();
			return null;
		}
		
		return null;
	}

    public void reset() {
    	setContent("");
    	
		ResultView resultView = ResultView.getResultView(false);
		if(resultView != null)
		  resultView.reset();
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)
		  contentView.reset();    
    	
    }
    
	/**
	 * @return
	 */
	public Action getCreateRuleAction() {
		return createRuleAction;
	}

	public void enableActions()
	{
		AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
		if(model == null || model.getContextInstance() == null)
		{
			previousEventAction.setEnabled(false);
			nextEventAction.setEnabled(false);
			firstEventAction.setEnabled(false);
			lastEventAction.setEnabled(false);
			createRuleAction.setEnabled(false);
			showLog.setEnabled(false);
			
			return;
			
		}

		firstEventAction.setEnabled(model.getEventCount() > 0);
		lastEventAction.setEnabled(model.getEventCount() > 0);
		createRuleAction.setEnabled(true);

		showLog.setEnabled(true);
		  
		previousEventAction.setEnabled(model.hasPrevious());
		nextEventAction.setEnabled(model.hasNext());
	}


	protected void enableEditActions()
	{
		ActionHandlerListener.DEFAULT.connectPart(this);
	}

	protected void showLog()
	{
		ContentView view = ContentView.getContentView(true);
		if(view != null)
			view.showLog();
	}

}
