/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;

public interface Agent {
	
	/**
	 * Add a listener that will inform you when this Agent
	 * becomes active/inactive.
	 * @see AgentListener
	 */
	void addAgentListener(AgentListener listener);
	
	/**
	 * Remove a listener previously added with addAgentListener.
	 * @see AgentListener
	 */
	void removeAgentListener(AgentListener listener);
	
	/**
	 * Get the process which this agent is a member of.
	 * @return the Process if it exists, null otherwise.
	 */
	Process getProcess();
	
	/**
	 * If autoAttach is set, this is like registering an interest
	 * in this agent.  It may not exist yet within the process,
	 * but you want to be attached to the agent as soon as it
	 * becomes active.
	 */
	void setAutoAttach(boolean auto);
	
	/**
	 * Is this agent configured for AutoAttach.
	 */
	boolean isAutoAttach();
	
	/**
	 * Attach to the agent.
	 */
	void attach() throws InactiveAgentException, InactiveProcessException;
	
	/**
	 * Detach from the agent.
	 */
	void detach() throws InactiveAgentException, InactiveProcessException;
	
	/**
	 * Retrieve the type name of this agent.
	 * @return the type of the agent if known, null otherwise.
	 */
	String getType();
	
	/**
	 * Retrieve the name of this agent.
	 * @return the name of the agent if known, null otherwise.
	 */
	String getName();
	
	/**
	 * Retrieve the UUID of this agent.
	 * @return the uuid of the agent if it is known, null otherwise.
	 */
	 String getUUID();
	
	
	/**
	 * Determine whether this agent is currently active.
	 */
	boolean isActive();
	
	/**
	 * Determine if this agent is currently being monitored.
	 */
	boolean isMonitored();
	
	/**
	 * Determine is this agent is currently attached to a client.
	 */
	boolean isAttached();
	
	/**
	 * Start monitoring the data output of this agent using the specified
	 * DataProcessor.  You must be attached to the agent before you can
	 * start monitoring it.
	 */
	void startMonitoring(DataProcessor processor) throws InactiveAgentException;
	
	/**
	 * Stop monitoring this agent.
	 */
	void stopMonitoring() throws InactiveAgentException;
	
	/**
	 * Set the configuration for the agent.
	 */
	void setConfiguration(AgentConfiguration config);
	
	/**
	 * Get the configuration object for this agent.
	 * @return the AgentConfiguration if it exists, null otherwise.
	 */
	AgentConfiguration getConfiguration();
	
	/**
	 * Publish's the AgentConfiguration to an active agent
	 */
	void publishConfiguration() throws InactiveAgentException;
	
	/**
	 * Send a custom command to the agent for processing.
	 */
	void invokeCustomCommand(CustomCommand command) throws InactiveAgentException;
	
	/**
	 * Determine if this agent is sending data to a profiling file
	 */
	public boolean isToProfileFile();
	
	/**
	 * Get the fully quality path of profiling file
	 * @return String
	 */
	public String getProfileFile();

	/**
	 * Set the fully quality path of profiling file
	 * @param _profileFile The _profileFile to set
	 */
	public void setProfileFile(String _profileFile);

}

