/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.views;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.ContextIds;
import org.eclipse.hyades.log.ui.internal.util.Splitter;
import org.eclipse.hyades.log.ui.internal.util.TerminalNode;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.IViewSelectionChangedListener;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.ViewForm;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Sash;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.help.WorkbenchHelp;

public class LogViewerUI implements SelectionListener, ControlListener,
									IViewSelectionChangedListener

{
    private TraceViewerPage fPage;
	private LogPaneTreeViewer fTreeViewer;
	private Composite            fParent;
	private ViewForm logPane;
	private ViewForm analPane;
	private LogRecordPageBook genRecPage;
	private LogAnalysisPageBook genAnalPage;
	private Splitter hSplitter;
	private ViewForm recPane;
	static final int SASH_WIDTH = 2;
	private Composite viewContainer;
	private Sash vSash;
	private String _viewerRole;
	private double viewerProp = 0.4;	

	public LogViewerUI(Composite parent, TraceViewerPage page, String role)
	{
		super();
		
		fPage = page;
		_viewerRole = role;
		fParent = parent;
	}
	public void controlMoved(ControlEvent e) {
	}
	public void controlResized(ControlEvent event) {
		resizeContent();
	}
	private Control createTreeControl(Composite parent) {
		

		fTreeViewer = new LogPaneTreeViewer(this, parent, _viewerRole);
	
		fTreeViewer.setMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager menu) {
				fillTreeContextMenu(menu);
			}
		});
		
		fTreeViewer.setContentProvider(new LogContentProvider(logPane,(LogViewer)fPage.getTraceViewer()));
		LogLabelProvider labelProvider = new LogLabelProvider();
		DecoratingLabelProvider decLabel =
			new DecoratingLabelProvider(labelProvider, labelProvider);
		fTreeViewer.setLabelProvider(decLabel);
	
		//popup menu contribution :begin
		MenuManager menuMgr= new MenuManager("TreeViewContextMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
		public void menuAboutToShow(IMenuManager menu) {
			fillTreeContextMenu(menu);
		}
		});
		Menu fContextMenu= menuMgr.createContextMenu(fTreeViewer.getTree());
		fTreeViewer.getTree().setMenu(fContextMenu);
		
		// Register viewer with site. This must be done before making the actions.
		IWorkbenchPartSite site= fPage.getTraceViewer().getSite();
		site.registerContextMenu(menuMgr, fTreeViewer);
		site.setSelectionProvider(fTreeViewer);
		
		UIPlugin.getDefault().addViewSelectionChangedListener(this);
		
		//popup menu contribution :end		
		// Create the viewers
		return fTreeViewer.getControl();
	}
	/**
	 * Creates the context menu for the method viewer
	 */
	private void fillTreeContextMenu(IMenuManager menu) {
		// viewer entries
		fTreeViewer.fillContextMenu(menu);
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (2/14/2001 2:07:21 PM)
	 * @return com.ibm.swt.widgets.Control
	 */
	public Control getControl() {
		return viewContainer;
		//return fTreeViewer.getControl();
	}

	/**
	* Lays out the list and text widgets according to the new
	* positions of the sashes.
	*/
	void layout() {
		Rectangle viewBounds = viewContainer.getClientArea();
		Rectangle vSashBounds = vSash.getBounds();

		logPane.setBounds(0, 0, vSashBounds.x, viewBounds.height);
		hSplitter.setBounds(
			vSashBounds.x + vSashBounds.width,
			0,
			viewBounds.width - (vSashBounds.x + vSashBounds.width),
			viewBounds.height);
	}
	public void initialize()
	{
		
		viewContainer = new Composite(fParent,SWT.NULL);
		viewContainer.setLayoutData(GridUtil.createFill());
		
		logPane = new ViewForm(viewContainer, SWT.NULL);
		Control viewerControl = createTreeControl(logPane);
		logPane.setContent(viewerControl);

		hSplitter = new Splitter(viewContainer, SWT.VERTICAL, true);
		vSash = new Sash(viewContainer, SWT.VERTICAL);
		vSash.addSelectionListener(this);
		viewContainer.addControlListener(this);

		ViewForm recPane = new ViewForm(hSplitter, SWT.NONE);
		recPane.setLayoutData(new Splitter.Weight(500));
		// create the Record PageBook
		genRecPage = new LogRecordPageBook(recPane, SWT.NONE);
		recPane.setContent(genRecPage);

		hSplitter.addSash();

		ViewForm analPane = new ViewForm(hSplitter, SWT.NONE);
		analPane.setLayoutData(new Splitter.Weight(500));
		// create the Analysis PageBook
		genAnalPage = new LogAnalysisPageBook(analPane, SWT.NULL);
		analPane.setContent(genAnalPage);
		
		genRecPage.setAnalysisPageBook(genAnalPage);
		
		genRecPage.addSelectionChangedListener(genAnalPage);
		fTreeViewer.addSelectionChangedListener(genRecPage);
		fTreeViewer.addSelectionChangedListener(genAnalPage);
		fTreeViewer.addSelectionListener(this);

		WorkbenchHelp.setHelp(logPane, ContextIds.ACTLOG_VIEW_PANE_LOG);
		WorkbenchHelp.setHelp(recPane, ContextIds.ACTLOG_VIEW_PANE_RECORD);
		WorkbenchHelp.setHelp(analPane, ContextIds.ACTLOG_VIEW_PANE_ANALYSIS);

		
		CLabel label = new CLabel(logPane, SWT.NONE);
		logPane.setTopLeft(label);
		label.setText(LogUIPlugin.getResourceString("STR_LOG_PANE_TITLE"));
		
		fTreeViewer.setInput(fPage.getMOFObject());
		Object sel = UIPlugin.getDefault().getSelectionModel(HyadesUtil.getMofObject()).getFirstElement();
		if(sel!=null){	
			fTreeViewer.setSelection(new StructuredSelection(sel));
		}
		
	}
	/**
	 * Resizes the content of the viewer  
	 * Preserves log pane proportion to the default value set in LogViewerUI#viewerProp 
	 * Creation date: (10/17/2000 2:08:36 PM)
	 */
	public void resizeContent() {
		/* Get the client area for the shell */
		Rectangle viewBounds = viewContainer.getClientArea();

		if (viewBounds.height == 0)
			return;

		/*
		* Make list 1 half the width and half the height of the tab leaving room for the sash.
		* Place list 1 in the top left quadrant of the tab.
		*/
		Rectangle treeBounds =
			new Rectangle(0, 0, (int) Math.round((viewBounds.width - SASH_WIDTH) * viewerProp), viewBounds.height);
		logPane.setBounds(
			treeBounds.x,
			treeBounds.y,
			treeBounds.width,
			treeBounds.height);

		/*
		* Make list 2 half the width and half the height of the tab leaving room for the sash.
		* Place list 2 in the top right quadrant of the tab.
		*/
		Rectangle analBounds =
			new Rectangle(
				treeBounds.width + SASH_WIDTH,
				0,
				viewBounds.width - (treeBounds.width + SASH_WIDTH),
				treeBounds.height);
		hSplitter.setBounds(analBounds);

		// Position the sash 
		vSash.setBounds(treeBounds.width, 0, SASH_WIDTH, treeBounds.height);
		
		genRecPage.getTableControl().getControl().redraw();
		fTreeViewer.getControl().redraw();					
	}
	/**
	 * Refreshes the tree viewer and resets the input of the log record properties
	 * table viewer
	 * Creation date: (3/21/2001 4:02:10 PM)
	 */
	public void update() {
		fTreeViewer.getControl().setRedraw(false);		
		fTreeViewer.refresh();		
//		refreshSelection();		
		fTreeViewer.getControl().setRedraw(true);
		
	}
	public void widgetDefaultSelected(SelectionEvent e) {
		
		Object data = e.item.getData();
		if(e.item instanceof TreeItem && data instanceof TerminalNode){
			EObject sel = (EObject)((TerminalNode)data).getParent();
			if(sel!=null){
				fTreeViewer.setSelection(new StructuredSelection(sel));
			}
		}

	}
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == vSash && e.detail != SWT.DRAG) {
			vSash.setBounds(e.x, e.y, e.width, e.height);
			layout();
			Rectangle viewBounds = viewContainer.getClientArea();
			Rectangle vSashBounds = vSash.getBounds();
			viewerProp = 1-(double)(viewBounds.width - vSashBounds.x)/viewBounds.width;
		}
		if(e.item instanceof TreeItem && e.item.getData()!=null){		
			UIPlugin.getDefault().getSelectionModel(HyadesUtil.getMofObject()).add(e.item.getData());
			
			ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
			event.setSource(fPage.getMOFObject());
			UIPlugin.getDefault().notifyViewSelectionChangedListener(event);			
		}
		
	}
	private void refreshSelection() {
				
		if (genRecPage != null && genRecPage.getTableControl() != null)
		{
			Object sel = null;
			ISelection selection = fTreeViewer.getSelection();
			if(selection != null && !selection.isEmpty()
			   && selection instanceof IStructuredSelection)
			   sel = ((IStructuredSelection)selection).getFirstElement();
			   
			genRecPage.getTableControl().getControl().setRedraw(false);
			genRecPage.getTableControl().setInput(sel);
			genRecPage.getTableControl().getControl().setRedraw(true);			
			
			if (genAnalPage != null && sel == null) {
				genAnalPage.setRedraw(false);
				genAnalPage.setInputNull();
				genAnalPage.setRedraw(true);
			}
		}
	}
	
	public void dispose()
	{
		UIPlugin.getDefault().removeViewSelectionChangedListener(this);

	   genRecPage.removeSelectionChangedListener(genAnalPage);
	   fTreeViewer.removeSelectionChangedListener(genAnalPage);
	   fTreeViewer.removeSelectionChangedListener(genRecPage);
	   fTreeViewer.dispose();	

	   if(genRecPage != null)
		  genRecPage.dispose();
	   genRecPage = null;      

	   if(genAnalPage != null)
		  genAnalPage.dispose();
	   genAnalPage = null;      

	   if(logPane != null)
		   logPane.dispose();
	   logPane = null;	   
		   
	   if(recPane != null)
	     recPane.dispose();	   
	   recPane = null;
	   
	   if(analPane != null)
	      analPane.dispose();
	   analPane = null;
	   
	   if(hSplitter != null)
	      hSplitter.dispose();
	   hSplitter = null;
	   
	   if(viewContainer != null && !viewContainer.isDisposed())
	      viewContainer.dispose();
	   viewContainer = null;      

	   if(vSash != null)
	      vSash.dispose();
	   vSash = null;      
	   
	   fTreeViewer = null;
	   fPage = null;
	   	   
	}
	
	public LogPaneTreeViewer getViewer(){
		return fTreeViewer;
	}
	
	/**
	 * Refreshes the tree viewer and resets the input for the record properties table viewer
	 */
	public void refresh() {
		if (fTreeViewer.getControl() != null && !fTreeViewer.getControl().isDisposed() && fTreeViewer.getControl().isVisible()) {
			fTreeViewer.refresh();
			refreshSelection();
		}
	}	
	
	/*
	 * This method handles a selection event
	 */
	public void selectionChanged() {
/*		
		if (fTreeViewer.getControl() != null && !fTreeViewer.getControl().isDisposed() && fTreeViewer.getControl().isVisible() && !fTreeViewer.getControl().isFocusControl()) {
			refreshSelection();
		}
*/		
	}	
	/**
	 * Returns the fPage.
	 * @return TraceViewerPage
	 */
	public TraceViewerPage getFPage() {
		return fPage;
	}
	
	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event)
	{
		if (fTreeViewer.getControl() != null && !fTreeViewer.getControl().isDisposed() 
		    && !fTreeViewer.getControl().isFocusControl())
		{
			Object selection = UIPlugin.getDefault().getSelectionModel(HyadesUtil.getMofObject()).getFirstElement();
			if(selection!=null){
				fTreeViewer.setSelection(new StructuredSelection(selection));
			}
		}
	}
	

}