/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.console;

import java.util.*;

import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.jface.text.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;

/**
 * The class manages trace console  
 */
public class ConsoleManager
							implements ISelectionListener, IDocumentListener

{

	/**
	 * Colors to be used in the console ui
	 */
	protected ColorManager fColorManager= new ColorManager();

	/**
	 * The mappings of processes to their console documents.
	 */
	protected Map fConsoleDocuments= new HashMap(3);

	/**
	 * The mappings of processes to their console data.
	 */
	protected Map fConsoleData= new HashMap(3);

	/**
	 * The process that is/can provide output to the console
	 * view.
	 */
	protected TRCProcessProxy fCurrentProcess= null;
	protected TRCProcessProxy fProcessTemp;


    /**
     * 
     * @return ColorManager
     */
	public ColorManager getColorManager() {
		return fColorManager;
	}

	/**
	 * Remove the console for this process
	 * @param process
	 */
	public void deregisterLaunchProcess(TRCProcessProxy process)
	{
		fProcessTemp = process;
		
		Display display= getDisplay();
		if (display != null) {
			display.syncExec(new Runnable () {
				public void run() {
					if (fProcessTemp != null)
					{
						TraceConsoleDocument doc= (TraceConsoleDocument)getConsoleDocument(fProcessTemp);

						if(doc != null)
						{
							doc.removeDocumentListener(ConsoleManager.this);
							doc.close();
						}
						setConsoleDocument(fProcessTemp, null);
						setConsoleData(fProcessTemp, null);
						fCurrentProcess= null;
						fProcessTemp = null;
					}
				}
			});
		}
	}
	
	/**
	 * Returns the current process to use as input for the console view.
	 */
	public TRCProcessProxy determineCurrentProcess()
	{
		PDProjectExplorer viewer = UIPlugin.getDefault().getViewer();

		if(viewer != null)
		{
			TreeItem item = ((PDProjectViewer)viewer.getViewer()).getTreeSelection();
			if(item == null || item.getData() == null)
			  return null;

			Object data = item.getData();
			if(data instanceof TRCProcessProxy)
			  return ((TRCProcessProxy)data);

			if(data instanceof TRCAgentProxy)
			  return ((TRCAgentProxy)data).getProcessProxy();
		}

		return null;
	}
	
	/**
	 * Returns the correct document for the process, determining the current
	 * process if specified.
	 */
	public ConsoleDataProcessor getConsoleData(TRCProcessProxy process, boolean determineCurrentProcess)
	{
		if (process != null)
		{
			ConsoleDataProcessor document= (ConsoleDataProcessor) fConsoleData.get(process);
			if (document != null)
				return document;
			
			document= new ConsoleDataProcessor(null);
			fConsoleData.put(process, document);
			return document;
		}
		
		if (determineCurrentProcess)
		{
			if (getCurrentProcess() == null)
				setCurrentProcess(determineCurrentProcess());

			TRCProcessProxy currentProcess= getCurrentProcess();
			if (currentProcess != null)
			{
				ConsoleDataProcessor document= (ConsoleDataProcessor) fConsoleData.get(currentProcess);
				if (document != null)
					return document;
				
				document= new ConsoleDataProcessor(null);
				fConsoleData.put(currentProcess, document);
				return document;
			}
		}

		return new ConsoleDataProcessor(null);
	}

	/**
	 * Returns the correct document for the process, determining the current
	 * process if required (process argument is null).
	 */
	public IDocument getConsoleDocument(TRCProcessProxy process)
	{
		return getConsoleDocument(process, true);
	}
	/**
	 * Returns the correct document for the process, determining the current
	 * process if specified.
	 */
	public IDocument getConsoleDocument(TRCProcessProxy process, boolean determineCurrentProcess)
	{
		if (process != null)
		{
			IDocument document= (IDocument) fConsoleDocuments.get(process);
			if (document != null)
				return document;

			ConsoleDataProcessor console = (ConsoleDataProcessor) fConsoleData.get(process);
			document= new TraceConsoleDocument(console);
			fConsoleDocuments.put(process, document);
			return document;
		}
		
		if (determineCurrentProcess)
		{
			if (getCurrentProcess() == null)
				setCurrentProcess(determineCurrentProcess());

			TRCProcessProxy currentProcess= getCurrentProcess();
			if (currentProcess != null)
			{
				IDocument document= (IDocument) fConsoleDocuments.get(currentProcess);
				if (document != null)
					return document;

				ConsoleDataProcessor console = (ConsoleDataProcessor) fConsoleData.get(currentProcess);
				document= new TraceConsoleDocument(console);
				fConsoleDocuments.put(currentProcess, document);
				return document;
			}
		}

		return new TraceConsoleDocument(null);
	}
	
	/**
	 * 
	 * @return TRCProcessProxy
	 */
	public TRCProcessProxy getCurrentProcess() {
		return fCurrentProcess;
	}

    /**
     * 
     * @param process
     */
	public void setCurrentProcess(TRCProcessProxy process)
	{
		if (fCurrentProcess != null) {
			getConsoleDocument(fCurrentProcess).removeDocumentListener(this);
		}
		fCurrentProcess= process;
		if (fCurrentProcess != null) {
			getConsoleDocument(fCurrentProcess).addDocumentListener(this);
		}
	}
	
	/**
	 * Sets the console document for the specified process.
	 * If the document is <code>null</code> the mapping for the
	 * process is removed.
	 */
	private void setConsoleData(TRCProcessProxy process, ConsoleDataProcessor doc)
	{
		if (doc == null)
		{
			Object dataProcessor = fConsoleData.get(process);
			if(dataProcessor != null && dataProcessor instanceof ConsoleDataProcessor)
			{
				((ConsoleDataProcessor)dataProcessor).dispose();
			}
			fConsoleData.remove(process);			
		}
		else
			fConsoleData.put(process, doc);
	}
	

	/**
	 * Sets the console document for the specified process.
	 * If the document is <code>null</code> the mapping for the
	 * process is removed.
	 */
	private void setConsoleDocument(TRCProcessProxy process, IDocument doc)
	{
		if (doc == null)
		{
			fConsoleDocuments.remove(process);
		}
		else
			fConsoleDocuments.put(process, doc);
	}

	/**
	 * Sets the input console view viewer input for all
	 * consoles that exist in a thread safe manner.
	 */
	protected void setConsoleInput(final TRCProcessProxy process)
	{
		Display display= getDisplay();
		if (display != null) {
			display.asyncExec(new Runnable() {
				public void run() {
					IWorkbenchWindow[] windows= UIPlugin.getDefault().getWorkbench().getWorkbenchWindows();
					for (int j= 0; j < windows.length; j++) {
						IWorkbenchWindow window= windows[j];
						IWorkbenchPage[] pages= window.getPages();
						if (pages != null) {
							for (int i= 0; i < pages.length; i++) {
								IWorkbenchPage page= pages[i];
								TraceConsoleView consoleView= (TraceConsoleView)page.findView(PDPerspective.ID_CONSOLE_VIEW);
								if (consoleView != null) {
									consoleView.setViewerInput(process);
								}
							}
						}
					}
				}
			});
		}
		
	}

    /**
     * 
     * @param console
     */
	public void registerLaunchProcess(ConsoleDataProcessor console)
	{
		TRCProcessProxy newProcess= console.getProcess();
		fConsoleData.put(newProcess, console);

		setCurrentProcess(newProcess);
		setConsoleInput(newProcess);

		Display display= getDisplay();
		if (display != null) {
			display.syncExec(new Runnable () {
				public void run() {
					TraceConsoleDocument doc= new TraceConsoleDocument((ConsoleDataProcessor)fConsoleData.get(getCurrentProcess()));
					doc.startReading();
					setConsoleDocument(getCurrentProcess(), doc);
				}
			});
		}
	}
	
	/**
	 * 
	 * @param part
	 * @param sel
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection sel)
	{
		if (!(part instanceof PDProjectExplorer)) {
			return;
		}
		
		IWorkbenchWindow window= UIPlugin.getActiveWorkbenchWindow();
		IWorkbenchPage page= window.getActivePage();
		if (page == null) {
			return;
		}
		Object input= null;
		if (sel instanceof IStructuredSelection) {
			input= ((IStructuredSelection) sel).getFirstElement();
		}
		TraceConsoleView consoleView= (TraceConsoleView)page.findView(PDPerspective.ID_CONSOLE_VIEW);
		if (input == null) {
			if (getCurrentProcess() != null)
			{
				if(consoleView != null)
					consoleView.setViewerInput(getCurrentProcess());
			} else {
				TRCProcessProxy currentProcess= determineCurrentProcess();
				if (currentProcess == null) {
					setCurrentProcess(currentProcess);
					if (consoleView != null) {
						consoleView.setViewerInput(currentProcess);
					}
				}
			}
		} else {
			TRCProcessProxy processFromInput= getProcessFromInput(input);
			if (processFromInput == null)
			{
				if(consoleView != null)
					consoleView.setViewerInput(null, false);
				setCurrentProcess(null);
				return;
			}
			if (!processFromInput.equals(getCurrentProcess())) {
				setCurrentProcess(processFromInput);
				if (consoleView != null) {
					consoleView.setViewerInput(processFromInput);
				} else {
					IDocument doc= getConsoleDocument(processFromInput);
					if (doc != null) {
						doc.addDocumentListener(this);
					}
				}
			}
		}
	}
	
	/**
	 * @see IDocumentListener
	 */
	public void documentAboutToBeChanged(final DocumentEvent e)
	{
		/*
		// if the prefence is set, show the conosle
		if (!getPreferenceStore().getBoolean(PDCoreConstantsConstants.CONSOLE_OPEN)) {
			return;
		}
		*/
		Display display= getDisplay();
		if (display != null) {
			display.asyncExec(new Runnable() {
				public void run() {
					IWorkbenchWindow window = UIPlugin.getActiveWorkbenchWindow();
					if (window != null) {
						IWorkbenchPage page= window.getActivePage();
						if (page != null) {
							try { // show the console
								TraceConsoleView consoleView= (TraceConsoleView)page.findView(PDPerspective.ID_CONSOLE_VIEW);
								if(consoleView == null) {
									consoleView= (TraceConsoleView)page.showView(PDPerspective.ID_CONSOLE_VIEW);
									consoleView.setViewerInput(getCurrentProcess());
								}
								page.bringToTop(consoleView);
							} catch (PartInitException pie) {
							}
						}
					}
				}
			});
		}
	}
	/**
	 * @see IDocumentListener
	 */
	public void documentChanged(DocumentEvent e) {
	}
	
	
	/**
	 * UI thread safe access to a display
	 */
	public Display getDisplay()
	{
		IWorkbench workbench= UIPlugin.getDefault().getWorkbench();
		if (workbench != null) {
			IWorkbenchWindow[] windows= workbench.getWorkbenchWindows();
			Display display= null;
			if (windows != null && windows.length > 0) {
				return windows[0].getShell().getDisplay();
			}
		}
		return null;
	}

    /**
     * 
     * @param input
     * @return TRCProcessProxy
     */
	protected TRCProcessProxy getProcessFromInput(Object input)
	{
		TRCProcessProxy processInput= null;
		if (input instanceof TRCProcessProxy)
			processInput= (TRCProcessProxy) input;
		else if(input instanceof TRCAgentProxy)
		   processInput = ((TRCAgentProxy)input).getProcessProxy();

		return processInput;
	}
	
}
