/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.piclient;

import java.util.ArrayList;
import java.util.Enumeration;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.AgentPeerListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.local.control.Variable;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCEnvironmentVariable;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.console.ConsoleDataProcessor;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;


public class PIProcessListener implements ProcessListener, AgentPeerListener {
    //~ Instance fields ----------------------------------------------------------------------------

    protected ArrayList _monitoredAgents = new ArrayList();
    protected ArrayList _processVariableList = new ArrayList();
    protected TRCNode fNode;
    protected TRCProcessProxy fProcess;
    protected boolean _autoMonitoring = true;
    protected boolean _monitor = false;
    protected int _launchMode = 0;

    //~ Constructors -------------------------------------------------------------------------------

    /**
     * PIProcessListener constructor comment.
     */
    public PIProcessListener(TRCNode node) {
        super();
        fNode = node;
    }

    public PIProcessListener(TRCNode node, TRCProcessProxy process) {
        this(node);

        fProcess = process;
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * @param b
     */
    public void setAutoMonitoring(boolean b) {
        _autoMonitoring = b;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 5:37:22 PM)
     * @param mode int
     */
    public void setLaunchMode(int mode) {
        _launchMode = mode;
    }

    /**
     *
     * @return HashMap
     */
    public ArrayList getProcessVariableList() {
        return _processVariableList;
    }

    public synchronized void agentActive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        if (_monitoredAgents.contains(ag.getName())) {
            _monitor = false;
        } else {
            _monitoredAgents.add(ag.getName());
        }

        if (!_monitor) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,false);
		
        if (agentProxy == null) {
            return;
        }

        agentProxy.setRuntimeId(ag.getUUID());
        LoadersUtils.registerAgentInstance(agentProxy, ag);

        agentProxy.setAttached(true);
        agentProxy.setProfileFile(ag.getProfileFile());

        if (agentProxy.isActive()) {
            final TRCAgentProxy agentTemp = agentProxy;

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        //update ui
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(agentTemp);
                        event.setType(ProfileEvent.START_MONITOR);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });

            return; //this is an attach scenario
        }

        agentProxy.setActive(true);

        try {
            ag.publishConfiguration();

            CustomCommand command = new CustomCommand();

            command.setData("APPLYFILTERS");
            ag.invokeCustomCommand(command);

            //			XMLTraceDataProcessor processor = (XMLTraceDataProcessor)agent.getDataProcessor();
            XMLTraceDataProcessor processor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(agentProxy);

            if (processor == null) {
                processor = new XMLTraceDataProcessor(agentProxy);

                //				agent.setDataProcessor(processor);
                //				LoadersUtils.loadRootEvent(processor);
                LoadersUtils.registerDataProcessor(agentProxy, processor);
            }

            if (agentProxy.isToProfileFile()) {
                processor.setProfileFileName(agentProxy.getProfileFile());
                processor.createWriter();

                if (processor.isNewProfileFile()) {
                    processor.writeXMLVersion(processor.getWriter());
                    processor.startTrace(processor.getWriter());
                }
            }

            agentProxy.setAttached(true);

            if (_autoMonitoring) {
                ag.startMonitoring(processor);
                agentProxy.setMonitored(true);
            }

            agentProxy.setCollectionData(true);

            command.setData("RESUME");
            ag.invokeCustomCommand(command);
        } catch (InactiveAgentException exc) {
            exc.printStackTrace();
        }

        final TRCAgentProxy agentTemp = agentProxy;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.START_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 10:36:27 AM)
     * @param command com.ibm.etools.logging.tracing.control.Agent
     */
    public synchronized void agentInactive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,true);
        if (agentProxy == null) {
            return;
        }

        agentProxy.setActive(false);
        agentProxy.setAttached(false);
        agentProxy.setMonitored(false);
        fProcess.setActive(false);

        final TRCAgentProxy agentTemp = agentProxy;
        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.STOP_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }

    public void dispose() {
        _monitoredAgents.clear();
        fNode = null;
        fProcess = null;
        _processVariableList.clear();
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 11:03:54 AM)
     * @return com.ibm.etools.perftrace.TRCAgent
     */
    /**
     * Invoked when an error is recieved from the agent.
     */
    public void error(Agent agent, String errorId, String errStr) {
        final String errorMsg = errStr;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, errorMsg, null);

                    ErrorDialog.openError(new Shell(), UIPlugin.getResourceString("LAUNCH_JAVA_PROBLEM_ERROR_"), UIPlugin.getResourceString("PROBLEM_WHILE_PROCESS_WARN_"), // no special message
                                          err);
                }
            });
    }

    public void handleCommand(Agent agent, CommandElement command) {
    }

    /*
     * return false if the agent should ignore the agentActive event
     */
    public void monitor(boolean monitor) {
        _monitor = monitor;
        _monitoredAgents.clear();
    }

    /**
     * Invoked when an agent requests to be monitored because this client
     * is currently monitoring another agent.
     */
    public void peerWaiting(Agent agent, Agent peer) {
        final TRCMonitor monitor = fNode.getMonitor();

        //get the right node for the peer
        final Process peerProcess = peer.getProcess();
        final Agent _peer = peer;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    TRCNode node = null;

                    try {
                        Node peerNode = peerProcess.getNode();

                        node = PDCoreUtil.createNode(monitor, peerNode.getName(), String.valueOf(peerNode.getConnection().getPort()));
                    } catch (InactiveProcessException exc) {
                        exc.printStackTrace();

                        return;
                    }

                    // Get/Create the proper process and agent for the peer 
                    TRCProcessProxy proxy = PDCoreUtil.createProcess(node, peerProcess);

                    proxy.setActive(true);

                    final TRCAgentProxy ag = PDCoreUtil.createAgent(proxy, _peer);

                    //		PIProcessListener pl = (PIProcessListener)ag.getAgentListener();
                    PIProcessListener pl = (PIProcessListener) LoadersUtils.locateAgentListener(ag);

                    if (pl == null) {
                        pl = new PIProcessListener(node, proxy);

                        //		   ag.setAgentListener(pl);
                        LoadersUtils.registerAgentListener(ag, pl);
                        _peer.addAgentListener(pl);
                    } else {
                        _peer.addAgentListener(pl);
                    }

                    pl.monitor(true);

                    // Set the reference to the agent
                    //		ag.setAgentInstance(_peer);
                    LoadersUtils.registerAgentInstance(ag, _peer);

                    //       XMLTraceDataProcessor aprocessor = (XMLTraceDataProcessor)ag.getDataProcessor();
                    XMLTraceDataProcessor aprocessor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(ag);

                    if (aprocessor == null) {
                        aprocessor = new XMLTraceDataProcessor(ag);

                        LoadersUtils.registerDataProcessor(ag, aprocessor);

                        //					   LoadersUtils.loadRootEvent(aprocessor);
                    }

                    if (_peer.isAttached()) {
                        try {
                            _peer.startMonitoring(aprocessor);
                        } catch (Exception exc) {
                        }

                        ag.setCollectionData(true);
                        ag.setActive(true);
                        ag.setAttached(true);
                        ag.setMonitored(true);

                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(ag);
                        event.setType(ProfileEvent.START_MONITOR);
                        UIPlugin.getDefault().notifyProfileEventListener(event);

                        return;
                    }

                    // Attach to the peer and start monitoring 		
                    final XMLTraceDataProcessor processor = aprocessor;

                    _peer.addAgentListener(new AgentListener() {
                            public void agentActive(Agent agent) {
                                try {
                                    if (!agent.isMonitored()) {
                                        ag.setCollectionData(true);
                                        agent.startMonitoring(processor);
                                    }
                                } catch (Exception e) {
                                    Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, e.toString(), null);

                                    ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("LAUNCH_JAVA_PROBLEM_ERROR_"), UIPlugin.getResourceString("PROBLEM_WHILE_PROCESS_WARN_"), // no special message
                                                          err);

                                    e.printStackTrace();
                                }
                            }

                            public void agentInactive(Agent agent) {
                            }

                            public void error(Agent agent, String errorId, String message) {
                            }

                            public void handleCommand(Agent agent, CommandElement command) {
                            }
                        });

                    try {
                        _peer.attach();
                    } catch (InactiveAgentException exc) {
                        exc.printStackTrace();
                    } catch (InactiveProcessException exc) {
                        exc.printStackTrace();
                    }

                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(ag);
                    event.setType(ProfileEvent.UNSPECIFIED);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 9:07:12 AM)
     * @param result com.ibm.etools.logging.tracing.control.Process
     */
    public synchronized void processExited(Process result) {
        if (fProcess != null) {
            fProcess.setActive(false);
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 9:07:12 AM)
     * @param result com.ibm.etools.logging.tracing.control.Process
     */
    public synchronized void processLaunched(Process result) {
        try {
            if ((result.getProcessId() == null) || result.getProcessId().equals("-1")) {
                return;
            }

            fProcess = createProcess(result);

            //register the trace
            UIPlugin.getDefault().getTraceManager().registerTrace(fProcess, TraceConstants.LAUNCH_MODE);

            if (fProcess == null) {
                return;
            }

            ConsoleDataProcessor consoleData = new ConsoleDataProcessor(fProcess);

            consoleData.setConsole(result.getConsole());

            /*  RKD:  We also need the console to know who it's data processor is so
               that the data gets forwarded.
             */
            result.getConsole().setDataProcessor(consoleData);
            UIPlugin.getDefault().registerLaunchProcess(consoleData);

            //create agents
            Enumeration agents = result.listAgents();

            while (agents.hasMoreElements()) {
                createAgent((Agent) agents.nextElement(), fProcess);
            }
        } catch (InactiveProcessException ex) {
            String msg = UIPlugin.getResourceString("LAUNCH_ERROR_");

            msg = TString.change(msg, "%1", result.getName());

            String text = UIPlugin.getResourceString("RAC_CONFIG_ERROR_");

            Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

            ErrorDialog.openError(UIPlugin.getDefault().getViewer().getShell(), UIPlugin.getResourceString("TRACE_MSG"), msg, // no special message
                                  err);

            return;
        } catch (Exception exc) {
            exc.printStackTrace();
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 12:50:32 PM)
     * @param agent com.ibm.etools.perftrace.TRCAgent
     */
    public Agent sendConfigurationToAgent(TRCAgentProxy agent, String host, String processId) {
        try {
            Node node = PDCoreUtil.profileConnect(host, String.valueOf(agent.getProcessProxy().getNode().getPort()));

            if (node == null) {
                return null;
            }

            Process p = node.getProcess(processId);

            if (p != null) {
                Agent a = p.getAgent(agent.getName());

                PDCoreUtil.setAgentConfiguration(agent, a);

                if ((a != null) && a.isActive()) {
                    a.publishConfiguration();

                    //			  a.invokeCustomCommand(new ApplyFiltersCommand());
                    CustomCommand command = new CustomCommand();

                    command.setData("APPLYFILTERS");
                    a.invokeCustomCommand(command);
                }

                return a;
            }
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }



    private void addEnvironment(TRCProcessProxy process) {
        for (int idx = 0; idx < _processVariableList.size(); idx++) {
            Variable var = (Variable) _processVariableList.get(idx);

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

            if (!var.getName().equals("UICLASSPATH")) {
                TRCEnvironmentVariable env = factory.createTRCEnvironmentVariable();

                env.setName(var.getName());
                env.setValue(var.getValue());
                process.getEnvironmentVariables().add(env);
            } else {
                process.setClasspath(var.getValue());
            }
        }
    }

    private TRCAgentProxy createAgent(Agent a, TRCProcessProxy process) {
        if (process == null) {
            return null;
        }

        TRCAgentProxy agent = PDCoreUtil.createAgent(process, a);

        //    agent.setAgentListener(this);
        LoadersUtils.registerAgentListener(agent, this);

        return agent;
    }

    private TRCProcessProxy createProcess(Process p, TRCNode pNode) {
        try {
            final TRCNode node = pNode;
            TRCMonitor monitor = node.getMonitor();

            String processId = p.getProcessId();

            if (processId == null) {
                processId = "-1";
            }

            String pName = p.getParameters();

            if (pName == null) {
                pName = "unknown";
            }

            String params = "";
            String vmparam = "";

            if (p.getExecutable().startsWith("java")) {
                String exeName = p.getParameters();

                if (exeName == null) {
                    exeName = "unknown";
                }

                if (!exeName.startsWith("-X")) { //log agents
                    exeName = "-X " + exeName;
                }

                int idx = exeName.indexOf(" ");

                if ((idx != -1) && (idx < (exeName.length() + 1))) {
                    exeName = exeName.substring(idx + 1);
                }

                while (exeName.startsWith("-")) {
                    idx = findVMendIdx(exeName);

                    if (idx != -1) {
                        vmparam += (" " + exeName.substring(0, idx));
                        exeName = exeName.substring(idx + 1);
                    } else {
                        vmparam = exeName;
                        exeName = "";
                    }
                }

                idx = exeName.indexOf(" ");

                if (idx != -1) {
                    params = exeName.substring(idx);
                    exeName = exeName.substring(0, idx);
                }

                pName = exeName;
            }

            String rId = ((ProcessImpl) p).getUUID();

            if (rId == null) {
                rId = "";
            }

            String fileName = new StringBuffer(monitor.getName()).append("_").append(node.getName()).append("_").append(TString.change(pName, " ", "")).append(processId).append("_").append(rId).append(".").append(TraceConstants.PROCESS_EXT).toString();

            String folderPath = node.eResource().getURI().toString();
            IPath path = new Path(folderPath);

            if (path.segmentCount() > 1) {
                folderPath = path.removeLastSegments(1).toString();
            }

            IPath filePath = new Path(folderPath).append(fileName);

            URI uri = URI.createURI(filePath.toString());

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();
            Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
            pDoc.setModified(true);
            EList pExt = pDoc.getContents();

            UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

//            //*** adding support for multiple files
//            SaveUtil.addDocument(pDoc);

            final TRCProcessProxy process = factory.createTRCProcessProxy();

            process.setPid(Integer.parseInt(processId));
            process.setRuntimeId(rId);
            process.setName(pName);
            process.setLocation(((ProcessImpl) p).getlocation());
            process.setNode(node);

            addEnvironment(process);

            process.setParameters(params.trim());
            process.setVmArguments(vmparam.trim());
            process.setLaunchMode(_launchMode);
            process.setActive(true);
            pExt.add(process);

            node.getProcessProxies().add(process);

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(null);
                        event.setType(ProfileEvent.UNSPECIFIED);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });

            _processVariableList.clear();

            return process;
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 11:03:54 AM)
     * @return com.ibm.etools.perftrace.TRCAgent
     */
    private TRCProcessProxy createProcess(Process p) {
        return createProcess(p, fNode);
    }

    private int findVMendIdx(String vmargs) {
        String space = " ";
        String quote = "\"";
        int startIdx = 0;
        int endIdx;
        int qIdx;

        endIdx = vmargs.indexOf(space);

        if (endIdx != -1) {
            qIdx = vmargs.substring(startIdx, endIdx).indexOf(quote);

            if (qIdx != -1) {
                startIdx = endIdx + vmargs.substring(endIdx + 1).indexOf(quote) + 1;
                endIdx = startIdx + findVMendIdx(vmargs.substring(startIdx + 1)) + 1;
            }
        }

        return endIdx;
    }
}
