/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.net.UnknownHostException;
import java.util.List;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.CorrelationContainer;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.EMFUtil;
import org.eclipse.hyades.models.hierarchy.util.HierarchyXMIResourceImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.ui.IDeleteListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.piclient.PIProcessListener;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;

/**
 */
public class DeleteUtil {
	/**
	 * Insert the method's description here. Creation date: (10/02/2000 11:51:54 AM)
	 * 
	 * @param agent
	 *            com.ibm.etools.perftrace.TRCAgent
	 */
	public static void deleteAgent(TRCAgentProxy agent, boolean refresh, boolean deleteContents) {
		if (agent == null)
			return;

		TRCProcessProxy process = agent.getProcessProxy();

		if (process == null)
			return;

		//remove documents
		removeDoc(agent, deleteContents);
		agent.setProcessProxy(null);

		if (process.getAgentProxies().size() == 0) {
			deleteProcess(process, refresh, deleteContents);
		}

		if (refresh) {
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);

		}
	}

	/**
	 * Insert the method's description here. Creation date: (10/02/2000 11:51:17 AM)
	 * 
	 * @param monitor
	 *            com.ibm.etools.perftrace.TRCMonitor
	 */
	public static void deleteMonitor(TRCMonitor monitor, boolean refresh, boolean deleteContents) {
		if (monitor == null)
			return;

		IPath path = new Path(TString.resourcePath(monitor.eResource().getURI()));
		if (path.segmentCount() > 1)
			path = path.removeLastSegments(1);

		PDContentProvider.removeMonitor(path.toOSString(), monitor);

		//remove documents
		removeDoc(monitor, deleteContents);

		if (refresh) {

			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);

		}

	}

	/**
	 * Insert the method's description here. Creation date: (10/02/2000 11:51:37 AM)
	 * 
	 * @param node
	 *            com.ibm.etools.perftrace.TRCNode
	 */
	public static void deleteNode(TRCNode node, boolean refresh, boolean deleteContents) {
		if (node == null)
			return;

		TRCMonitor monitor = node.getMonitor();

		if (monitor == null)
			return;

		if (monitor.getNodes().size() == 1) {
			deleteMonitor(monitor, refresh, deleteContents);
		} else {
			//remove documents
			removeDoc(node, deleteContents);
			node.setMonitor(null);
		}

		if (refresh) {
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);

		}
	}

	public static void deleteProcess(TRCProcessProxy process, boolean refresh, boolean deleteContents) {

		if (process == null)
			return;

		TRCNode node = process.getNode();

		if (node == null)
			return;

		if (node.getProcessProxies().size() == 1) {
			deleteNode(node, refresh, deleteContents);
		} else {

			//remove documents
			removeDoc(process, deleteContents);
			process.setNode(null);
		}

		if (refresh) {
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);
		}
	}

	/**
	 * Insert the method's description here. Creation date: (11/03/2000 5:36:48 PM)
	 * 
	 * @param project
	 *            com.ibm.itp.core.api.resources.IProject
	 */
	public static void deleteContainer(IContainer container, boolean deleteContents) throws CoreException {

		if (container == null || !container.exists())
			return;

		IResource[] members = container.members();
		if (members.length > 0) {
			for (int i = 0; i < members.length; i++) {
				if (members[i] instanceof IFolder) {
					IFolder member = (IFolder) members[i];
					deleteContainer(member, deleteContents);
				}
			}
		}

		List monitors = PDContentProvider.getMonitors(container);
		for (int idx = 0; idx < monitors.size(); idx++) {
			if (!(monitors.get(idx) instanceof TRCMonitor))
				continue;

			TRCMonitor mon = (TRCMonitor) monitors.get(idx);
			if (mon != null)
				deleteMonitor(mon, true, deleteContents);
		}

		PDContentProvider.removeContainer(container);

		container.delete(true, null);

	}

	/**
	 * Insert the method's description here. Creation date: (10/05/2000 2:20:27 PM)
	 * 
	 * @param resource
	 *            com.ibm.itp.core.api.resources.IResource
	 */
	public static void deleteResource(IResource resource, boolean deleteContents) {
		if (resource == null)
			return;

		try {

			IContainer parent = resource.getParent();

			if (resource == null || !resource.exists()) {
				return;
			}
			//remove documents
			else if (resource instanceof IProject || resource instanceof IFolder)
				deleteContainer((IContainer) resource, deleteContents);
			else
				resource.delete(true, null);

			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);

		} catch (CoreException exc) {
			exc.printStackTrace();
		}

	}

	public static void removeDoc(TRCAgentProxy agent, boolean deleteContents) {
		if (agent == null || agent.getAgent() == null || agent.getAgent().eResource() == null)
			return;

		Object dataProcessor = LoadersUtils.locateDataProcessor(agent);
		if (dataProcessor != null && dataProcessor instanceof XMLTraceDataProcessor) {
			((XMLTraceDataProcessor) dataProcessor).dispose();
		}
		LoadersUtils.deregisterDataProcessor(agent);

		Object listener = LoadersUtils.locateAgentListener(agent);
		if (listener != null && listener instanceof PIProcessListener) {
			((PIProcessListener) listener).dispose();
		}
		LoadersUtils.deregisterAgentListener(agent);

		LoadersUtils.deregisterAgentInstance(agent);
		LoadersUtils.deregisterAgentProxyContext(agent);

		if (deleteContents) {
			IPath path = new Path(TString.resourcePath(agent.getAgent().eResource().getURI()));
			IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);
			deleteResource(res, true);
		}

		UIPlugin.getDefault().getTraceManager().deregisterTrace(agent);
		List list = UIPlugin.getDefault().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(agent);
		}

		TRCProcessProxy process = agent.getProcessProxy();
		TRCNode trcnode = process.getNode();

		if (process == null || trcnode == null)
			return;

		//process.getAgentProxies().remove(agent);

		try {
			String nodeName = trcnode.getName();
			Node node = NodeFactory.createNode(nodeName);
			if (node.isConnected()) {

				Object instance = LoadersUtils.locateAgentInstance(agent);
				if (instance != null && instance instanceof Agent) {
					Agent a = (Agent) instance;
					if (a.isAttached()) {
						a.detach();
					}
				}
			}

		} catch (UnknownHostException uhe) {
			//do nothing
		} catch (Exception exc) {
			exc.printStackTrace();
		}

		//unload resource
		unloadDoc(agent.getAgent());
		for (Iterator iter = EMFUtil.delete(agent, agent.eResource().getResourceSet()).iterator();
		iter.hasNext();
		) {
			Resource res = (Resource) iter.next();

			if (res != null) {
				res.setModified(true);
			}
		}
	}
	/**
	 * Insert the method's description here. Creation date: (11/07/2000 12:54:24 PM)
	 * 
	 * @param monitor
	 *            com.ibm.etools.perftrace.TRCMonitor
	 */
	public static void removeDoc(TRCMonitor mon, boolean deleteContents) {
		if (mon == null)
			return;

		List list = UIPlugin.getDefault().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(mon);
		}

		if (mon.eResource() != null) {
			//			SaveUtil.removeDocument(mon.eResource());

			if (deleteContents) {
				IPath path = new Path(TString.resourcePath(mon.eResource().getURI()));
				IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);
				deleteResource(res, true);
			}

		}

		Object[] nodes = mon.getNodes().toArray();
		for (int idx = 0; idx < nodes.length; idx++) {
			TRCNode node = (TRCNode) nodes[idx];

			if (node.eIsProxy())
				continue;

			removeDoc(node, deleteContents);
		}

		//unload resource
		unloadDoc(mon);

	}

	/**
	 * Insert the method's description here. Creation date: (11/07/2000 12:54:43 PM)
	 * 
	 * @param node
	 *            com.ibm.etools.perftrace.TRCNode
	 */
	public static void removeDoc(TRCNode node, boolean deleteContents) {
		if (node == null)
			return;

		List list = UIPlugin.getDefault().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(node);
		}

		if (node.eResource() != null) {
			//			SaveUtil.removeDocument(node.eResource());
			if (deleteContents) {
				IPath path = new Path(TString.resourcePath(node.eResource().getURI()));
				IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);
				deleteResource(res, true);
			}

		}

		Object[] processes = node.getProcessProxies().toArray();
		for (int idx = 0; idx < processes.length; idx++) {

			TRCProcessProxy process = (TRCProcessProxy) processes[idx];

			if (process.eIsProxy())
				continue;

			removeDoc(process, deleteContents);
		}

		TRCMonitor monitor = node.getMonitor();
		if (monitor == null)
			return;

		monitor.getNodes().remove(node);

		if (deleteContents) { //update the monitor xmi file
			IPath path = new Path(TString.resourcePath(monitor.eResource().getURI()));
			IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);

			if (res != null) {
				try {
					SaveUtil.saveResource(null, monitor.eResource());
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		//unload resource
		unloadDoc(node);

	}

	public static void removeDoc(TRCProcessProxy process, boolean deleteContents) {
		if (process == null)
			return;

		if (deleteContents) {
			IPath path = new Path(TString.resourcePath(process.eResource().getURI()));
			IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);
			deleteResource(res, true);
		}

		Object[] agents = process.getAgentProxies().toArray();
		for (int idx = 0; idx < agents.length; idx++) {
			TRCAgentProxy agent = (TRCAgentProxy) agents[idx];

			if (agent.eIsProxy())
				continue;

			removeDoc(agent, deleteContents);
		}

		TRCNode node = process.getNode();
		if (node != null) {
			node.getProcessProxies().remove(process);

			if (deleteContents) { //update the node xmi file
				IPath path = new Path(TString.resourcePath(node.eResource().getURI()));
				IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);

				if (res != null) {
					try {
						SaveUtil.saveResource(null, node.eResource());
					} catch (Exception exc) {
						exc.printStackTrace();
					}
				}
			}
		}

		UIPlugin.getDefault().getTraceManager().deregisterTrace(process);
		UIPlugin.getDefault().deregisterLaunchProcess(process);

		List list = UIPlugin.getDefault().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(process);
		}

		//unload resource
		unloadDoc(process);

	}

	public static void unloadDoc(EObject object) {

		if (object == null || object.eResource() == null)
			return;

		UIPlugin.getDefault().removeSelectionModel(object);
		((HierarchyXMIResourceImpl) object.eResource()).doShallowDelete();

	}

	/**
	 * @param proxy
	 * @param b
	 * @param deleteContents
	 */
	public static void deleteCorrelationContainer(CorrelationContainerProxy correlationContainerProxy, boolean refresh, boolean deleteContents) {
		if (correlationContainerProxy == null)
			return;


		//remove documents
		CorrelationContainer correlationContainer = correlationContainerProxy.getCorrelationContainer();
		if (correlationContainer != null && !correlationContainer.eIsProxy()) {
			if (deleteContents) { //update the node xmi file
				IPath path = new Path(TString.resourcePath(correlationContainer.eResource().getURI()));
				IResource res = UIPlugin.getPluginWorkbench().getRoot().findMember(path);
				deleteResource(res, true);

			}
			((HierarchyXMIResourceImpl) correlationContainer.eResource()).doShallowDelete();
			correlationContainer.setCorrelationContainerProxy(null);
		}

		Resource res = correlationContainerProxy.getMonitor().eResource();
		
		correlationContainerProxy.setMonitor(null);
		
		if (res != null && deleteContents ) {
			try {
				SaveUtil.saveResource(null, res);
			} catch (Exception exc) {
				exc.printStackTrace();
			}
		}
		
		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
			TRCAgentProxy element = (TRCAgentProxy) iter.next();
			if (element.getInternalCorrelations().contains(correlationContainerProxy))
				element.getInternalCorrelations().remove(correlationContainerProxy);
		}
		correlationContainerProxy.getCorrelatedAgents().clear();

		if (refresh) {
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			UIPlugin.getDefault().notifyProfileEventListener(event);

		}

	}

}
