/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import org.apache.xerces.jaxp.SAXParserFactoryImpl;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;


/**
 * @author slavescu
 */
/**
 * This class wraps an XMLLoader with a SAX DefaultHandler.
 */
class SAXFragmentHandler extends DefaultHandler implements XMLFragmentHandler {
    //~ Instance fields ----------------------------------------------------------------------------

    private InputSource inputSource = null;
    protected ParserPipedInputStream xmlStream = null;
    private SAXParser parser;
    private Thread parseThread;
    private IXMLLoader handler;

    //~ Constructors -------------------------------------------------------------------------------

	public void setXMLLoader(IXMLLoader handler) {
        try {
            parser = makeParser();
            xmlStream = new ParserPipedInputStream();

            parseThread = new Thread(Thread.currentThread().getThreadGroup(), "xmlParserThread") {
                    public void run() {
                        if (xmlStream == null) {
                            return;
                        }

                        parse();
                    }
                };
            parseThread.start();
        } catch (Exception e) {
            e.printStackTrace();
        }

        this.handler = handler;
    }

    //~ Methods ------------------------------------------------------------------------------------

    public void setDocumentLocator(Locator locator) {
    }

    public void characters(char[] ch, int start, int length) throws SAXException {
        //		System.err.println("characters="+new String(ch,start,length));
        handler.characters(ch, start, length);
    }

    public void endDocument() throws SAXException {
        //		System.err.println("endDocument");
        handler.endDocument(null, 0);
    }

    public void endElement(String uri, String localName, String qName) throws SAXException {
        //		System.err.println("endElement="+qName);
        handler.endElement(qName, 0);
    }

    public void endPrefixMapping(String prefix) throws SAXException {
        //      handler.endPrefixMapping(prefix);
    }

    public void error(SAXParseException e) throws SAXException {
//        terminateParser();
        throw e;
    }

    public void fatalError(SAXParseException e) throws SAXException {
//        terminateParser();
        throw e;
    }

    public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
        // handler.ignorableWhitespace(ch, start, length);
    }

    public void notationDecl(String name, String publicId, String systemId) throws SAXException {
        // handler.notationDecl(name, publicId, systemId);
    }

    public void processingInstruction(String target, String data) throws SAXException {
        // handler.processingInstruction(target, data);
    }

    public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
        // handler.resolveEntity(publicId, systemId);
        return null;
    }

    public void scanContent(byte[] fragment, int offset, int length) throws InvalidXMLException {
        if (xmlStream != null) {
            xmlStream.writeBuf(fragment, offset, length);
        }
    }

    public void skippedEntity(String name) throws SAXException {
        // handler.skippedEntity(name);
    }

    public void startDocument() throws SAXException {
        //		System.err.println("startDocument");
        handler.startDocument();
    }

    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
        //		System.err.println("startElement="+qName);
        handler.startElement(qName, false, attributes.getLength() == 0);

        for (int i = 0; i < attributes.getLength(); i++) {
            handler.attributeName(attributes.getQName(i));
            handler.attributeValueCharacters(attributes.getValue(i));
        }
    }

    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        //      handler.startPrefixMapping(prefix, uri);
    }

    public void terminateParser() {
        if (xmlStream != null) {
            xmlStream.makeClosed();
            xmlStream = null;
        }

        notifyAll();
        parseThread.interrupt();
    }

    public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName) throws SAXException {
        // handler.unparsedEntityDecl(name, publicId, systemId, notationName);
    }

    public void warning(SAXParseException e) throws SAXException {
        //      handler.warning(xmi);
    }

    /**
     * Make either a validating or non-validating parser;
     * throw an if one could not be made.
     */
    protected SAXParser makeParser() throws ParserConfigurationException, SAXException {
//        SAXParserFactory f = SAXParserFactory.newInstance();
		SAXParserFactoryImpl f = new SAXParserFactoryImpl();
		
        f.setValidating(false);

        //        f.setNamespaceAware(true);
        return f.newSAXParser();
    }

    protected synchronized void parse() throws InvalidXMLException {
        try {
            inputSource = new InputSource(xmlStream);
            parser.parse(inputSource, this);
        } catch (Exception e) {
            handler.error(new InvalidXMLException(e));
        }
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    public static class ParserPipedInputStream extends InputStream {
        private byte[] inBuf;
        private boolean closed = false;
        private int inCount;
		private long inTotalLength;
        private int inPos;

        /**
         * Constructor MyPipedInputStream.
         * @param buf
         */
        public ParserPipedInputStream() {
			inTotalLength=0;
            inCount = 0;
            inPos = 0;
        }

        /**
         * @see java.io.InputStream#available()
         */
        public int available() throws IOException {
            waitForNewData();

            return inCount;
        }

        /* (non-Javadoc)
         * @see java.io.InputStream#close()
         */
        public void close() throws IOException {
            super.close();
            closed = true;

            //            notifyClose();
        }

        public boolean hasEmptyBuffer() {
            return inCount == 0;
        }

        /* (non-Javadoc)
         * @see java.io.InputStream#mark(int)
         */
        public synchronized void mark(int arg0) {
            super.mark(arg0);
        }

        /**
         * @see java.io.InputStream#markSupported()
         */
        public boolean markSupported() {
            return false;
        }

        /**
         * @see java.io.InputStream#read()
         */
        public int read() throws IOException {
            if (available() == 0) {
                return -1;
            } else {
            	inCount--;
                return inBuf[inPos++];
            }
        }

        /**
         * @see java.io.InputStream#read(byte[], int, int)
         */
        public int read(byte[] outBuf, int offset, int length) throws IOException {
            if (available() == 0) {
                return -1;
            }

            int readBytes = Math.min(inCount, length);

            System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);

            inPos = inPos + readBytes;
            inCount = inCount - readBytes;

            return readBytes;
        }

        /**
         * @see java.io.InputStream#read(byte[])
         */
        public int read(byte[] arg0) throws IOException {
            return read(arg0, 0, arg0.length);
        }

        public synchronized void reset() throws IOException {
            super.reset();
        }

        /**
         * @see java.io.InputStream#skip(long)
         */
        public long skip(long arg0) throws IOException {
            arg0 = arg0 + inPos;

            if (arg0 < inTotalLength) {
				inPos = (int) arg0;
				return arg0;
            } else {
				return -1;
            }
        }

        /**
         * Sets the inBuf.
         * @param inBuf The inBuf to set
         */
        public synchronized void writeBuf(byte[] buf, int offset, int length) {
            if ((buf == null) || (length == 0)) {
                return;
            }

            inBuf = buf;
            inPos = offset;
            inTotalLength = offset+length;
            inCount = length;
            notifyAll();
            waitForEmptyBuffer();
        }

        /**
         *
         */
        protected synchronized void makeClosed() {
            inBuf = null;
            inCount = 0;
            inPos = 0;
            closed = true;
            notifyAll();
        }

        /**
         *
         */
        private void waitForEmptyBuffer() {
            while (!hasEmptyBuffer() && !closed) {
                try {
                    wait(1000);
                } catch (InterruptedException e) {
                    if (hasEmptyBuffer() || closed) {
                        return;
                    }
                }
            }
        }

        /**
         *
         */
        private synchronized void waitForNewData() {
            boolean notify = true;

            while (hasEmptyBuffer()) {
                if (notify) {
                    notifyAll();
                    notify = false;
                }

                if (closed) {
                    return;
                }

                try {
                    wait(1000);
                } catch (InterruptedException e) {
                    if (closed) {
                        return;
                    }
                }
            }
        }
    }
}
