package org.eclipse.hyades.logging.events;

import java.util.Date;
import java.util.List;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Factory to construct ICommonBaseEvent objects.
 * 
 * @version 1.0.1
 * @since 1.0
 */
public class CommonBaseEventFactory implements IEventFactory {

    /** 
     * The name of this factory instance (so that it can be fetched by
     * <code>FactoryHome</code>, if multiple components are sharing it.
     */
    private String name = null;

    /** 
     * A 'settings' event, whose values will be copied into any new
     * event created by this factory's <code>createEvent()</code> method.
     */
    private CommonBaseEventImpl settings = null;

    /**
     * Protected constructor for CommonBaseEventFactory.
     * Applications should use <code>FactoryHome</code>
     * rather than directly creating event factories.
     */
    protected CommonBaseEventFactory() {
        super();
    }

    /**	
     * Get the name of this factory instance.
     *	Instance names are immutable once set.
     *
     *	@return	The instance name.
     **/
    public String getName() {
        return this.name;
    }

    /**	Get the fully-qualified interface name for the events
     *  created by this factory. This should either be:
     *	<code>org.eclipse.hyades.logging.events.ICommonBaseEvent</code>
     *	or an interface which extends it.
     *
     *	@return	The interface name.
     **/
    public String getEventInterfaceClassName() {
        return ICommonBaseEvent.class.getName();
    }

    /**	Get the optional event settings for this factory
     *	instance. The values of this object (if any) will
     *  be used as defaults when creating new events.
     * 
     *	@return	The current event settings, or
     *				<code>null</code>	if no settings
     *				have been defined.
     **/
    public ICommonBaseEvent getEventSettings() {
        return this.settings;
    }

    /**	Set the optional event settings for this factory
     *	instance. The values of this object (if any) will
     *  be used as defaults when creating new events.
     * 
     *  <p>The settings object should be an event which was
     *  created by this factory's <code>createEmptyEvent()</code>
     *  method, then populated with values.</p>
     *
     *	@param	The new event settings, or <code>null</code>
     *			if no event settings should be used.
     **/
    public void setEventSettings(ICommonBaseEvent s) {
        this.settings = (CommonBaseEventImpl) s;
    }

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event,
     *							or <code>null</code> to use the
     *							current system time.
     *
     *	@param sourceCompId	The source component ID for the event,
     *							or <code>null</code> to copy the value
     *							from the event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    public ICommonBaseEvent createEvent(Date creationTime, IComponentIdentification sourceCompId) {
        long timeVal = CURRENT_TIME;

        if (creationTime != null) {
            timeVal = creationTime.getTime();
        }

        return createEvent(timeVal, sourceCompId);

    } /* end method createEvent() */

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event, or
     * 							<code>CURRENT_TIME</code> to use the
     *							current system time.
     *
     *	@param sourceCompId	The source component ID for the event,
     *							or <code>null</code> to copy the value
     *							from the event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    public ICommonBaseEvent createEvent(long creationTime, IComponentIdentification sourceCompId) {
        // If no creation time specified, use the current time.

        if (creationTime == CURRENT_TIME) {
            creationTime = System.currentTimeMillis();
        }

        // If no source component id specified, make sure that
        // the settings define one, or return null if not.

        if (sourceCompId == null) {

            if (settings != null) {
                sourceCompId = settings.getSourceComponentId();
            }

            if (sourceCompId == null) {
                return null;
            }
        }

        ICommonBaseEvent event = new CommonBaseEventImpl();

        event.setCreationTime(creationTime);
        event.setSourceComponentId(sourceCompId);

        if (settings == null) {
            return event;
        }

        // If settings are available, copy all defined values
        // into the new event.

        String localInstId = settings.getLocalInstanceId();
        if (localInstId != null) {
            event.setLocalInstanceId(localInstId);
        }

        String globalInstId = settings.getLocalInstanceId();
        if (globalInstId != null) {
            event.setGlobalInstanceId(globalInstId);
        }

        short severity = settings.getSeverity();
        if (severity != UNSPECIFIED_SEVERITY) {
            event.setSeverity(severity);
        }

        short priority = settings.getPriority();
        if (priority != UNSPECIFIED_PRIORITY) {
            event.setPriority(priority);
        }

        IComponentIdentification reporterCompId = settings.getReporterComponentId();
        if (reporterCompId != null) {
            event.setReporterComponentId(reporterCompId);
        }

        ISituation situation = settings.getSituation();
        if (situation != null) {
            event.setSituation(situation);
        }

        List contextData = settings.getContextDataElements();
        if (contextData != null) {
            for (int i = 0; i < contextData.size(); i++) {
                event.addContextDataElement(((IContextDataElement)(contextData.get(i))));
            }
        }

        IMsgDataElement msgData = settings.getMsgDataElement();
        if (msgData != null) {
            event.setMsgDataElement(msgData);
        }

        String msg = settings.getMsg();
        if (msg != null) {
            event.setMsg(msg);
        }

        short repeatCount = settings.getRepeatCount();
        if (repeatCount != UNSPECIFIED_REPEAT_COUNT) {
            event.setRepeatCount(repeatCount);
        }

        long elapsedTime = settings.getElapsedTime();
        if (elapsedTime != UNSPECIFIED_ELAPSED_TIME) {
            event.setElapsedTime(elapsedTime);
        }

        List assocEvents = settings.getAssociatedEvents();
        if (assocEvents != null) {
            for (int i = 0; i < assocEvents.size(); i++) {
                event.addAssociatedEvent(((IAssociatedEvent)(assocEvents.get(i))));
            }
        }

        String extensionName = settings.getExtensionName();
        if (extensionName != null) {
            event.setExtensionName(extensionName);
        }

        List extendedData = settings.getExtendedDataElements();
        if (extendedData != null) {
            for (int i = 0; i < extendedData.size(); i++) {
                event.addExtendedDataElement((IExtendedDataElement)(extendedData.get(i)));
            }
        }

        long sequenceNum = settings.getSequenceNumber();
        if (sequenceNum != UNSPECIFIED_SEQUENCE_NUM) {
            event.setSequenceNumber(sequenceNum);
        }

        String[] otherData = settings.getOtherData();
        if (otherData != null) {
            event.setOtherData(otherData);
        }

        return event;

    } /* end method createEvent() */

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event,
     *							or <code>null</code> to use the
     *							current system time.
     *
     *	@param localInstId		An optional local-instance identifier,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param globalInstId	An optional global-instance identifier,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param severity		The optional event severity, or
     *							<code>UNSPECIFIED_SEVERITY</code> to copy
     *							the value (if any) from the event settings.
     *
     *	@param priority		The optional event priority, or
     *							<code>UNSPECIFIED_PRIORITY</code> to copy 
     *							the value (if any) from the event settings
     *
     *	@param reporterCompId	The optional reporter component ID for
     *							the event, or <code>null</code> to copy
     *							the value (if any) from event settings.
     *
     *	@param sourceCompId	The required source component ID for
     *							the event, or <code>null</code> to copy
     *							the value from the event settings.
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@param situationType	An optional situation-type string,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param contextData		An optional array of context data
     *							elements, or <code>null</code> to
     *							copy the value (if any) from the
     *							event settings.
     *
     *	@param msgData			Optional data about the message, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param msg				The optional message text, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param repeatCount		The optional repeat count, or
     *							<code>UNSPECIFIED_REPEAT_COUNT</code>
     *							to copy	the value (if any) from the
     *							event settings.
     *
     *	@param elapsedTime		The optional time (in microseconds)
     *							for the number of events specified
     *							in the repeat count, or 
     *							<code>UNSPECIFIED_ELAPSED_TIME</code>
     *							to copy	the value (if any) from the
     *							event settings.
     *
     *	@param assocEvents		An optional array of associated events,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param extensionName	An optional extension name, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param extendedData	An optional array of extended data
     *							elements, or <code>null</code> to
     *							copy the value (if any) from the
     *							event settings.
     *
     *	@param sequenceNum		An optional sequence number for the event,
     *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
     *							to copy the value (if any) from the
     *							event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
      * @deprecated Use the createEvent(Date, String, String, short, short, IComponentIdentification, IComponentIdentification, ISituation, IContextDataElement[], IMsgDataElement, String, short, long, IAssociatedEvent, String, IExtendedDataElement, long,String[]) as of 1.0.1.
     **/
    public ICommonBaseEvent createEvent(Date creationTime, String localInstId, String globalInstId, short severity, short priority, IComponentIdentification reporterCompId, IComponentIdentification sourceCompId, String situationType, IContextDataElement[] contextData, IMsgDataElement msgData, String msg, short repeatCount, long elapsedTime, IAssociatedEvent[] assocEvents, String extensionName, IExtendedDataElement[] extendedData, long sequenceNum) {
        long timeVal = CURRENT_TIME;

        if (creationTime != null) {
            timeVal = creationTime.getTime();
        }

        return createEvent(timeVal, localInstId, globalInstId, severity, priority, reporterCompId, sourceCompId, situationType, contextData, msgData, msg, repeatCount, elapsedTime, assocEvents, extensionName, extendedData, sequenceNum);

    } /* end method createEvent() */

    /**	Create a new event.
      * 
      *	@param creationTime	The creation time for the event, or
      * 						<code>CURRENT_TIME</code> to use the
      *							current system time.
      *
      *	@param localInstId		An optional local-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param globalInstId	An optional global-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param severity		The optional event severity, or
      *							<code>UNSPECIFIED_SEVERITY</code> to copy
      *							the value (if any) from the event settings.
      *
      *	@param priority		The optional event priority, or
      *							<code>UNSPECIFIED_PRIORITY</code> to copy 
      *							the value (if any) from the event settings
      *
      *	@param reporterCompId	The optional reporter component ID for
      *							the event, or <code>null</code> to copy
      *							the value (if any) from event settings.
      *
      *	@param sourceCompId	The required source component ID for
      *							the event, or <code>null</code> to copy
      *							the value from the event settings.
      *
      *	<p>NOTE: If the source component ID is not specified and is
      *	not found in the factory event settings, the event will not
      *	be created. Use <code>createEmptyEvent()</code> to create an
      *	event with no creation time or source component ID.</p>
      *
      *	@param situation    	A required ISituation or <code>null</code> 
      *                         to copy the value from the event settings.
      *
      *	@param contextData		An optional array of context data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param msgData			Optional data about the message, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param msg				The optional message text, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param repeatCount		The optional repeat count, or
      *							<code>UNSPECIFIED_REPEAT_COUNT</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param elapsedTime		The optional time (in microseconds)
      *							for the number of events specified
      *							in the repeat count, or 
      *							<code>UNSPECIFIED_ELAPSED_TIME</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param assocEvents		An optional array of associated events,
      *							
      *
      *	@param extensionName	An optional extension name, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extendedData	    An optional array of extended data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param sequenceNum		An optional sequence number for the event,
      *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
      *							to copy the value (if any) from the
      *							event settings.
      *
      *	@param otherData		An optional string array of other data
      *                         or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	<p>If event settings are defined for this factory instance,
      *	they will be used to set any fields not specified above.</p>
      *
      *	@return	The new event, or <code>null</code> if no value
      *				was found for the required source component ID.
      *
      *	<p>The new event will implement the interface reported by
      *	   <code>getEventInterfaceClassName()</code>.</p>
      * @since 1.0.1
	  **/
    public ICommonBaseEvent createEvent(Date creationTime, String localInstId, String globalInstId, short severity, short priority, IComponentIdentification reporterCompId, IComponentIdentification sourceCompId, ISituation situation, IContextDataElement contextData[], IMsgDataElement msgData, String msg, short repeatCount, long elapsedTime, IAssociatedEvent assocEvents[], String extensionName, IExtendedDataElement extendedData[], long sequenceNum, String[] otherData) {

        long timeVal = CURRENT_TIME;

        if (creationTime != null) {
            timeVal = creationTime.getTime();
        }

        return createEvent(timeVal, localInstId, globalInstId, severity, priority, reporterCompId, sourceCompId, situation, contextData, msgData, msg, repeatCount, elapsedTime, assocEvents, extensionName, extendedData, sequenceNum, otherData);
    }

    /**	Create a new event.
      * 
      *	@param creationTime	The creation time for the event, or
      * 						<code>CURRENT_TIME</code> to use the
      *							current system time.
      *
      *	@param localInstId		An optional local-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param globalInstId	An optional global-instance identifier,
      *							or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param severity		The optional event severity, or
      *							<code>UNSPECIFIED_SEVERITY</code> to copy
      *							the value (if any) from the event settings.
      *
      *	@param priority		The optional event priority, or
      *							<code>UNSPECIFIED_PRIORITY</code> to copy 
      *							the value (if any) from the event settings
      *
      *	@param reporterCompId	The optional reporter component ID for
      *							the event, or <code>null</code> to copy
      *							the value (if any) from event settings.
      *
      *	@param sourceCompId	The required source component ID for
      *							the event, or <code>null</code> to copy
      *							the value from the event settings.
      *
      *	<p>NOTE: If the source component ID is not specified and is
      *	not found in the factory event settings, the event will not
      *	be created. Use <code>createEmptyEvent()</code> to create an
      *	event with no creation time or source component ID.</p>
      *
      *	@param situation    	A required ISituation or <code>null</code> 
      *                         to copy the value from the event settings.
      *
      *	@param contextData		An optional array of context data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param msgData			Optional data about the message, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param msg				The optional message text, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param repeatCount		The optional repeat count, or
      *							<code>UNSPECIFIED_REPEAT_COUNT</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param elapsedTime		The optional time (in microseconds)
      *							for the number of events specified
      *							in the repeat count, or 
      *							<code>UNSPECIFIED_ELAPSED_TIME</code>
      *							to copy	the value (if any) from the
      *							event settings.
      *
      *	@param assocEvents		An optional array of associated events,
      *							
      *
      *	@param extensionName	An optional extension name, or
      *							<code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	@param extendedData	    An optional array of extended data
      *							elements, or <code>null</code> to
      *							copy the value (if any) from the
      *							event settings.
      *
      *	@param sequenceNum		An optional sequence number for the event,
      *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
      *							to copy the value (if any) from the
      *							event settings.
      *
      *	@param otherData		An optional string array of other data
      *                         or <code>null</code> to copy the value
      *							(if any) from the event settings.
      *
      *	<p>If event settings are defined for this factory instance,
      *	they will be used to set any fields not specified above.</p>
      *
      *	@return	The new event, or <code>null</code> if no value
      *				was found for the required source component ID.
      *
      *	<p>The new event will implement the interface reported by
      *	   <code>getEventInterfaceClassName()</code>.</p>
      * @since 1.0.1
      **/
    public ICommonBaseEvent createEvent(long creationTime, String localInstId, String globalInstId, short severity, short priority, IComponentIdentification reporterCompId, IComponentIdentification sourceCompId, ISituation situation, IContextDataElement contextData[], IMsgDataElement msgData, String msg, short repeatCount, long elapsedTime, IAssociatedEvent assocEvents[], String extensionName, IExtendedDataElement extendedData[], long sequenceNum, String[] otherData) {

        ICommonBaseEvent event = this.createEvent(creationTime, sourceCompId);

        if (event == null) {
            return null;
        }

        // Apply all specified parameters to the event

        if (localInstId != null) {
            event.setLocalInstanceId(localInstId);
        }

        if (globalInstId != null) {
            event.setGlobalInstanceId(globalInstId);
        }

        if (severity != UNSPECIFIED_SEVERITY) {
            event.setSeverity(severity);
        }

        if (priority != UNSPECIFIED_PRIORITY) {
            event.setPriority(priority);
        }

        if (reporterCompId != null) {
            event.setReporterComponentId(reporterCompId);
        }

        if (situation != null) {
            event.setSituation(situation);
        }

        if (contextData != null) {
            for (int i = 0; i < contextData.length; i++) {
                if (contextData[i] != null) {
                    event.addContextDataElement(contextData[i]);
                }
            }
        }

        if (msgData != null) {
            event.setMsgDataElement(msgData);
        }

        if (msg != null) {
            event.setMsg(msg);
        }

        if (repeatCount != UNSPECIFIED_REPEAT_COUNT) {
            event.setRepeatCount(repeatCount);
        }

        if (elapsedTime != UNSPECIFIED_ELAPSED_TIME) {
            event.setElapsedTime(elapsedTime);
        }

        if (assocEvents != null) {
            for (int i = 0; i < assocEvents.length; i++) {
                event.addAssociatedEvent(assocEvents[i]);
            }
        }

        if (extensionName != null) {
            event.setExtensionName(extensionName);
        }

        if (extendedData != null) {
            for (int i = 0; i < extendedData.length; i++) {
                if (extendedData[i] != null) {
                    event.addExtendedDataElement(extendedData[i]);
                }
            }
        }

        if (sequenceNum != UNSPECIFIED_SEQUENCE_NUM) {
            event.setSequenceNumber(sequenceNum);
        }

        if (otherData != null) {
            event.setOtherData(otherData);
        }

        return event;
    }

    /**	Create a new event.
     * 
     *	@param creationTime	The creation time for the event, or
     * 							<code>CURRENT_TIME</code> to use the
     *							current system time.
     *
     *	@param localInstId		An optional local-instance identifier,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param globalInstId	An optional global-instance identifier,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param severity		The optional event severity, or
     *							<code>UNSPECIFIED_SEVERITY</code> to copy
     *							the value (if any) from the event settings.
     *
     *	@param priority		The optional event priority, or
     *							<code>UNSPECIFIED_PRIORITY</code> to copy 
     *							the value (if any) from the event settings
     *
     *	@param reporterCompId	The optional reporter component ID for
     *							the event, or <code>null</code> to copy
     *							the value (if any) from event settings.
     *
     *	@param sourceCompId	The required source component ID for
     *							the event, or <code>null</code> to copy
     *							the value from the event settings.
     *
     *	<p>NOTE: If the source component ID is not specified and is
     *	not found in the factory event settings, the event will not
     *	be created. Use <code>createEmptyEvent()</code> to create an
     *	event with no creation time or source component ID.</p>
     *
     *	@param situationType	An optional situation-type string,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param contextData		An optional array of context data
     *							elements, or <code>null</code> to
     *							copy the value (if any) from the
     *							event settings.
     *
     *	@param msgData			Optional data about the message, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param msg				The optional message text, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param repeatCount		The optional repeat count, or
     *							<code>UNSPECIFIED_REPEAT_COUNT</code>
     *							to copy	the value (if any) from the
     *							event settings.
     *
     *	@param elapsedTime		The optional time (in microseconds)
     *							for the number of events specified
     *							in the repeat count, or 
     *							<code>UNSPECIFIED_ELAPSED_TIME</code>
     *							to copy	the value (if any) from the
     *							event settings.
     *
     *	@param assocEvents		An optional array of associated events,
     *							or <code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param extensionName	An optional extension name, or
     *							<code>null</code> to copy the value
     *							(if any) from the event settings.
     *
     *	@param extendedData	An optional array of extended data
     *							elements, or <code>null</code> to
     *							copy the value (if any) from the
     *							event settings.
     *
     *	@param sequenceNum		An optional sequence number for the event,
     *							or <code>UNSPECIFIED_SEQUENCE_NUM</code>
     *							to copy the value (if any) from the
     *							event settings.
     *
     *	<p>If event settings are defined for this factory instance,
     *	they will be used to set any fields not specified above.</p>
     *
     *	@return	The new event, or <code>null</code> if no value
     *				was found for the required source component ID.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
      * @deprecated Use the createEvent(long, String, String, short, short, IComponentIdentification, IComponentIdentification, ISituation, IContextDataElement[], IMsgDataElement, String, short, long, IAssociatedEvent, String, IExtendedDataElement, long,String[]) as of 1.0.1.
     **/
    public ICommonBaseEvent createEvent(long creationTime, String localInstId, String globalInstId, short severity, short priority, IComponentIdentification reporterCompId, IComponentIdentification sourceCompId, String situationType, IContextDataElement[] contextData, IMsgDataElement msgData, String msg, short repeatCount, long elapsedTime, IAssociatedEvent[] assocEvents, String extensionName, IExtendedDataElement[] extendedData, long sequenceNum) {
        ICommonBaseEvent event = this.createEvent(creationTime, sourceCompId);

        if (event == null) {
            return null;
        }

        // Apply all specified parameters to the event

        if (localInstId != null) {
            event.setLocalInstanceId(localInstId);
        }

        if (globalInstId != null) {
            event.setGlobalInstanceId(globalInstId);
        }

        if (severity != UNSPECIFIED_SEVERITY) {
            event.setSeverity(severity);
        }

        if (priority != UNSPECIFIED_PRIORITY) {
            event.setPriority(priority);
        }

        if (reporterCompId != null) {
            event.setReporterComponentId(reporterCompId);
        }

        if (situationType != null) {
            event.setSituationType(situationType);
        }

        if (contextData != null) {
            for (int i = 0; i < contextData.length; i++) {
                if (contextData[i] != null) {
                    event.addContextDataElement(contextData[i]);
                }
            }
        }

        if (msgData != null) {
            event.setMsgDataElement(msgData);
        }

        if (msg != null) {
            event.setMsg(msg);
        }

        if (repeatCount != UNSPECIFIED_REPEAT_COUNT) {
            event.setRepeatCount(repeatCount);
        }

        if (elapsedTime != UNSPECIFIED_ELAPSED_TIME) {
            event.setElapsedTime(elapsedTime);
        }

        if (assocEvents != null) {
            for (int i = 0; i < assocEvents.length; i++) {
                event.addAssociatedEvent(assocEvents[i]);
            }
        }

        if (extensionName != null) {
            event.setExtensionName(extensionName);
        }

        if (extendedData != null) {
            for (int i = 0; i < extendedData.length; i++) {
                if (extendedData[i] != null) {
                    event.addExtendedDataElement(extendedData[i]);
                }
            }
        }

        if (sequenceNum != UNSPECIFIED_SEQUENCE_NUM) {
            event.setSequenceNumber(sequenceNum);
        }

        return event;
    }

    /**	Create an empty event.
     *
     *	<p>No fields will be set for this event. Any event settings
     *	for this factory instance will be ignored.</p>
     *
     *	@return	The empty new event.
     *
     *	<p>The new event will implement the interface reported by
     *	   <code>getEventInterfaceClassName()</code>.</p>
     **/
    public ICommonBaseEvent createEmptyEvent() {
        return new CommonBaseEventImpl();

    } /* end method createEmptyEvent() */

    /**	Initialization method. This should not be called by
     *	applications, and has no effect after this factory
     *	instance is initialized.
     *
     *	@param	The new name of this factory instance.
     **/
    public void init(String n) {
        if (this.name == null) {
            this.name = n;
        }

    } /* end method init() */

} /* end class CommonBaseEventFactory */
