/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.testgen.ui;

import java.text.MessageFormat;

import org.eclipse.hyades.execution.testgen.TestgenPlugin;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * @author ejessee
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class TestgenUIUtility
{
	
	/**
		 * exception thrown by the RecorderHarnessUIUtility class when problems occur while createing or laying out various controls.
		 */
		public static class TestGenUIException extends Exception
		{
	
			/**
			 * 
			 */
			public TestGenUIException()
			{
				super();
			}

			/**
			 * @param message
			 */
			public TestGenUIException(String message)
			{
				super(message);
			}

			

	}
	
	
	
	/**
	 * Utility method for laying out a control in a form layout by specifying percentage-of-parent-width and border regions
	 *  If the parent of the control is not a FormLayout, an exception will be thrown
	 * @param Control control - the control to be layed out
	 * @param intpixelsToTop - pixels from top of the parent control to the parent of the control of focus
	 * @param int verticalSpanPixels - the vertical span of the control being layed out
	 * @param int leftPercentWidth - percentage of total parent width from left for left of target
	 * @param int rightPercentWidth - percentage of total parent width from left for right of target 
	 * @param int leftBorder - pixels for left border region
	 * @param int bottomBorder - pixels for bottom border region
	 * @param int topBorder - pixels for top border region
	 * @param int rightBorder - pixels for bottom border region
	 * @throws TestGenUIException
	 */
	public static void layoutCompositelInFormLayout(Control composite,
								 int pixelsToTop,
								 int verticalSpanPixels,
								 int leftPercentWidth, 
								 int rightPercentWidth, 
								 int leftBorder, 
								 int bottomBorder,
								 int topBorder, 
								 int rightBorder) throws TestGenUIException
	{
		if(!FormLayout.class.isInstance(composite.getParent().getLayout()))
		{
			String[] layoutName = {"org.eclipse.swt.layout.FormLayout"}; //$NON-NLS-1$
			String errorMessage = MessageFormat.format(TestgenPlugin.getResourceString("TestGenUIUtility.INCORRECT_LAYOUT_MESSAGE_FORMAT_STRING"),layoutName); //$NON-NLS-1$
			throw new TestGenUIException(errorMessage);
		}
		
		FormData formData = new FormData();
		formData.left = new FormAttachment(leftPercentWidth,leftBorder);
		formData.right = new FormAttachment(rightPercentWidth,-rightBorder);
		formData.top = new FormAttachment(0,pixelsToTop+topBorder);
		formData.bottom = new FormAttachment(0,pixelsToTop+verticalSpanPixels-bottomBorder);
		composite.setLayoutData(formData);
	}
	
	
	
		/** 
		 * Positions a control on it's parent FormLayout.
		 * @param Control control - the Control to be positioned
		 * @param int topBorder - the top border for this control in pixels
		 * @param int leftBorder - the left border for this control in pixels.
		 * @param int bottomBorder - the bottom border for this control in pixels
		 * @param int rightBorder - the right border for this control in pixels
		 */
		public static void layoutControlInFormLayout(Control control,
												 int topBorder, 
												 int leftBorder, 
												 int bottomBorder, 
												 int rightBorder) throws TestGenUIException
		{
			if(!FormLayout.class.isInstance(control.getParent().getLayout()))
			{
				String[] layoutName = {"org.eclipse.swt.layout.GridLayout"}; //$NON-NLS-1$
				String errorMessage = MessageFormat.format(TestgenPlugin.getResourceString("TestGenUIUtility.INCORRECT_LAYOUT_MESSAGE_FORMAT_STRING"),layoutName); //$NON-NLS-1$
				throw new TestGenUIException(errorMessage);
			}
	
			FormData formData = new FormData();
			formData.left = new FormAttachment(0,leftBorder);
			formData.right = new FormAttachment(100,-rightBorder);
			formData.top = new FormAttachment(0,topBorder);
			formData.bottom = new FormAttachment(100,-bottomBorder);
				
			control.setLayoutData(formData);
		}
	


	
		/**
		 * method for laying out a control in a grid layout by providing columnSpan and any "or'ed" combination of GridData options
		 * @param Control control
		 * @param int columnSpan
		 * @param int gridDataOptions
		 * @throws RecorderHarnessUIException
		 */
		public static void layoutControlInGrid(Control control, int columnSpan, int gridDataOptions) throws TestGenUIException
		{
//			if(!GridLayout.class.isInstance(control.getParent().getLayout()))
//			{
//				throw new TestGenUIException("Parent Composite does not have an assigned grid Layout (org.eclipse.swt.layout.GridLayout)");
//			}
			GridData gridData = new GridData(gridDataOptions);
			gridData.horizontalSpan=columnSpan;
			control.setLayoutData(gridData);	
		}



		/**
		 * method for assigning a FormLayoout to a composite
		 * @param Composite parent
		 */
		public static void assignFormLayout(Composite parent)
		{
			FormLayout layout = new FormLayout();
			layout.marginHeight=0;
			layout.marginWidth=0;
			parent.setLayout(layout);
		}



		/**
		 * method for assigning a GridLayout to a Composite
		 * @param Composite compsite
		 * @param int numberColumns
		 * @param int columnsEqual	 
		 */
		public static void assignGridLayout(Composite compsite, int numberColumns)
		{
			GridLayout gridLayout = new GridLayout(numberColumns,false);
			compsite.setLayout(gridLayout);
		}



		/**
		 * method for creating a label
		 * @param Composite parent
		 * @param String labelText
		 * @param Font font
		 * @return Label
		 */
		public static Label createLabel(Composite composite, String labelText, Font font)
		{
			Label label = new Label(composite,SWT.WRAP);
			label.setText(labelText);
		
			if(font!=null)
				label.setFont(font);
		
			return label;
		}



		/**
		 * method for creating an Edit Control
		 * @param Composite parent
		 * @param String defaultText
		 * @param Font font
		 * @return Text
		 */
		public static Text createEditControl(Composite composite, String textValue, Font font)
		{
			Text text = new Text(composite,SWT.BORDER);
			text.setText(textValue);
		
			if(font!=null)
				text.setFont(font);
			
			return text;
		}
	
	
		/**
		 * method for creating a TreeViewer
		 * @param composite
		 * @param font
		 * @return Tree
		 */
		public static TreeViewer createTreeViewer(Composite composite, Font font)
		{
			TreeViewer tree = new TreeViewer(composite);
			tree.setUseHashlookup(true);
			tree.getControl().setFont(font);
			return tree;
		}
	
		/**
		 * method for creating a control group
		 * @param controlArea
		 * @param text
		 * @return Group
		 */
		public static Group createGroup(Composite controlArea, String text)
		{
			Group recorderHostGroup = new Group(controlArea,SWT.SHADOW_ETCHED_IN);
	
			assignFormLayout(controlArea);
			
			recorderHostGroup.setText(text);
	
			return recorderHostGroup;
		}
	
		/**
		 * method for creating a Button
		 * @param parent
		 * @param text
		 * @param swtOptions
		 * @return Button
		 */
		public static Button createButton(Composite parent, String text, int swtOptions)
		{
			Button button = new Button(parent,swtOptions);
			button.setText(text);
			return button;
	
		}

}
