/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.extractors;
import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.impl.Extractor;
import org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
/**
 * @author sluiman
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class SimpleExtractor extends Extractor
{
   private int arraySize = 8;
   private int messageArraySize = 8;
   private int lastAddedString = -1;
   private int lastAddedMessage = -1;
   private int lastRemovedString = -1;
   protected String searchString;
   private String[] inputStrings = new String[arraySize];
   private StringPosition start, end, adjustedStart, adjustedEnd;
   private MessageString[] finalResult;
   //TODO: HS determine if using line.separator is actually appropriate given files may not be from the local system
   private String localLineSeparator = System.getProperty("line.separator");
   /**
    * @see org.eclipse.hyades.logging.adapter.IExtractor#processStrings(java.lang.String[])
    */
   public MessageString[] processStrings(String[] strings)
   {
      finalResult = new MessageString[messageArraySize];
      // add new input to left over input if any
      // null is a valid input, particularly during a flush opperation
      if (strings != null)
      {
         for (int i = 0; i < strings.length; i++)
         {
            if (strings[i] != null)
               addString(strings[i]);
            strings[i] = null;
         }
      }
      if (lastAddedString == lastRemovedString)
         return null;
      if (!getContainsLineBreaks())
         processSeparateLines();
      else
         processMultipleLines();
      if (finalResult != null)
         for (int i = 0; i < finalResult.length; i++)
            if (finalResult[i] != null)
            {
               log(Messages.getString("HyadesGA_CBE_Simple_Extractor_Message_Info",finalResult[i].getValue()), AdapterSensor.FINEST_LEVEL);
            }
      return finalResult;
   }
   /**
    * A single line is processed in isolation of all other data.
    * Because of this, if no start pattern is provided, it is assumed the beginning of the line
    * is the beginning of the message. Also if no end is provided, the end of the line is 
    * concidered to be an end pattern match. If a pattern is provided it is used, thus lines may be
    * discarded if there is not a complete pair of pattern matches.
    * @return MessageString[] or null
    */
   private void processSeparateLines()
   {
      int startOffset;
      // search out chunks of the searchString until it is used up
      for (int i = lastRemovedString + 1; i <= lastAddedString; i++)
      {
         startOffset = 0;
         searchString = buildString(i, startOffset, i, inputStrings[i].length() - 1);
         int endOffset = searchString.length() - 1;
         processSearchString(startOffset, endOffset);
         // if the last lien was just processed and there is an unused piece, preserve it for next time
         String temp = null;
         if (i == lastAddedString && start != null && end == null)
            temp = inputStrings[i].substring(start.getStartPosition(), endOffset);
         lastRemovedString = i;
         if (i == lastAddedString && start != null && end == null)
            addString(temp);
      }
      return;
   }
   /**
    * All the current input strings are concatentated and processed as one string
    * @return
    */
   private void processMultipleLines()
   {
      int startOffset;
      startOffset = 0;
      searchString =
         buildString(
            lastRemovedString + 1,
            startOffset,
            lastAddedString,
            inputStrings[lastAddedString].length() - 1);
      // note that length -1 is the correct offset however in many cases the offset + 1
      // is used because this is how String substring behaves. It requires an offest that 
      // is 1 past the end in order to be able to do simple length calculations.		
      int endOffset = searchString.length() - 1;
      processSearchString(startOffset, endOffset);
      // if the last line was just processed and there is an unused piece, preserve it for next time
      String temp = null;
      if (start != null && end == null)
         temp = searchString.substring(start.getStartPosition(), endOffset + 1);
      // if the end was really a start that needs to be reused	
      if (end != null && (getEndPattern() == null || getEndPattern().length() < 1))
         temp = searchString.substring(end.getStartPosition(), endOffset + 1);
      lastRemovedString = lastAddedString;
      // if the extractor is being flushed and there is a input left that is a valid message	
      // push it out
      if (flushingMode
         && (getEndPattern() == null || getEndPattern().length() < 1)
         && start != null
         && end == null)
      {
         adjustAndAddMessage(start, new StringPosition(endOffset + 1, endOffset + 1));
      }
      else //			if (start != null && end == null)
         if (temp != null)
            addString(temp);
      return;
   }
   /**
    * processSearchString is the heart fo the Extractor. The role is to extract Messages form
    * the input array of strings, ignoring un-needed input, and saving meaningful left over
    * input for the next processing step.
    * The actual search is delegated to the search method so that it can be speciallized.
    * @param startOffset
    * @param endOffset
    */
   private void processSearchString(int startOffset, int endOffset)
   {
      MessageString[] result = new MessageString[lastAddedString - lastRemovedString];
      boolean lineIsUsedUp = false;
      while (!lineIsUsedUp)
      {
         start = null;
         end = null;
         adjustedStart = null;
         adjustedEnd = null;
         if (getStartPattern() != null
            && getStartPattern().length() > 0) //search the input stream for a start
         {
            start = search(getStartPattern(), startOffset);
         }
         else // use the beginning of the String
            {
            start = new StringPosition(startOffset, startOffset);
         }
         // if start startPosition is greater than the end position
         // the match pattern was control characters

         // don't hunt for an end until there is a beginning
         if (start != null)
         {
            if (getEndPattern() != null && getEndPattern().length() > 0)
            {
               end = search(getEndPattern(), startOffset);
            }
            else // find the next start 
               {
               end = search(getStartPattern(), start.getEndPosition() + 1);
            }
         }
         // if no end found or it was found at the end of the input
         // signal end of loop and keep the left over input
         if (end == null || end.getEndPosition() >= endOffset)
            lineIsUsedUp = true;
         else //bump to a new start position
            if (getIncludeEndPattern())
               startOffset = end.getEndPosition() + 1;
            else
            {
               startOffset = end.getStartPosition();
            }
         // if end comes before start, try again
         if (end != null && start != null && start.getStartPosition() >= end.getEndPosition())
         {
            start = null;
            startOffset = end.getEndPosition() + 1;
         }
         //
         // adjust beginning and end if they exist
         if (start != null && end != null)
         {
            adjustAndAddMessage(start, end);
         }
      }
      return;
   }
   /**
    * Method adjustAndAddMessage adjusts the start and end positions based on the 
    * configuration, and then has the newly bound String set in a MessageString instance.
    * @param startPosition
    * @param endPosition
    */
   protected void adjustAndAddMessage(StringPosition startPosition, StringPosition endPosition)
   {
      //TODO: HS the ^ should be special cased only for regex extractors
      if (getIncludeStartPattern()
         || getStartPattern().compareTo("^") == 0
         || getStartPattern().length() < 1)
         adjustedStart = startPosition;
      else
         adjustedStart =
            new StringPosition(startPosition.getEndPosition() + 1, startPosition.getEndPosition() + 1);
      if (getIncludeEndPattern())
      {
         //TODO: HS the $ should be a special case for the regex extractor	
         int tempEndEnd = searchString.length() - 1;
         int tempEndStart = searchString.length() - 1;
         if (end.getEndPosition() > tempEndEnd || end.getStartPosition() > tempEndStart)
            adjustedEnd = new StringPosition(tempEndStart, tempEndEnd);
         else
            adjustedEnd = endPosition;
      }
      else
         adjustedEnd =
            new StringPosition(endPosition.getStartPosition() - 1, endPosition.getStartPosition() - 1);
      addMessageString(createAndPopulateMessageString(adjustedStart, adjustedEnd));
   }
   //
   //
   // The following are all convienence methods
   /**
    * createAndPopulateMessageString creates a single string between the start and end positions. 
    * If there are line breaks the line break symbol is inserted if provided.
    * @param start
    * @param end
    * @return
    */
   //
   private MessageString createAndPopulateMessageString(StringPosition start, StringPosition end)
   {

      int beginOffset = start.getStartPosition();
      int tailOffset = end.getEndPosition();
      if (beginOffset < 0 || tailOffset < 0)
         return null;
      MessageString result = new MessageString();
      //TODO: HS this is an ugly hack for logical end of line support
      if (beginOffset > tailOffset)
         result.setValue("");
      else
         result.setValue(searchString.substring(beginOffset,
         // add 1 to tailOffset because that is how substring works
         tailOffset + 1));
      return result;
   }
   /**
    * addMessageString adds the input MessageString to the array of MessageStrings. If needed the array will be 
    * compressed or extended as needed.
    * @param newString
    */
   private void addMessageString(MessageString newMessage)
   {
      if (newMessage == null)
         return;
      if (lastAddedMessage + 1 >= finalResult.length)
         increaseMessageStringArraySize();
      finalResult[++lastAddedMessage] = newMessage;
   }
   /**
    * increaseMessageStringArray doubles the size fo the current array being created
    */
   private void increaseMessageStringArraySize()
   {
      MessageString[] tempArray = new MessageString[messageArraySize * 2];
      System.arraycopy(finalResult, 0, tempArray, 0, messageArraySize);
      messageArraySize = messageArraySize * 2;
      finalResult = tempArray;
   }
   /**
    * getNextString returns the next input string awaiting processing.
    * @return
    */
   // 
   private String getNextString()
   {
      if (lastAddedString == -1 || lastRemovedString == lastAddedString)
         return null;
      return inputStrings[lastRemovedString + 1];
   }
   /**
    * addString adds the input String to the array of Stringsa toe be processed. If needed the array will be 
    * compressed or extended as needed.
    * @param newString
    */
   private void addString(String newString)
   {
      if (lastAddedString + 1 >= inputStrings.length)
         increaseStringArraySize();
      inputStrings[++lastAddedString] = newString;
   }
   /**
    * increaseArray doubles the size fo the current array being processed
    */
   private void increaseStringArraySize()
   {
      String[] tempArray = new String[arraySize * 2];
      arraySize = arraySize * 2;
      compressStringArrayInto(tempArray);
      inputStrings = tempArray;
   }
   /**
    * compressStringArrayInto compresses the current string array and moves the entries still 
    * active to the front of the array passed in. The input array is assumed to have enough room.
    * @param targetArray
    */
   private void compressStringArrayInto(String[] targetArray)
   {
      System.arraycopy(
         inputStrings,
         lastRemovedString + 1,
         targetArray,
         0,
         lastAddedString - lastRemovedString);
      lastAddedString = lastAddedString - lastRemovedString - 1;
      lastRemovedString = -1;
   }
   /**
    * search locates the pattern between the boundaries defined by the other parameters.
    * This method resolves a match pattern into a String that exists in the input.
    * To provide more complex pattern mathcing, this is the only method that needs to be
    * overridden.
    * @param pattern
    * @param arrayStartIndex
    * @param startPosition
    * @param arrayStopIndex
    * @param stopPosition
    * @return Position
    */
   protected StringPosition search(String pattern, int startPosition)
   {
      int foundPosition = -1;
      if (pattern != null)
      {
         foundPosition = searchString.indexOf(pattern, startPosition);
      }
      //		return result;
      if (foundPosition > -1)
         if (pattern.length() < 1)
            return new StringPosition(foundPosition, foundPosition);
         else
            return new StringPosition(foundPosition, foundPosition + pattern.length() - 1);
      else
         return null;
   }
   /**
    * Method buildString creates a single String from an input array of Strings
    * and inserts a line break character sequence between then as directed by the 
    * configuration.
    * @param arrayStartIndex
    * @param startPosition
    * @param arrayStopIndex
    * @param stopPosition
    * @return String
    */
   private String buildString(int arrayStartIndex, int startPosition, int arrayStopIndex, int stopPosition)
   {
      //TODO: HS change this to use StringBuffers
      String result = "";
      int copyStart, copyEnd;
      for (int i = arrayStartIndex; i <= arrayStopIndex; i++)
      {
         if (i == arrayStartIndex)
            copyStart = startPosition;
         else
         {
            copyStart = 0;
         }
         if (i == arrayStopIndex)
            copyEnd = stopPosition + 1;
         else
            copyEnd = inputStrings[i].length();
         if (i != arrayStartIndex)
         {
            if (getReplaceLineBreaks())
            {
               if (!(getLineBreakSymbol() == null))
                  result = result + getLineBreakSymbol();
            }
            else
               result = result + localLineSeparator;

         }

         result = result + inputStrings[i].substring(copyStart, copyEnd);
      }
      return result;
   }
   // inner classes for convience function
   /**
    * Class StringPosition is used as a simple structure for holding the 
    * begin and end offsets of a substring.
    */
   protected class StringPosition
   {
      private int startPosition, endPosition;
      StringPosition(int foundStartPosition, int foundEndPosition)
      {
         startPosition = foundStartPosition;
         endPosition = foundEndPosition;
      }
      int getStartPosition()
      {
         return startPosition;
      }
      int getEndPosition()
      {
         return endPosition;
      }
   }
   /**
    * Position extends StringPosition to allow the substring to span an array of Strings
    */
   protected class Position extends StringPosition
   {
      private int arrayIndex, startPosition, endPosition;
      Position(int foundArrayIndex, int foundStartPosition, int foundEndPosition)
      {
         super(foundStartPosition, foundEndPosition);
         arrayIndex = foundArrayIndex;
      }
      int getArrayIndex()
      {
         return arrayIndex;
      }
   }
}
