/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.PatternCompiler;
import org.apache.oro.text.regex.PatternMatcher;
import org.apache.oro.text.regex.Perl5Compiler;
import org.apache.oro.text.regex.Perl5Matcher;
import org.apache.oro.text.regex.Util;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.impl.ProcessUnit;
import org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This is the base implementation of a parser. 
 * 
 * @version 0.1
 * @since 0.1
 */
public class Parser extends ProcessUnit implements IParser {
	
	
	/*
	 * The regular expression compiler that can be used by all instances
	 * of the parser.  
	 */
	static private PatternCompiler compiler;
	static private PatternMatcher matcher;
	
	static {
		compiler=new Perl5Compiler();
		matcher=new Perl5Matcher();
		
		/* Initialize the matcher and the compiler */
		((Perl5Matcher)matcher).setMultiline(true);
		
	}
	
	/* The sets of rules this parser is configured to work with.
	 */
	private List ruleSets;
	 
	 /*
	  * The separator for fields within a message
	  */
	 private String separator;
	 
	 /*
	  * The hashing character(s).
	  */
	 private String hashing;
	 
	
	 private Pattern separatorPattern;
	
	/**
	 * 
	 * @return
	 */
	public static final PatternCompiler getCompiler() {
		return compiler;
	}
	
	public static PatternMatcher getPatternMatcher() {
		return matcher;
	}
	
	/**
	 * Overrides the update implementation provided by the ProcessUnit
	 * superclass.  Updates the configuration.
	 * 
	 * THIS IMPLEMENTATION IS TIGHTLY COUPLED WITH THE SCHEMA.  If the schema
	 * changes for the parser this will need to be changed accordingly.
	 * 
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig{
		
		/* Let our superclass update itself first */
		super.update();
		
		/* Get the configuration that the adapter assigned us.  We
		 * are actually handed the cc:Parser node.  We are interested
		 * in the Parser:Rules child andwe will only consider the first
		 * one.  There should be only one!
		 */
		Element element = getConfiguration();
		
		if(!element.getNodeName().equals(Messages.getString("HyadesGAParserTagName"))) {
			throw new AdapterInvalidConfig("cc:Parser expected to contain parsing rules");
		}
		
		
		
		/* Extract our configuration information from the config.
		 * RKD:  The original code from research created a RulesNode
		 * and told it to prepare itself using the element tht is the
		 * root of the configuration.  It then proceeded to delegate
		 * all of the work to the RulesNode.  I have changed that so
		 * that all of the sets of rules are held directly by this
		 * parser and each ruleset contains a set of rules.
		 */
		 
		NodeList rules=element.getChildNodes();
		
		
		separator=element.getAttribute(Messages.getString("HyadesGASeparatorTokenAttributeName"));
		hashing=element.getAttribute(Messages.getString("HyadesGADesignationTokenName"));
		
		if(separator!=null && !separator.equals("")) {
			try {
				separatorPattern=compiler.compile(separator);
			}
			catch(MalformedPatternException e) {
				throw new PreparationException(e);
			}
		}

		int ruleCount = rules.getLength();
		if (ruleCount >0) {
			ruleSets=new ArrayList();
			for (int i = 0; i < ruleCount; i++)	{
				Element rule=null;
				try {
					rule=(Element)rules.item(i);
				}
				catch(ClassCastException e) {
					/* We can ignore this child as it is not a Element */
					continue; 	 	
				}
				if(rule!=null) {
					IStatement statement=null;
					if(rule.getNodeName().endsWith(Messages.getString("HyadesGARuleElementTagName"))) {
						statement=new GroupStatement();
					}
					else if(rule.getNodeName().endsWith(Messages.getString("HyadesRuleAttributeTagName"))) {
						statement=new AttributeStatement();
					}
					else {
						throw new AdapterInvalidConfig();
					}
					
					/* Try and prepare the rules for processing */
					try {
						statement.prepare(rule, null);	
						ruleSets.add(statement);
					}
					catch(PreparationException e) {
						/* If the rule is bad we will log the fact it is bad and that we are ignoring this rule */
						log(Messages.getString("HyadesGAMalformedParserExpression")+" "+e.getDetails()+". ", AdapterSensor.CRITICAL_LEVEL);
						log(Messages.getString("HyadesGAMalformedParserExpressionIgnored")+" "+e.getDigraph(), AdapterSensor.CRITICAL_LEVEL);
					}
				}	
			}
		}
	}
	
	/**
	 * This implementation preocesses MessageString[] instances where each MessageString
	 * contains a single message.  
	 * @see org.eclipse.hyades.logging.adapter.IProcessUntit#processEventItems(java.lang.Object msgs)
	 */
	public Object[] processEventItems(Object[] msgs) {
		/* If we have no rules to run just return */
		if (ruleSets!=null)	{
			
			ArrayList collection=new ArrayList();
			
			for(int i=0; i<msgs.length; i++) {
				
				/* Don't try and process null messages */
				if(msgs[i]==null) {
					continue;
				}
				/* Prepare the line by first breaking it up into tokens.  We do this once
				 * then run the resulting list against each of the rules
				 */
				HashMap map=prepareEnvData(((MessageString)msgs[i]).getValue());
				
				/* Here we iterate through each of the rules..wouldn't it be better
				 * if we just ran the rules with each of the tokens until we get a match..could there be multiple
				 * match statements..this if Big-O n^2, we should be able to make this 
				 * linear
				 */
				Iterator iter=ruleSets.listIterator();
				
				ArrayList result=new ArrayList();
				
				while(iter.hasNext()) {
					IStatement statement=(IStatement)iter.next();
					statement.run(((MessageString)msgs[i]).getValue(), map, result);
				}
				
				/* Add the hashmap to the arraylist */
				collection.add(result);
			}
			return collection.toArray();
		}
		return null;
	}
	
	/**
	 * 
	 * @see org.eclipse.hyades.logging.adapter.IProcessUntit#processEventItems(java.lang.Object msgs)
	 */
	public Object[] testProcessEventItems(Object[] msgs) throws AdapterInvalidConfig {
		if(!(msgs instanceof MessageString[])) {
			throw new AdapterInvalidConfig("This parser will only accept arrays of MessageString");
		}
		return msgs;		
	}
	
	/**
	 * Separate the message into individual properties based upon the separator.  
	 * This method will split the names and vales apart if there is a '=' or ':'
	 * in the name.  This is probably incorrect behaviour and this should use only the
	 * hashing string.
	 * 
	 * @param line
	 * @param env
	 */
	private HashMap prepareEnvData(String line) {
		
		HashMap result=new HashMap();
		
		List list = new ArrayList();
		
		/* If we have no separatorToken we just return */
		if(separator==null || separator.equals("")) {
			return result;
		}
		
		/* We use the split method to split up the 
		 * line into tokens.
		 */
		Util.split(list,matcher,separatorPattern,line);
		
		/* Iterate over each of the tokens and ensure there are no quotes */
		ListIterator iterator=list.listIterator();
		boolean startsWithQuote=false;
		boolean insideHash=false;
	
		int i1 = -1;
		int start = 0, len = line.length();
		
		while(iterator.hasNext()) {

			/* The key and the value to place in the hash */
			StringBuffer hashVal = new StringBuffer();
			String hashKey = "";	 
		
			String val="";
			
			String current=(String)iterator.next();
			// Break our message up based upon the hashing substring
			if((i1=current.indexOf(hashing))>=0 || insideHash)  {
				if (i1<current.length()-1) {
					if (!insideHash && i1>=0) {
						hashKey = current.substring(0,i1);
						val = current.substring(i1+1);
					}
					if (!insideHash && (val.startsWith("\"") || val.startsWith("\\'")) ) {
						insideHash=true;
						val=val.substring(1);
	
						if (val.startsWith("\"") || val.startsWith("\\'")) 
						{
							startsWithQuote=true;
						}				
					}
				}
				if (startsWithQuote) {
					if (val.endsWith("\"") || val.endsWith("\\'")) {
						hashVal.append(val.substring(0, val.length()-1));
						insideHash = false;	
						result.put(hashKey.trim(),hashVal.toString().trim());	 
						startsWithQuote=false;
					}
					else {
						hashVal.append(val);
						if (start+1<len) {
							hashVal.append(line.substring(start,start+1));
						} 	
					}
				}		  
				else {	
					insideHash = false;	
					result.put(hashKey.trim(),val.trim());
				}	
			}
		}
		return result;
	}

}
