/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;

/**
 * This class provides the ability to process 'Position' attribute
 * values.
 * 
 * @author rduggan
 * @version 0.1
 * @since 0.1
 */
public class PositionParser {
	
	/* The separator for position statements */
	static final String POSITION_SEPARATOR = "@@";
	
	/* The hash start for position statements */
	static final String HASH_START = "$h(";
	
	/* The hash end for position statements */
	static final String HASH_END = ")";


	/**
	 * Parse a position attribute value and return the tokens that are embedded within
	 * the string.
	 * 
	 * @param strPositions - the position string to parse
	 * @param bStripHash - whether to strip the hash information from the attributes 
	 * @return - the tokenized position information
	 * @since 0.1
	 */
	public static List getPositionedString(String strPositions, boolean bStripHash) {
		if(strPositions==null) {
			return null; 
		} 
		
		List list = new ArrayList();
		List newList=new ArrayList();
		
		/* Tokenize the string based upon the separator */
		StringTokenizer s = new StringTokenizer(strPositions, POSITION_SEPARATOR);
	    while (s.hasMoreTokens()) {
				list.add(s.nextToken());
		}
		
		/* Iterate over the tokenized string and process each token. */
		ListIterator iterator=list.listIterator();
		while(iterator.hasNext()) {
			String current=(String)iterator.next();
			if(current!=null) {
				current.trim();
				if(!current.equals("")) {
					Object result=null;
					
					/* If we have to strip the hash information do it now */ 
					if(bStripHash && isHashPosition(current)) {
						result = current.substring(4,current.length()-2).trim();
					}
					else {
						/* TODO: RKD:  Why is this changed to a numeric here? */
						try { 
							result=Long.decode(current);
						}
						catch(Exception e) {
							/* This is not a number, ignore it */
						}
					}
					
					if(result!=null) {
						newList.add(result);
					}
					else if(!bStripHash) {
						newList.add(current);
					}
				}	
			}
		}
		return newList;
	  }

	/**
	 * Determines if this token is a hashed position token.
	 * 
	 * @param text
	 * @return
	 * @since 0.1
	 */
	private static boolean isHashPosition(String text) {
	  if (text !=null) {
		 return text.startsWith(HASH_START) && text.endsWith(HASH_END);
	  }
	  return false;		
	}

}
