/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;

import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.Perl5Substitution;
import org.apache.oro.text.regex.Substitution;
import org.apache.oro.text.regex.Util;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.Element;

/**
 * @author rduggan
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class SubstituteStatement implements IStatement {
	
	/* A single csubstitution to be used by all SustituteStatements
	 */
	static private Substitution substitution=new Perl5Substitution();
	
	/* The fully qualified name for this substitution rule */
	String name;
	
	/* The positions string as specified in the rule */
	private String positions;
	
	/* The match string as specified in the rule */
	private String match;
	
	/* The substitute string as specified in the rule */
	private String substitute;
	
	/* The compiled pattern for this SubstitutionStatement */
	private Pattern pattern;

	/* The list of positions that will be built based upon the positions string */
	private List positionsList;
	
	/* The builtin function indicator */
	boolean useBuiltin;
	
	/* The parsed path for this rule.  This is loaded into the IDirectedGraph to identify 
	 * this rule uniquely to the next component in the chain.
	 */
	private List path;
	
	private IDirectedGraph resultGraph;

	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
		positions = node.getAttribute(Messages.getString("HyadesGAPositionsAttributeName"));
		match=node.getAttribute(Messages.getString("HyadesGAMatchAttributeName"));
		substitute=node.getAttribute(Messages.getString("HyadesGASubstituteAttributeName"));
		
		String builtin=node.getAttribute(Messages.getString("HyadesGAuseBuiltInFunctionAttributeName"));
		if(builtin!=null && !builtin.equals("")) {
			if(builtin.equals("true")) {
				useBuiltin=true;
			}
			else {
				useBuiltin=false;
			}
		}
		else {
			useBuiltin=false;
		}
		
		/* Save the name.  We actaully never use this so we don't need this variable */
		this.name=path;
		
		/* Clear the result graph */
		resultGraph=null;
		
		/* Store the path as a list. The path may contain indexes for elements
		 * which are multiples.
		 */
		this.path=new ArrayList();
		StringTokenizer s = new StringTokenizer(path, ".");
		while (s.hasMoreTokens()) {
			String subPath=s.nextToken();
			int indexStart=subPath.indexOf('[');
			if(indexStart>0) {
				String index=subPath.substring(indexStart+1, subPath.length()-1);
				subPath=subPath.substring(0, indexStart);
				this.path.add(subPath);
				this.path.add(index);
			}
			else {
				this.path.add(subPath);
			}
		}
		
		/* Parse the positions attribute value into a tokenized list */
		if(positions!=null) {
			positionsList=PositionParser.getPositionedString(positions, true);
		}
		
	
		/* RKD:  The initial research code did not compile the match statement here */
		if (match!=null && !match.equals("")) {
			try {
				pattern=Parser.getCompiler().compile(match);
			}
			catch(MalformedPatternException e) {
				/* This is a badly formed regular expression.  Throw it as an error */
				PreparationException details=new PreparationException(Messages.getString("HyadesGAMalformedParserExpression"), e);
				details.setDetails(e.getLocalizedMessage()+" \""+match+"\"|");
				details.setDigraph(this.name);
				throw details;
			}				
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String)
	 */
	public boolean run(String line, HashMap inputs, List outputs) {
		String newLine;
		boolean result=false;
		
		
		/* If we are running a builtin function then ignore the reset of the content */
		if(useBuiltin) {
			resultGraph=new DirectedGraphImpl();
			resultGraph.setPath(this.path);
			resultGraph.setValue("##BUILTIN");
			outputs.add(resultGraph);
			return true;
		}
		
		/* If we have a positions string modify our inpts accordingly */
		if(positions!=null && !positions.equals("")) {
			line=positionsToString(line, inputs);
		}
		
		/* If we have a match pattern to look for then lets get the results.  Otherwise
		 * we just place the substitution string in the value.
		 */
		if (pattern!=null && !pattern.equals(""))	{
			newLine=matchAndSubstitute(line);
			if (newLine!=null && line.length()==newLine.length()) {
				if(!Parser.getPatternMatcher().contains(line,pattern)) {
					result=false;
				} 	
			}
			result=(newLine!=null);	
		}
		else {
			if (substitute!=null && !substitute.equals("")) {
				newLine=substitute;	
			}
			else {
				newLine=line;	
			}
			result=true;
		}
		
		
		/* If we have a match add this match to the result set.  This
		 * is quite a bit of object creation.  Perhaps we could clean this
		 * up some more.
		 */
		if(result) {
			/* RKD:  We are instantiating a new instance here each pass.  This isn't
			 * necessary providing the contract between the Parser and the next element
			 * in the chain is on the same thread and the contract states they will
			 * not be changed.  I am leaving the instantiation because I believe we should
			 * support each component running in its own thread. 
			 */
			resultGraph=new DirectedGraphImpl();
			resultGraph.setPath(this.path);
			resultGraph.setValue(newLine);
			outputs.add(resultGraph);
			return true;
		}
		else {
			resultGraph=null;
		}
		
		return false;
	}
	
	private String matchAndSubstitute (String line) {
		if(substitute!= null && !substitute.equals("")) {
			if(!Parser.getPatternMatcher().contains(line,pattern)) {
				return null;
			} 
			((Perl5Substitution)substitution).setSubstitution(substitute);
			String result = Util.substitute(Parser.getPatternMatcher(),pattern,substitution,line, Util.SUBSTITUTE_ALL);
			return result.trim(); 
		}
		return null;
		
	}
	
	/**
	 * This method iterates over the positions string and builds a new string
	 * based upon the environment entires.  For each position that is a key into
	 * the enviornment we extract the value and append it to the previous.  All 
	 * of the values are separated by the postion separator.
	 * 
	 * @param line
	 * @param inputs
	 * @return
	 */
	private String positionsToString(String line, HashMap inputs) {
		if (positionsList!=null)	{
			
			/* Iterate over the tokenized postion information and build
			 * a new line
			 */
			 StringBuffer result=new StringBuffer();
			 ListIterator iterator=positionsList.listIterator();
			 boolean first=true;
			 boolean emptyString=false;
			 while(iterator.hasNext()) {
			 	/* Grab the next token.  It may be a String or a Long */
			 	Object current=iterator.next();
			 	if(first) {
					first=!first;	
			 	}
			 	else {
			 		result.append(PositionParser.POSITION_SEPARATOR);
			 	}
			 	if(current instanceof String) {
					String val=(String)inputs.get(current);
					if (val!=null && val.length()==0) {
						emptyString=true;
					}
					else {
						result.append(val);
					}
			 	}
			 	else if(current instanceof Long) {
					Long item = (Long)current;
					String val=null;
					int index = item.intValue();
				
					if (index >=0 && index < positionsList.size()) {
						val = (String)positionsList.get(index);
						result.append(val);
					}

			 	}
			 }
			if(result.length()>0)	{
				return result.toString();
			}
			else if(emptyString) {
				return "";
			}
		}
		return line;	
	}

	/**
	 * Retrieve the IDirectedGraph created as a result of this rull being run.
	 * @return
	 */
	public IDirectedGraph getResultGraph() {
		return resultGraph;
	}

}
