package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The AvailableSituation deals with the situations reported from the component, 
 * regarding its operational state and availability. This situation provides a 
 * context for operations that can be performed on the component by distinguishing 
 * if a product is installed, operational and ready to process functional requests, 
 * or operational and ready/not ready to process management requests. Existing message 
 * include words like those that now ready to take requests, online, and offline, for 
 * example:
 * 
 * <code>
 * ADMC0013I: SOAP connector available at port 8880
 * ADMC0026I: RMI Connector available at port 2809
 * </code>
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 * @see IStartSituation
 * @see IAvailableSituation
 */
public class AvailableSituationImpl extends SituationTypeImpl implements IAvailableSituation, Serializable, Cloneable {

    //Define the serialiVersionUID for serialization:
    //NOTE: If this object is ever changed which breaks backward compatibility,
    //this value MUST be changed.  Every effort should be made not to break 
    //backward compatibility.
    static final long serialVersionUID = -3888096669772662967L;

    //Define the fields that will be serialized: 
    //NOTE: If any new variables are added to the class this table should be 
    //updated along with the readObject and writeObject methods.
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("operationDisposition", String.class), new ObjectStreamField("processingDisposition", String.class), new ObjectStreamField("availabilityDisposition", String.class)};

    //Class instance variable(s):
    protected String operationDisposition = null;
    protected String processingDisposition = null;
    protected String availabilityDisposition = null;

    /**
     * No-argument protected constructor for AvailableSituationImpl.
     */
    protected AvailableSituationImpl() {
		super();
    }

    /**
     * Returns the operationDisposition.
     * 
     * This property specifies the operation state of the component reported 
     * by the situation. The operationalDisposition is a string with the 
     * following set of values:
     * 
     * STARTABLE
     * NONSTARTABLE
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @return String The returned operationDisposition.
     * @since 1.0.1
     */
    public String getOperationDisposition() {
        return operationDisposition;
    }

    /**
     * Returns the processingDisposition.
     * 
     * This property specifies the processing disposition of a component 
     * opertation that caused the situation to be reported.  The 
     * processingDisposition is a string with the following set of values:
     * 
     * FUNCTION_PROCESS
     * FUNCTION_BLOCK
     * MGMTTASK_PROCESS
     * MGMTTASK_BLOCKED
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @return String The returned processingDisposition.
     * @since 1.0.1
     */
    public String getProcessingDisposition() {
        return processingDisposition;
    }

    /**
     * Returns the availabilityDisposition.
     * 
     * This property specifies the availability disposition of an entity or 
     * component that caused the situation to be reported.  The availableDisposition 
     * is a string with the following set of values:
     * 
     * AVAILABLE
     * NOT AVAILABLE
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @return String The returned availabilityDisposition.
     * @since 1.0.1
     */
    public String getAvailabilityDisposition() {
        return availabilityDisposition;
    }

    /**
     * Sets the operationDisposition.
     * 
     * This property specifies the operation state of the component reported 
     * by the situation. The operationalDisposition is a string with the 
     * following set of values:
     * 
     * STARTABLE
     * NONSTARTABLE
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @param operationDisposition The operationDisposition to be set.
     * @since 1.0.1
     */
    public void setOperationDisposition(String operationDisposition) {
        this.operationDisposition = operationDisposition;
    }

    /**
     * Sets the processingDisposition.
     * 
     * This property specifies the processing disposition of a component 
     * opertation that caused the situation to be reported.  The 
     * processingDisposition is a string with the following set of values:
     * 
     * FUNCTION_PROCESS
     * FUNCTION_BLOCK
     * MGMTTASK_PROCESS
     * MGMTTASK_BLOCKED
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @param processingDisposition The processingDisposition to be set.
     * @since 1.0.1
     */
    public void setProcessingDisposition(String processingDisposition) {
        this.processingDisposition = processingDisposition;
    }

    /**
     * Sets the availabilityDisposition.
     * 
     * This property specifies the availability disposition of an entity or 
     * component that caused the situation to be reported.  The availableDisposition 
     * is a string with the following set of values:
     * 
     * AVAILABLE
     * NOT AVAILABLE
     * 
     * This is a REQUIRED property and once it is set it MUST NOT change. 
     * The string length for this property MUST NOT exceed 64 characters.
     * 
     * @param availabilityDisposition The availabilityDisposition to be set.
     * @since 1.0.1
     */
    public void setAvailabilityDisposition(String availabilityDisposition) {
        this.availabilityDisposition = availabilityDisposition;
    }

    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return true if this object is the same as the parameter <code>object</code>, false otherwise.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        //Check if the parameter object's super class is equal to the this object's super class:
        if (super.equals(object)) {

            AvailableSituationImpl availableSituationImpl = ((AvailableSituationImpl) (object));

            //Compare the value(s) of all object properties:
            String thisObjectsProperty = getOperationDisposition();
            String parameterObjectsProperty = availableSituationImpl.getOperationDisposition();

            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                thisObjectsProperty = getProcessingDisposition();
                parameterObjectsProperty = availableSituationImpl.getProcessingDisposition();

                if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                    thisObjectsProperty = getAvailabilityDisposition();
                    parameterObjectsProperty = availableSituationImpl.getAvailabilityDisposition();

                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null)))
                        return true;
                }
            }
        }

        return false;
    }
    private void writeObject(ObjectOutputStream out) throws IOException {

        //Set the values of the serializable field(s):
        ObjectOutputStream.PutField fields = out.putFields();

        fields.put("operationDisposition", operationDisposition);
        fields.put("processingDisposition", processingDisposition);
        fields.put("availabilityDisposition", availabilityDisposition);

        //Write the values of the serializable field(s):
        out.writeFields();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

        //Prepare and read the alternate persistent field(s):
        ObjectInputStream.GetField fields = null;

        fields = in.readFields();

        //Retrieve the alternate persistent field(s):
        operationDisposition = ((String) (fields.get("operationDisposition", operationDisposition)));
        processingDisposition = ((String) (fields.get("processingDisposition", processingDisposition)));
        availabilityDisposition = ((String) (fields.get("availabilityDisposition", availabilityDisposition)));
    }

    /**
     * Return a clone (copy by value) of this object. This is a deep copy version, 
     * in which all the objects within this object will be copied by value.
     *  
     * @return Object The clone of this object.
     */
    public Object clone() throws CloneNotSupportedException {
        return (super.clone());
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * @see org.eclipse.hyades.logging.events.ISituationType#init()
     */
    public void init() {

        super.init();

        operationDisposition = null;
        processingDisposition = null;
        availabilityDisposition = null;
    }
}