package org.eclipse.hyades.logging.events;

import org.eclipse.hyades.logging.core.Guid;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * This is the concrete implementation of the SimpleEventFactory.  It
 * implements the ISimpleEventFactory interface.
 * 
 * The SimpleEventFactory is a singleton as prescribed by the 
 * Abstract Factory pattern (Gamma et.al.).  It contains simple
 * constructors to build CommonBaseEvent and related complex 
 * type objects.
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author John K. Gerken
 * @version 1.0.1
 * @since 1.0
 */
public class SimpleEventFactoryImpl implements ISimpleEventFactory {

    private static final SimpleEventFactoryImpl instance = new SimpleEventFactoryImpl();

	/**
	 * Prefix added to all GUIDS produced from this class.
	 * 
	 * @see #createGlobalInstanceId()
	 */
	private static final String GUID_PREFIX = "CBE";

    /**
     * Constructor for SimpleEventFactoryImpl.
     */
    private SimpleEventFactoryImpl() {
        super();
    }

    /**
     * Method IMsgCatalogToken.
     * 
     * Create and return a concrete MsgCatalogToken object.  Objects
     * of this type are encapsulated into MsgDataElement objects.
     * 
     * @return IMsgCatalogToken
     */
    public IMsgCatalogToken createMsgCatalogToken() {
        return new MsgCatalogTokenImpl();
    }

    /**
     * Method IAssociationEngine.
     * 
     * Create and return a concrete AssociationEngine object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IAssociationEngine
     */
    public IAssociationEngine createAssociationEngine() {
        return new AssociationEngineImpl();
    }

    /**
     * Method createCommonBaseEvent.
     * 
     * Create and return a concrete CommonBaseEvent object.
     * 
     * @return ICommonBaseEvent
     */
    public ICommonBaseEvent createCommonBaseEvent() {
        return new CommonBaseEventImpl();
    }

    /**
     * Method createExtendedDataElement.
     * 
     * Create and return a concrete ExtendedDataElement object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IExtendedDataElement
     */
    public IExtendedDataElement createExtendedDataElement() {
        return new ExtendedDataElementImpl();
    }

    /**
     * Method createAssociatedEvent.
     * 
     * Create and return a concrete AssociatedEvent object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IAssociatedEvent
     */
    public IAssociatedEvent createAssociatedEvent() {
        return new AssociatedEventImpl();
    }

    /**
     * Method createContextDataElement.
     * 
     * Create and return a concrete ContextDataElement object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IContextDataElement
     */
    public IContextDataElement createContextDataElement() {
        return new ContextDataElementImpl();
    }

    /**
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createAnyElement()
     * @deprecated Use the otherData as of 1.0.1.
     */
    public ICommonBaseEventAnyElement createAnyElement() {
        return new CommonBaseEventAnyElementImpl();
    }

    /**
     * Method createComponentIdentification.
     * 
     * Create and return a concrete ComponentIdentification object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IComponentIdentification
     */
    public IComponentIdentification createComponentIdentification() {
        return new ComponentIdentificationImpl();
    }

    /**
     * Method createMsgDataElement.
     * 
     * Create and return a concrete MsgDataElement object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IMsgDataElement
     */
    public IMsgDataElement createMsgDataElement() {
        return new MsgDataElementImpl();
    }

    /**
     * Method createMsgDataElement.
     * 
     * Create and return a concrete Situation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return ISituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createSituation()
     */
    public ISituation createSituation() {
        return (new SituationImpl());
    }

    /**
     * Method createStartSituation.
     * 
     * Create and return a concrete StartSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IStartSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createStartSituation()
     */
    public IStartSituation createStartSituation() {
        return (new StartSituationImpl());
    }

    /**
     * Method createStopSituation.
     * 
     * Create and return a concrete StopSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IStopSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createStopSituation()
     */
    public IStopSituation createStopSituation() {
        return (new StopSituationImpl());
    }

    /**
     * Method createConnectSituation.
     * 
     * Create and return a concrete ConnectSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IConnectSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createConnectSituation()
     */
    public IConnectSituation createConnectSituation() {
        return (new ConnectSituationImpl());
    }

    /**
     * Method createReportSituation.
     * 
     * Create and return a concrete ReportSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IReportSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createReportSituation()
     */
    public IReportSituation createReportSituation() {
        return (new ReportSituationImpl());
    }

    /**
     * Method createFeatureSituation.
     * 
     * Create and return a concrete FeatureSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IFeatureSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createFeatureSituation()
     */
    public IFeatureSituation createFeatureSituation() {
        return (new FeatureSituationImpl());
    }

    /**
     * Method createConfigureSituation.
     * 
     * Create and return a concrete ConfigureSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IConfigureSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createConfigureSituation()
     */
    public IConfigureSituation createConfigureSituation() {
        return (new ConfigureSituationImpl());
    }

    /**
     * Method createDependencySituation.
     * 
     * Create and return a concrete DependencySituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IDependencySituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createDependencySituation()
     */
    public IDependencySituation createDependencySituation() {
        return (new DependencySituationImpl());
    }

    /**
     * Method createCreateSituation.
     * 
     * Create and return a concrete CreateSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return ICreateSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createCreateSituation()
     */
    public ICreateSituation createCreateSituation() {
        return (new CreateSituationImpl());
    }

    /**
     * Method createDestroySituation.
     * 
     * Create and return a concrete DestroySituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IDestroySituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createDestroySituation()
     */
    public IDestroySituation createDestroySituation() {
        return (new DestroySituationImpl());
    }

    /**
     * Method createAvailableSituation.
     * 
     * Create and return a concrete AvailableSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IAvailableSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createAvailableSituation()
     */
    public IAvailableSituation createAvailableSituation() {
        return (new AvailableSituationImpl());
    }

    /**
     * Method createRequestSituation.
     * 
     * Create and return a concrete RequestSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IRequestSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createRequestSituation()
     */
    public IRequestSituation createRequestSituation() {
        return (new RequestSituationImpl());
    }

    /**
     * Method createOtherSituation.
     * 
     * Create and return a concrete OtherSituation object.  Objects
     * of this type are encapsulated into CommonBaseEvent objects.
     * 
     * @return IOtherSituation
     * @since 1.0.1
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createOtherSituation()
     */
    public IOtherSituation createOtherSituation() {
        return (new OtherSituationImpl());
    }

    /**
     * Method getInstance.
     * 
     * Accessor method to retrieve the instance of the singleton factory.
     * @return SimpleEventFactoryImpl
     */
    public static ISimpleEventFactory getInstance() {
        return instance;
    }

    /**
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createCommonBaseEvent(java.lang.String, long)
     */
    public ICommonBaseEvent createCommonBaseEvent(String extensionName, long creationTime) {

        ICommonBaseEvent commonBaseEvent = new CommonBaseEventImpl();
        commonBaseEvent.setExtensionName(extensionName);
        commonBaseEvent.setCreationTime(creationTime);

        return commonBaseEvent;
    }

    /**
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createCommonBaseEvent(java.lang.String)
     */
    public ICommonBaseEvent createCommonBaseEvent(String extensionName) {

        ICommonBaseEvent commonBaseEvent = new CommonBaseEventImpl();
        commonBaseEvent.setExtensionName(extensionName);

        return commonBaseEvent;
    }

    /**
     * @see org.eclipse.hyades.logging.events.ISimpleEventFactory#createGlobalInstanceId()
     */
    public String createGlobalInstanceId() {
		return (GUID_PREFIX.concat(new Guid().toString()));
    }
}
