/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.CorrelationContainer;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.CorrelationEngine;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.impl.CorrelationEngineImpl;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;

public class LoadersUtils {
    //~ Static fields/initializers -----------------------------------------------------------------

    private final static Map threads = new HashMap();

    private static SimpleDateFormat partialCIMDateFormatDashDelimiter = new SimpleDateFormat("yyyy'-'MM'-'dd'T'HH:mm:ss");
    private static SimpleDateFormat partialCIMDateFormatPlusDelimiter = new SimpleDateFormat("yyyy'+'MM'+'dd'T'HH:mm:ss");
    private static SimpleDateFormat partialCIMDateFormatNoDelimiter = new SimpleDateFormat("yyyyMMdd'T'HH:mm:ss");

	private final static Map correlationMap = new HashMap();

    //~ Methods ------------------------------------------------------------------------------------

    public static void setDefaultContext(HierarchyContext context) {
        TRCMonitor monitor = context.getMonitor();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();

        /* If we don't know the node, place it in the default file */
        if (node == null) {
            /* If we already have a default node we use it */
            node = LoadersUtils.locateNodeFromMonitor(monitor, "Unknown", "0", "0");

            if (node == null) {
                EList nodeExt = new BasicEList();
                EList nExt = new BasicEList();
                String monitorFolder = getMonitorFolder(monitor);
                Resource nDoc = null;

                if (monitorFolder != null) {
                    String nName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + "Unknown";

                    String nodeFileName = monitorFolder + nName;

                    nDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(nodeFileName + ".trcnxmi")).createResource(SaveUtil.createURI(nodeFileName + ".trcnxmi"));
                    HierarchyResourceSetImpl.getInstance().getResources().add(nDoc);

//                    //*** adding support for multiple files
//                    SaveUtil.addDocument(nDoc);
                }

                node = HierarchyFactory.eINSTANCE.createTRCNode();
                node.setIpAddress("0");
                node.setRuntimeId("0");
                node.setName("Unknown");
                monitor.getNodes().add(node);
                node.setMonitor(monitor);
                context.setNode(node);

                if (nDoc != null) {
                    nDoc.getContents().add(node);
                    nDoc.setModified(true);
                }
            }

            context.setNode(node);
        }

        if (process == null) {
            /* Do we already have a default process? If not create it */
            process = LoadersUtils.locateProcessFromNode(node, "0");

            if (process != null) {
                context.setProcessProxy(process);

                return;
            }

            //create the process
            EList procExt = new BasicEList();
            String monitorFolder = getMonitorFolder(monitor);
            Resource procDoc = null;

            if (monitorFolder != null) {
                String pName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + node.getName().trim() + "_" + "-1" /* The _pid */
                +"_" + "0"; /* The _processId */

                String pFileName = monitorFolder + pName;

                procDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcpxmi")).createResource(SaveUtil.createURI(pFileName + ".trcpxmi"));
                HierarchyResourceSetImpl.getInstance().getResources().add(procDoc);

//                //*** adding support for multiple files
//                SaveUtil.addDocument(procDoc);
            }

            process = HierarchyFactory.eINSTANCE.createTRCProcessProxy();
            node.getProcessProxies().add(process);
            process.setNode(node);
            process.setPid(-1);
            process.setRuntimeId("0");
            process.setName("Unknown");

            context.setProcessProxy(process);

            if (procDoc != null) {
                procDoc.getContents().add(process);
                procDoc.setModified(true);
            }
        }
    }

    /**
     * Returns the signed time zone offset from GMT or Zulu time in minutes of the parameter XML schema dateTime datatype time stamp.
     * 
     * Assumption:  The caller is responsible for removing any leading and/or trailing whitespace.
     * 
     * @param creationTime XML schema dateTime datatype time stamp.
     * @return short Signed time zone offset from GMT in minutes.
     * @throws IllegalArgumentException The parameter time stamp is not a valid XML schema dateTime datatype. 
     */
    public static short getDeltaTimeZoneInMinutesFromCIMFormat(String creationTime) throws IllegalArgumentException {

        //XML schema dateTime datatype (http://www.w3.org/TR/xmlschema-2/#dateTime):
        //y...y[-/+]MM[-/+]ddTHH:mm:ss[.S...S][-/+HH:mm]/[Z]
        //For example: 2003-06-03T15:48:03.783455-04:00
        //             2003-06-03T19:48:03.12Z
        //             2003+06+03T19:48:03Z
        //             2000030603T19:48:03Z

        //Calculate the length of the creationTime time stamp:
        int creationTimeLength = creationTime.length();

        short timeZoneOffset = 0;

        //If the time stamp is in UTC (i.e. GMT), return a zero offset:
        if (creationTime.charAt(creationTimeLength - 1) != 'Z') {

            try {

                //Parse the time zone hour offset (i.e. [HH]):
                timeZoneOffset = Short.parseShort(creationTime.substring((creationTimeLength - 5), (creationTimeLength - 3)));

                //Convert the time zone hour offset to minutes:
                timeZoneOffset *= 60;

                //Parse the time zone minute offset (i.e. [mm]):
                timeZoneOffset += Short.parseShort(creationTime.substring(creationTimeLength - 2));

                //Change the sign of the time zone offset:
                if (creationTime.charAt(creationTimeLength - 6) == '-')
                    timeZoneOffset *= -1;
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid parameter time stamp format.");
            }
        }

        return timeZoneOffset;
    }

    /**
     * Returns the microsecond representation of the parameter XML schema dateTime datatype time stamp, 
     * adjusted for the local time zone.
     * 
     * Assumption:  The caller is responsible for removing any leading and/or trailing whitespace.   
     * 
     * @param creationTime XML schema dateTime datatype time stamp.
     * @return double Microsecond representation of the parameter XML schema dateTime datatype time stamp.
     * @throws IllegalArgumentException The parameter time stamp is not a valid XML schema dateTime datatype. 
     */
    public static double getMicrosecondsFromCIMFormat(String creationTime) throws IllegalArgumentException {

        //XML schema dateTime datatype (http://www.w3.org/TR/xmlschema-2/#dateTime):
        //y...y[-/+]MM[-/+]ddTHH:mm:ss[.S...S][-/+HH:mm]/[Z]
        //For example: 2003-06-03T15:48:03.783455-04:00
        //             2003-06-03T19:48:03.12Z
        //             2003+06+03T19:48:03Z
        //             2000030603T19:48:03Z

        //Microsecond representation of the creationTime time stamp:
        double creationTimeInMicros = 0.0;

        //'T' character is the reference index in the creationTime since this character is required according to the XML schema dateTime datatype specification:
        //NOTE: The reference character (i.e. 'T') is required due to variable length year and fractional second fields.
        int tCharIndex = creationTime.indexOf('T');

        //creationTime must have the reference character (i.e. 'T') according to the XML schema dateTime datatype specification:
        if (tCharIndex != -1) {
            try {
                //For performance reasons, attempt to parse the date and time (second granularity) part of the creationTime with the year/month/day delimiter as the '-' character (most common format):
                //NOTE:  1 millisecond = 1000 microseconds
                creationTimeInMicros = (partialCIMDateFormatDashDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);
            }

            //The year/month/day delimiter is not the '+' character:
            catch (ParseException p) {

                try {

                    //If the year/month/day delimiter is not the '-' character, check if the year/month/day delimiter is the '+' character:
                    //NOTE:  1 millisecond = 1000 microseconds
                    if (creationTime.charAt(tCharIndex - 3) == '+')
                        creationTimeInMicros = (partialCIMDateFormatPlusDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);

                    //If the year/month/day delimiter is not neither the '-' or '+' characters, no year/month/day delimiter is used:
                    //NOTE:  1 millisecond = 1000 microseconds
                    else
                        creationTimeInMicros = (partialCIMDateFormatNoDelimiter.parse(creationTime.substring(0, (tCharIndex + 9))).getTime() * 1000.0);
                }
                catch (ParseException pp) {
                    throw new IllegalArgumentException("Invalid parameter time stamp format.");
                }
            }

            //Convert the microsecond representation of the creationTime to UTC (i.e. GMT or Zulu time):
            //NOTE:  1 minute = 60000000 microseconds
            creationTimeInMicros -= (((double) (getDeltaTimeZoneInMinutesFromCIMFormat(creationTime)) * 60000000.0));

            //Retrieve, calculate and add the fractional second value of the creationTime if exists:
            try {

                //Retrieve the fractional second value of the creationTime if exists (i.e. [.S...S] and convert to microseconds:
                //NOTE:  1 second = 1000000 microseconds
                creationTimeInMicros += (Double.parseDouble("0.".concat(creationTime.substring((tCharIndex + 10), (creationTime.endsWith("Z") ? (creationTime.length() - 1) : (creationTime.length() - 6))))) * 1000000.0);
            }
            catch (Exception e) {
                //The fractional second is not included in this creationTime:
            }
        }
        else
            throw new IllegalArgumentException("Invalid parameter time stamp format.");

        return creationTimeInMicros;
    }

    // get the folder name of a monitor
    public static String getMonitorFolder(TRCMonitor monitor) {
        if (monitor == null) {
            return null;
        }

        Resource res = monitor.eResource();

        if (res == null) {
            return null;
        }

        String uri = res.getURI().toString();

        uri = uri.replace('\\', '/');

        int index = uri.lastIndexOf("/");

        if (index >= 0) {
            uri = uri.substring(0, index);
        }

        if (uri.startsWith("/")) {
            uri = uri.substring(1);
        }

        if (!uri.endsWith("/")) {
            uri += "/";
        }

        return uri;
    }

    /**
     * @deprecated Use getPartialCIMDateFormatDashDelimiter().
     */
    public static SimpleDateFormat getPartialCIMDateFormat() {
        return (getPartialCIMDateFormatDashDelimiter());
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatDashDelimiter() {
        return partialCIMDateFormatDashDelimiter;
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatPlusDelimiter() {
        return partialCIMDateFormatPlusDelimiter;
    }

    /**
     * @return SimpleDateFormat
     */
    public static SimpleDateFormat getPartialCIMDateFormatNoDelimiter() {
        return partialCIMDateFormatNoDelimiter;
    }

    /**
     * @param value
     * @return
     */
    public static double convertCreationTime(String value) {
        return getMicrosecondsFromCIMFormat(value);
    }

    /**
     * @return
     */
    public static TRCAgent getOrCreateAgent(HierarchyContext context) {
    	TRCAgent agent = context.getAgent();
    	if(agent==null)
    	{
    		agent = LoadersUtils.createAgent(context);
    	}
    	return agent;
    }
    
    
    /**
     * @param context
     * @return
     */
    public static TRCAgent createAgent(HierarchyContext context) {
        setDefaultContext(context);

        TRCAgent agent = context.getAgent();
        TRCAgentProxy agentProxy = context.getAgentProxy();
        TRCProcessProxy process = context.getProcessProxy();
        TRCNode node = context.getNode();
        TRCMonitor monitor = context.getMonitor();

        String agentName = "Unknown";
        String monitorFolder = LoadersUtils.getMonitorFolder(monitor);
        Resource agDoc = null;

        double startTime = ((double) (new Date()).getTime()) / 1000;

        if (monitorFolder != null) {
            String aName = StringUtil.change(monitor.getName().trim(), " ", "") + "_" + node.getName().trim() + "_" + process.getPid() + "_" + process.getRuntimeId() + "_" + StringUtil.change(agentName.trim(), " ", "");

            String pFileName = monitorFolder + aName;

            agDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcaxmi")).createResource(SaveUtil.createURI(pFileName + ".trcaxmi"));
			agDoc.setModified(true);
			
            HierarchyResourceSetImpl.getInstance().getResources().add(agDoc);
//            SaveUtil.addDocument(agDoc);
        }

        agent = HierarchyFactory.eINSTANCE.createTRCAgent();
        if (agentProxy == null) {
            agentProxy = HierarchyFactory.eINSTANCE.createTRCAgentProxy();
            agentProxy.setRuntimeId("" + startTime);
            agent.setRuntimeId("" + startTime);

            agent.setName(agentName);
            agentProxy.setName(agentName);
            
            agent.setType("Logging");
            agentProxy.setType("Logging");
            
            agent.setStartTime(startTime);
            agentProxy.setStartTime(startTime);

            agent.setAgentProxy(agentProxy);

            agentProxy.setProcessProxy(process);

            agent.setCollectionMode(context.getCollectionMode());
            agentProxy.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
            context.setAgentProxy(agentProxy);
        }
        else {
            agent.setRuntimeId(agentProxy.getRuntimeId());

            agent.setName(agentProxy.getName());
            if(agentProxy.getType()!=null)
	            agent.setType(agentProxy.getType());
            else
            {	
	            agent.setType("Logging");
	            agentProxy.setType("Logging");
            }
            agent.setStartTime(agentProxy.getStartTime());

            agent.setAgentProxy(agentProxy);

            agent.setCollectionMode(context.getCollectionMode());

            context.setAgent(agent);
        }

        if (context.getContextURI() == null) {
            context.setContextURI(agDoc.getURI().toString());
            LookupServiceExtensions.getInstance().register(null, context);

        }

        if (agDoc != null) {
            agDoc.getContents().add(agent);
            agDoc.setModified(true);
        }

        return agent;
    }

    public static void deregisterAgentInstance(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentInstance(null);
        }
    }

    public static void deregisterAgentListener(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentListener(null);
        }
    }

    public static void deregisterAgentProxyContext(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            LookupServiceExtensions.getInstance().deregister(null, c);
        }
    }

    public static void deregisterDataProcessor(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setDataProcessor(null);
        }
    }

    public static TRCAgent locateAgentFromMonitor(TRCMonitor monitor, String agentRId) {
        EList nodeList = monitor.getNodes();
        Iterator nodeIterator = nodeList.iterator();

        while (nodeIterator.hasNext()) {
            TRCNode node = (TRCNode) nodeIterator.next();

            if (!node.eIsProxy()) {
                EList processList = node.getProcessProxies();
                Iterator processIterator = processList.iterator();

                while (processIterator.hasNext()) {
                    TRCProcessProxy process = (TRCProcessProxy) processIterator.next();

                    if (!process.eIsProxy()) {
                        EList agents = process.getAgentProxies();
                        Iterator agentIterator = agents.iterator();

                        while (agentIterator.hasNext()) {
                            TRCAgentProxy agent = (TRCAgentProxy) agentIterator.next();

                            if (!agent.eIsProxy()) {
                                if ((agent.getRuntimeId() != null) && agent.getRuntimeId().equals(agentRId)) {
                                    return agent.getAgent();
                                }
                            }
                        }
                    }
                }
            }
        }

        return null;
    }

    public static TRCAgent locateAgentFromProcess(TRCProcessProxy process, String aName, String agentRId) {
        EList agents = process.getAgentProxies();
        Iterator i = agents.iterator();

        while (i.hasNext()) {
            TRCAgentProxy agent = (TRCAgentProxy) i.next();

            if (!agent.eIsProxy()) {
                if (agent.getName() != null && agent.getName().equals(aName)) {
                    if ((agent.getRuntimeId() == null) || agent.getRuntimeId().equals("") || agent.getRuntimeId().equals(agentRId)) {
                        return agent.getAgent();
                    }
                }
            }
        }

        return null;
    }

    public static Object locateAgentInstance(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getAgentInstance();
        }
        else {
            return null;
        }
    }

    public static Object locateAgentListener(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getAgentListener();
        }
        else {
            return null;
        }
    }

    public static XMLLoader locateDataProcessor(TRCAgentProxy agentProxy) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            return c.getDataProcessor();
        }
        else {
            return null;
        }
    }

    /**
     * @param agent
     * @return
     */
    public static HierarchyContext locateHierarchyContext(TRCAgent agent) {
        HierarchyContext context = null;

        if (agent.eResource() != null) {
            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, agent.eResource().getURI().toString());
        }
        if (context == null) {
            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, agent.getRuntimeId());
        }

        return context;
    }

    public static TRCNode locateNodeFromMonitor(TRCMonitor monitor, String hostName, String ipaddr, String nRId) {
        EList list = monitor.getNodes();
        Iterator i = list.iterator();

        try {
            InetAddress addr = InetAddress.getByName(hostName);

            while (i.hasNext()) {
                TRCNode node = (TRCNode) i.next();

                if (!node.eIsProxy()) {
                    String name = node.getName();

                    try {
                        InetAddress naddr = InetAddress.getByName(name);

                        if (addr.equals(naddr)) {
                            return node;
                        }
                    }
                    catch (UnknownHostException exc) {
                        String ipaddress = node.getIpAddress();

                        if (name.equals(hostName) || ((ipaddress != null) && ipaddress.equals(ipaddr))) {
                            node.setIpAddress(ipaddr);
                            node.setRuntimeId(nRId);

                            return node;
                        }
                    }
                }
            }
        }
        catch (UnknownHostException exc) {
            while (i.hasNext()) {
                TRCNode node = (TRCNode) i.next();
                String name = node.getName();
                String ipaddress = node.getIpAddress();

                if (((name != null) && (hostName != null) && name.equals(hostName)) || ((ipaddress != null) && ipaddress.equals(ipaddr))) {
                    node.setIpAddress(ipaddr);
                    node.setRuntimeId(nRId);

                    return node;
                }
            }
        }

        return null;
    }

    public static TRCProcessProxy locateProcessFromNode(TRCNode node, String pRId) {
        EList processList = node.getProcessProxies();
        Iterator i = processList.iterator();

        while (i.hasNext()) {
            TRCProcessProxy process = (TRCProcessProxy) i.next();

            if (!process.eIsProxy()) {
                if ((process.getRuntimeId() != null) && (process.getRuntimeId().compareTo(pRId) == 0)) {
                    return process;
                }
            }
        }

        return null;
    }

    public static TRCProcessProxy locateProcessFromNode(TRCNode node, int pId, String pRId) {
        EList processList = node.getProcessProxies();
        Iterator i = processList.iterator();

        while (i.hasNext()) {
            TRCProcessProxy process = (TRCProcessProxy) i.next();

            if (!process.eIsProxy()) {
                int processId = process.getPid();

                if (processId == pId) {
                    String runtimeId = process.getRuntimeId();

                    if ((runtimeId == null) || runtimeId.equals("") || runtimeId.equals(pRId)) {
                        process.setRuntimeId(pRId);

                        return process;
                    }
                }
            }
        }

        return null;
    }

    /**
     * @param e
     */
    public static void log(Exception e) {
        e.printStackTrace();
    }

    public static String makeString(byte[] data, int offset, int length) {
        if (data == null) {
            return null;
        }

        try {
            return new String(data, offset, length, "UTF8");
        }
        catch (Exception ex) {
            return null;
        }
    }

    public static void registerAgentInstance(TRCAgentProxy agentProxy, Object ai) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentInstance(ai);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setAgentInstance(ai);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    public static void registerAgentListener(TRCAgentProxy agentProxy, Object al) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setAgentListener(al);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setAgentListener(al);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    public static void registerDataProcessor(TRCAgentProxy agentProxy, XMLLoader processor) {
        AgentProxyContext c = (AgentProxyContext) LookupServiceExtensions.getInstance().locate(null, AgentProxyContext.class, LoadersUtils.getLookUpKey(agentProxy.hashCode()));

        if (c != null) {
            c.setDataProcessor(processor);
        }
        else {
            c = new AgentProxyContext();
            c.setAgentProxy(agentProxy);
            c.setDataProcessor(processor);
            LookupServiceExtensions.getInstance().register(null, c);
        }
    }

    /**
     * @param processor
     */
    public static void loadRootEvent(XMLLoader processor) {
        processor.loadEvent("<TRACE>".getBytes(), "<TRACE>".getBytes().length);
    }

    public static TRCAgent locateAgent(String agentIdRef, TRCAgent agent) {
        AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agentIdRef));
        if (agentsContext != null) {
            return agentsContext.getActiveAgent(agent);
        }
        return null;
    }

    public static boolean registerAgent(String agentIdRef, TRCAgent agent) {
        AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agentIdRef));
        if (agentsContext == null) {
            agentsContext = new AgentsContext(agentIdRef);
            LookupServiceExtensions.getInstance().register(null, agentsContext);
        }

        return agentsContext.registerAgent(agent);
    }

    public static Object getLookUpKey(int key) {
        return new Integer(key);
    }

    public static Object getLookUpKey(long key) {
        return new Integer(new Long(key).hashCode());
    }

    public static Object getLookUpKey(short key) {
        return new Short(key);
    }
    public static Object getLookUpKey(String key) {
        if (key == null)
            return new Integer(0);
        return new Integer(key.hashCode());
    }

    public static GenericLookUpEntry getGenericLookUpEntry(Thread thread) {
        Integer key = new Integer(thread.hashCode());
        GenericLookUpEntry entry = (GenericLookUpEntry) threads.get(key);
        if (entry == null) {
            entry = new GenericLookUpEntry();
            threads.put(key, entry);
        }
        return entry;
    }

    public static void registerGenericLookUpEntry(HierarchyContext context, Object key, Object value) {
        GenericLookUpEntry entry = LoadersUtils.getGenericLookUpEntry(Thread.currentThread());
        entry.setKey(key);
        entry.setValue(value);
        LookupServiceExtensions.getInstance().register(context, entry);
        entry.reset();
    }

	public static int getHashCode(byte[] data, int offset, int length) {
		int h = 0;

		for (int i = 0; i < length; i++) {
			h = (31 * h) + data[offset + i];
		}

		return h;
	}

    public static int getHashCode(String value) {
        int h = 0;
        int len = value.length();
        for (int i = 0; i < len; i++) {
            h = 31 * h + value.charAt(i);
        }
        return h;
    }
	public static void setSessionStartTime(double time, TRCAgentProxy agentProxy) {
		if(agentProxy == null)
			return;

		if (!agentProxy.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_StartTime())) {
			agentProxy.setStartTime(time);

		}

		TRCAgent agent = agentProxy.getAgent();

		if ((agent != null) && (!agent.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgent_StartTime()))) {
			agent.setStartTime(time);
		}
	}

	/**
	 * @param string
	 * @return
	 */
	public static CorrelationEngine findCorrelationEngine(String string) {
		return (CorrelationEngine)LookupServiceExtensions.getInstance().locate(null,CorrelationEngineImpl.class,string);
	}

	/**
	 * @param correlationEngine
	 */
	public static void addCorrelationEngine(CorrelationEngine correlationEngine) {
		Resource resource = HierarchyResourceSetImpl.getInstance().getResource(URI.createURI("http://www.eclipse.org/hyades/correlations.xmi"),false);
		resource.getContents().add(correlationEngine);
	}

	/**
	 * 
	 * @param agentProxy
	 * @param correlationEngine
	 * @return the correlation container proxy for an internal correlation container
	 */
	public static CorrelationContainerProxy getInternalCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID) {
		URI internalCorrelationURI = createCorrelationContainerURI(agentProxy, correlationEngineID, "internalcc");
		for (Iterator iter = agentProxy.getInternalCorrelations().iterator(); iter.hasNext();) {
			CorrelationContainerProxy correlationContainerProxy = (CorrelationContainerProxy) iter.next();
			CorrelationContainer correlationContainer = correlationContainerProxy.getCorrelationContainer();
			if(correlationContainer!=null)
			{	
				URI ccURI = correlationContainer.eIsProxy() ? ((InternalEObject)correlationContainer).eProxyURI() : correlationContainer.eResource().getURI();
				if(!correlationContainerProxy.eIsProxy() && ccURI.equals(internalCorrelationURI))
					return correlationContainerProxy;
			}
			else
			{
				correlationContainer = HierarchyFactory.eINSTANCE.createCorrelationContainer();
				correlationContainer.setCorrelationContainerProxy(correlationContainerProxy);
				createCorrelationContainerResource(agentProxy,correlationEngineID,correlationContainer,"internalcc");
				return correlationContainerProxy;
			}
		}
		CorrelationContainerProxy correlationContainerProxy = HierarchyFactory.eINSTANCE.createCorrelationContainerProxy();
		correlationContainerProxy.getCorrelatedAgents().add(agentProxy);
		agentProxy.getInternalCorrelations().add(correlationContainerProxy);
		correlationContainerProxy.setCreationTime(new Date().getTime());
		correlationContainerProxy.setName("InternalCorrelation_"+correlationEngineID);
		correlationContainerProxy.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
		agentProxy.getProcessProxy().eResource().setModified(true);
		agentProxy.getProcessProxy().getNode().getMonitor().eResource().setModified(true);
		return correlationContainerProxy;
	}

	public static CorrelationContainerProxy getCorrelationContainerProxy(TRCAgentProxy agentProxy, String correlationEngineID,String correlationTag) {
		CorrelationContainerProxy correlationContainerProxy = HierarchyFactory.eINSTANCE.createCorrelationContainerProxy();
		CorrelationContainer correlationContainer = HierarchyFactory.eINSTANCE.createCorrelationContainer();
		correlationContainer.setCorrelationContainerProxy(correlationContainerProxy);
		createCorrelationContainerResource(agentProxy,correlationEngineID,correlationContainer,correlationTag);

		correlationContainerProxy.getCorrelatedAgents().add(agentProxy);
//		agentProxy.getInternalCorrelations().add(correlationContainerProxy);
		correlationContainerProxy.setCreationTime(new Date().getTime());
		correlationContainerProxy.setName(correlationTag+"_"+correlationEngineID);
		correlationContainerProxy.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
//		agentProxy.getProcessProxy().eResource().setModified(true);
		agentProxy.getProcessProxy().getNode().getMonitor().eResource().setModified(true);
		return correlationContainerProxy;
	}
	
	public static URI createCorrelationContainerURI(TRCAgentProxy agentProxy, String correlationEngineID,String correlationTag) {
		URI internalCorrelationURI = agentProxy.getAgent().eResource().getURI();
		String fileExt = internalCorrelationURI.fileExtension();
		internalCorrelationURI = internalCorrelationURI.trimFileExtension();
		String lastSegment = internalCorrelationURI.lastSegment()+"_"+(correlationTag+"-"+correlationEngineID+"-"+(new Date().getTime())).hashCode();
		internalCorrelationURI = internalCorrelationURI.trimSegments(1);
		internalCorrelationURI = internalCorrelationURI.appendSegment(lastSegment);
		internalCorrelationURI = internalCorrelationURI.appendFileExtension("corrxmi");
		return internalCorrelationURI;
	}

	/**
	 * @param container
	 */
	public static void createCorrelationContainerResource(TRCAgentProxy proxy, String correlationEngineID, CorrelationContainer container,String correlationTag) {
		URI internalCorrelationURI = createCorrelationContainerURI(proxy, correlationEngineID,correlationTag);
		Resource resource = HierarchyResourceSetImpl.getInstance().createResource(internalCorrelationURI);
		resource.setModified(true);
		resource.getContents().add(container);
	}

	/**
	 * @param uri
	 * @return
	 */
	public static String getPostfix(URI uri) {
		String postfix = uri.toString();
		int index = postfix.lastIndexOf("__");
		if(index>0)
			postfix = postfix.substring(index);
		else
			postfix = postfix.substring(postfix.indexOf('.'));
		return postfix;
	}

//	/**
//	 * @param correlationContainerProxy
//	 */
//	public static void addCorrelation(CorrelationContainerProxy correlationContainerProxy) {
//		String key="";
//		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
//			TRCAgentProxy element = (TRCAgentProxy) iter.next();
//			key = key+"_"+element.hashCode();
//		}
//		correlationMap.put(key,correlationContainerProxy.eResource().getURI());
//	}
//	/**
//	 * @param correlationContainerProxy
//	 */
//	public static void getCorrelation(List agents) {
//		String key=correlationContainerProxy.getCorrelationEngine().getId();
//		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
//			TRCAgentProxy element = (TRCAgentProxy) iter.next();
//			key = key+"_"+element.hashCode();
//		}
//		correlationMap.put(key,correlationContainerProxy.eResource().getURI());
//	}
}
