/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.extension;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.form.ConfigurableObjectOverviewForm;
import org.eclipse.hyades.test.ui.internal.editor.form.IPropertyGroupForm;
import org.eclipse.hyades.test.ui.internal.editor.form.PropertyGroupForm;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.internal.editor.form.util.WidgetFactory;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

/**
 * This is the base class for Hyades configuration editor extension.
 * 
 * @author bjiang
 * @since  1.3
 */
public class ConfigurableObjectEditorExtension extends BaseEditorExtension implements IExtensibleEditorExtension
{	
	IConfigurationElement[] registeredExtensionElements;
	private Hashtable extensionPagesMap = new Hashtable();

	protected static final int PAGE_OVERVIEW = 0;	
	private ConfigurableObjectOverviewForm overviewForm;	
	private WidgetFactory widgetFactory;
	
	public ConfigurableObjectEditorExtension()
	{
		super();	
		registeredExtensionElements = collectRegisteredExtensionElements();		
	}

	protected IConfigurationElement[] collectRegisteredExtensionElements() 
	{
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint("org.eclipse.hyades.test.ui.configurableObjectExtension"); //$NON-NLS-1$
		if( extPoint != null )
			return extPoint.getConfigurationElements();
		else
			return new IConfigurationElement[0];
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		overviewForm.dispose();
		
		Enumeration elements = extensionPagesMap.elements();
		while(elements.hasMoreElements())
		{
			((EditorForm)elements.nextElement()).dispose();
		}		
		extensionPagesMap.clear();		
		
		if(widgetFactory != null)
			widgetFactory.dispose();
		
		super.dispose();
	}

	/**
	 * Returns the data object that is manipulated by this editor extension.
	 * @return EObject 
	 */
	public EObject getEditorObject()
	{
		return (EObject)getHyadesEditorPart().getEditorObject();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	public void createPages()
	{
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		widgetFactory = new WidgetFactory();
		
		this.overviewForm = createOverviewPage(widgetFactory);
		overviewForm.setRegisteredExtensions( registeredExtensionElements );		
		getHyadesEditorPart().addPage(overviewForm.createControl());
		getHyadesEditorPart().setPageText(PAGE_OVERVIEW, TestUIPlugin.getString("W_OVERVIEW")); //$NON-NLS-1$
		overviewForm.updateTitle();
		
		createOtherStaticPages(widgetFactory);		
		
		// TODO create forms from extensions for existing property groups.
		createExtensionPages(widgetFactory);
	}
	
	protected ConfigurableObjectOverviewForm createOverviewPage(WidgetFactory widgetFactory)
	{
		return new ConfigurableObjectOverviewForm(this, widgetFactory);
	}

	protected void createOtherStaticPages(WidgetFactory widgetFactory)
	{
	}
	
	protected void createExtensionPages(WidgetFactory widgetFacotory)
	{
		// TODO change to IConfigurableObject when data model is settled.
		EObject editorObject = getEditorObject();
		EList propertyGroups = null;
		String editorType = null;
		if(editorObject instanceof CFGLocation)
		{
			propertyGroups = ((CFGLocation)editorObject).getPropertyGroups();
			editorType = PropertyGroupForm.EDITOR_TYPE_LOCATION;
		}
		else if(editorObject instanceof CFGArtifact)
		{
			propertyGroups = ((CFGArtifact)editorObject).getPropertyGroups();
			editorType = PropertyGroupForm.EDITOR_TYPE_ARTIFACT;
		}
		else if(editorObject instanceof TPFDeployment)
		{
			propertyGroups = ((TPFDeployment)editorObject).getPropertyGroups();
			editorType = PropertyGroupForm.EDITOR_TYPE_DEPLOYMENT;
		}		
		if(propertyGroups == null || propertyGroups.isEmpty())
			return;
			
		Iterator it = propertyGroups.iterator();
		CFGPropertyGroup propGroup = null;
		String propGroupID = null;
		boolean extensionFound = false;
		while( it.hasNext() )
		{		
			propGroup = (CFGPropertyGroup)it.next();
			if(propGroup.getProperties().isEmpty())
				continue;
				
			propGroupID = propGroup.getPropertyGroupID();			
			for(int i = 0; i < registeredExtensionElements.length; i++)
			{
				if( registeredExtensionElements[i].getAttribute("propertyGroupID").equals(propGroupID) ) //$NON-NLS-1$
				{
					addExtensionPage(registeredExtensionElements[i]);
					extensionFound = true;
					break;
				}
			}
			
			if(extensionFound == false)
			{
				IPropertyGroupForm defaultForm = new PropertyGroupForm();
				defaultForm.init(propGroup, this, widgetFactory);
				defaultForm.setAppliesto(editorType);

				int index = getHyadesEditorPart().addPage(defaultForm.createControl());
				getHyadesEditorPart().setPageText(index, propGroup.getName());					
				extensionPagesMap.put(new Integer(index), defaultForm);
				overviewForm.addSection(defaultForm, index); 
			}
		}
	}
	
	/*protected void setOverviewForm(ConfigurationOverviewForm ovForm)
	{
		this.overviewForm = ovForm;
	}*/
	
	protected ConfigurableObjectOverviewForm getOverviewForm()
	{
		return this.overviewForm;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IRefreshable#refreshContent(java.lang.Object)
	 */
	public void refreshContent(Object data)
	{
		overviewForm.load();
		overviewForm.updateTitle();
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#getSelection()
	 */
	public IStructuredSelection getSelection()
	{
		int index = getHyadesEditorPart().getActivePage();
		if(index == PAGE_OVERVIEW) // overview page, just return the data object
			return new StructuredSelection(getEditorObject());
		else if(extensionPagesMap.containsKey(new Integer(index)))
			return ((IPropertyGroupForm)extensionPagesMap.get(new Integer(index))).getSelection();
		
		return StructuredSelection.EMPTY;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#pageActivated(int)
	 */
	public boolean pageActivated(int index)
	{
		if(index == PAGE_OVERVIEW)
			return overviewForm.activated();
		else if(extensionPagesMap.containsKey(new Integer(index)))
			return ((EditorForm)extensionPagesMap.get(new Integer(index))).activated();
		
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#setSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void setSelection(IStructuredSelection structuredSelection)
	{
		if(structuredSelection.size() == 1)
		{
			Object object = structuredSelection.getFirstElement();
			if(object == getEditorObject())
			{
				// don't do anything.
			}
			else if(object instanceof IConfigurationElement) // selection on extensions
			{
			}
			else if(object instanceof CFGComparableProperty)
			{
				EObject container = ((CFGComparableProperty)object).eContainer();
				Enumeration pages = extensionPagesMap.elements();
				IPropertyGroupForm page = null;
				while(pages.hasMoreElements())
				{
					page = (IPropertyGroupForm)pages.nextElement(); 
					if( page.getPropertyGroup() == container )
					{
						page.selectReveal(structuredSelection);
						break;
					}
				}
			}
		}
	}
		
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.IExtensibleEditorExtension#addExtensionPage(org.eclipse.core.runtime.IConfigurationElement)
	 */
	public int addExtensionPage(IConfigurationElement extensionElement) 
	{
		// check if this page is already activated.
		//String extensionID = extensionElement.getAttribute("id");
		String propertyGroupID = extensionElement.getAttribute("propertyGroupID"); //$NON-NLS-1$
		Enumeration keys = extensionPagesMap.keys();
		Integer key;
		IPropertyGroupForm page = null;
		while( keys.hasMoreElements() )
		{
			key = (Integer)keys.nextElement();
			page = (IPropertyGroupForm)extensionPagesMap.get(key);
			if( page.getPropertyGroup().getPropertyGroupID().equals( propertyGroupID ) )
				return key.intValue();
		}
			 
		IPropertyGroupForm formExt = null;
		try
		{
			formExt = (IPropertyGroupForm)extensionElement.createExecutableExtension("class"); //$NON-NLS-1$
		}
		catch(Throwable e)
		{
			e.printStackTrace();
			// TODO properly handle exception when failed to instantiate extension.
		}
		
		formExt.init(getCFGPropertyGroup(extensionElement), this, widgetFactory);
		formExt.setAppliesto(extensionElement.getAttribute("appliesTo")); //$NON-NLS-1$
		formExt.setControlType(extensionElement.getAttribute("controlType")); //$NON-NLS-1$
		
		int index = getHyadesEditorPart().addPage(formExt.createControl());
		getHyadesEditorPart().setPageText(index, formExt.getPropertyGroup().getName());					
		extensionPagesMap.put(new Integer(index), formExt);
		overviewForm.addSection(formExt, index); 
		
		return index;
	}
	
	private CFGPropertyGroup getCFGPropertyGroup(IConfigurationElement extensionElement)
	{
		//TODO use CFGConfigurableObject interface instead
		EObject editorObject = getEditorObject();
		EList propertyGroups = null; 
		if(editorObject instanceof CFGLocation)
			propertyGroups = ((CFGLocation)editorObject).getPropertyGroups();
		else if(editorObject instanceof CFGArtifact)
			propertyGroups = ((CFGArtifact)editorObject).getPropertyGroups();
		else if(editorObject instanceof TPFDeployment)
			propertyGroups = ((TPFDeployment)editorObject).getPropertyGroups();

		if(propertyGroups == null)
			return null;
			
		CFGPropertyGroup propertyGroup = null;
		String propertyGroupID = extensionElement.getAttribute("propertyGroupID"); //$NON-NLS-1$
		String name = extensionElement.getAttribute("name"); //$NON-NLS-1$
		String description = extensionElement.getAttribute("description"); //$NON-NLS-1$
		for( Iterator i = propertyGroups.iterator(); i.hasNext(); )
		{			
			propertyGroup = (CFGPropertyGroup)i.next();
			if(propertyGroupID.equals(propertyGroup.getPropertyGroupID()))
				break;
			else
				propertyGroup = null;			
		}

		if( propertyGroup == null ) 
		{
			propertyGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
			propertyGroup.setPropertyGroupID(propertyGroupID);
			propertyGroup.setName(name);
			propertyGroup.setDescription(description);
			propertyGroups.add(propertyGroup);
		}
	
		return propertyGroup;
	}

}
