/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form.util;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.test.ui.adapter.TestWorkbenchAdapter;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.StructuredViewerPart;
import org.eclipse.hyades.test.ui.internal.model.ui.ChildrenSelectionAction;
import org.eclipse.hyades.ui.internal.provider.WorkbenchAdapterLabelProvider;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.actions.SelectionListenerAction;

/**
 * This class provides a EObject table view section with configurable buttons in it.
 * The sole constructor of this class takes all the button actions in an array which
 * will be used to create the buttons and will be invoked when the button is selected.
 * 
 * @author bjiang
 * @since  1.3
 */
public class ExtensibleEObjectTableSection extends StructuredViewerSection implements ISelectionChangedListener, IDoubleClickListener
{
	private int tableStyle = SWT.MULTI | SWT.FULL_SELECTION;
	private int numColumns;
	private IAction[] buttonActions;
	private EStructuralFeature eStructuralFeature;
	private ITableLabelProvider tableLabelProvider;

	/**
	 * @param editorForm
	 * @param buttonLabels
	 */
	public ExtensibleEObjectTableSection(EditorForm editorForm, EStructuralFeature eStructuralFeature, int numOfColumns, IAction[] theButtonActions) 
	{
		super(editorForm);
		this.eStructuralFeature = eStructuralFeature;
		this.numColumns = numOfColumns;
		this.buttonActions = theButtonActions;
		String[] buttonLabels = new String[buttonActions.length];
		for(int i = 0; i < buttonActions.length; i++)
		{
			buttonLabels[i] = buttonActions[i].getText();
		}
		
		super.setButtonLabels(buttonLabels);
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#dispose()
	 */
	public void dispose()
	{
		eStructuralFeature = null;
		
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
				buttonActions[i] = null;
		}

		super.dispose();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.StructuredViewerSection#createViewerPart(java.lang.String[])
	 */
	protected StructuredViewerPart createViewerPart(String[] buttonLabels) 
	{
		StructuredViewerPart tablePart = new StructuredViewerPart(buttonLabels)
		{
			/*
			 * @see StructuredViewerPart#createStructuredViewer(Composite, FormWidgetFactory)
			 */
			protected StructuredViewer createStructuredViewer(Composite parent, int style, FormWidgetFactory factory)
			{				
				// create a table with numColumns columns
				Table table = new Table(parent, style);
				//GridData gridData = new GridData(GridData.FILL_BOTH);
				//gridData.grabExcessVerticalSpace = true;
				//gridData.grabExcessHorizontalSpace = true;
				//table.setLayoutData(gridData);	
				table.setLinesVisible(true);
				table.setHeaderVisible(false);
				for(int i = 0; i < numColumns; i++)
				{
					TableColumn column = new TableColumn(table, SWT.NONE);
					column.setResizable(true);
				}
				table.addControlListener( new ControlListener() {
					public void controlMoved(ControlEvent e) {}

					public void controlResized(ControlEvent e) 
					{
						if(e.getSource() instanceof Table)
						{
							Table table = ((Table)e.getSource());
							int width = table.getSize().x;
							int gridLineWidth = table.getGridLineWidth();
							TableColumn[] columns = table.getColumns();
							if( columns.length == 3 )
							{
								columns[0].setWidth(width*2/5);
								columns[1].setWidth(width/5);
								columns[2].setWidth(width*2/5 + gridLineWidth);
							}
							else
							{
								for(int i = 0; i < columns.length; i++)
									columns[i].setWidth(width/columns.length);
							}
						}

					}
				});
			
				parent.pack();
				
				TableViewer tableViewer = new TableViewer(table);
				tableViewer.addSelectionChangedListener(ExtensibleEObjectTableSection.this);
				tableViewer.addDoubleClickListener(ExtensibleEObjectTableSection.this);
				return tableViewer;
			}
			public void buttonSelected(Button button, int index)
			{
				ExtensibleEObjectTableSection.this.buttonSelected(index);
				if(isHandlingDefaultButton())
					button.getShell().setDefaultButton(null);
			}			
		};
		
		//Table table = tablePart.getTableViewer().getTable();
		return tablePart;
	}
	
	protected void buttonSelected(int index)
	{
		if(!isReadOnly())
		{
			if( index >=0 && index < buttonActions.length )
				buttonActions[index].run();
		}
	
		updateActionsAndButtons(getStructuredSelection());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection#createClient(org.eclipse.swt.widgets.Composite, org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory)
	 */
	public Composite createClient(Composite parent, FormWidgetFactory factory) 
	{
		setFormWidgetFactory(factory);
		Composite container = createClientContainer(parent, 2, getFormWidgetFactory());
		createViewerPartControl(container, getTableStyle(), 2, getFormWidgetFactory());
		getTableViewer().getControl().setFocus();

		adjustClient(parent);

		getFormWidgetFactory().paintBordersFor(container);
		return container;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseTreeSection#adjustClient(org.eclipse.swt.widgets.Composite)
	 */
	protected void adjustClient(Composite parent)
	{
		getTableViewer().setContentProvider(createContentProvider());
		getTableViewer().setLabelProvider(getLabelProvider());

		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
			{
				getTablePart().setButtonEnabled(i, buttonActions[i].isEnabled());
				if(buttonActions[i] instanceof ChildrenSelectionAction)
					((ChildrenSelectionAction)buttonActions[i]).setStructuredViewer(getTableViewer());
			}
				
		}
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseStructuredViewerSection#updateActionsAndButtons(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected void updateActionsAndButtons(IStructuredSelection structuredSelection)
	{
		if(buttonActions == null || buttonActions.length < 1)
			return;
			
		for(int i = 0; i < buttonActions.length; i++)
		{
			if(buttonActions[i] != null)
			{
				if(buttonActions[i] instanceof SelectionListenerAction)
					((SelectionListenerAction)buttonActions[i]).selectionChanged(structuredSelection);
				getTablePart().setButtonEnabled(i, buttonActions[i].isEnabled());
			}
		}
	}

	/**
	 * Creates this section's content provider
	 * @return IContentProvider
	 */
	protected IContentProvider createContentProvider()
	{
		return new EObjectTreeSection.EObjectContainerContentProvider(getEditorForm(), getEStructuralFeature())
		{			
			protected void updateViewer(Notification msg)
			{
				if(getViewer() == null)
					return;

				if(!(getViewer() instanceof StructuredViewer))
				{
					getViewer().refresh();
					return;
				}

				if(msg.getNotifier() != getContainer())
				{
					switch(msg.getEventType())
					{
						case Notification.SET:
						case Notification.UNSET:
							attributeChanged(msg);
							return;
					}
				}

				if(!(getViewer() instanceof TableViewer))
				{
					getViewer().refresh();
					return;
				}
		
				if(shouldUpdateViewer(msg))
				{	
					Object parent = null;
					switch(msg.getEventType())
					{
						case Notification.ADD:
							parent = getParent(msg.getNewValue());
							if(parent == null)
								parent = getViewer().getInput();
								
							if(msg.getNewValue() instanceof CFGComparableProperty)
							{
								CFGComparableProperty prop = (CFGComparableProperty)msg.getNewValue();
								TableItem item = new TableItem( ((TableViewer)getViewer()).getTable(), SWT.NONE );
								String[] values = new String[]{prop.getName(), prop.getOperator(), prop.getValue()};
								item.setText(values);
							}
							else if(msg.getNewValue() instanceof CFGArtifactLocationPair)
							{
								CFGArtifactLocationPair pair = (CFGArtifactLocationPair)msg.getNewValue();
								TableItem item = new TableItem( ((TableViewer)getViewer()).getTable(), SWT.NONE );
								String[] values = new String[]{pair.getArtifact().getName(), pair.getLocation().getName()};
								item.setText(values);
							}
							//((TableViewer)getViewer()).add(msg.getNewValue());
							getViewer().setSelection(new StructuredSelection(msg.getNewValue()), true);
							break;

						case Notification.ADD_MANY:
							Object[] addedObjects = ((Collection)msg.getNewValue()).toArray();
							if(addedObjects != null && addedObjects.length > 0)
							{
								parent = getParent(addedObjects[0]);
								if(parent == null)
									parent = getViewer().getInput();
								for(int i = 0; i < addedObjects.length; i++)
								{
									if(addedObjects[i] instanceof CFGComparableProperty)
									{
										CFGComparableProperty prop = (CFGComparableProperty)msg.getNewValue();
										TableItem item = new TableItem( ((TableViewer)getViewer()).getTable(), SWT.NONE );
										String[] values = new String[]{prop.getName(), prop.getOperator(), prop.getValue()};
										item.setText(values);
									}
									else if(addedObjects[i] instanceof CFGArtifactLocationPair)
									{
										CFGArtifactLocationPair pair = (CFGArtifactLocationPair)msg.getNewValue();
										TableItem item = new TableItem( ((TableViewer)getViewer()).getTable(), SWT.NONE );
										String[] values = new String[]{pair.getArtifact().getName(), pair.getLocation().getName()};
										item.setText(values);
									}									
								}
								//((TableViewer)getViewer()).add(addedObjects);
								getViewer().setSelection(new StructuredSelection(addedObjects), true);
							}
							break;
					
						case Notification.REMOVE:
							((TableViewer)getViewer()).remove(msg.getOldValue());
							Object value = ((EObject)msg.getNotifier()).eGet((EStructuralFeature)msg.getFeature());
							if(value instanceof List)
							{
								List values = (List)value;
								if(values.isEmpty())
								{
									getViewer().getControl().setFocus();
								}
								else
								{
									int position = msg.getPosition();
									position = (position<values.size())?position:(values.size()-1);
									getViewer().setSelection(new StructuredSelection(values.get(position)), true);
								}
							}
							else
							{
								getViewer().getControl().setFocus();
							}
							break;

						case Notification.REMOVE_MANY:
							((TableViewer)getViewer()).remove(((Collection)msg.getOldValue()).toArray());
							getViewer().getControl().setFocus();
							break;

						case Notification.MOVE:
							getViewer().refresh();
							break;
					}
				}
			}
		};
	}
	
	protected EStructuralFeature getEStructuralFeature()
	{
		return eStructuralFeature;
	}
	
	/**
	 * Creates this section's label provider
	 * @return ILabelProvider
	 */
	protected IBaseLabelProvider getLabelProvider()
	{
		if(tableLabelProvider == null)
			return new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class);
		else
			return tableLabelProvider;
	}
	
	public void setLabelProvider(ITableLabelProvider provider)
	{
		this.tableLabelProvider = provider;
	}
	
	/**
	 * Sets the table style.  The default value is 
	 * <code>SWT.FULL_SELECTION | SWT.MULTI</code>
	 * @param tableStryle
	 */
	public void setTableStryle(int tableStryle)
	{
		this.tableStyle = tableStryle;
	}	
	
	/**
	 * Gets the table style.  The default value is 
	 * <code>SWT.FULL_SELECTION | SWT.MULTI</code>
	 * @return int
	 */
	public int getTableStyle()
	{
		return tableStyle;
	}
	
	/**
	 * Returns this section's table part.
	 * @return TablePart
	 */
	public StructuredViewerPart getTablePart()
	{
		return (StructuredViewerPart)getViewerPart();
	}

	/**
	 * Returns this section's table viewer.
	 * @return TableViewer
	 */
	public TableViewer getTableViewer()
	{
		return (TableViewer)getTablePart().getViewer();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) 
	{
		selectionChanged((IStructuredSelection)event.getSelection());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event) 
	{
		handleDoubleClick((IStructuredSelection)event.getSelection());
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
	 */
	public String getColumnText(Object element, int columnIndex) 
	{
		return element.toString();
	}

}
