/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.util.internal;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;

/*
 * 
 * @author sduguet
 *
 * A folder data item of the table tree viewer associated to a list of TRCPackage or TRCClass
 * All of these EMF objects refer to the same Java source item from one or several runs
 */

public class CoverageFolder extends Coverage {

	private int methods;
	private int methodsHit;
	private ArrayList childs;

	/**
	 * @param analysis
	 * @param parent
	 */
	public CoverageFolder(CoverageAnalysis analysis, Coverage parent) {
		super(analysis, parent);
		childs = new ArrayList();
	}
	
	/**
	 * @param analysis
	 * @param parent
	 * @param from
	 */
	public CoverageFolder(CoverageAnalysis analysis, Coverage parent, CoverageFolder from) {
		super(analysis, parent);
		methods = from.getMethods();
		methodsHit = from.getMethodsHit();
		setCalls(from.getCalls());
		childs = null;
	}
	
	/**
	 * Parses the model
	 * @param list
	 */
	public void parse(EList list) {
		for (int i = 0; i < list.size(); i++) {
			EObject echild = (EObject)list.get(i);
			if (echild instanceof TRCMethod) {
				CoverageLeaf child = analysis.FindCoverageLeaf(this, echild);
				child.parse(echild);
			} else {
				EList subList = null;
				if (echild instanceof TRCClass && 
					!this.analysis.isFiltered((TRCClass)echild)) {
						subList = ((TRCClass)echild).getMethods();
				}
				else if (echild instanceof TRCPackage) {
					subList = ((TRCPackage)echild).getClasses();
				}
				if (subList != null) {
					CoverageFolder child = analysis.FindCoverageFolder(this, echild);
					child.parse(subList);
				}
			}
		}
	}
	
	/**
	 * Computes the childs and synthesizes the result in this folder
	 */
	public void compute() {
		methods = 0;
		methodsHit = 0;
		for (int i = 0; i < getChilds().size(); i++) {
			Coverage child = (Coverage)childs.get(i);
			if (child instanceof CoverageFolder) {
				((CoverageFolder)child).compute();
				methods += child.getMethods();
				methodsHit += child.getMethodsHit();
				setCalls(getCalls()+child.getCalls());
			} else {
				methods++;
				if (child.getCalls() > 0) {
					methodsHit ++;
					setCalls(getCalls()+child.getCalls());
				}
			}
		}
	}

	/**
	 * @see org.eclipse.hyades.trace.views.util.internal.Coverage#getMethods()
	 */
	public int getMethods() {
		return methods;
	}

	/**
	 * @see org.eclipse.hyades.trace.views.util.internal.Coverage#getMethodsHit()
	 */
	public int getMethodsHit() {
		return methodsHit;
	}

	/**
	 * @see org.eclipse.hyades.trace.views.util.internal.Coverage#getMethodsMissed()
	 */
	public int getMethodsMissed() {
		return methods - methodsHit;
	}

	/**
	 * @see org.eclipse.hyades.trace.views.util.internal.Coverage#getChilds()
	 */
	public ArrayList getChilds() {
		return childs;
	}

	/**
	 * @param childs
	 */
	public void setChilds(ArrayList childs) {
		this.childs = childs;
	}

	/**
	 * @see org.eclipse.hyades.trace.views.util.internal.Coverage#getPercentage()
	 */
	public double getPercentage() {
		if (getMethods() == 0) {
			return 0.0;
		}
		return (double)getMethodsHit() / (double)getMethods();
	}
	
}