/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.actions;

import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.*;

public abstract class AbstractListenerActionDelegate
	extends AbstractProfileActionDelegate
	implements IProfileEventListener, IPartListener, IPageListener {

	/**
	 * The window associated with this action delegate
	 */
	private IWorkbenchWindow fWindow;

	/**
	 * @see IPartListener#partActivated(IWorkbenchPart)
	 */
	public void partActivated(IWorkbenchPart part) {
	}

	/**
	 * @see IPartListener#partBroughtToTop(IWorkbenchPart)
	 */
	public void partBroughtToTop(IWorkbenchPart part) {		
	}

	/**
	 * @see IPartListener#partClosed(IWorkbenchPart)
	 */
	public void partClosed(IWorkbenchPart part) {
		if (part.equals(getView())) {
			dispose();
		}
	}

	/**
	 * @see IPartListener#partDeactivated(IWorkbenchPart)
	 */
	public void partDeactivated(IWorkbenchPart part) {
	}

	/**
	 * @see IPartListener#partOpened(IWorkbenchPart)
	 */
	public void partOpened(IWorkbenchPart part) {
	}

	/**
	 */
	public void dispose() {
		
		UIPlugin.getDefault().removeProfileEventListener(this);
		getWindow().removePageListener(this);
		if (getView() != null) {
			getView().getViewSite().getPage().removePartListener(this);
		}
		
		fWindow = null;
		
		super.dispose();
		
	}

	/**
	 */
	public void handleProfileEvent(final ProfileEvent event) {
		if (getPage() == null || getAction() == null) {
			return;
		}
		final Shell shell = getWindow().getShell();
		if (shell == null || shell.isDisposed()) {
			return;
		}
		Runnable r = new Runnable() {
			public void run() {
				if (shell == null || shell.isDisposed()) {
					return;
				}
				doHandleProfileEvent(event);
			}
		};

		getPage().getWorkbenchWindow().getShell().getDisplay().asyncExec(r);
	}

	/**
	 * Returns the page that this action works in.
	 */
	protected IWorkbenchPage getPage() {
		if (getWindow() != null) {
			return getWindow().getActivePage();
		} else {
			return UIPlugin.getActiveWorkbenchWindow().getActivePage();
		}
	}

	protected IWorkbenchWindow getWindow() {
		return fWindow;
	}

	protected void setWindow(IWorkbenchWindow window) {
		fWindow = window;
	}

	/**
	 * Default implementation to update on specific profile events.
	 * Subclasses should override to handle events differently.
	 */
	protected void doHandleProfileEvent(ProfileEvent event) {
		switch (event.getType()) {
			case ProfileEvent.TERMINATE :
				update(getAction(), getSelection());
				break;
			case ProfileEvent.ATTACH :
				update(getAction(), getSelection());
				break;
			case ProfileEvent.DETACH :
				update(getAction(), getSelection());
				break;
			case ProfileEvent.START_MONITOR :
				update(getAction(), getSelection());
				break;
			case ProfileEvent.STOP_MONITOR :
				update(getAction(), getSelection());
				break;
			case ProfileEvent.CLEAN_UP :
				update(getAction(), StructuredSelection.EMPTY);
				break;

		}
	}

	/**
	 * @see IWorkbenchWindowActionDelegate#init(IWorkbenchWindow)
	 */
	public void init(IWorkbenchWindow window) {
		super.init(window);
		UIPlugin.getDefault().addProfileEventListener(this);
		setWindow(window);
		window.addPageListener(this);
	}

	/**
	 * @see IViewActionDelegate#init(IViewPart)
	 */
	public void init(IViewPart view) {
		super.init(view);
		UIPlugin.getDefault().addProfileEventListener(this);
		setWindow(view.getViewSite().getWorkbenchWindow());
		view.getViewSite().getPage().addPartListener(this);
		view.getViewSite().getWorkbenchWindow().addPageListener(this);
	}

	/**
	 * @see IPageListener#pageActivated(IWorkbenchPage)
	 */
	public void pageActivated(final IWorkbenchPage page) {
		
		if (getPage() != null && getPage().equals(page)) {
			Runnable r = new Runnable() {
				public void run() {
					if (getPage() != null) {
						IWorkbenchWindow window = getPage().getWorkbenchWindow();
						if (window != null
							&& window.getShell() != null
							&& !window.getShell().isDisposed()) {
							ISelection selection = page.getSelection(PDPerspective.ID_PD_NAVIGATOR_VIEW);
							update(getAction(), selection);
						}
					}
				}
			};

			getPage().getWorkbenchWindow().getShell().getDisplay().asyncExec(r);
		}
	}

	/**
	 * @see IPageListener#pageClosed(IWorkbenchPage)
	 */
	public void pageClosed(IWorkbenchPage page) {
		if (page.equals(getPage())) {
			dispose();
		}
	}

	/**
	 * @see IPageListener#pageOpened(IWorkbenchPage)
	 */
	public void pageOpened(IWorkbenchPage page) {
	}
		
}
