/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.ui.internal.views;

import java.io.File;

import org.eclipse.hyades.logging.adapter.MessageString;
import org.eclipse.hyades.logging.adapter.model.internal.sensor.SensorConfigType;
import org.eclipse.hyades.logging.adapter.model.internal.unit.PropertyType;
import org.eclipse.hyades.logging.adapter.ui.AcadEditorPlugin;
import org.eclipse.hyades.logging.adapter.ui.AcadEditorPluginImages;
import org.eclipse.hyades.logging.adapter.ui.ContextIds;
import org.eclipse.hyades.logging.adapter.ui.internal.actions.ActionHandlerListener;
import org.eclipse.hyades.logging.adapter.ui.internal.util.AcadPerspective;
import org.eclipse.hyades.logging.adapter.ui.internal.util.GridUtil;
import org.eclipse.hyades.logging.adapter.ui.internal.util.ResultQueueEntry;
import org.eclipse.hyades.logging.events.CbeFormatter;
import org.eclipse.hyades.logging.events.ICommonBaseEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.part.ViewPart;


/**
 */
public class LogView extends ViewPart 
{
	private final String LOG_TITLE = AcadEditorPlugin.getPlugin().getString("TEMPLATE_LOG");

	private TextViewer eventsViewer;		
	
	private Action createRuleAction;
	private Action guessRuleAction;
	private Action selectAllAction;
	private Action nextEventAction;
	private Action previousEventAction;
	private Action firstEventAction;
	private Action lastEventAction;	
	private Action refreshAction;
	private Action showLog;
	private Action setStartRecord;

	/*
	 * 
	 */
	class SetStartRecordDlg extends Dialog implements ModifyListener {

		protected int _recordNb;
		protected Text _recordText;
		
		public SetStartRecordDlg(Shell shell, int recordNb) {
			super(shell);
			
			_recordNb = recordNb;

		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(AcadEditorPlugin.getPlugin().getString("STR_SETSTART_ACTION"));
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			layout = new GridLayout();
			layout.numColumns = 2;
			layout.verticalSpacing = 10;
			layout.marginWidth = 10;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 300;
			result.setLayoutData(data);
			
			Label label = new Label(result, SWT.WRAP);
			label.setText(AcadEditorPlugin.getPlugin().getString("STR_SETSTART_DESC"));
			data = new GridData();
			data.horizontalSpan = 2;
			data.widthHint = 280;			
			label.setLayoutData(data);
			
			label = new Label(result, SWT.NULL);
			label.setText(AcadEditorPlugin.getPlugin().getString("STR_REC_NUMBER"));
			_recordText = new Text(result, SWT.BORDER);
			_recordText.setLayoutData(GridUtil.createHorizontalFill());
			
			_recordText.addModifyListener(this);				
			
			_recordText.setText(String.valueOf(_recordNb));
			
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _recordText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
							_recordText.getText().trim() != "");
			}
		}

		protected void okPressed() {

			try {
				_recordNb = Integer.parseInt(_recordText.getText().trim());
			}
			catch(NumberFormatException exc)
			{
				_recordNb = 1;
			}						
			super.okPressed();
		}
		
		public int getRecordNumber()
		{
			return _recordNb;
		}

	}
	
	
	/**
	 * Constructor
	 */
	public LogView() {
		super();
		
	}

	/**
	 * @see IViewPart.init(IViewSite)
	 */
	public void init(IViewSite site) throws PartInitException {
		super.init(site);
	}

	/**
	 * Initializes this view with the given view site.  A memento is passed to
 	 * the view which contains a snapshot of the views state from a previous
	 * session.  	
	 */
	public void init(IViewSite site,IMemento memento) throws PartInitException {
		init(site);
		
		enableEditActions();
		
	}
	
	/**
	 * @see IWorkbenchPart#createPartControl(Composite)
	 */
	public void createPartControl(Composite parent) {
		
		SashForm sashForm = new SashForm(parent, SWT.HORIZONTAL);
		sashForm.setLayout(new FillLayout());
		
		// Create viewer.
		eventsViewer = new TextViewer(sashForm, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
		eventsViewer.setEditable(false);
		
		
		// Create menu and toolbars.
		createActions();
		createToolbar();
		createContextMenu();

		set_Title();
		
		WorkbenchHelp.setHelp(parent, ContextIds.ACAD_EDITOR_LOGVIEW);
						
	}
	
	/**
	 * @see WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		if(eventsViewer != null && eventsViewer.getControl().isDisposed())
			eventsViewer.getControl().setFocus();
	}

	/**
	 * Create the actions.
	 */
	public void createActions() {
		
		showLog = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LOG_ACTION")) {
		public void run() {
							showLog();
						}
					};
		showLog.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LOG_ACTION")); 		
		AcadEditorPluginImages.setImageDescriptors(showLog,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_SHOW_LOG);	
		
		nextEventAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_NEXTE_ACTION")) {
			public void run() { 
								showNextEvent();
							}
						};
		nextEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_NEXTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(nextEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_NEXT_EVET);	
		
		previousEventAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_PREVE_ACTION")) {
			public void run() {
								showPreviousEvent();
							}
						};
		previousEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_PREVE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(previousEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_PREVIOUS_EVENT);	
		
		firstEventAction= new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_FIRSTE_ACTION")) {
		public void run() {
							showFirstEvent();
						}
					};
		firstEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_FIRSTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(firstEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_FIRST_EVENT);	
		
		lastEventAction= new Action(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LASTE_ACTION")) {
		public void run() {
							showLastEvent();
						}
					};
		lastEventAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SHOW_LASTE_ACTION")); 
		AcadEditorPluginImages.setImageDescriptors(lastEventAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_LAST_EVENT);	
			
		refreshAction = new Action(AcadEditorPlugin.getPlugin().getString("STR_RERUN_ACTION")) {
			public void run() { 
				refreshData();
			}
		};
		refreshAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_RERUN_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(refreshAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_REAPPLY);	
		
		createRuleAction = new Action(AcadEditorPlugin.getPlugin().getString("ADD_HASH_ACTION")) {
			public void run() { 
				createRule();
			}
		};
		createRuleAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("ADD_HASH_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(createRuleAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_CREATE_RULE);	
		
		guessRuleAction = new Action(AcadEditorPlugin.getPlugin().getString("GUESS_RULE_ACTION")) {
			public void run() {
				guessRule();
			}
		};
		guessRuleAction.setToolTipText(AcadEditorPlugin.getPlugin().getString("GUESS_RULE_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(guessRuleAction,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_GUESS_RULE);	
		
		setStartRecord = new Action(AcadEditorPlugin.getPlugin().getString("STR_SETSTART_ACTION")) {
			public void run() { 
				openSetStartRecordDlg();
			}
		};
		setStartRecord.setToolTipText(AcadEditorPlugin.getPlugin().getString("STR_SETSTART_ACTION"));
		AcadEditorPluginImages.setImageDescriptors(setStartRecord,AcadEditorPluginImages.T_LCL,AcadEditorPluginImages.IMG_UI_SETSTART);	
		
		
		// Add selection listener.
		eventsViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateActionEnablement();
			}
		});
	}

	/**
	 * Create toolbar.
	 */
	private void createToolbar() {
		IToolBarManager mgr = getViewSite().getActionBars().getToolBarManager();
		mgr.add(refreshAction);
		mgr.add(setStartRecord);		
		mgr.add(new Separator());
		mgr.add(firstEventAction);		
		mgr.add(previousEventAction);
		mgr.add(nextEventAction);
		mgr.add(lastEventAction);
		mgr.add(new Separator());
		mgr.add(showLog);
	}
		
	/**
	 * Create context menu.
	 */
	private void createContextMenu() {
		// Create menu manager.
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});
		
		// Create menu.
		Menu menu = menuMgr.createContextMenu(eventsViewer.getControl());
		eventsViewer.getControl().setMenu(menu);
		
		// Register menu for extension.
		getSite().registerContextMenu(menuMgr, eventsViewer);
	}

	/**
	 * Hook global actions
	 */
	private void hookGlobalActions() {
		IActionBars bars = getViewSite().getActionBars();
		bars.setGlobalActionHandler(IWorkbenchActionConstants.SELECT_ALL, selectAllAction);
		bars.setGlobalActionHandler(IWorkbenchActionConstants.DELETE, guessRuleAction);
		eventsViewer.getControl().addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent event) {
				if (event.character == SWT.DEL && 
					event.stateMask == 0 && 
				guessRuleAction.isEnabled()) 
				{
					guessRuleAction.run();
				}
			}
		});
	}
		
	private void fillContextMenu(IMenuManager mgr) {
		mgr.add(createRuleAction);
		mgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		mgr.add(guessRuleAction);

	}

	private void updateActionEnablement() {
		TextSelection sel = (TextSelection)eventsViewer.getSelection();
		createRuleAction.setEnabled(!sel.isEmpty());
		guessRuleAction.setEnabled(!sel.isEmpty());
	}
					
	private void showNextEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;

		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getNextEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		set_Title();
		enableActions();
		
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)		
		   contentView.showLog();
		
	}
	
	private void showPreviousEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getPreviousEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		
		set_Title();
		enableActions();
		
	}
	
	protected void refreshData()
	{
		BusyIndicator.showWhile(Display.getCurrent(), new Runnable() {
			public void run() {
				AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
				
				if(model != null) {
					model.loadTemplateFile();
				}				
		
			}
		});
		
	}
	
	
	public void set_Title()
	{
		AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
		
		StringBuffer title = (new StringBuffer(LOG_TITLE)).append(" ");
		
		if(model != null)
		{
			long currentEvent = model.getCurrentCount();
			if(currentEvent >=0 ) 
			  currentEvent++;
			else 
			  currentEvent = 0;  
			
			if(model.getContextInstance() == null || model.getContextInstance().eContainer() == null)
			{  
				title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_EXTRACTOR"));
			}
			else
			{
				SensorConfigType sensor = AcadGuiModelManager.getLogForConfiguration(model);
				if(sensor != null)
				{  
//					 TODO check this fragment
					//					SensorType type = sensor.getType();
//				
//					switch(type.getValue())
//					{
//						case SensorType.ADAPTER_CBE_SENSOR:
//						  title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
//						  break;
//						case SensorType.SINGLE_FILE_SENSOR:
//						  SingleFileSensorType fileSensor = sensor.getSingleFileSensor();
//						  if(fileSensor == null)
//							title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
//						  else
//							title.append(fileSensor.getDirectory()).append(File.separator)
//								  .append(fileSensor.getFileName());	
//						  break;
//					}
					PropertyType fileName = AcadGuiModel.getProperty(sensor,"fileName");
					PropertyType directory = AcadGuiModel.getProperty(sensor,"directory");
					if(fileName!=null && directory!=null)
					{
						title.append(directory.getPropertyValue()).append(File.separator)
						  .append(fileName.getPropertyValue());	
						
					}
					else
						  title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
						
					
					String msg = AcadEditorPlugin.getPlugin().getString("STR_LOG_TITLE", new Object[] {String.valueOf(currentEvent), String.valueOf(model.getEventCount())});
					title.append(" ").append(msg);						
				}
				else
				{
					title.append(AcadEditorPlugin.getPlugin().getString("STR_NO_SENSOR"));
				}
				
			}
				 				 			
		}
		
		super.setTitle(title.toString());
		
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)		
		   contentView.set_Title();
		ResultView resultView = ResultView.getResultView(false);
		if(resultView != null)		
		   resultView.set_Title();
		
		
	}
	
	/**
	 * 
	 *
	 */
	public void showFirstEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		  
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getFirstEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
		}
		set_Title();
		enableActions();
	}
	
	/**
	 * 
	 *
	 */
	private void showLastEvent() 
	{
		if(AcadGuiModelManager.getCurrentModel() == null)
		  return;
		
		ResultQueueEntry result=AcadGuiModelManager.getCurrentModel().getLastEvent();
		String extractorOutput=((MessageString)result.extractorOutput).getValue();
		eventsViewer.setDocument(new Document(extractorOutput));
		if(result.formatterOutput!=null) {
			ResultView resultsView=ResultView.getResultView(true);
			if(resultsView!=null) {
				String formatterOutput=(CbeFormatter.toCanonicalXMLString(((ICommonBaseEvent)result.formatterOutput), true));
				resultsView.setResult(formatterOutput);
			}
	
		}
		
		set_Title();
		enableActions();
	}
	
	private void createRule() 
	{
		MessageDialog.openInformation(Display.getCurrent().getActiveShell()
		   , "Generic Adaptor Message"
		   , "Not implemented");
	}
	
	
	private void guessRule() {
		MessageDialog.openInformation(Display.getCurrent().getActiveShell()
		   , "Generic Adaptor Message"
		   , "Not implemented");
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPart#dispose()
	 */
	public void dispose() 
	{
		// Auto-generated method stub
		super.dispose();
	}
	
	public void setContent(String content)
	{
		if(eventsViewer != null)		
			eventsViewer.setDocument(new Document(content));
	}
	
	public String getText()
	{
		return ((StyledText)eventsViewer.getControl()).getText();
	}
	/**
	 * 
	 * @return
	 */
	public static final LogView getLogView(boolean show)
	{
		try {
			IWorkbenchPage page = AcadEditorPlugin.getDefault().getActivePage();
			if(page == null)
			  return null;
			
			IViewPart view =
				page.findView(AcadPerspective.ID_LOGVIEW);
				
			if (view != null)
			{
				if(show)
					page.showView(AcadPerspective.ID_LOGVIEW);
			}
			else if(show)
			{
				page.showView(AcadPerspective.ID_LOGVIEW);
									
			}
			if(view instanceof LogView)
				return (LogView) view;
		}
		catch(Exception exc)
		{
			exc.printStackTrace();
			return null;
		}
		
		return null;
	}

    public void reset() {
    	setContent("");
    	
		ResultView resultView = ResultView.getResultView(false);
		if(resultView != null)
		  resultView.reset();
		ContentView contentView = ContentView.getContentView(false);
		if(contentView != null)
		  contentView.reset();    
    	
    }
    
	/**
	 * @return
	 */
	public Action getCreateRuleAction() {
		return createRuleAction;
	}

	public void enableActions()
	{
		AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
		if(model == null || model.getContextInstance() == null)
		{
			previousEventAction.setEnabled(false);
			nextEventAction.setEnabled(false);
			firstEventAction.setEnabled(false);
			lastEventAction.setEnabled(false);
			createRuleAction.setEnabled(false);
			showLog.setEnabled(false);
			
			return;
			
		}

		firstEventAction.setEnabled(model.getEventCount() > 0);
		lastEventAction.setEnabled(model.getEventCount() > 0);
		createRuleAction.setEnabled(true);

		showLog.setEnabled(true);
		  
		previousEventAction.setEnabled(model.hasPrevious());
		nextEventAction.setEnabled(model.hasNext());
	}


	protected void enableEditActions()
	{
		ActionHandlerListener.DEFAULT.connectPart(this);
	}

	protected void showLog()
	{
		ContentView view = ContentView.getContentView(true);
		if(view != null)
			view.showLog();
	}

	/**
	 * 
	 *
	 */
	protected void openSetStartRecordDlg() 
	{
		AcadGuiModel model = AcadGuiModelManager.getCurrentModel();
		if(model == null)
		  return;
		
		SetStartRecordDlg dialog =
			new SetStartRecordDlg(Display.getDefault().getActiveShell(), model.getStartRecord());
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {
		
			  model.setStartRecord(dialog.getRecordNumber());
		}
		  
		set_Title();
		enableActions();
	}

}
