/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.hyades.statistical.ui.*;
import org.eclipse.hyades.statistical.ui.variableloader.internal.*;
import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.*;
import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.*;

import org.eclipse.emf.common.util.*;

import org.eclipse.hyades.model.statistical.*;

import org.eclipse.hyades.models.cbe.*;

import java.util.*;
import java.text.*;

import org.eclipse.swt.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.custom.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.dnd.*;

import org.eclipse.hyades.models.hierarchy.*;

import org.eclipse.swt.events.*;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import org.eclipse.emf.ecore.*;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.*;
import org.eclipse.emf.ecore.resource.impl.*;

import java.io.*;

import org.eclipse.ui.dialogs.*;
import org.eclipse.ui.part.*;

import org.eclipse.emf.ecore.xmi.*;

/**
 * Creates a viewer/editor of a statistical model and its aggregates
 * the aim is to be able to see some kind of tree of the statistical model
 * and then pick any set of observations you like to be graphed
 * 
 */
public class CopyOfStatisticalModelGraphViewer implements DisposeListener, SelectionListener, DropTargetListener, MouseListener, StatConInterface {

private static final boolean DEBUG_LOADING = true;

protected ArrayList disposables = new ArrayList();

protected HashMap module_refs = new HashMap();
protected ArrayList modules = new ArrayList();

protected int next = 0;
protected int[] colours = new int[]{
	SWT.COLOR_RED,	
	SWT.COLOR_GREEN,	
	SWT.COLOR_BLUE,	
	SWT.COLOR_YELLOW,	
	SWT.COLOR_BLACK,	
	SWT.COLOR_CYAN,	
	SWT.COLOR_MAGENTA,	
};

protected TabFolder tabfolder;
protected SashForm sashfolder;
protected Composite leftsash;
protected Composite rightsash;

protected GraphWindow graph_window;

protected Tree chooser_tree;

protected Menu chooser_menu;
protected ArrayList chooser_disposables;
protected MenuItem chooser_add;

//SDDescriptor selected_model;

protected ImageManager img;

protected HashMap reverse_map = new HashMap();			//key=SDDescriptor, value=TreeItem
	
protected HashMap trcagent_map = new HashMap();			//key=TreeItem, value=TRCAgent
protected HashMap msglist_map = new HashMap();			//key=TreeItem, value=EList (TRCAgent.getDefaultEvents())
protected HashMap sddescriptor_map = new HashMap();		//key=TreeItem, value=SDDescriptor
protected HashMap counter_map = new HashMap();			//key=TreeItem, value=SDSnapshotObservation
protected HashMap modifiable_map = new HashMap();			//key=TreeItem, value=SDRepresentation

protected HashMap graph_map = new HashMap();				//key=TreeItem, value=Graph
protected HashMap bar_map = new HashMap();				//key=TreeItem, value=ModifiableVariableBar
protected HashMap bar_slider_map = new HashMap();			//key=TreeItem, value=ZoomSlider
protected HashMap source_map = new HashMap();				//key=TreeItem, value=GraphSource
protected HashMap source_graph_map = new HashMap();		//key=GraphSource, value=Graph

protected Dirtiable dirtiable;

protected ArrayList agents = new ArrayList();

protected ArrayList global_actions = new ArrayList();
protected ArrayList drop_listeners = new ArrayList();
protected ArrayList root_nodes = new ArrayList();

protected ArrayList model_change_listeners = new ArrayList();
protected ArrayList agent_change_listeners = new ArrayList();
protected ArrayList node_change_listeners = new ArrayList();

protected URI current_project;

	public void setCurrentProjectURI(URI uri) {
		current_project = uri;	
	}
	
	public URI getCurrentProjectURI() {
		return current_project;	
	}
	
	public void showTab(int i) {
		graph_window.showTab(i);	
	}

	public CopyOfStatisticalModelGraphViewer(Composite parent, int style, boolean tabfolder) {		
		EditorPlugin.DBG.info("constructor");
		initAsSash(parent);
	}
	
	public CopyOfStatisticalModelGraphViewer(Composite parent, int style, boolean tabfolder, Dirtiable d, URI current_project) {		
		EditorPlugin.DBG.info("constructor");
		dirtiable = d;

		this.current_project = current_project;

		initAsSash(parent);

		loadModules();

	}

	public void addModuleNodeChangeListener(ModuleNodeChangeListener listener) {
		node_change_listeners.add(listener);	
	}

	public void removeModuleNodeChangeListener(ModuleNodeChangeListener listener) {
		node_change_listeners.remove(listener);	
	}
	
	public void addModelChangeListener(ModelChangeListener listener) {
		model_change_listeners.add(listener);	
	}

	public void removeModelChangeListener(ModelChangeListener listener) {
		model_change_listeners.remove(listener);	
	}

	public void addAgentChangeListener(AgentChangeListener listener) {
		agent_change_listeners.add(listener);	
	}

	public void removeAgentChangeListener(AgentChangeListener listener) {
		agent_change_listeners.remove(listener);	
	}

	public String[] getModuleRefs() {
		String[] refs = new String[modules.size()];
		for (int i = 0; i < modules.size(); i++) {
			refs[i] = ((StatConModule)modules.get(i)).getModuleRef();
		}
		return refs;
	}
	
	public StatConModule getModuleByRef(String ref) {
		return (StatConModule)module_refs.get(ref);
	}
	
	public void unloadModules() {
		EditorPlugin.DBG.info("unloading modules");
		
		for (int i = 0; i < modules.size(); i++) {
			try {
				StatConModule module = (StatConModule)modules.get(i);
				EditorPlugin.DBG.info("unloading module "+i);
				module.unload();	
			} catch (Throwable t) {
				EditorPlugin.DBG.error("failed to unload module",t);
			}
		}
		
		modules.clear();
		module_refs.clear();
	}
	
	public void loadModules() {
		EditorPlugin.DBG.info("loading modules");
		
		IPluginRegistry registry = Platform.getPluginRegistry();
		IExtensionPoint point = registry.getExtensionPoint(EditorPlugin.PACKAGE_STATCON_MODULE);
		IExtension[] extensions = point.getExtensions();

		EditorPlugin.DBG.info("found "+extensions.length+" modules");

		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] configs = extensions[i].getConfigurationElements();
			IPluginDescriptor pdescriptor = extensions[i].getDeclaringPluginDescriptor();
			
			for (int z = 0; z < configs.length; z++) {
				String clazz = configs[z].getAttribute("class");

				StatConModule module = null;

				try {			
					EditorPlugin.DBG.info("loading module "+i);
					ClassLoader cloader = pdescriptor.getPluginClassLoader();	
					Class c = cloader.loadClass(clazz);
					module = (StatConModule)c.newInstance();
					module.load(this);
					module_refs.put(module.getModuleRef(),module);
					modules.add(module);
					EditorPlugin.DBG.info("loaded OK");
				} catch (Throwable e) {
					try {
						modules.remove(module);
					} catch (Throwable x) {}
					try {
						module_refs.remove(module);
					} catch (Throwable x) {}
					EditorPlugin.DBG.logVisibleError(e,"Failed to load StatCon Module \""+clazz+"\" from plugin \""+pdescriptor+"\" ",false);
					EditorPlugin.DBG.error("failed to load module",e);
				}
			}	
		}

		EditorPlugin.DBG.info("finished loading modules");
	}		

	private BasicAction findAction(String id, ArrayList actions) {
		for (int i = 0; i < actions.size(); i++) {
			BasicAction action = (BasicAction)actions.get(i);
			if (action.id.equals(id)) {
				return action;
			}	
		}
		return null;
	}
	
	private RootNode getRootNodeName(String name) {
		for (int i = 0; i < root_nodes.size(); i++) {
			RootNode node = (RootNode)root_nodes.get(i);
			if (node.name.equals(name)) {
				return node;
			}	
		}
		return null;
	}
	private RootNode getRootNodeID(String id) {
		for (int i = 0; i < root_nodes.size(); i++) {
			RootNode node = (RootNode)root_nodes.get(i);
			if (node.id.equals(id)) {
				return node;
			}	
		}
		return null;
	}

	private StatAgent getStatAgent(TRCAgent agent) {
		for (int i = 0; i < agents.size(); i++) {
			StatAgent node = (StatAgent)agents.get(i);
			if (node.agent == agent) {
				return node;
			}	
		}
		return null;
	}

	private StatAgent getStatAgent(EList events) {
		for (int i = 0; i < agents.size(); i++) {
			StatAgent node = (StatAgent)agents.get(i);
			if (node.agent.getDefaultEvents() == events) {
				return node;
			}	
		}
		return null;
	}

	public void addGlobalAction(String id, String name, Image image, Runnable runnable) {
		BasicAction action = findAction(id,global_actions);
		if (action == null) {
			action = new BasicAction();
			global_actions.add(action);
		}
		action.id = id;
		action.name = name;
		action.image = image;
		action.runnable = runnable;
		action.menuitem = new MenuItem(chooser_menu,0);
		action.menuitem.setText(name);
		action.menuitem.setImage(image);
		action.menuitem.addSelectionListener(action);
	}

	public void removeGlobalAction(String id) {
		BasicAction action = findAction(id,global_actions);
		if (action != null) {
			action.dispose();
			global_actions.remove(action);
		}	
	}

	public void addDropTargetListener(String extension, StatConDropListener listener) {
		DropListener drop = new DropListener();
		drop.extension = extension;
		drop.listener = listener;
		
		drop_listeners.add(drop);
	}
	
	public void removeDropTargetListener(String extension) {
		for (int i = 0; i < drop_listeners.size(); i++) {
			DropListener drop = (DropListener)drop_listeners.get(i);
			if (drop.extension.equals(extension)) {
				drop_listeners.remove(drop);
			}	
		}
	}
	public void addModuleNode(String name, String id, Image image) throws EntityExistsException {
		if (getRootNodeID(id) != null) {
			throw new EntityExistsException(EditorPlugin.getString("ROOT_NODE_EXISTS")+" "+name+" "+EditorPlugin.getString("ROOT_NODE_EXISTS_2"));	
		}
		
		RootNode node = new RootNode();
		node.name = name;
		node.image = image;
		node.id = id;

		node.item = new TreeItem(chooser_tree,0,0);
		node.item.setText(name);
		node.item.setImage(image);
		node.item.setGrayed(true);

		root_nodes.add(node);
		
		for (int i = 0; i < node_change_listeners.size(); i++) {
			ModuleNodeChangeListener listener = (ModuleNodeChangeListener)node_change_listeners.get(i);	
			listener.nodeAdded(id);
		}
	}

	public void addModuleNode(String name, String id, String parent_id, Image image) throws EntityExistsException, NullPointerException {
		if (getRootNodeID(id) != null) {
			throw new EntityExistsException(EditorPlugin.getString("ROOT_NODE_EXISTS")+" "+name+" "+EditorPlugin.getString("ROOT_NODE_EXISTS_2"));	
		}

		RootNode parent = getRootNodeID(parent_id);
		if (parent == null) throw new NullPointerException("");
		
		RootNode node = new RootNode();
		node.name = name;
		node.image = image;
		node.id = id;
		node.parent_id = parent_id;

		node.item = new TreeItem(parent.item,0,0);
		node.item.setText(name);
		node.item.setImage(image);
		node.item.setGrayed(true);

		parent.node_children.add(node);

		root_nodes.add(node);

		for (int i = 0; i < node_change_listeners.size(); i++) {
			ModuleNodeChangeListener listener = (ModuleNodeChangeListener)node_change_listeners.get(i);	
			listener.nodeAdded(id);
		}
	}
	
	public void removeModuleNode(String id) {
		RootNode node = getRootNodeID(id);
		if (node != null) {
			
			if (node.parent_id != null) {
				RootNode parent = getRootNodeID(node.parent_id);
				parent.node_children.remove(node);
			}

			for (int i = 0; i < node.node_children.size(); i++) {
				RootNode child = (RootNode)node.node_children.get(i);
				removeModuleNode(child.id);	
			}
			
			node.dispose();
			root_nodes.remove(node);
		}	

		for (int i = 0; i < node_change_listeners.size(); i++) {
			ModuleNodeChangeListener listener = (ModuleNodeChangeListener)node_change_listeners.get(i);	
			listener.nodeRemoved(id);
		}
	}
	
	public void addModuleNodeAction(String node_id, String action_id, String action_name, Image action_image, Runnable runnable) {
		RootNode node = getRootNodeID(node_id);
		if (node != null) {
			BasicAction action = findAction(action_id,node.actions); 
			if (action == null) {
				action = new BasicAction();
				node.actions.add(action);	
			}

			action.id = action_id;
			action.name = action_name;
			action.image = action_image;
			action.runnable = runnable;
	
		}	
	}

	public void removeModuleNodeAction(String node_id, String action_id) {
		RootNode node = getRootNodeID(node_id);
		if (node != null) {
			for (int i = 0; i < node.actions.size(); i++) {
				BasicAction action = (BasicAction) node.actions.get(i);
				if (action.id.equals(action_id)) {
					node.actions.remove(i);
					break;
				}
			}
		}
	}
	
	public SDDescriptor[] addAgent(IFile ifile) throws IOException {
		return addAgent(ifile.getProjectRelativePath());
	}
	
	private Resource createResource(URI uri) {

		ResourceSet resourceSet = new ResourceSetImpl();
		Resource resource = resourceSet.getResource(uri,false);
		if (resource == null) return resourceSet.createResource(uri);
		return resource;		
	}
	
	public SDDescriptor[] addAgent(IPath ipath) throws IOException {

		EditorPlugin.DBG.info("add trace agent");

		URI fileURI = URI.createPlatformResourceURI(ipath.toString());

		EditorPlugin.DBG.info("agent URI:"+fileURI);

		if (fileURI == null) throw new IOException(EditorPlugin.getString("NULL_URI_WHILE_LOADING")+" "+ipath);

		Resource resource = createResource(fileURI);
		if (resource == null) throw new IOException(EditorPlugin.getString("NULL_RESOURCE_WHILE_LOADING")+" "+ipath);

		if (resource instanceof XMLResource) {
			EditorPlugin.DBG.info("found XML resource for agent");
			XMLResource xresource = (XMLResource)resource;
			
			String msg = "";
			
			try {
				//try to load as a non-zip file
				EditorPlugin.DBG.info("trying to load resource as an unzipped file");
				xresource.setUseZip(false);
				xresource.load(Collections.EMPTY_MAP);
				EditorPlugin.DBG.info("loaded resource OK as an unzipped file");
			} catch (Throwable e) {

				EditorPlugin.DBG.info("load failed as an unzipped file");

				msg += e.toString()+" / ";

				resource = createResource(fileURI);
				xresource = (XMLResource)resource;
				
				try {
					//try to load as a zip file
					EditorPlugin.DBG.info("trying to load resource as a zipped file");
					xresource.setUseZip(true);
					xresource.load(Collections.EMPTY_MAP);
					EditorPlugin.DBG.info("loaded resource OK as a zipped file");
				} catch (Throwable t) {
					msg += t.toString();
					throw new IOException(msg); 
				}
			
			}
			
		} else {
			EditorPlugin.DBG.info("found a non-XML resource for agent - trying to load anyway");
			if (!resource.isLoaded()) resource.load(Collections.EMPTY_MAP);
		}

		EList list = resource.getContents();
		EditorPlugin.DBG.info("got contents of agent resource (size "+list.size()+")");
		
		return addAgentFromTree(list,fileURI);
	}
	
	private SDDescriptor[] addAgentFromTree(EList list, URI fileURI) {
		
		EditorPlugin.DBG.info("Searching Resource Contents ("+list.size()+")");
		
		//check the list for a TRCAgent, if we find one add it
		for (int k = 0; k < list.size(); k++) {
			Object o = list.get(k);
			EditorPlugin.DBG.info("resource content "+k+" = "+o.getClass());

			if (o instanceof TRCAgentProxy) {
				TRCAgentProxy proxy = (TRCAgentProxy)o;
				return addAgent(proxy.getAgent(),fileURI);
			}

			if (o instanceof TRCAgent) {
				return addAgent((TRCAgent)o,fileURI);
			}
		}
		//no TRCAgent found - recursively search on all its children
		SDDescriptor[] descriptors = null;
		for (int k = 0; k < list.size(); k++) {
			EObject o = (EObject)list.get(k);
			if (o instanceof TRCProcessProxy) {
				EditorPlugin.DBG.info("resource subcontent ProcessProxy");
				descriptors = addAgentFromTree(((TRCProcessProxy)o).getAgentProxies(),fileURI);
			} else if (o instanceof TRCNode) {
				EditorPlugin.DBG.info("resource subcontent Node");
				descriptors = addAgentFromTree(((TRCNode)o).getProcessProxies(),fileURI);
			} else if (o instanceof TRCMonitor) {
				EditorPlugin.DBG.info("resource subcontent Monitor");
				descriptors = addAgentFromTree(((TRCMonitor)o).getNodes(),fileURI);
			} else {
			EditorPlugin.DBG.info("resource subcontent "+k+" = "+o.getClass());
				descriptors = addAgentFromTree(o.eContents(),fileURI);
			}
			if (descriptors != null) {
				return descriptors;
			}
		}
		return null;
	}

	public SDDescriptor[] addAgent(TRCAgent agent) {
		return addAgent(agent,null);
	}

	SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
	private StringBuffer cbeToString(CBECommonBaseEvent e, StringBuffer sb) {
		
		Date create = new Date((long)e.getCreationTime());
		
		sdf.format(create,sb,new FieldPosition(0));
		sb.append(" ");
	
		short sev = e.getSeverity();
		if (sev <= 10) {
			sb.append("(info)");
		} else if (sev <= 30) {
			sb.append("(warning)");
		} else if (sev <= 40) {
			sb.append("(error)");
		} else if (sev <= 60) {
			sb.append("(critical)");
		}

		sb.append(" ");

		if (e.getRepeatCount() > 0) {
			sb.append("(x");				
			sb.append(e.getRepeatCount());
			sb.append(") ");				
		}

//		sb.append(e.getAgent().getName());
//
//		sb.append(" / ");

		sb.append(e.getSourceComponentId().getComponent());
		
		if (e.getSourceComponentId().getSubComponent() != null) {
			sb.append(" / ");
			sb.append(e.getSourceComponentId().getSubComponent());
		}

		sb.append(" - ");
		sb.append(e.getMsg());

		sb.append("\n");

		return sb;
	}
	
	private boolean filterCBE(CBECommonBaseEvent e) {
		
		TRCAgent agent = e.getAgent();
		
		if (agent == null) return false;
		
		StatAgent stagent = getStatAgent(agent);
		
		if (stagent == null) return false;
		if (stagent.msg_info_item == null) return false;
		
		//check general message filtering
		if (!stagent.msg_item.getChecked()) return false;
		
		//check severity filtering
		
		int severity = e.getSeverity();
		if (severity <= 10) {
			if (!stagent.msg_info_item.getChecked()) return false; 
		} else if (severity <= 30) {
			if (!stagent.msg_warn_item.getChecked()) return false; 
		} else {
			if (!stagent.msg_error_item.getChecked()) return false; 
		}
		
		//check location filtering
		CBEComponentIdentification sid = e.getSourceComponentId();
		String component = sid.getComponent();
		String subcomponent = sid.getSubComponent();
		
		TreeItem citem = (TreeItem)stagent.msg_locations.get(component);
		if (citem == null) {
			citem = new TreeItem(stagent.msg_loc_item,0);
			citem.setImage(img.getImage(ImageManager.IMG_SMODEL_CBESOURCE));
			citem.setText(component);
			citem.setChecked(true);
			stagent.msg_locations.put(component,citem);
		}
		if (!citem.getChecked()) return false;
		
		if (subcomponent != null) {
			String key = component+"<%$%~%>"+subcomponent;
			TreeItem sitem = (TreeItem)stagent.msg_locations.get(key);
			if (sitem == null) {
				sitem = new TreeItem(citem,0);
				sitem.setImage(img.getImage(ImageManager.IMG_SMODEL_CBESOURCE));
				sitem.setText(subcomponent);
				sitem.setChecked(true);
				stagent.msg_locations.put(key,sitem);
			}
			if (!sitem.getChecked()) return false;
		}
		
		TreeItem[] items = stagent.msg_loc_item.getItems();

		return true;
	}
	
	private int appendCBEs(java.util.List events, int index) {
		StringBuffer sb = new StringBuffer();
		for (int i = index; i < events.size(); i++) {
			CBECommonBaseEvent e = (CBECommonBaseEvent)events.get(i);
			if (filterCBE(e)) {
				cbeToString(e,sb);
			}
		}
		graph_window.appendCBEText(sb.toString());
		return events.size();
	}

	private int setCBEs(java.util.List events, int index) {
		StringBuffer sb = new StringBuffer();
		for (int i = index; i < events.size(); i++) {
			CBECommonBaseEvent e = (CBECommonBaseEvent)events.get(i);
			if (filterCBE(e)) {
				cbeToString(e,sb);
			}
		}
		graph_window.setCBEText(sb.toString());
		return events.size();
	}

	private void sortCBEs(java.util.List events) {
		Collections.sort(events,new CBEComparator());	
	}

	private void appendCBEs(StatAgent agent) {
		if (agent.show_messages) {
			//append all unappended messages from this agent
			agent.msgindex = appendCBEs(agent.agent.getDefaultEvents(),agent.msgindex);
		}	
	}

	private void appendAllCBEs() {
		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);
			//append all unappended messages from this agent
			agent.msgindex = appendCBEs(agent.agent.getDefaultEvents(),agent.msgindex);
		}	
	}
	
	private void redoAllCBEs() {
		ArrayList all_events = new ArrayList();
		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);

			if (agent.show_messages) {
				agent.msgindex = (agent.agent.getDefaultEvents().size());
				all_events.addAll(agent.agent.getDefaultEvents());
			}

		}
		sortCBEs(all_events);
		setCBEs(all_events,0);
	}

	public void updateCBELog() {
		appendAllCBEs();
	}

	public void reloadCBELog() {
		redoAllCBEs();
	}

	public SDDescriptor[] updateAgent(TRCAgent agent) {

		for (int i = 0; i < agents.size(); i++) {
			StatAgent node = (StatAgent)agents.get(i);
			EditorPlugin.DBG.info("agent "+i+" = "+node.agent.getName());
		}

		StatAgent statagent = getStatAgent(agent);
		
		if (statagent == null) {
			EditorPlugin.DBG.info("update requested on non-existent agent "+agent.getName());
			return null;
		}

		EList direct_children = agent.eContents();

		EditorPlugin.DBG.info("checking agent's "+direct_children.size()+" children for Statistical Models");

		ArrayList added = new ArrayList();
	
		for (int z = 0; z < direct_children.size(); z++) {
			Object co = direct_children.get(z);

			EditorPlugin.DBG.info("agent child "+z+" = "+co.getClass());
		
			if (co instanceof SDDescriptor) {
				EditorPlugin.DBG.info("adding agent SDDescriptor (statistical model)");

				//if model root already there, update it, otherwise add it
				if (statagent.model_roots.contains(co)) {
					updateModel((SDDescriptor)co,statagent);
					added.add(co);
				} else { //new model - add it
					addModel((SDDescriptor)co,statagent);
					added.add(co);
				}
			}
		}

		SDDescriptor[] des = new SDDescriptor[added.size()];
		for (int i = 0; i < des.length; i++) {
			des[i] = (SDDescriptor)added.get(i);	
		}

		updateTreeItems();

		return des;
	}		

	public SDDescriptor[] addAgent(TRCAgent agent, URI fileURI) {

		if (agent == null) {
			EditorPlugin.DBG.logVisibleError(new Exception("Null Agent"),"Null Agent "+fileURI,false);
			return null;
		}
		if (agent.getName() == null) {
			EditorPlugin.DBG.logVisibleError(new Exception("Null Agent Name"),"Null Agent Name "+fileURI,false);
			return null;
		}

		//
		// This bit is just for our info
		//
		EditorPlugin.DBG.info("add valid agent "+agent.getName()+" / "+fileURI);
		for (int i = 0; i < agents.size(); i++) {
			StatAgent node = (StatAgent)agents.get(i);
			EditorPlugin.DBG.info("agent "+i+" = "+node.agent.getName());
		}
		
		if (fileURI != null) {
			//dont add models that reference an already loaded file
			//(dont load same model twice)
			for (int i = 0; i < agents.size(); i++) {
				StatAgent node = (StatAgent)agents.get(i);
				URI uri = node.uri;
				if (uri != null) {
					if (uri.equals(fileURI)) {
						EditorPlugin.DBG.info("URI matched to an existing agent - returning without adding");
						return null;	
					}	
				}	
			}
		}

		EditorPlugin.DBG.info("creating new agent");

		//
		// Create the agent object and add it
		//		
		StatAgent statagent = getStatAgent(agent);
		if (statagent == null) {
			statagent = new StatAgent();
			agents.add(statagent);
		}
		statagent.agent = agent;
		statagent.uri = fileURI;
		
		TreeItem item = new TreeItem(chooser_tree,0);	
		item.setText(agent.getName());
		item.setChecked(true);
		item.setGrayed(false);

		trcagent_map.put(item,agent);
		reverse_map.put(agent,item);

		//
		// Add a remove action
		//
		addAgentAction(statagent.agent,"AGENT_REMOVE",EditorPlugin.getString("AGENT_REMOVE"),img.getImage(ImageManager.IMG_SMODEL_REMOVE),new AgentRemoveAction(statagent.agent));

		//
		// Add all the messages from this agent
		//
		EList events = agent.getDefaultEvents();
		
		TreeItem msg = new TreeItem(item,0);	
		msg.setImage(img.getImage(ImageManager.IMG_SMODEL_CBEMESSAGE));
		msg.setText("Messages");
		msg.setChecked(true);
		msg.setGrayed(false);

		TreeItem msg_severity = new TreeItem(msg,0);	
		msg_severity.setImage(img.getImage(ImageManager.IMG_SMODEL_CBESEVERITY));
		msg_severity.setText("Severity");
		msg_severity.setChecked(true);
		msg_severity.setGrayed(true);

		TreeItem msg_location = new TreeItem(msg,0);	
		msg_location.setImage(img.getImage(ImageManager.IMG_SMODEL_CBESOURCE));
		msg_location.setText("Source Locations");
		msg_location.setChecked(true);
		msg_location.setGrayed(true);
		
		TreeItem msg_info = new TreeItem(msg_severity,0);
		msg_info.setImage(img.getImage(ImageManager.IMG_SMODEL_CBEINFO));
		msg_info.setText("Info");
		msg_info.setChecked(true);
		msg_info.setGrayed(false);

		TreeItem msg_warning = new TreeItem(msg_severity,0);
		msg_warning.setImage(img.getImage(ImageManager.IMG_SMODEL_CBEWARNING));
		msg_warning.setText("Warning");
		msg_warning.setChecked(true);
		msg_warning.setGrayed(false);

		TreeItem msg_errors = new TreeItem(msg_severity,0);
		msg_errors.setImage(img.getImage(ImageManager.IMG_SMODEL_CBEERROR));
		msg_errors.setText("Error");
		msg_errors.setChecked(true);
		msg_errors.setGrayed(false);

		msglist_map.put(msg,events);
		reverse_map.put(events,msg);
		
		statagent.msg_item = msg;
		statagent.msg_info_item = msg_info;
		statagent.msg_warn_item = msg_warning;
		statagent.msg_error_item = msg_errors;
		statagent.msg_loc_item = msg_location;

		//
		// Add all CBE messages
		//	
//		appendCBEs(statagent);
		EditorPlugin.DBG.info("interleaving all agent CBE messages");
		redoAllCBEs();
		
		//
		// Add all statistical models under this agent
		//
		ArrayList added = new ArrayList();

		EList direct_children = agent.eContents();

		EditorPlugin.DBG.info("checking agent's "+direct_children.size()+" children for Statistical Models");
	
		for (int z = 0; z < direct_children.size(); z++) {
			Object co = direct_children.get(z);

			EditorPlugin.DBG.info("agent child "+z+" = "+co.getClass());
		
			if (co instanceof SDDescriptor) {
				EditorPlugin.DBG.info("adding agent SDDescriptor (statistical model)");

				if (!statagent.model_roots.contains(co)) {
					addModel((SDDescriptor)co,statagent);
					added.add(co);
				}
			}
		}

		SDDescriptor[] des = new SDDescriptor[added.size()];
		for (int i = 0; i < des.length; i++) {
			des[i] = (SDDescriptor)added.get(i);	
		}

		//
		// This bit is just for our information
		//
		EditorPlugin.DBG.info("agent added ok");
		for (int i = 0; i < agents.size(); i++) {
			StatAgent node = (StatAgent)agents.get(i);
			EditorPlugin.DBG.info("agent "+i+" = "+node.agent.getName());
		}

		EditorPlugin.DBG.info("notifying agent listeners");
		for (int i = 0; i < agent_change_listeners.size(); i++) {
			AgentChangeListener listener = (AgentChangeListener)agent_change_listeners.get(i);

			EditorPlugin.DBG.info("notifying agent listener "+listener.getClass().getName());

			try {
				listener.agentAdded(agent);
			} catch (Throwable t) {
				EditorPlugin.DBG.logVisibleError(t,"problem notifying listener of Agent Add",false);
			}	
		}
		
		updateTreeItems();
		
		return des;
	}

	public void updateModel(SDDescriptor model_node) {
		
		SDDescriptor parent = model_node.getParent();
		TreeItem parent_item = null;
		if (parent == null) {
			//descriptor has no parent - must be an agent root
			TRCAgent agent = model_node.getAgent();
			parent_item = (TreeItem)reverse_map.get(agent);
		} else {
			parent_item = (TreeItem)reverse_map.get(parent);
		}
		
		if (parent_item != null) {
			addModelToChooser(model_node,parent_item,true);
			EditorPlugin.DBG.info("model node updated - "+model_node.getName());
			treeItemSelected(parent_item);
		}
	}

	private void updateModel(SDDescriptor model_root, StatAgent statagent) {
		EditorPlugin.DBG.mark("updating model - "+model_root.getName());

		addModelToChooser(model_root,statagent.agent,true);	

		EditorPlugin.DBG.mark("model updated - "+model_root.getName());

		updateAgentImages();
	}

	private void addModel(SDDescriptor model_root, StatAgent statagent) {
			
		addModelToChooser(model_root,statagent.agent,false);
		statagent.model_roots.add(model_root);
		EditorPlugin.DBG.info("model added - "+model_root.getName());
//EditorPlugin.DBG.info("MODEL ROOT ADDED "+statagent.model_roots.size());			

//		addAgentAction(statagent.agent,"AGENT_REMOVE",EditorPlugin.getString("AGENT_REMOVE"),img.getImage(ImageManager.IMG_SMODEL_REMOVE),new AgentRemoveAction(statagent.agent));

		for (int i = 0; i < model_change_listeners.size(); i++) {
			ModelChangeListener listener = (ModelChangeListener)model_change_listeners.get(i);
			listener.modelAdded(model_root);	
		}

		EditorPlugin.DBG.info("notified listeners");

		updateAgentImages();
	}

	public void setAgentFile(TRCAgent trcagent, IFile ifile) {
		URI fileURI = URI.createPlatformResourceURI(ifile.getProjectRelativePath().toString());
		setAgentFile(trcagent,fileURI);		
	}

	public void setAgentFile(TRCAgent trcagent, URI model_uri) {
		StatAgent statmodel = getStatAgent(trcagent);

		EditorPlugin.DBG.info("set agent file "+trcagent.getName()+" / "+model_uri);
		
		if (statmodel != null) {
			URI uri = statmodel.uri;
			if (uri != null) {
				//model exists but they are different = dirty
				if (!uri.equals(model_uri)) {
					setDirty(true);	
				}
			} else {
				//model doesnt exist and model set to a real model = dirty
				if (model_uri != null) {
					setDirty(true);
				}		
			}

			statmodel.uri = model_uri;
		}
	}

	public void setAgentImage(TRCAgent trcagent, Image image) {
		EditorPlugin.DBG.info("set agent "+trcagent.getName()+" image");
		StatAgent agent = getStatAgent(trcagent);	
		if (agent != null) {
			agent.image = image;	
		}
		updateAgentImages();
	}

	private void updateAgentImages() {
		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);

			if (agent.agent != null) {
				TreeItem agent_item = (TreeItem)reverse_map.get(agent.agent);
				if (agent_item != null) {
					if (agent.image == null) {
						agent_item.setImage(img.getImage(ImageManager.IMG_SMODEL_TRCAGENT));
					} else {
						agent_item.setImage(agent.image);
					}		
				}
			}
				
		}		
	}
	
	public void addModelAction(TRCAgent trcagent, String action_id, String action_name, Image image, Runnable runnable) {
		StatAgent agent = getStatAgent(trcagent);
		if (agent == null) {
			//we allow adding actions before the model itself has been added
			agent = new StatAgent();
			agent.agent = trcagent;
			
			agents.add(agent);
		}
		BasicAction action = findAction(action_id,agent.modelactions);
		if (action == null) {
			action = new BasicAction();
			agent.modelactions.add(action);	
		}

		action.id = action_id;
		action.name = action_name;
		action.image = image;
		action.runnable = runnable;
	}

	public void removeModelAction(TRCAgent trcagent, String action_id) {
		StatAgent agent = getStatAgent(trcagent);
		if (agent != null) {
			for (int i = 0; i < agent.modelactions.size(); i++) {
				BasicAction action = (BasicAction)agent.modelactions.get(i);
				if (action.id.equals(action_id)) {
					agent.modelactions.remove(i);
					break;	
				}
			}
		}	
	}
	
	public void addAgentAction(TRCAgent trcagent, String action_id, String action_name, Image image, Runnable runnable) {
		StatAgent agent = getStatAgent(trcagent);
		if (agent == null) {
			//we allow adding actions before the model itself has been added
			agent = new StatAgent();
			agent.agent = trcagent;
			
			agents.add(agent);
		}
		BasicAction action = findAction(action_id,agent.actions);
		if (action == null) {
			action = new BasicAction();
			agent.actions.add(action);	
		}
		action.id = action_id;
		action.name = action_name;
		action.image = image;
		action.runnable = runnable;

	}

	public void removeAgentAction(TRCAgent trcagent, String action_id) {
		StatAgent agent = getStatAgent(trcagent);
		if (agent != null) {
			for (int i = 0; i < agent.actions.size(); i++) {
				BasicAction action = (BasicAction)agent.actions.get(i);
				if (action.id.equals(action_id)) {
					agent.actions.remove(i);
					break;	
				}
			}
		}	
	}

	public void setDirty(boolean b) {
		if (dirtiable != null) {
			dirtiable.setDirty(b);	
		}	
	}
	
	public Color parseColor(String col, Color def) {
		try {
			if (!col.startsWith("#")) throw new Exception();
			if (col.length() != 7) throw new Exception();

			String sr = col.substring(1,3);
			String sg = col.substring(3,5);
			String sb = col.substring(5,7);

			int r = Integer.parseInt(sr,16);
			int g = Integer.parseInt(sg,16);
			int b = Integer.parseInt(sb,16);

			RGB rgb = XMLConfigUtil.parseColor(col, def.getRGB());

			Color c = new Color(graph_window.getDisplay(),rgb);
			disposables.add(c);
			return c;
			
		} catch (Throwable e) {
			return def;
		}		
	}
	
	public void applyConfig(InputStream config_xml) throws Exception {
		
		try {
		
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			byte[] buf = new byte[4096];
			int n = 0;
			while (n != -1) {
				n = config_xml.read(buf,0,buf.length);
				if (n > 0) {
					bout.write(buf,0,n);	
				}	
			}
			String xml = new String(bout.toByteArray());		
			
			applyConfig(xml);
		
		} catch (Exception t) {
			EditorPlugin.DBG.warning("error applying configuration",t);
			throw t;	
		}
	}

	public void applyConfig(String config_xml) throws Exception {
		
		try {
			
			ByteArrayInputStream config_stream = new ByteArrayInputStream(config_xml.getBytes());
			Document doc = XMLConfigUtil.getDocumentBuilder().parse(config_stream);
	
			EditorPlugin.DBG.info("applying configuration");
			applyConfig(doc);
	
			EditorPlugin.DBG.info("applying configuration to all modules");
			//
			// Apply config to all modules
			//
			for (int i = 0; i < modules.size(); i++) {
				StatConModule module = (StatConModule)modules.get(i);
				module.applyConfig(config_xml);
			}

		} catch (Exception t) {
			EditorPlugin.DBG.warning("error applying configuration",t);
			throw t;	
		}
	}
	

	private void applyConfig(Document doc) throws Exception {
		
		try {
		
			Element doc_element = doc.getDocumentElement();
			
			if (doc_element.getTagName().equals("smgconfig") == false) {
				EditorPlugin.DBG.error("bad configuration - no valid root <smgconfig> element");
				throw new Exception(EditorPlugin.getString("BAD_ROOT_XML_NODE"));	
			}

			EditorPlugin.DBG.info("loading all module data");
			
			ArrayList moduledats = XMLConfigUtil.getAllElements(doc_element.getChildNodes(),"module_data");
			for (int k = 0; k < moduledats.size(); k++) {
				
				try {
				
					Element mod_dat = (Element)moduledats.get(k);
					String ref = mod_dat.getAttribute("ref");
					String data = mod_dat.getAttribute("data");
					
					StatConModule module = (StatConModule)module_refs.get(ref);
					if (module != null) {
						module.loadModuleData(HexData.hexStringToByteArray(data));
					} else {
						EditorPlugin.DBG.warning("Found data for missing module "+ref);	
					}
					
				} catch (Exception e) {
					EditorPlugin.DBG.error("Error loading data for module "+k,e);
				}
			}
			
			ArrayList xml_models = XMLConfigUtil.getAllElements(doc_element.getChildNodes(),"model");
			EditorPlugin.DBG.info("found "+xml_models.size()+" agent references");
	
			//make a first pass and load any models
			for (int i = 0; i < xml_models.size(); i++) {
				
				try {
				
					Element model = (Element)xml_models.get(i);
		
					String file = model.getAttribute("file");
		
					EditorPlugin.DBG.info("trying to load agent "+file);
		
					//get the model as a file			
	//				EditorPlugin.DBG.info("LOAD MODEL FILE "+ifile);
				
					IPath path = new Path(file); 
					IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
					IFile ifile = workspaceRoot.getFile(path);
	//EditorPlugin.DBG.info("File Path = "+ifile.getFullPath());
	//EditorPlugin.DBG.info("File Relative Path = "+ifile.getProjectRelativePath());
		
					addAgent(ifile);
	
				} catch (Throwable t) {
					EditorPlugin.DBG.logVisibleError(t,"Error loading agent from configuration",true);	
				}
			}
	
			ArrayList graphs = XMLConfigUtil.getAllElements(doc_element.getChildNodes(),"graph");
			EditorPlugin.DBG.info("found "+graphs.size()+" graph configurations");
			
			//make a second pass and apply any graph configurations
			for (int i = 0; i < graphs.size(); i++) {
				
				try {
				
					Element graph = (Element)graphs.get(i);
		
					if (EditorPlugin.DBG.check("configuration")) EditorPlugin.DBG.info("applying graph configuration");
		
					Color color = parseColor(graph.getAttribute("color"),graph_window.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
					boolean match_color = XMLConfigUtil.parseBoolean(graph.getAttribute("match_colors"),false);
		
					graph_window.setSlidersMatchGraph(match_color);
					graph_window.setGraphBackground(color);
		
					ArrayList ysliders = XMLConfigUtil.getAllElements(graph.getChildNodes(),"yslider");
					ArrayList xsliders = XMLConfigUtil.getAllElements(graph.getChildNodes(),"xslider");

					EditorPlugin.DBG.info("X slider count of "+xsliders.size());
					EditorPlugin.DBG.info("Y slider count of "+ysliders.size());
		
					//set the correct number of y sliders
					while (graph_window.getYSliderCount() > ysliders.size()) {
						graph_window.removeYSlider(0);
					}
					while (graph_window.getYSliderCount() < Math.max(1,ysliders.size())) {
						graph_window.addYSlider();
					}
		
					//set the correct number of x sliders
					while (graph_window.getXSliderCount() > xsliders.size()) {
						graph_window.removeXSlider(0);
					}
					while (graph_window.getXSliderCount() < Math.max(1,xsliders.size())) {
						graph_window.addXSlider();
					}
		
					for (int k = 0; k < ysliders.size(); k++) {
						Element yslider = (Element)ysliders.get(k);	
		
						String title = yslider.getAttribute("title");
						double min_limit = XMLConfigUtil.parseDouble(yslider.getAttribute("min_limit"),GraphWindow.YSLIDER_DEFMIN);
						double max_limit = XMLConfigUtil.parseDouble(yslider.getAttribute("max_limit"),GraphWindow.YSLIDER_DEFMAX);
						double min_vis = XMLConfigUtil.parseDouble(yslider.getAttribute("min_vis"),0);
						double max_vis = XMLConfigUtil.parseDouble(yslider.getAttribute("max_vis"),5000);
						double resolution = XMLConfigUtil.parseDouble(yslider.getAttribute("resolution"),0.01d);
						boolean min_locked = XMLConfigUtil.parseBoolean(yslider.getAttribute("min_locked"),false);
						boolean max_locked = XMLConfigUtil.parseBoolean(yslider.getAttribute("max_locked"),false);

						EditorPlugin.DBG.info("Configuring Y slider "+k+" "+title);
						
						ZoomSlider slider = graph_window.getYSlider(k);
						slider.setTitle(title);
						slider.lock(ZoomSlider.MIN,min_locked);
						slider.lock(ZoomSlider.MAX,max_locked);
						slider.configure(min_limit,max_limit,min_vis,max_vis,resolution);
					}
		
					for (int k = 0; k < xsliders.size(); k++) {
						Element xslider = (Element)xsliders.get(k);	
		
						String title = xslider.getAttribute("title");
						double min_limit = XMLConfigUtil.parseDouble(xslider.getAttribute("min_limit"),0);
						double max_limit = XMLConfigUtil.parseDouble(xslider.getAttribute("max_limit"),1000 * 60 * 60 * 2); //2 hours
						double min_vis = XMLConfigUtil.parseDouble(xslider.getAttribute("min_vis"),0);
						double max_vis = XMLConfigUtil.parseDouble(xslider.getAttribute("max_vis"),1000 * 60 * 3); //3 mins
						double resolution = XMLConfigUtil.parseDouble(xslider.getAttribute("resolution"),0.01d);
						boolean min_locked = XMLConfigUtil.parseBoolean(xslider.getAttribute("min_locked"),false);
						boolean max_locked = XMLConfigUtil.parseBoolean(xslider.getAttribute("max_locked"),false);
						boolean following = XMLConfigUtil.parseBoolean(xslider.getAttribute("following"),false);

						EditorPlugin.DBG.info("Configuring X slider "+k);
						
						ZoomSlider slider = graph_window.getXSlider(k);
						slider.setTitle(title);
						slider.lock(ZoomSlider.MIN,min_locked);
						slider.lock(ZoomSlider.MAX,max_locked);
						slider.configure(min_limit,max_limit,min_vis,max_vis,resolution);
						graph_window.setTimeSliderFollowing((TimeZoomSlider)slider,following);
					}
	
				} catch (Throwable t) {
					EditorPlugin.DBG.logVisibleError(t,"Error applying graph configuration",true);	
				}
			}
	
			ArrayList agent_roots = XMLConfigUtil.getAllElements(doc_element.getChildNodes(),"agent");
			EditorPlugin.DBG.info("found "+agent_roots.size()+" agent/model configurations");
	
			//make a second pass and apply any descriptor configurations
			for (int i = 0; i < agent_roots.size(); i++) {
				try {
					Element agent_root = (Element)agent_roots.get(i);
					
					String rootname = agent_root.getAttribute("name");
		
//					EditorPlugin.DBG.info("looking for agent named "+rootname);
					boolean found = false;
		
					for (int k = 0; k < agents.size(); k++) { 
						StatAgent agent = (StatAgent)agents.get(k);
						if (agent.agent != null) {
							if (agent.agent.getName().equals(rootname)) {
								EditorPlugin.DBG.info("Applying configuration to agent "+rootname+"");
								found = true;
								applyAgentConfig(agent,agent_root);	
							}
						}
					}

					if (!found) {
						EditorPlugin.DBG.warning("Could not find agent "+rootname+" - unable to apply this config");
					}
	
				} catch (Throwable t) {
					EditorPlugin.DBG.logVisibleError(t,"Error applying agent/model configuration",true);	
				}
			}
	
			//
			// After we have applied the config, generate selection events on all the models
			// so that they refactor themselves accordingly
			//
			TreeItem[] items = chooser_tree.getItems();
			for (int i = 0; i < items.length; i++) {
				try {
					treeItemSelected(items[i]);
				} catch (Throwable t) {
					EditorPlugin.DBG.logVisibleError(t,"Error updating agent/model configuration",true);	
				}
			}
	
			//
			// Update Graph Window tables
			//
			graph_window.updateTables();
		
		} catch (Exception t) {
			EditorPlugin.DBG.warning("error applying configuration",t);
			throw t;
		}

	}

	private HashMap mapChildrenByName(StatAgent agent) {
		HashMap map = new HashMap(128,0.75f);
		
		ArrayList list = agent.model_roots;
		for (int i = 0; i < list.size(); i++) {
			SDDescriptor tmp = (SDDescriptor)list.get(i);
			map.put(tmp.getName(),tmp);
		}
		
		return map;
	}

	private HashMap mapChildrenByName(TRCAgent agent) {
		HashMap map = new HashMap(128,0.75f);
		
		EList list = agent.getDescriptor();
		for (int i = 0; i < list.size(); i++) {
			SDDescriptor tmp = (SDDescriptor)list.get(i);
			map.put(tmp.getName(),tmp);
		}
		
		return map;
	}

	private HashMap mapChildrenByName(SDDescriptor descriptor) {
		HashMap map = new HashMap(128,0.75f);
		
		EList list = descriptor.getChildren();
		for (int i = 0; i < list.size(); i++) {
			SDDescriptor tmp = (SDDescriptor)list.get(i);
			map.put(tmp.getName(),tmp);
		}
		
		return map;
	}

	private void applyCBEConfig(TreeItem item, Element element, boolean parent_checked) throws Exception {

		if (element == null) return;	

		boolean ticked = XMLConfigUtil.parseBoolean(element.getAttribute("ticked"),true);
		boolean expanded = XMLConfigUtil.parseBoolean(element.getAttribute("expanded"),false);

		item.setChecked(ticked);
		item.setExpanded(expanded);

		if (parent_checked) {
			item.setGrayed(false);	
		} else {
			item.setGrayed(true);	
		}
				
		TreeItem[] items = item.getItems();
		HashMap item_map = new HashMap();
		
		for (int i = 0; i < items.length; i++) {
			item_map.put(items[i].getText(),items[i]);
		}

		ArrayList cbeconfigs = XMLConfigUtil.getAllElements(element.getChildNodes(),"cbelog");
		
		for (int i = 0; i < cbeconfigs.size(); i++) {
			Element config = (Element)cbeconfigs.get(i);
			String name = config.getAttribute("name");
			
			if (name != null) {
				TreeItem subitem = (TreeItem)item_map.get(name);
				if (subitem != null) {
					applyCBEConfig(subitem,config,parent_checked && ticked);	
				}	
			}
		}
		
	}

	private void applyAgentConfig(StatAgent agent, Element element) throws Exception {

		boolean sd_ticked = XMLConfigUtil.parseBoolean(element.getAttribute("ticked"),true);
		boolean sd_expanded = XMLConfigUtil.parseBoolean(element.getAttribute("expanded"),false);

		TreeItem item = (TreeItem)reverse_map.get(agent.agent);
		item.setChecked(sd_ticked);
		item.setExpanded(sd_expanded);

		Element cbeconfig = XMLConfigUtil.getFirstElement(element.getChildNodes(),"cbelog");

		if (cbeconfig != null) {
			applyCBEConfig(agent.msg_item,cbeconfig,sd_ticked);
		}

//		if (cbeconfig != null) {
//			EditorPlugin.DBG.info(" agent CBE log configuration");
//			applyCBEConfig(agent.agent.getDefaultEvents(),cbeconfig,sd_ticked);
//		}

		ArrayList descriptor_roots = XMLConfigUtil.getAllElements(element.getChildNodes(),"descriptor");
		
		EditorPlugin.DBG.info("found "+descriptor_roots.size()+" model configurations");
		
//		EditorPlugin.DBG.info("searching for models beneath agent");

		//get a map of the descriptors under this agent
		HashMap dmap = mapChildrenByName(agent);

		//make a second pass and apply any descriptor configurations
		for (int i = 0; i < descriptor_roots.size(); i++) {
			Element model_root = (Element)descriptor_roots.get(i);
			
			String rootname = model_root.getAttribute("name");
			boolean found = false;

			SDDescriptor descriptor = (SDDescriptor)dmap.get(rootname);

			if (descriptor != null) {
				if (EditorPlugin.DBG.check("configuration")) EditorPlugin.DBG.info("Applying configuration to model "+rootname);
				try {
					applyModelConfig(descriptor,model_root,sd_ticked);	
				} catch (Throwable e) {
					EditorPlugin.DBG.error("problem applying model configuration",e);
				}
			} else {
				EditorPlugin.DBG.warning("Could not find model "+rootname+" - unable to apply this model configuration");
			}
			
/*
			for (int m = 0; m < agent.model_roots.size(); m++) {
			
				SDDescriptor descriptor = (SDDescriptor)agent.model_roots.get(m);

				try {
					if (descriptor != null) {
						if (descriptor.getName().equals(rootname)) {
							EditorPlugin.DBG.info("found model "+rootname);
							found = true;

							try {
								applyModelConfig(descriptor,model_root,sd_ticked);	
							} catch (Throwable e) {
								EditorPlugin.DBG.error("problem applying model configuration",e);
							}
						}
					}
				} catch (Throwable e) {
					EditorPlugin.DBG.error("problem searching for model",e);
				}
			
			}
			
			if (!found) {
				EditorPlugin.DBG.warning("Could not find model "+rootname+" - unable to apply this model configuration");
			}
*/
		}
	}
	
	private void applyModelConfig(SDDescriptor descriptor, Element element, boolean parent_checked) throws Exception {

		boolean sd_ticked = XMLConfigUtil.parseBoolean(element.getAttribute("ticked"),true);
		boolean sd_expanded = XMLConfigUtil.parseBoolean(element.getAttribute("expanded"),false);
		
		TreeItem item = (TreeItem)reverse_map.get(descriptor);
		item.setChecked(sd_ticked);
		item.setExpanded(sd_expanded);
		
		if (parent_checked) {
			item.setGrayed(false);	
		} else {
			item.setGrayed(true);	
		}
		
		//get graph elements
		ArrayList obs_graphs = XMLConfigUtil.getAllElements(element.getChildNodes(),"obs_graph");

		//configure the graphs
		for (int i = 0; i < obs_graphs.size(); i++) {
			Element obs_graph = (Element)obs_graphs.get(i);

			boolean ticked = XMLConfigUtil.parseBoolean(obs_graph.getAttribute("ticked"),true);
			int obs_index = (int)XMLConfigUtil.parseDouble(obs_graph.getAttribute("obs_index"),0);
			int x_slider = (int)XMLConfigUtil.parseDouble(obs_graph.getAttribute("x_slider"),0);			
			int y_slider = (int)XMLConfigUtil.parseDouble(obs_graph.getAttribute("y_slider"),0);			
			Color color = parseColor(obs_graph.getAttribute("color"),graph_window.getDisplay().getSystemColor(colours[next++]));
			if (next >= colours.length) next = 0;
			int line_width = (int)XMLConfigUtil.parseDouble(obs_graph.getAttribute("line_width"),1);
			int line_style = XMLConfigUtil.parseLineStyle(obs_graph.getAttribute("line_style"),SWT.LINE_SOLID);
			double multiplier = XMLConfigUtil.parseDouble(obs_graph.getAttribute("multiplier"),1.0);
			
			if (EditorPlugin.DBG.check("configuration")) EditorPlugin.DBG.info("Applying graph configuration for "+descriptor.getName()+" "+ticked+" "+color);
			
			if (x_slider < 0) x_slider = 0;
			if (x_slider > graph_window.getXSliderCount()) x_slider = 0;
			if (y_slider < 0) y_slider = 0;
			if (y_slider > graph_window.getYSliderCount()) y_slider = 0;
			
			if (descriptor instanceof SDMemberDescriptor) {
				SDMemberDescriptor memdescriptor = (SDMemberDescriptor)descriptor;
				EList obs_list = memdescriptor.getSnapshotObservation();
				
				if (obs_index >= 0 && obs_index < obs_list.size()) {
					
					SDSnapshotObservation observation = (SDSnapshotObservation)obs_list.get(obs_index);
				
					TreeItem obs_item = (TreeItem)reverse_map.get(observation);
					obs_item.setChecked(ticked);
					obs_item.setGrayed(!(sd_ticked && parent_checked));
				
//					if (ticked && !obs_item.getGrayed()) {
						//set up a graph over the above observation
						//
						//note - graphs etc are created right when a model is loaded, so there should always
						//be corresponding graphs for everything

						treeItemSelected(obs_item);

						BasicGraphSource source = (BasicGraphSource)source_map.get(obs_item);
						if (source != null) {
							//must use the source graph map here otherwise it might be null if the graph is not checked
							Graph graph = (Graph)source_graph_map.get(source);
							if (graph != null) {
								
								graph.setXSlider(graph_window.getXSlider(x_slider));
								graph.setYSlider(graph_window.getYSlider(y_slider));
								graph.setForeground(color);
								graph.setLineWidth(line_width);
								graph.setLineStyle(line_style);
								graph.setStaticScaling(multiplier);

//								EditorPlugin.DBG.info("applied graph configuration for "+descriptor.getName());
	
							} else {
								EditorPlugin.DBG.warning("Could not find graph - no graph found for "+obs_item.getText());
							}
						} else {
							EditorPlugin.DBG.warning("Could not find graph - no graph source found for "+obs_item.getText());
						}

//					}
				} else {
					EditorPlugin.DBG.warning("Could not find graph - observation index out of range "+obs_index+" / "+obs_list.size());
				}
			} else {
				EditorPlugin.DBG.warning("Could not find graph - descriptor is not a member descriptor");
			}
			
		}
		
		//get the control elements
		ArrayList obs_controls = XMLConfigUtil.getAllElements(element.getChildNodes(),"control");
		
		//configure the controls
		for (int i = 0; i < obs_controls.size(); i++) {
			Element obs_control = (Element)obs_controls.get(i);
			
			boolean ticked = XMLConfigUtil.parseBoolean(obs_control.getAttribute("ticked"),true);
			int y_slider = (int)XMLConfigUtil.parseDouble(obs_control.getAttribute("y_slider"),0);			

			if (EditorPlugin.DBG.check("configuration")) EditorPlugin.DBG.info("Applying control configuration for "+descriptor.getName()+" (yslider:"+y_slider+")");

			if (descriptor instanceof SDMemberDescriptor) {
				SDMemberDescriptor memdescriptor = (SDMemberDescriptor)descriptor;
				SDRepresentation representation = memdescriptor.getRepresentation();
				
				if (representation instanceof SDModifiableVariableRepresentation) {
					SDModifiableVariableRepresentation rep = (SDModifiableVariableRepresentation)representation;
					
					try {
						
						TreeItem rep_item = (TreeItem)reverse_map.get(rep);
						rep_item.setChecked(ticked);
						rep_item.setGrayed(!(sd_ticked && parent_checked));

						if (y_slider >= 0 && y_slider < graph_window.getYSliderCount()) {
							bar_slider_map.put(rep_item,graph_window.getYSlider(y_slider));
						}

	//					if (ticked && !rep_item.getGrayed()) {
							//set up a control bar over the above representation
//							rep_item.setChecked(true);
//							treeItemSelected(rep_item,true);
							treeItemSelected(rep_item);
	
							ModifiableVariableBar bar = (ModifiableVariableBar)bar_map.get(rep_item);

							if (bar != null) {
								//ok for this to be null
								//may be a control with no graphical bar
								if (bar.bar != null) {
									graph_window.barMoved(bar.bar,y_slider);
								}
							} else {
								EditorPlugin.DBG.warning("Could not find bar for representation "+rep_item.getText());	
							}

						if (y_slider >= 0 && y_slider < graph_window.getYSliderCount()) {
							bar_slider_map.put(rep_item,graph_window.getYSlider(y_slider));
						}
						
	
	//					}

					} catch (NullPointerException e) {
						EditorPlugin.DBG.warning("Could not find rep item",e);
					}
				} else {
					EditorPlugin.DBG.warning("Could not find control - representation is not a modifiable variable representation");
				}
			} else {
				EditorPlugin.DBG.warning("Could not find control - descriptor is not a member descriptor");
			}
					
		}
		
		ArrayList descriptor_children = XMLConfigUtil.getAllElements(element.getChildNodes(),"descriptor");

		HashMap dmap = mapChildrenByName(descriptor);

		//recurse on the children of this descriptor
		for (int i = 0; i < descriptor_children.size(); i++) {
			Element child = (Element)descriptor_children.get(i);
			String childname = child.getAttribute("name");

			SDDescriptor child_descriptor = (SDDescriptor)dmap.get(childname);
			if (child_descriptor != null) {
				if (EditorPlugin.DBG.check("configuration")) EditorPlugin.DBG.info("Applying configuration to descriptor "+childname);
				applyModelConfig(child_descriptor,child,sd_ticked && parent_checked);	
			} else {
				EditorPlugin.DBG.warning("Could not find descriptor - descriptor of name "+childname+" was not found under descriptor "+descriptor.getName());
			}
/*
			boolean found = false;
			EList list = descriptor.getChildren();
			
			//search for an actual descriptor with the specified name
			for (int k = 0; k < list.size(); k++) { 
				SDDescriptor child_descriptor = (SDDescriptor)list.get(k);
				if (child_descriptor.getName().equals(childname)) {
					found = true;
					applyModelConfig(child_descriptor,child,sd_ticked && parent_checked);	
				}
			}

			if (!found) {
				EditorPlugin.DBG.warning("descriptor configuration search failure - descriptor of name "+childname+" was not found under descriptor "+descriptor.getName());
			}
*/			
		}

	}	

	public String generateConfig() {
		return generateConfig("");
	}

	public String generateConfig(String extra_xml) {
		StringBuffer xml = new StringBuffer();

		xml.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
		xml.append("<!-- Statistical Model Graph Viewer configuration file -->\n");
		xml.append("<!-- Not intended to be human editable - edit at your own risk -->\n");
		xml.append("<smgconfig>\n");
		
		xml.append(extra_xml);
		
		for (int i = 0; i < modules.size(); i++) {
			StatConModule module = (StatConModule)modules.get(i);
			String moduleref = module.getModuleRef();
			
			try {
				byte[] moduledat = module.saveModuleData();
				String hex = HexData.byteArrayToHexString(moduledat);
				
				xml.append("<module_data ref=\""+moduleref+"\" data=\""+hex+"\" />\n");
//				xml.append(module.generateConfig());
			} catch (Exception e) {
				EditorPlugin.DBG.logVisibleError(e,"Error saving module data for "+moduleref,false);
			}
		}
		

		//
		// Append any loaded models which have URI references
		//
		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);
			URI uri = agent.uri;
			if (uri != null) {
				xml.append("<model file=\"").append(uri).append("\" />\n");
			}
		}

		//
		// Append graph configuration data
		//
		xml.append("<graph color=\"").append(XMLConfigUtil.colorToString(graph_window.getGraphBackground())).append("\" match_colors=\"").append(graph_window.getSlidersMatchGraph()).append("\" >\n");

		for (int i = 0; i < graph_window.getYSliderCount(); i++) {
			ZoomSlider slider = graph_window.getYSlider(i);
			
			xml.append("  <yslider ");
			xml.append("title=\"").append(slider.getTitle()).append("\" ");
			xml.append("min_limit=\"").append(slider.getMinLimit()).append("\" ");
			xml.append("max_limit=\"").append(slider.getMaxLimit()).append("\" ");
			xml.append("min_vis=\"").append(slider.getMinVisible()).append("\" ");
			xml.append("max_vis=\"").append(slider.getMaxVisible()).append("\" ");
			xml.append("resolution=\"").append(slider.getResolution()).append("\" ");
			xml.append("min_locked=\"").append(slider.isLocked(ZoomSlider.MIN)).append("\" ");
			xml.append("max_locked=\"").append(slider.isLocked(ZoomSlider.MAX)).append("\" ");
			xml.append("/>\n");
						
		}
		
		for (int i = 0; i < graph_window.getXSliderCount(); i++) {
			ZoomSlider slider = graph_window.getXSlider(i);

			xml.append("  <xslider ");
			xml.append("title=\"").append(slider.getTitle()).append("\" ");
			xml.append("min_limit=\"").append(slider.getMinLimit()).append("\" ");
			xml.append("max_limit=\"").append(slider.getMaxLimit()).append("\" ");
			xml.append("min_vis=\"").append(slider.getMinVisible()).append("\" ");
			xml.append("max_vis=\"").append(slider.getMaxVisible()).append("\" ");
			xml.append("resolution=\"").append(slider.getResolution()).append("\" ");
			xml.append("min_locked=\"").append(slider.isLocked(ZoomSlider.MIN)).append("\" ");
			xml.append("max_locked=\"").append(slider.isLocked(ZoomSlider.MAX)).append("\" ");
			xml.append("following=\"").append(graph_window.getTimeSliderFollowing((TimeZoomSlider)slider)).append("\" ");
			xml.append("/>\n");
			
		}

		xml.append("</graph>\n");

		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);

			generateConfig(xml,agent,0);			
			
		}
	
		xml.append("</smgconfig>\n");

		return xml.toString();
	}

	/**
	 * Generates a configuration branch for an agent.
	 * NOTE: does not include "<smgconfig>" tags - to make this a standalone 
	 * valid config file these must be added
	 * @param agent the TRCAgent in statcon to generate a configuration for
	 * @return a string containing the XML configuration file
	 */
	public String generateConfig(TRCAgent agent) {
		StringBuffer sb = new StringBuffer();
//		sb.append("<smgconfig>\n");

		StatAgent statagent = getStatAgent(agent);
		generateConfig(sb,statagent,0);

//		sb.append("</smgconfig>\n");	
		return sb.toString();
	}

	private void generateConfig(StringBuffer xml, StatAgent agent, int depth) {
		TreeItem agent_item = (TreeItem)reverse_map.get(agent.agent);

		if (agent_item == null) return;

		xml.append("<agent ");
		xml.append("name=\"").append(agent.agent.getName()).append("\" ");
		xml.append("ticked=\"").append(agent_item.getChecked()).append("\" ");
		xml.append("expanded=\"").append(agent_item.getExpanded()).append("\" ");
		xml.append(">\n");

		generateCBEConfig(xml,agent.msg_item,depth+1);		
		
		for (int m = 0; m < agent.model_roots.size(); m++) {
			SDDescriptor root = (SDDescriptor)agent.model_roots.get(m);
			generateConfig(xml,root,depth+1);
		}

		xml.append("</agent>");
	}

	private void generateCBEConfig(StringBuffer xml, TreeItem item, int depth) {

		for (int i = 0; i < depth; i++) {
			xml.append("  ");
		}

		xml.append("<cbelog ");
		xml.append("name=\"").append(item.getText()).append("\" ");
		xml.append("ticked=\"").append(item.getChecked()).append("\" ");
		xml.append("expanded=\"").append(item.getExpanded()).append("\" ");
		xml.append(">\n");
		
		TreeItem[] items = item.getItems();
		for (int i = 0; i < items.length; i++) {
			generateCBEConfig(xml,items[i],depth+1);
		}

		for (int i = 0; i < depth; i++) {
			xml.append("  ");
		}
		xml.append("</cbelog>\n");
	}
/*
	private void generateConfig(StringBuffer xml, EList events, int depth) {
		
		TreeItem msg_item = (TreeItem)reverse_map.get(events);
		
		if (msg_item != null) {
		
//			xml.append("  <cbelog ");
//			xml.append("ticked=\"").append(msg_item.getChecked()).append("\" ");
//			xml.append("expanded=\"").append(msg_item.getExpanded()).append("\" ");
//			xml.append(">\n");
	
			generateCBEConfig(xml,msg_item,1);
	
			xml.append("<cbeitem ");
			xml.append("name=\"severity\" ");
			xml.append("ticked=\"severity\" ");
			xml.append("name=\"severity\" ");
	
			ArrayList locations = new ArrayList();
	
			for (int i = 0; i < events.size(); i++) {
				CBECommonBaseEvent cbe = (CBECommonBaseEvent)events.get(i);
	
				CBEComponentIdentification source = cbe.getSourceComponentId();
	
				if (!locations.contains(source)) {
					locations.add(source);
	
					TreeItem comp_item = (TreeItem)reverse_map.get(source);
					
					if (comp_item != null) {
					
						xml.append("    <component ");
						xml.append("componentIdType=\"").append(source.getComponentIdType()).append("\" ");
						xml.append("component=\"").append(source.getComponent()).append("\" ");
						xml.append("subcomponent=\"").append(source.getSubComponent()).append("\" ");
						xml.append("ticked=\"").append(comp_item.getChecked()).append("\" ");
						xml.append("expanded=\"").append(comp_item.getExpanded()).append("\" ");
						xml.append(" />\n");
					
					}
				}
				
			}		

		
		
//
// APPEND ALL SOURCE LOCATIONS HERE?
//
// WONDER IF LOCATIONS SHOULD BE NESTED?  PROBABLY
// DONT NEED TO BE FOR NOW THOUGH
//
// IF NESTED, I SUPPOSE AN EVENT BELONGS TO THE SUBCOMPONENT
// BUT SUBCOMPONENT MAY BE NULL
//
	
//			xml.append("  </cbelog>\n");
		
		}
	}
*/	

	/**
	 * Generates a configuration branch for a statistical model.
	 * NOTE: does not include "<smgconfig>" tags - to make this a standalone 
	 * valid config file these must be added
	 * @param model_root the model held in statcon to generate a configuration for
	 * @return a string containing the XML configuration file
	 */
	public String generateConfig(SDDescriptor model_root) {
		try {
			StringBuffer sb = new StringBuffer();

			EditorPlugin.DBG.info("generating StatCon configuration");
	//		sb.append("<smgconfig>\n");
			generateConfig(sb,model_root,0);
	//		sb.append("</smgconfig>\n");

			return sb.toString();
		} catch (Throwable e) {
			EditorPlugin.DBG.error("problem generating StatCon configuration",e);
		}
		return null;
	}
	
	private void generateConfig(StringBuffer xml, SDDescriptor descriptor, int depth) {

		TreeItem sd_item = (TreeItem)reverse_map.get(descriptor);

		if (descriptor == null) return;
		if (sd_item == null) return;

		//
		// Append descriptor information
		//
		for (int t = 0; t < depth; t++) {
			xml.append("  ");	
		} 
		xml.append("<descriptor ");
		xml.append("name=\"").append(descriptor.getName()).append("\" ");
		xml.append("ticked=\"").append(sd_item.getChecked()).append("\" ");
		xml.append("expanded=\"").append(sd_item.getExpanded()).append("\" ");
		xml.append(">\n");
		
		//
		// Append descriptor graph informationm
		//
		if (descriptor instanceof SDMemberDescriptor) {
			SDMemberDescriptor mem_descriptor = (SDMemberDescriptor)descriptor;
			EList observations = mem_descriptor.getSnapshotObservation();
			
			for (int i = 0; i < observations.size(); i++) {
				Object o = observations.get(i);
				if (o instanceof SDContiguousObservation || o instanceof SDTextObservation) {
					TreeItem obs_item = (TreeItem)reverse_map.get(o);
					BasicGraphSource source = (BasicGraphSource)source_map.get(obs_item);
					Graph graph = (Graph)source_graph_map.get(source);
//					Graph graph = (Graph)graph_map.get(obs_item);

					if (graph != null) {

						for (int t = 0; t < depth; t++) {
							xml.append("  ");	
						} 
						xml.append("<obs_graph ");
						xml.append("ticked=\"").append(obs_item.getChecked()).append("\" ");
						xml.append("obs_index=\"").append(i).append("\" ");
						xml.append("x_slider=\"").append(graph_window.getXSlider(graph.getXSlider())).append("\" ");
						xml.append("y_slider=\"").append(graph_window.getYSlider(graph.getYSlider())).append("\" ");
						xml.append("color=\"").append(XMLConfigUtil.colorToString(graph.getForeground())).append("\" ");
						xml.append("line_width=\"").append(graph.getLineWidth()).append("\" ");
						xml.append("line_style=\"").append(XMLConfigUtil.lineStyleToString(graph.getLineStyle())).append("\" ");
						xml.append("multiplier=\"").append(graph.getStaticScaling()).append("\" ");
						xml.append("/>\n");
					
					}
				}
				
			}	
		}
		
		//
		// Append descriptor control information
		//
		if (descriptor instanceof SDMemberDescriptor) {
			SDMemberDescriptor mem_descriptor = (SDMemberDescriptor)descriptor;
			SDRepresentation rep = mem_descriptor.getRepresentation();
			
			if (rep instanceof SDModifiableVariableRepresentation) {
				TreeItem rep_item = (TreeItem)reverse_map.get(rep);

				ZoomSlider slider = (ZoomSlider)bar_slider_map.get(rep_item);
				if (slider != null) {
					for (int t = 0; t < depth; t++) {
						xml.append("  ");	
					} 
					
					xml.append("<control ");
					xml.append("ticked=\"").append(rep_item.getChecked()).append("\" ");
					xml.append("y_slider=\"").append(graph_window.getYSlider(slider)).append("\" ");
					xml.append("/>\n");
				}
				
			}
			
		}
		
		//
		// Append descriptor children
		//
		EList list = descriptor.getChildren();
		for (int i = 0; i < list.size(); i++) {
			SDDescriptor child = (SDDescriptor)list.get(i);
			generateConfig(xml,child,depth+1);	
		}
		
		for (int t = 0; t < depth; t++) {
			xml.append("  ");	
		} 
		xml.append("</descriptor>\n");
		
	}

	public GraphWindow getGraphWindow() {
		return graph_window;
	}

protected HashMap updates = new HashMap();
	public void setConstantUpdate(String request_id, boolean request_on, long ms_refresh) {
		Boolean val = new Boolean(request_on);
		updates.put(request_id,val);

		boolean on = false;
		
		Iterator requests = updates.values().iterator();
		while (requests.hasNext()) {
			Boolean b = (Boolean)requests.next();
			if (b.booleanValue()) {
				on = true;
				break;	
			}
		}
		
		if (request_on) {
			graph_window.setConstantUpdate(on,ms_refresh);
		} else {
			graph_window.setConstantUpdate(on);
		}
	}

	public void widgetDisposed(DisposeEvent e) {
		dispose();
	}

	public void dispose() {
		for (int i = 0; i < disposables.size(); i++) {
			try {
			Object o = disposables.get(i);
			if (o != null) {
				if (o instanceof Widget) {
					((Widget)o).dispose();	
				} else if (o instanceof Color) {
					((Color)o).dispose();	
				} else if (o instanceof Image) {
					((Image)o).dispose();	
				} else {
					try {
						EditorPlugin.disposeObject(o);
					} catch (Throwable e) {
						EditorPlugin.DBG.warning("dispose object failed ",e);
					}
				}
			}
			} catch (Throwable e) {
				e.printStackTrace();
			}
		}		
	}

	private void initAsSash(Composite parent) {
		sashfolder = new SashForm(parent,0);
		sashfolder.setOrientation(SWT.HORIZONTAL);

		leftsash = new Group(sashfolder,0);
		rightsash = new Composite(sashfolder,0);

		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.horizontalSpacing = 0;
		layout.verticalSpacing = 0;

		leftsash.setLayout(layout);
		rightsash.setLayout(new FillLayout());

		parent.addDisposeListener(this);

		img = EditorPlugin.img;
		
		//
		// Add a test graph tab in
		//		
		initChooser(leftsash);

		GridData data = new GridData();
		data.grabExcessHorizontalSpace = true;
		data.grabExcessVerticalSpace = true;
		data.horizontalAlignment = GridData.FILL;
		data.verticalAlignment = GridData.FILL;

		chooser_tree.setLayoutData(data);

		//
		// Add the aggregate graph tab
		//
		initAggregateGraph(rightsash);

		sashfolder.setWeights(new int[]{30,70});

		disposables.add(sashfolder);
		disposables.add(leftsash);
		disposables.add(rightsash);
	}

	private void initChooser(Composite parent) {
		chooser_tree = new Tree(parent, SWT.CHECK);
		
		disposables.add(chooser_tree);

		chooser_menu = new Menu(chooser_tree);
		chooser_add = new MenuItem(chooser_menu,0);

		chooser_add.setText(EditorPlugin.getString("ADD_NEW_TRACE"));
		chooser_add.setImage(img.getImage(ImageManager.IMG_SMODEL_TRCAGENT));

		disposables.add(chooser_menu);
		disposables.add(chooser_add);
		
		chooser_tree.addMouseListener(this);
		chooser_add.addSelectionListener(this);
				
		chooser_tree.addSelectionListener(this);

		DropTarget target = new DropTarget(chooser_tree,DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK);
		target.addDropListener(this);

		Transfer[] types = new Transfer[] {
//			TextTransfer.getInstance(),
//			FileTransfer.getInstance(),
			ResourceTransfer.getInstance(),
//			EditorInputTransfer.getInstance(),
//			EditorInputTransfer.getInstance(),
			};
		target.setTransfer(types);
		
//		for (int i = 0; i < model_roots.size(); i++) {
//			SDDescriptor sdroot = (SDDescriptor) model_roots.get(i);
		for (int i = 0; i < agents.size(); i++) {
			StatAgent agent = (StatAgent)agents.get(i);
			for (int m = 0; m < agent.model_roots.size(); m++) {
				SDDescriptor sdroot = (SDDescriptor)agent.model_roots.get(m);
				addModelToChooser(sdroot,agent.agent,false);
			}
		}

		updateAgentImages();
	}

	private void addModelToChooser(SDDescriptor sd, TRCAgent agent, boolean update) {

		TreeItem agent_item = (TreeItem)reverse_map.get(agent);
		addModelToChooser(sd,agent_item, update);
				
	}
	
	private void updateTreeItems() {
		TreeItem[] items = chooser_tree.getItems();
		for (int i = 0; i < items.length; i++) {
			treeItemSelected(items[i]);
		}
	}

	private String getTreeName(SDDescriptor sd) {
		String basicname = null;
		//
		// Get the basic name for this node and its duplicates
		//
		if (sd.getDescription() != null) {
			basicname = sd.getName()+" ("+sd.getDescription()+")";
		} else {
			basicname = sd.getName();
		}
		
		if (basicname == null) {
			basicname = "";	
		}
		if (basicname.length() == 0) {
			basicname = "(unknown)";	
		}
		
		return basicname;
	}	
	private void addObservations(SDDescriptor sd, TreeItem item, boolean update) {
		//
		// Objects related to a SnapshotObservation (graph)
		//
		EList obs = null;
		Image obs_image = null;

		//
		// Search for observations
		//
		if (sd instanceof SDMemberDescriptor) {
			SDMemberDescriptor sdmem = (SDMemberDescriptor)sd;
			
			//
			//Check to see if it has a valid observation (metric)
			//		
			obs = sdmem.getSnapshotObservation();
			if (obs != null) {
				if (obs.size() > 0) {
					obs_image = img.getImage(ImageManager.IMG_SMODEL_COUNTER);	
				} else {
					obs = null;	
				}
			}
		}		

		//
		//Set up the graph node if necessary
		//
		if (obs != null) {
			for (int i = 0; i < obs.size(); i++) {
			
				SDSnapshotObservation o = (SDSnapshotObservation)obs.get(i);
				
				//
				// These types of observations are graphable
				//
				if (o instanceof SDContiguousObservation
					|| o instanceof SDDiscreteObservation
					|| o instanceof SDTextObservation) {

					TreeItem obs_item = (TreeItem)reverse_map.get(o);
					
					if (!update || obs_item == null) {
						obs_item = new TreeItem(item,0,0);		
			
						obs_item.setText(sd.getName()+" (Graph "+(i+1)+")");			
						obs_item.setImage(obs_image);
						obs_item.setGrayed(false);
						obs_item.setChecked(true);
	
						counter_map.put(obs_item,o);			
						reverse_map.put(o,obs_item);
						sddescriptor_map.put(obs_item,sd);
					}
				}
			
			}
		}

	}
	
	private void addRepresentation(SDDescriptor sd, TreeItem item, boolean update) {
		//
		// Objects related to a ModifiableVariableRepresentation (control)
		//
		SDModifiableVariableRepresentation rep = null;
		Image rep_image = null;

		//
		// Search for representations
		//
		if (sd instanceof SDMemberDescriptor) {
			SDMemberDescriptor sdmem = (SDMemberDescriptor)sd;

			//
			//Check to see if it has a valid modifiable representation (control)
			//
			SDRepresentation reptmp = sdmem.getRepresentation();	
			if (reptmp instanceof SDModifiableVariableRepresentation) {
				rep = (SDModifiableVariableRepresentation)reptmp;
				rep_image = rep.getImage();	
			}
		}		
		
		//
		//Set up the control node if necessary
		//
		if (rep != null) {
			TreeItem rep_item = (TreeItem)reverse_map.get(rep);
			
			if (!update || rep_item == null) {
				rep_item = new TreeItem(item,0,0);		
			
				rep_item.setText(sd.getName()+" (Control)");			
				rep_item.setImage(rep_image);
				if (rep_image == null) {
					rep_item.setImage(img.getImage(ImageManager.IMG_SMODEL_MODIFIABLEREP));
				}
				rep_item.setGrayed(false);
				rep_item.setChecked(true);
	
				modifiable_map.put(rep_item,rep);						
				reverse_map.put(rep,rep_item);
				sddescriptor_map.put(rep_item,sd);
			
			}
		}		
	}

	private void addModelToChooser(SDDescriptor sd, TreeItem parent_item, boolean update) {

		//ignore null nodes
		if (sd == null) return;

		TreeItem item = (TreeItem)reverse_map.get(sd);
		
		if (update && item != null) {
			//
			// this descriptor already exists - leave it alone and just check its children
			//
				
			//
			// update any descriptions
			//
			String tmp = getTreeName(sd);
			if (!tmp.equals(item.getText())) {
				item.setText(tmp);
			}

				
		} else {
			//
			// this descriptor does not exist or we are not updating - add it
			//
		
//			TreeItem item = null;
			if (parent_item == null) {
				//no parent item - make it a root of the tree
				item = new TreeItem(chooser_tree,0);	
			} else {
				//parent item, create under parent
				item = new TreeItem(parent_item,0);		
			}
	
			//
			// Get the basic descriptor to the tree
			//
			item.setText(getTreeName(sd));
			item.setImage(img.getImage(ImageManager.IMG_SMODEL_DESCRIPTOR));
			item.setChecked(true);
			item.setGrayed(false);

			reverse_map.put(sd,item);
			sddescriptor_map.put(item,sd);

		}

		addObservations(sd,item,update);
		addRepresentation(sd,item,update);
		//
		// Recurse on its children
		//
		EList children = sd.getChildren();
		if (children == null) return;
		
		for (int i = 0; i < children.size(); i++) {
			SDDescriptor csd = (SDDescriptor)children.get(i);
			addModelToChooser(csd,item,update);	
		}
	}
	
	public void initAggregateGraph(Composite parent) {
		graph_window = new GraphWindow(parent,0,dirtiable);

		graph_window.addUpdateRunnable(new CBEUpdate());
		
		disposables.add(graph_window);
	}
	
	public Control initTestGraph(Composite parent) {	
		Color black = parent.getShell().getDisplay().getSystemColor(SWT.COLOR_BLACK);
		Color cyan = parent.getShell().getDisplay().getSystemColor(SWT.COLOR_CYAN);
		Color magenta = parent.getShell().getDisplay().getSystemColor(SWT.COLOR_MAGENTA);
	
		//the execution history should be loaded ok so I can add whatever editors I want now	
		GraphWindow window = new GraphWindow(parent,0);
		disposables.add(window);
//		GraphCanvas graph_canvas = window.getGraphCanvas();
		
		window.addXSlider();
		window.addYSlider();
		
		GraphSource sine_source = new TestGraphSource(-1000000,1000000,TestGraphSource.SINE_WAVE);
		LineGraph sine_line = new LineGraph(LineGraph.TYPE_AVERAGE,sine_source,window.getXSlider(0),window.getYSlider(0),cyan);
		sine_line.setLineWidth(2);
	
		GraphSource cosine_source = new TestGraphSource(-1000000,1000000,TestGraphSource.COSINE_WAVE);
		HistogramGraph cosine_line = new HistogramGraph(cosine_source,window.getXSlider(1),window.getYSlider(1),magenta);
		cosine_line.setLineWidth(2);
			
		graph_window.addGraph("Test Sine","A test sine wave graph",sine_line,null);
		graph_window.addGraph("Test Cosine","A test cosine wave graph",cosine_line,null);
		
		return window;
	}
	
	protected SDDescriptor selected_descriptor = null;
	
	public SDDescriptor getSelectedDescriptor() {
		return selected_descriptor;
	}
	
	public void setDescriptorImage(SDDescriptor descriptor, Image image) {
		TreeItem item = (TreeItem)reverse_map.get(descriptor);
		if (item == null) return;
		item.setImage(image);
	}

	//////////////////////////////
	// tree DND listener
	//////////////////////////////

	public void dragEnter(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;
	}
	public void dragLeave(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;
	} 
	public void dragOperationChanged(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;
	} 
	public void dragOver(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;
	} 
	public void dropAccept(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;
	}  
	public void drop(DropTargetEvent event) {
		event.detail = DND.DROP_COPY;

		if (event.data == null) { // no data to copy, indicate failure in event.detail
			event.detail = DND.DROP_NONE;
			return;
		}
//		EditorPlugin.DBG.info("DRAG AND DROP "+event.data.getClass()+" / "+event.data);

		StringBuffer errors = new StringBuffer();

		try {
			IResource[] resources = (IResource[])event.data;

			for (int i = 0; i < resources.length; i++) {

				String path = resources[i].getFullPath().toString();
				String extension = path.substring(path.lastIndexOf(".")+1);

				EditorPlugin.DBG.info("got dragged resource "+path);

				if (extension.equalsIgnoreCase("trcmxmi") 
					|| extension.equalsIgnoreCase("trcnxmi")
					|| extension.equalsIgnoreCase("trcpxmi")
					|| extension.equalsIgnoreCase("trcaxmi")
					) {

					try {
	
						addAgent(resources[i].getFullPath());
						setDirty(true);

					} catch (Throwable e) {
						errors.append(EditorPlugin.getString("INVALID_TRACE_FILE")+":"+resources[i].getFullPath()+"\n");
						EditorPlugin.DBG.warning("bad trace/model file drop ",e);
//						e.printStackTrace();
					}
				
				} else {

					try {
						boolean accepted = false;
						
						for (int k = 0; k < drop_listeners.size(); k++) {
							DropListener drop = (DropListener)drop_listeners.get(k);
							if (drop.extension.equalsIgnoreCase(extension)) {
								if (drop.listener.drop(this,resources[i])) {
									//DROP ACCEPTED
									accepted = true;
//									return;	
								}
							}
						}
							
						if (!accepted) {
							event.detail = DND.DROP_NONE;
							throw new Exception(EditorPlugin.getString("NO_MODULE_ACCEPTED_DROP"));
						}
					
					} catch (Throwable e) {
						String msg = e.getMessage();
						if (msg == null) msg = "";
						if (msg.length() > 0) {
							errors.append(EditorPlugin.getString("INVALID_FILE")+":"+resources[i].getFullPath()+" ("+e.getMessage()+")\n");
						} else {
							errors.append(EditorPlugin.getString("INVALID_FILE")+":"+resources[i].getFullPath()+" (no further information)\n");
						}
						EditorPlugin.DBG.warning("bad module file drop "+resources[i].getFullPath(),e);
					}
				}
			}
			
		} catch (Throwable e) {
			errors.append(EditorPlugin.getString("INVALID_FILE_DROP")+":"+e+"\n");
			EditorPlugin.DBG.error("bad file drop",e);
			event.detail = DND.DROP_NONE;
			return;
		}
		if (errors.length() > 0) {
			MessageBox mbox = new MessageBox(graph_window.getShell(),SWT.OK|SWT.ICON_WARNING);	
			mbox.setText(EditorPlugin.getString("INVALID_FILE_DROP_TITLE"));
			mbox.setMessage(errors.toString());
			mbox.open();
		}
	}
	
	//////////////////////////////
	// tree popup listener
	//////////////////////////////

	public void mouseDoubleClick(MouseEvent e) {}
	public void mouseDown(MouseEvent e) {}
	public void mouseUp(MouseEvent e) {
		Point pt = new Point(e.x,e.y);
		TreeItem item = chooser_tree.getItem(pt);

		boolean toplevel = false;		
		
		ArrayList subactions = new ArrayList();
		
		if (item != null) {
			TRCAgent trcagent = (TRCAgent)trcagent_map.get(item);
//			SDDescriptor descriptor = (SDDescriptor)sddescriptor_map.get(item);
			
			StatAgent agent = getStatAgent(trcagent);
			if (agent != null) {
				//item is a statistical model root node

//				selected_model = model.root;
				
				subactions = agent.actions;
				
			} else {
				//item may be a top level node - see if it has associated actions

				for (int k = 0; k < root_nodes.size(); k++) {
					RootNode node = (RootNode)root_nodes.get(k);
					if (node.item == item) {
						subactions = node.actions;	
						toplevel = true;
					}	
				}
				
			}
		}
		
		ArrayList slideractions = new ArrayList();
		if (item != null && !toplevel) {
			//dont show set slider associations if the node is a top level node (a module node)

			for (int i = 0; i < graph_window.getYSliderCount(); i++) {
				ZoomSlider slider = graph_window.getYSlider(i);
				BasicAction action = new BasicAction();
				action.name = EditorPlugin.getString("SET_ALL_VERTICAL_TO")+" "+slider.getTitle();
				action.image = img.getImage(ImageManager.IMG_MISC_VSLIDER);
				action.runnable = new SetSliderAction(false,slider,item);
				
				slideractions.add(action);
			}
			for (int i = 0; i < graph_window.getXSliderCount(); i++) {
				ZoomSlider slider = graph_window.getXSlider(i);
				BasicAction action = new BasicAction();
				action.name = EditorPlugin.getString("SET_ALL_HORIZONTAL_TO")+" "+slider.getTitle();
				action.image = img.getImage(ImageManager.IMG_MISC_HSLIDER);
				action.runnable = new SetSliderAction(true,slider,item);
				
				slideractions.add(action);
			}
		}
		
		if (item != null) {
			SDDescriptor descriptor = (SDDescriptor)sddescriptor_map.get(item);
			selected_descriptor = descriptor;
			
			if (descriptor != null) {

				while (descriptor.getParent() != null) {
					descriptor = descriptor.getParent();	
				}
				
				for (int z = 0; z < agents.size(); z++) {
					StatAgent node = (StatAgent)agents.get(z);
					if (node.model_roots.contains(descriptor)) {
						subactions.addAll(node.modelactions);
						break;
					}	
				}

			}
		}
		
		if (chooser_disposables != null) {
			for (int i = 0; i < chooser_disposables.size(); i++) {
				MenuItem mitem = (MenuItem)chooser_disposables.get(i);
				mitem.dispose();	
			}	
		}

		chooser_disposables = new ArrayList();

		if (subactions.size() > 0) {
			MenuItem sep = new MenuItem(chooser_menu,SWT.SEPARATOR);
			chooser_disposables.add(sep);	
			
			for (int k = 0; k < subactions.size(); k++) {
				BasicAction action = (BasicAction)subactions.get(k);
				action.menuitem = new MenuItem(chooser_menu,0);
				action.menuitem.setText(action.name);
				action.menuitem.setImage(action.image);
				action.menuitem.addSelectionListener(action);
				chooser_disposables.add(action.menuitem);		
			}
		}
		
		if (slideractions.size() > 0) {
			MenuItem sep = new MenuItem(chooser_menu,SWT.SEPARATOR);
			chooser_disposables.add(sep);	

			for (int k = 0; k < slideractions.size(); k++) {
				BasicAction action = (BasicAction)slideractions.get(k);
				action.menuitem = new MenuItem(chooser_menu,0);
				action.menuitem.setText(action.name);
				action.menuitem.setImage(action.image);
				action.menuitem.addSelectionListener(action);
				chooser_disposables.add(action.menuitem);		
			}
		}
		
		
		if (e.button > 1) {
			chooser_menu.setVisible(true);
		}
		
		
	}	
	
	//////////////////////////////
	// tree selection listener
	//////////////////////////////

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	} 
	
	public void widgetSelected(SelectionEvent e) {
		Object o = e.getSource();
		widgetSelected(e.item,o);
	}

	public void widgetSelected(Widget widget, Object o) {
	
		if (o == chooser_tree) {
	
			//return on invalid tree items
			if (widget == null) return;
			if (widget instanceof TreeItem) {
	
				TreeItem item = (TreeItem) widget;
				boolean checked = item.getChecked();
				
				treeItemSelected(item);
			
			}
		
		} else if (o == chooser_add) {

			EditorPlugin.DBG.info("add agent file - opening ResourceSelectionDialog");
			
			ResourceSelectionDialog dialog = new ResourceSelectionDialog(graph_window.getShell(),ResourcesPlugin.getWorkspace().getRoot(),"Please choose one or more statistical models");

			dialog.open();
			
			Object[] results = dialog.getResult();
			
			if (results != null) {
			
				for (int i = 0; i < results.length; i++) {
					EditorPlugin.DBG.info(""+results[i].getClass());
					
					if (results[i] instanceof IFile) {
						try {
							IFile file = (IFile)results[i];

							EditorPlugin.DBG.info("trying to add "+file);
							
							addAgent(file.getFullPath());	

							setDirty(true);

							chooser_tree.redraw();

						} catch (Throwable x) {
							EditorPlugin.DBG.error("model add failed",x);
						}
					}
				}
			
			}

		}

	}

protected long prev_used = 0;	
	
	class AgentRemoveAction implements Runnable {
		TRCAgent ragent;
		public AgentRemoveAction(TRCAgent ragent) {
			this.ragent = ragent;
		}
		public void run() {
			removeAgent(ragent);
			redoAllCBEs();
		}
	}

	public void removeAgent(TRCAgent agent) {
		
		StatAgent statagent = getStatAgent(agent);
		agents.remove(statagent);
		if (statagent == null) return;

		for (int i = 0; i < agent_change_listeners.size(); i++) {
			AgentChangeListener listener = (AgentChangeListener)agent_change_listeners.get(i);
			listener.agentRemoved(agent);	
		}

		for (int m = 0; m < statagent.model_roots.size(); m++) {
			SDDescriptor model_root = (SDDescriptor)statagent.model_roots.get(m);
			for (int i = 0; i < model_change_listeners.size(); i++) {
				ModelChangeListener listener = (ModelChangeListener)model_change_listeners.get(i);
				listener.modelRemoved(model_root);	
			}
		}


//		SDDescriptor descriptor = statmodel.root;
//		models.remove(statmodel);
		
		TreeItem item = (TreeItem)reverse_map.get(agent);
						
		// remove all the graphs and stuff associated with this model tree
		removeTreeItemRef(item);

		graph_window.updateTables();

		if (item != null) {
			item.dispose();
			disposables.remove(item);
		}

		setDirty(true);

EditorPlugin.DBG.info("Reverse Map Items = "+reverse_map.values().size());

EditorPlugin.DBG.info("SDDescriptor Map Items = "+sddescriptor_map.values().size());
EditorPlugin.DBG.info("Counter Map Items = "+counter_map.values().size());
EditorPlugin.DBG.info("Modifiable Map Items = "+modifiable_map.values().size());

EditorPlugin.DBG.info("Graph Map Items = "+graph_map.values().size());
EditorPlugin.DBG.info("Bar Map Items = "+bar_map.values().size());
EditorPlugin.DBG.info("Bar Slider Map Items = "+bar_slider_map.values().size());
EditorPlugin.DBG.info("Source Map Items = "+source_map.values().size());
EditorPlugin.DBG.info("Source Graph Map Items = "+source_graph_map.values().size());

EditorPlugin.DBG.info("Disposables Items = "+disposables.size());
EditorPlugin.DBG.info("Model Items = "+agents.size());
//EditorPlugin.DBG.info("Model Roots Items = "+model_roots.size());
//EditorPlugin.DBG.info("Model Files Items = "+model_files.size());

Runtime r = Runtime.getRuntime();
r.gc();
			
//try {
//	Thread.sleep(250);	
//} catch (Exception e) {
//}
long used = r.totalMemory() - r.freeMemory();
EditorPlugin.DBG.info("USED MEMORY = "+(used/1024)+"k (+"+(used-prev_used)+")");
prev_used = used;
//try {
//	Thread.sleep(750);	
//} catch (Exception e) {
//}

	}	
	
	private void removeTreeItemRef(TreeItem item) {

		if (item == null) return;

		Object agent = trcagent_map.remove(item);
		TreeItem trcagent_item = (TreeItem)reverse_map.remove(agent);

		Object descriptor = sddescriptor_map.remove(item);
		TreeItem descriptor_item = (TreeItem)reverse_map.remove(descriptor);
		TreeItem rep_item = null;
		TreeItem obs_item = null;

		if (descriptor instanceof SDMemberDescriptor) {
			SDMemberDescriptor mem_descriptor = (SDMemberDescriptor)descriptor;			
			
			SDRepresentation rep = mem_descriptor.getRepresentation();
			rep_item = (TreeItem)reverse_map.remove(rep);
			SDModifiableVariableRepresentation rep_mod = (SDModifiableVariableRepresentation)modifiable_map.remove(rep_item);

			try {
				if (rep_mod != null) {
					graph_window.removeControl(rep_mod);
				}
			} catch (Throwable e) {
				EditorPlugin.DBG.error("problem removing control from graph window",e);
			}

			ZoomSlider rep_slider = (ZoomSlider)bar_slider_map.remove(rep_item);
			ModifiableVariableBar bar = (ModifiableVariableBar)bar_map.remove(rep_item);
			if (bar != null) {
				try {
					if (bar.bar.getZoomSlider() != null) {
						bar.bar.getZoomSlider().removeZoomControlBar(bar.bar);
						bar.bar.dispose();
						disposables.remove(bar.bar);
					}
				} catch (NullPointerException e) {
					EditorPlugin.DBG.warning("bar zoomslider not found",e);
				}
			}
			
			EList obs_list = mem_descriptor.getSnapshotObservation();
			for (int i = 0; i < obs_list.size(); i++) {
				Object obs = obs_list.get(i);
				obs_item = (TreeItem)reverse_map.remove(obs);
				SDSnapshotObservation obs_obs = (SDSnapshotObservation)counter_map.remove(obs_item);
				Graph obs_graph = (Graph)graph_map.remove(obs_item);

				try {
					if (obs_graph != null) {
						obs_graph.setUseIndicator(false);
						graph_window.removeGraph(obs_graph);
					}
				} catch (Throwable e) {
					EditorPlugin.DBG.error("problem removing graph from graph window",e);
				}
				
				BasicGraphSource source = (BasicGraphSource)source_map.remove(obs_item);
				if (source != null) {
					source_graph_map.remove(source);	
				}
			}

		}
		
		TreeItem[] children = item.getItems();
		for (int i = 0; i < children.length; i++) {
			removeTreeItemRef(children[i]);	
		}
		
		item.dispose();
		if (rep_item != null) {
			rep_item.dispose();
		}
		if (obs_item != null) {
			obs_item.dispose();
		}
	}

		
	public void treeItemSelected(TreeItem item) {
		
		try {
		
			TreeItem msgparent = item;
			while (msgparent != null) {

				if (msgparent.getText().equals("Messages")) {
					redoAllCBEs();
					break;
				}
				
				msgparent = msgparent.getParentItem();
			}
		
			boolean parents_all_checked = true;
	
			TreeItem parent = item.getParentItem();
	
			while (parent != null) {
				if (parent.getChecked() == false) {
					parents_all_checked = false;
					break;	
				}
				
				parent = parent.getParentItem();
			}
			
			//preserve graying on root nodes
			if (item.getParentItem() == null) {
				boolean b = item.getGrayed();
				treeItemSelected(item, parents_all_checked);
				item.setGrayed(b);	

			} else {
				treeItemSelected(item, parents_all_checked);
			}

			graph_window.redraw();
			
		} catch (Throwable t) {
			EditorPlugin.DBG.logVisibleError(t,"Error updating StatCon tree",false);	
		}
	}
		
	public void treeItemSelected(TreeItem item, boolean parents_all_checked) {
		boolean checked = item.getChecked();

		if (!parents_all_checked) {
			checked = false;
			if (!item.getGrayed()) {
				setDirty(true);
			}
			item.setGrayed(true);
		} else {
			if (item.getGrayed()) {
//this has really bad big O - every root node we add we do like 10000 searches for it in perfmon (doh)			
//				RootNode node = getRootNodeName(item.getText());
//				if (node == null) {
					setDirty(true);
//				}
			}
			item.setGrayed(false);	
		}

		EList events = (EList)msglist_map.get(item);
		if (events != null) {
			StatAgent agent = getStatAgent(events);
			if (checked && !agent.show_messages) {
				agent.show_messages = true;
				redoAllCBEs();
				setDirty(true);
			} else if (!checked && agent.show_messages) {
				agent.show_messages = false;
				redoAllCBEs();
				setDirty(true);
			}
		} 

		//
		// is it a graphable node?
		//
		SDSnapshotObservation obs = (SDSnapshotObservation)counter_map.get(item);
		if (obs != null) {

			Graph graph = (Graph)graph_map.get(item);
			BasicGraphSource source = (BasicGraphSource)source_map.get(item);
			
			//only need to do this once, no point in removing and adding sources all the time
			if (source == null) {
				
				if (obs instanceof SDContiguousObservation) {
					source = new SDSnapshotObservationGraphSource((SDContiguousObservation)obs);
				} else if (obs instanceof SDDiscreteObservation) {
					source = new SDSnapshotObservationGraphSource((SDDiscreteObservation)obs);
				} else if (obs instanceof SDTextObservation) {
					source = new SDTextObservationGraphSource((SDTextObservation)obs);
				} else {
					EditorPlugin.DBG.info("SnapshotObservation neither Contiguous nor Discrete!");
					return;
				}
				
				source_map.put(item,source);	
			}

			//
			// Create a new graph and add it
			//
			if (graph == null && checked == true) {
				setDirty(true);

				graph = (Graph)source_graph_map.get(source);

				if (graph == null) {
					
					Color col = item.getDisplay().getSystemColor(colours[next++]);
					
					if (next >= colours.length) {
						next = 0;
					}
	
					if (source instanceof GraphSource) { 
						graph = new LineGraph(LineGraph.TYPE_AVERAGE,(SDSnapshotObservationGraphSource)source,graph_window.getXSlider(0),graph_window.getYSlider(0),col);
					} else if (source instanceof EnumerationGraphSource) {
						graph = new EnumerationGraph(graph_window.getXSlider(0),graph_window.getYSlider(0),col,(SDTextObservationGraphSource)source);
					}	

					source_graph_map.put(source,graph);

				}

				//if x slider has been disposed, default to x slider 0
				if (graph.getXSlider().isDisposed()) {
					graph.setXSlider(graph_window.getXSlider(0));	
				}

				//if y slider has been disposed, default to x slider 0
				if (graph.getYSlider().isDisposed()) {
					graph.setYSlider(graph_window.getYSlider(0));	
				}

				SDDescriptor sd = (SDDescriptor)sddescriptor_map.get(item);
				
				graph.setUseIndicator(true);
				
				graph_window.addGraph(item.getText(),sd.getDescription(),graph,sd);

			//
			// Dispose the graph and remove it
			//
			} else if (graph != null && checked == false) {
				setDirty(true);
				
				graph.setUseIndicator(false);

				graph_window.removeGraph(graph);

			}
			
			if (checked) {
				graph_map.put(item,graph);
			} else {
				graph_map.remove(item);	
			}
			
			return;
		}
		
		//
		// is it a control node?
		//
		SDModifiableVariableRepresentation rep = (SDModifiableVariableRepresentation)modifiable_map.get(item);
		if (rep != null) {
			ModifiableVariableBar mbar = (ModifiableVariableBar)bar_map.get(item);
			
			//
			//create a new bar and add it
			//
			if (mbar == null && checked == true) {
				setDirty(true);

				ZoomSlider slider = (ZoomSlider)bar_slider_map.get(item);

				//if slider is null, default to Y slider 0
				if (slider == null) {
					slider = graph_window.getYSlider(0);	
				}
				
				//check to see if slider exists
				if (slider.isDisposed()) {
					slider = graph_window.getYSlider(0);	
				}

				Image enabled = rep.getImage(); 
//				if (enabled == null) {
//					enabled = img.getImage(ImageManager.IMG_SMODEL_MODIFIABLEREP);
//				}
				
				ZoomImageBar varbar = null;
				if (rep.getModifierType() != SDMeasuredVariableRepresentation.STRING && enabled != null && rep.hasBar()) {
					varbar = new ZoomImageBar(slider,(double)((Number)rep.getLastRequestedValue()).doubleValue(), enabled, enabled, enabled.getBounds().height+6);
					varbar.setToolTipText(item.getText());
					mbar = new ModifiableVariableBar(this,varbar,rep,dirtiable);
					varbar.addZoomControlBarListener(mbar);
				} else {
					//varbar==null
					mbar = new ModifiableVariableBar(this,varbar,rep,dirtiable);
				}

				SDDescriptor sd = (SDDescriptor)sddescriptor_map.get(item);
				graph_window.addControl(item.getText(),sd.getDescription(),enabled,varbar,rep);

				if (varbar != null) {
					disposables.add(varbar);
				}
				
				if (rep.getModifierType() != SDMeasuredVariableRepresentation.STRING && enabled != null && rep.hasBar()) {
					mbar.bar.getZoomSlider().addZoomControlBar(mbar.bar);
				}

			//
			//dispose the bar and remove it
			//
			} else if (mbar != null && checked == false) {
				setDirty(true);
				
				ZoomControlBar zbar = mbar.bar;
				
				try {
					bar_slider_map.put(item,zbar.getZoomSlider());
					
					zbar.dispose();
					zbar.getZoomSlider().removeZoomControlBar(zbar);
					
					disposables.remove(zbar);
				} catch (Throwable x) {
					//?
				}
				
				graph_window.removeControl(rep);
			}
			
			if (checked) {
				bar_map.put(item,mbar);
			} else {
				bar_map.remove(item);	
			}
			
			return;
		}
		
		//
		// If it is just a descriptor node
		//
		//(need to recurse down here and add/remove all the children)
		//probably easiest to get all of its children and their children in an arraylist and then call this
		//method for each of them?

		TreeItem[] children = item.getItems();

		for (int i = 0; i < children.length; i++) {
			treeItemSelected(children[i],checked);	
		}

	}  
	
	private void setTreeItemSlider(TreeItem item, boolean horizontal, ZoomSlider slider) {

		ModifiableVariableBar bar = getVariableBar(item);
		if (!horizontal) {
			bar_slider_map.put(item,slider);
			try {
				if (bar.bar.getZoomSlider() != null) {
					//move bar to other slider (if its vertical)
					bar.bar.getZoomSlider().removeZoomControlBar(bar.bar);
					bar.bar.setZoomSlider(slider);
					bar.bar.getZoomSlider().addZoomControlBar(bar.bar);
				}
			} catch (NullPointerException e) {
				EditorPlugin.DBG.warning("bar zoomslider not found",e);
			}
		}
		
		Graph obs_graph = getGraph(item);
		if (obs_graph != null) {
			if (horizontal) {
				obs_graph.setXSlider((TimeZoomSlider)slider);	
			} else {
				obs_graph.setYSlider(slider);	
			}
		}

		TreeItem[] children = item.getItems();
		for (int i = 0; i < children.length; i++) {
			setTreeItemSlider(children[i],horizontal,slider);	
		}
	}

	/////////////////////////////
	// Tree utilities
	/////////////////////////////
	
	private SDDescriptor getDescriptor(TreeItem item) {
		SDDescriptor descriptor = (SDDescriptor)sddescriptor_map.get(item);
		return descriptor;
	}
	
	private SDMemberDescriptor getMemberDescriptor(TreeItem item) {
		SDDescriptor descriptor = (SDDescriptor)sddescriptor_map.get(item);
		if (descriptor instanceof SDMemberDescriptor) {
			return (SDMemberDescriptor)descriptor;			
		} else {
			return null;
		}
	}
	
	private SDModifiableVariableRepresentation getRepresentation(TreeItem item) {
		SDModifiableVariableRepresentation rep = (SDModifiableVariableRepresentation)modifiable_map.get(item);
		return rep;
	}
	
	private ModifiableVariableBar getVariableBar(TreeItem item) {
		ModifiableVariableBar bar = (ModifiableVariableBar)bar_map.get(item);
		return bar;
	}

	private SDSnapshotObservation getObservation(TreeItem item) {
		SDSnapshotObservation obs = (SDSnapshotObservation)counter_map.get(item);
		return obs;
	}
	
	private BasicGraphSource getGraphSource(TreeItem item) {
		BasicGraphSource obs_source = (BasicGraphSource)source_map.get(item);
		return obs_source;
	}
	
	private Graph getGraph(TreeItem item) {
		BasicGraphSource source = getGraphSource(item);
		if (source == null) return null;
		Graph obs_graph = (Graph)source_graph_map.get(source);
		return obs_graph;
	}
	
//	private void doRecursiveTreeAction(TreeItem item, TreeAction action) {
//	
//		action.run(item);
//
//		TreeItem[] children = item.getItems();
//		for (int i = 0; i < children.length; i++) {
//			doRecursiveTreeAction(children[i],action);	
//		}
//	}

	abstract class TreeAction {
		public abstract void run(TreeItem item);
	}
	
	//////////////////////////////
	// tree actions
	//////////////////////////////
	
	
	//////////////////////////////
	// inner classes
	//////////////////////////////


	class DropListener {
		String extension;
		StatConDropListener listener;	
	}

	class BasicAction implements SelectionListener {
		String id = "unknown";
		String name;
		Runnable runnable;	
		MenuItem menuitem;
		Image image;
			
		public void dispose() {
			if (menuitem != null) menuitem.dispose();
		}

		public void widgetSelected(SelectionEvent e) {
			runnable.run();
		}
		public void widgetDefaultSelected(SelectionEvent e) {
			runnable.run();
		}
	}
	
	class SetSliderAction implements Runnable {
		boolean horizontal;
		ZoomSlider slider;
		TreeItem item;
		public SetSliderAction (boolean horizontal, ZoomSlider slider, TreeItem item) {
			this.horizontal = horizontal;
			this.slider = slider;
			this.item = item;
		}
		public void run() {
			setTreeItemSlider(item,horizontal,slider);
			graph_window.redraw();
			graph_window.updateTables();
		}
	}

	class CBEComparator implements Comparator {
		public int compare(Object o1, Object o2) {
			CBECommonBaseEvent e1 = (CBECommonBaseEvent)o1;	
			CBECommonBaseEvent e2 = (CBECommonBaseEvent)o2;
			return (int)(e1.getCreationTime() - e2.getCreationTime());	
		}	
	}

	class CBEUpdate implements Runnable {
		public void run() {
			appendAllCBEs();	
		}	
	}

	class RootNode {
		String name;
		Image image;
		ArrayList actions = new ArrayList();
		TreeItem item;
		String id;
		String parent_id;
		ArrayList node_children = new ArrayList();
		
		public void dispose() {
			item.dispose();	
		}
	}
	
	class StatAgent {
		TRCAgent agent = null;
		Image image = null;
		ArrayList actions = new ArrayList();
		ArrayList modelactions = new ArrayList();
		ArrayList model_roots = new ArrayList();
		URI uri;
		int msgindex = 0;
		boolean show_messages = true;
		
		TreeItem msg_item = null;
		TreeItem msg_info_item = null;
		TreeItem msg_warn_item = null;
		TreeItem msg_error_item = null;
		TreeItem msg_loc_item = null;
		
		HashMap msg_locations = new HashMap();
	}

}
