/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.grapher.internal;

import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;

public class LineGraph implements Graph {

/** use nearest data to requested points */
public static final int TYPE_NEAREST = 0;
/** average data between requested points */
public static final int TYPE_AVERAGE = 1;
/** sum data between requested points */
public static final int TYPE_SUM = 2;

private static final boolean DRAW_ONLY_IF_DATA = true;
	
GraphSource source;
ZoomSlider yslider;
TimeZoomSlider xslider;

Color fg_col;
int line_width = 1;
int line_style= SWT.LINE_SOLID;

int type;

double scaling = 1.0;

double last_drawn_value = 0.0d;

boolean pixel_perfect = false;
boolean use_indicators = true;

ZoomIndicator indicator;

IndicatorSource isource;

	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col) {
		this.type = type;
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
		init();
	}
	
	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col, boolean pixel_perfect, boolean use_indicators) {
		this.type = type;
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
		this.pixel_perfect = pixel_perfect;
		this.use_indicators = use_indicators;
		init();
	}

	public BasicGraphSource getGraphSource() {
		return source;	
	}
		
	private void init() {
		if (use_indicators) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	public void setIndicatorSource(IndicatorSource source) {
		isource = source;
	}

	public void setUseIndicator(boolean b) {
		if (use_indicators && !b) {
			yslider.removeIndicator(indicator);
			indicator.dispose();			
		} else if (!use_indicators && b) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
		use_indicators = b;
	}
	
	public double getXMin() {
		return source.getMin();		
	}

	public double getXMax() {
		return source.getMax();
	}

	public double getYMin() {
		return source.getValueMin();		
	}

	public double getYMax() {
		return source.getValueMax();		
	}

	public void setForeground(Color col) {
		fg_col = col;
		if (use_indicators) {
			indicator.setColor(col);	
		}
	}
	
	public Color getForeground() {
		return fg_col;
	}

	public void setLineWidth(int w) {
		line_width = w;
	}
	
	public int getLineWidth() {
		return line_width;
	}
	
	public void setLineStyle(int swt_line_style) {
		line_style = swt_line_style;
	}
	
	public int getLineStyle() {
		return line_style;	
	}

	public void paintGraph(GC gc, int x, int y, int w, int h) {

		gc.setForeground(fg_col);
		gc.setLineWidth(line_width);	
		gc.setLineStyle(line_style);
		
//we always draw on the ticks, this way the graphs stay constant 
//since we are averaging over the same periods and thus the same
//data all the time
		ZoomSliderTick tick_start = xslider.getFirstTick();
		ZoomSliderTick tick_end = xslider.getLastTick();

		if (tick_start == null || tick_end == null) return;

//		double start = xslider.getMinVisible();
//		double end = xslider.getMaxVisible();

//		double start = xslider.pixel2Value(x);
//		double end = xslider.pixel2Value(x+w);

		double res = xslider.getIncrement();

		double start = tick_start.value - res;
		double end = tick_end.value;

		if (pixel_perfect) {
			res = xslider.getUnitIncrement();
		}

		boolean found_indicator = false;
		double indicator_loc = isource.getIndicatorLocation();
		double indicator_val = last_drawn_value;
		int indicator_type = isource.getSourceType();

		boolean first = true;

		double px = 0;
		double py = 0;
		
		double last_cy = 0;
		
		for (double i = start; i <= end; i+= res) {

			double cx = i;
			
			if (DRAW_ONLY_IF_DATA) {
				//dont draw past the maximum extent of the data
				if (i > (source.getMax()+res)) continue;
				if (i < (source.getMin()-res)) continue;
			}

			double pixel_x;
			double pixel_y;

			//dont draw lines on the first run, we wont have valid previous pixel data
			if (first) {
				first = false;

				double cy = source.getValueAt(cx);
	
				//apply static scaling
				cy = cy * scaling;
	
				last_cy = cy;
	
				pixel_x = xslider.value2Pixel(cx);
				pixel_y = yslider.value2Pixel(cy);
			
				gc.drawLine((int)pixel_x,(int)pixel_y,(int)pixel_x,(int)pixel_y);		
								
			} else {

				double cy = 0;

				if (type == TYPE_NEAREST) {
					cy = source.getValueAt(cx);
				} else if (type == TYPE_AVERAGE) {
					cy = source.getAverageBetween(cx-res,cx);
				} else if (type == TYPE_SUM) {
					cy = source.getSumBetween(cx-res,cx);
				} else {
					//unrecognised type, default to average
					cy = source.getAverageBetween(cx-res,cx);
				}
	
				//apply static scaling
				cy = cy * scaling;

				//Indicator source type TIME - specifies the time value to show values at 
				if (indicator_type == IndicatorSource.TYPE_TIME) {
					if (indicator_loc > cx-res && indicator_loc < cx) { 	
						found_indicator = true;
						indicator_val = cy;
					}
				}

				last_cy = cy;
	
				pixel_x = xslider.value2Pixel(cx);
				pixel_y = yslider.value2Pixel(cy);

				//Indicator source type PIXEL - specifies the time value to show values at 
				if (indicator_type == IndicatorSource.TYPE_PIXEL) {
					if (indicator_loc > px && indicator_loc < pixel_x) { 	
						found_indicator = true;
						indicator_val = cy;
					}
				}

				gc.drawLine((int)px,(int)py,(int)pixel_x,(int)pixel_y);
			}
			
			px = pixel_x;
			py = pixel_y;

		}
		
		//this will be the value that appears on the indicator
		last_drawn_value = last_cy;

		if (isource == null || !found_indicator) {
			indicator.setValue(last_drawn_value);
		} else {
			indicator.setValue(indicator_val);
		}
		yslider.updateIndicators();
	}
	
	public void paintGraphNearest(GC gc, int x, int y, int w, int h) {
	}

	public TimeZoomSlider getXSlider() {
		return xslider;	
	}

	public ZoomSlider getYSlider() {
		return yslider;	
	}
	public void setXSlider(TimeZoomSlider slider) {
		xslider = slider;
	}
	
	public void setYSlider(ZoomSlider slider) {
		if (use_indicators) {
			yslider.removeIndicator(indicator);
			indicator.dispose();
		}
		yslider = slider;
		if (use_indicators) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	public void setStaticScaling(double mult) {
		scaling = mult;
	}
	public double getStaticScaling() {
		return scaling;	
	} 

}