/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal;

import java.util.Vector;

import org.eclipse.hyades.statistical.ui.*;

import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;

import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;


import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.DisposeEvent;

public class ZoomSlider extends Canvas
{   
    public static final int MIN = 0;
    public static final int MAX = 1;

    public static final int TOP = 0;
    public static final int BOTTOM = 1;        

    public static final int MIN_TICKWIDTH = 10;
    public static final int MAX_TICKWIDTH = 25;
    
    public static final int NORTH = 0;
    public static final int SOUTH = 1;
    public static final int EAST = 2;
    public static final int WEST = 3;
    
    public static final int VERTICAL = 0;
    public static final int HORIZONTAL = 1;
    
    public static final int INCREASING = 1;
    public static final int DECREASING = -1;
    
    private static final double [] LINEAR_PROGRESSION = {1, 2, 5};
    
    
    /** Vector of Ticks */
    protected Vector tickVector = new Vector();
    
    /** Vector of ZoomSliderListeners registered on this Slider*/ 
    Vector listeners = new Vector(); 
    
    /** Collection of Bars */
    ZoomControlBarsCollection barsCollection = new ZoomControlBarsCollection(this);

    /** Collection of Indicators */
    ZoomIndicatorsCollection indicatorsCollection = new ZoomIndicatorsCollection(this);

    /** Stores last tick to be painted with text visible
        (used to avoid overlapping of long text when zoomSlider 
        orientation is HORIZONTAL)
    */
    transient ZoomSliderTick lastVisible;    
    
    
    /** Min and Max possible values */
    protected double [] limit = {0, 100};
    
    /** Min and Max visible values */
    protected double [] visible = {0, 50};
    
    /** Min and Max lock status */
    protected boolean [] locked = {false, false};
    
    /** Visual orientation */
    protected int orientation = VERTICAL;
    
    /** Visual direction */
    protected int direction = INCREASING;
    
    /** Resolution - minimum possible value difference between consecutive ticks */
    protected double resolution = 1;
    
    /** Pixels between component edge and first tick */
    protected int indent = 10;
    
    /** Image used for double buffering */
    protected Image image;
    protected GC gImage;
    
    /** Image used to buffer the scale (without the bars being drawn on it).
        This is used to cover up the indicators when their value is reduced */
    protected Image plainScale;
    protected GC gPlain;
    
    protected Image titleImage;
    protected GC gTitle;
    
    int titWidth, titHeight;
    
    Color backgroundColor;
    Color foregroundColor;
    Color titleColor;
    
    Font titleFont;
    
    protected boolean zoomable = true;
    protected boolean transposable = true;
    
    /* Current value-difference between consecutive ticks */
    transient double increment = 1;
    
    /* Order of magnitude of increment property */
    transient int incrementOOM;
    
    /* Current difference between consecutive selectable positions on the scale */
    transient double unitIncrement;
    
    /* x-pixel value of decimal point position (used only when orientation is VERTICAL) */
    transient int decimalPointPosition;
    
    /* determines whether the image buffer should be redrawn when repainting */        
    transient boolean scaleChanged = true;
    
    transient boolean indicatorsChanged;
    
    /* Zoom state */
    transient boolean zoomPrimed;
	transient boolean zoomConfigured;
	transient int zoomStart;
	transient double zoomValue;
	transient int zoomFixed;
	transient boolean zoomed;
	
    /* Transpose state */
	transient boolean transposePrimed;
	transient boolean transposeConfigured;
	transient int transposeStart;
	transient double transposeValue;
	transient double transposeRange;
    transient boolean transposed;
    
    String title;
    
    public int rawOffset;
    
   	int fontHeight;
    
    // CONSTRUCTOR
    /** Constructor */
	public ZoomSlider(Composite parent)
	{
		super(parent, SWT.NO_BACKGROUND);
		
		FontData fd = getFont().getFontData()[0];
		
        indent = (fd.getHeight() + 4)/2;
        fontHeight = fd.getHeight();

		backgroundColor = parent.getBackground();
		foregroundColor = parent.getForeground();
		
		if (backgroundColor == null) backgroundColor = getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		if (foregroundColor == null) foregroundColor = getDisplay().getSystemColor(SWT.COLOR_LIST_FOREGROUND);		
	    titleColor = foregroundColor;
		
		rawOffset = java.util.TimeZone.getDefault().getRawOffset();
		if (rawOffset < 0)
		{
		    rawOffset += 24 * 60 * 60 * 1000;
		}    

		// Listeners
		SliderMouseListener sliderMouseListener = new SliderMouseListener();
		addMouseListener(sliderMouseListener);
		
		SliderMouseMoveListener sliderMouseMoveListener = new SliderMouseMoveListener();
		addMouseMoveListener(sliderMouseMoveListener);

		SliderPaintListener sliderPaintListener = new SliderPaintListener();
		addPaintListener(sliderPaintListener);
		
		SliderControlListener sliderControlListener = new SliderControlListener();
		addControlListener(sliderControlListener);
		
		SliderDisposeListener sliderDisposeListener = new SliderDisposeListener();
		addDisposeListener(sliderDisposeListener);
	}
	
	public ZoomSlider(Composite parent, double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    this(parent);
	    
	    configure(minLimit, maxLimit, minVisible, maxVisible, resolution);
	}    
	
    public void addZoomControlBar(double value)
    {
        new ZoomControlBar(this, value);
        
        updateScale();
    }
    
    public void addZoomControlBar(ZoomControlBar bar)
    {
        barsCollection.add(bar);
        
        updateScale();
    }
    
    public void removeZoomControlBar(ZoomControlBar bar)
    {
        barsCollection.remove(bar);
        
        updateScale();
    }
    
    public boolean hasBars()
    {
        return barsCollection.hasBars();
    }
    
    public ZoomControlBarsCollection getZoomControlBarsCollection()
    {
        return barsCollection;
    }    
    
    public void addZoomIndicator(ZoomIndicator indicator)
    {
        indicatorsCollection.add(indicator);
        
        updateScale();
    }
    
    public void removeIndicator(ZoomIndicator indicator)
    {
        indicatorsCollection.remove(indicator);
        
        updateScale();
    }    
    
    public ZoomIndicatorsCollection getZoomIndicatorsCollection()
    {
        return indicatorsCollection;
    }    
    
    public Vector getTickVector()
    {
        return tickVector;
    }    
	
	public ZoomSliderTick getFirstTick()
	{
		if (tickVector.size() == 0) return null;
		return (ZoomSliderTick)tickVector.get(0);
	}

	public ZoomSliderTick getLastTick()
	{
		if (tickVector.size() == 0) return null;
		return (ZoomSliderTick)tickVector.get(tickVector.size()-1);
	}
	
	public void reinitialiseGraphics() 
	{
        if (image != null) image.dispose();
        if (gImage != null) gImage.dispose();
        if (plainScale != null) plainScale.dispose();
        if (gPlain != null) gPlain.dispose();
        
        if (getDisplay() == null) return;
        if (getBounds().width < 1) return;
		if (getBounds().height < 1) return;
        
        // Recreate the double-buffering image.
        image = new Image(getDisplay(), getBounds().width, getBounds().height);
        gImage = new GC(image);

        // Recreate the plainScale image
        plainScale = new Image(getDisplay(), getBounds().width, getBounds().height);
        gPlain = new GC(plainScale);

		// Find new decimal point position
		findDecimalPointPosition();
		
        createTicks();
	}
    
    // PAINT
    /** Paints this component */
    public void paint(PaintEvent e)
    {
    	if (isDisposed()) {
    		return;	
    	}
    	
        if (gImage == null) 
        {
            reinitialiseGraphics();
        }
        
        if (scaleChanged)
        {
            paintTicks(gImage);
            
            paintTicks(gPlain);
            
            scaleChanged = false;
            
            indicatorsCollection.paint(gImage, null);
        }
        
        if (indicatorsChanged)
        {
            indicatorsCollection.paint(gImage, plainScale);
        }    
        
        barsCollection.paint(image, gImage, e.gc, indicatorsChanged);
        
        indicatorsChanged = false;
    }
    
    public void updateScale()
    {
        scaleChanged = true;
        
        redraw();
    }
    
    public void recreateScale()
    {
        createTicks();
    
        updateScale();
    }
    
    public void updateIndicators()
    {
        indicatorsChanged = true;
        
        redraw();
    }    

    public void externalRepaint()
    {
        scaleChanged = true;
        
        redraw();
    }
    
    public void resizeScale()
    {
    	reinitialiseGraphics();
    	indicatorsCollection.calculatePositions();
    	recreateScale();
    }    
    
    public void layout()
    {
        super.layout();
        
        indicatorsCollection.calculatePositions();
        
        // Extend the range of visible values if the layout action would cause the 
        // maxTickWidth property to be exceeded
        if (isMaxTickWidthExceeded())
        {
            visible[MAX] = visible[MIN] + (getPixelRange()/MAX_TICKWIDTH * resolution);
            
            if (visible[MAX] > limit[MAX])
            {
                visible[MAX] = limit[MAX];
                
                visible[MIN] = visible[MAX] - (getPixelRange()/MAX_TICKWIDTH * resolution);
            }    
            
            updateScale();
        }    
    }
        
    // SET FONT
    /** Set the font for this component */
    public void setFont(Font font)
    {
		FontData fd = font.getFontData()[0];
		
        indent = (fd.getHeight() + 4)/2;
        fontHeight = fd.getHeight();

        super.setFont(font);
    }    
    
    // GET PIXEL RANGE
    /** Get the width or height of the visible area */
    protected int getPixelRange()
    {
        if (orientation == HORIZONTAL)
        {
            return getBounds().width - (indent * 2);
        }
        else
        { 
            return getBounds().height - (indent * 2);
        }    
    }
    
    
    
    // GET ZERO PIXEL VALUE 
    /** Get the value at the pixel indent */
    protected double getZeroPixelValue()
    {
        if (direction == DECREASING)
        {
            return visible[MIN];
        }
        else
        {
            return visible[MAX];
        }
    }    
    
    
    
    // GET VISIBLE VALUE RANGE
    /** Get range of visible values */
    protected double getVisibleValueRange()
    {
        return Math.abs(visible[MAX] - visible[MIN]);
    }    



    // GET SINGLE PIXEL RANGE
    /** Get value range across a single pixel */
    protected double getSinglePixelRange()
    {
        return getVisibleValueRange() / getPixelRange();
    }    

    
    
    // PIXEL TO VALUE
    /** Convert a pixel to its value */ 
    public double pixel2Value(double pixelPos)
    {
        return getZeroPixelValue() - (direction * ((pixelPos - indent) * getSinglePixelRange()));
    }
    
    
    
    // VALUE TO PIXEL 
    /** Convert a value to its pixel */ 
    public double value2Pixel(double value)
    {
        return (((getZeroPixelValue() - value) / direction) / getSinglePixelRange()) + indent;        
    }    



    // GET INCREMENT
    /** Get the value spacing between ticks */
    public double getIncrement()
    {
        return increment;
    }

    

    // GET INCREMENT OOM
    /** Get the order of magnitude of the value spacing between ticks */
    public int getIncrementOOM()
    {
        return incrementOOM;
    }
    
    
    // GET UNIT INCREMENT
    /** Get the value spacing between consecutive selectable positions on the scale */
    public double getUnitIncrement()
    {
        return unitIncrement;
    }    
    
    // GET DECIMAL POINT POSITION
    /** Get the pixel position of the decimal point */
    public int getDecimalPointPosition()
    {
        return decimalPointPosition;
    }    
    
    // GET SCALE IMAGE
    /** Get the image which is used to double-buffer the drawing of the scale */
    public Image getScaleImage()
    {
        return image;
    }
    
    public int getIndent()
    {
        return indent;
    }
    
    public int getFontHeight()
    {
    	return fontHeight;
    }    
    
    
    /** Tells some visiting thread whether or not the scale is currently changing */
    public boolean isChanging()
    {
        if (zoomConfigured)
        {
            return true;
        }
        
        if (transposeConfigured)
        {
            return true;
        }
        
        return false;
    }    
    
    
    
    // FIND DECIMAL POINT POSITION
    /** Find the pixel position of the decimal point */
    protected void findDecimalPointPosition()
    {
        if (orientation == VERTICAL)
        {
            int tail = 0;
            
            if (increment < 1)
            {
                GC gcTemp = new GC(this);
                tail =  Math.abs(incrementOOM) * gcTemp.stringExtent("0").x + gcTemp.stringExtent(".").x;
            }    
            
            decimalPointPosition = getBounds().width - tail - 5;
        }
    }
    
    class TickValueRange
    {
        public double min, max;
        
        public TickValueRange()
        {
            min = getSinglePixelRange() * MIN_TICKWIDTH;
            if (min < resolution) min = resolution;
            
            max = getSinglePixelRange() * MAX_TICKWIDTH;
            if (max < resolution) max = resolution;
        }
        
        public boolean outside()
        {
            return (increment < min || increment > max);
        }
    }    


    // CALCULATE INCREMENT
    /** Calculate the value spacing between consecutive ticks */
    protected void calculateIncrement()
    {
        TickValueRange tvr = new TickValueRange();
        
        if (tvr.outside())
        {
            int maxTickCount = getPixelRange() / MIN_TICKWIDTH;

            increment = getVisibleValueRange() / maxTickCount;
            
            int orderOfMagnitude = ZoomSliderUtilities.calculateOOM(increment);
            
            increment = LINEAR_PROGRESSION[0] * Math.pow(10, orderOfMagnitude);
            
            while (increment < tvr.min)
            {
                for (int i=0; i<3; i++)
                {
                    increment = LINEAR_PROGRESSION[i] * Math.pow(10, orderOfMagnitude);
                    
                    if (increment >= tvr.min) break;
                }
                orderOfMagnitude++;
            }
            
            incrementOOM = ZoomSliderUtilities.calculateOOM(increment);
            
            findDecimalPointPosition();
        }    
    }
    

    // CREATE TICKS
    /** Create the vector of ticks */
    protected void createTicks()
    {
        calculateIncrement();
        
        // Empty the tick vector
        tickVector.removeAllElements();
        
        // Refill the tick vector
        double tickValue = ZoomSliderUtilities.round(visible[MIN] - increment, increment);
        
        int loop = 0;
        
        while (tickValue <= visible[MAX] + increment)
        {
            double tickIn = tickValue;
            
            tickVector.addElement(new ZoomSliderTick(this, tickValue));

            tickValue += increment;
            
            tickValue = ZoomSliderUtilities.round(tickValue, increment);
            
            loop++;
            
            if (loop > 500)
            {
                break;
            }    
        }
        
        // Depending on the pixel spacing between ticks find a reasonable number of 
        // intermediate selecteable points (5 or 10)
		int selectablePoints = 10;
    	int tickSpacing = (int)(getPixelRange()/(getVisibleValueRange()/increment));
		if (tickSpacing < 20)
		{
		    selectablePoints = 5;
		}
		
        // Set the unit increment based on the intermediate selectable points
		unitIncrement = increment/selectablePoints;
		if (unitIncrement < resolution)
		{
		    unitIncrement = resolution;
		}
    }

    
    
    // PAINT TICKS
    /** Paint the ticks */
    private void paintTicks(GC g)
    {
        g.setBackground(backgroundColor);
        
        g.fillRectangle(0,0,getBounds().width, getBounds().height);
        
        g.setForeground(foregroundColor);
        
        // initialise lastVisible
        lastVisible = null;
        
        paintTitle(g);

        try
        {
            // Paint each tick
            for (int i=0; i<tickVector.size(); i++)
            {
                ((ZoomSliderTick)(tickVector.elementAt(i))).paintLine(g);
            }
            for (int i=0; i<tickVector.size(); i++)
            {
                ((ZoomSliderTick)(tickVector.elementAt(i))).paintText(g);
            }
        }
        catch (NullPointerException e) 
        {
            EditorPlugin.DBG.warning("tickVector elements null");
            return;
        }
        
        switch (orientation)
        {
            case VERTICAL:
                g.drawLine(getBounds().width - 8, indent, getBounds().width, indent);
                g.drawLine(getBounds().width - 8, getBounds().height - indent, getBounds().width, getBounds().height - indent);
                g.drawLine(0, indent, 8, indent);
                g.drawLine(0, getBounds().height - indent, 8, getBounds().height - indent);
                break;
                
            case HORIZONTAL:
                g.drawLine(indent, 0, indent, 8);
                g.drawLine(getBounds().width - indent, 0, getBounds().width - indent, 8);
                g.drawLine(getBounds().width - indent, getBounds().height - 8, getBounds().width - indent, getBounds().height);
                g.drawLine(indent, getBounds().height - 8, indent, getBounds().height);
                break;
        }        

        // Draw the border 
        g.setForeground(foregroundColor);
        g.drawRectangle(0, 0, getBounds().width - 1, getBounds().height - 1);
        
        //paintTitle(g);
    }
    
    private void paintTitle(GC g)
    {
        if (titleImage == null)
        {
            createTitle(g);
        }    
            
        if (titleImage != null)
        {
            if (orientation == HORIZONTAL)
            {
                //g.drawImage(titleImage, getSize().width/2 - titWidth/2, getSize().height - (titHeight + 5), null);
                g.drawImage(titleImage, getBounds().width - titWidth - (indent + 5), getBounds().height - (titHeight + 2));
            }
            else
            {
        	    //g.drawImage(titleImage, 5, getSize().height/2 - titWidth/2, null);
        	    
        	    g.drawImage(titleImage, 2, indent + 5);
            }
        }    
    }
    
    private void createTitle(GC g)
    {
        if (title != null)
        {
            if (!title.equals(""))
            {
				g.setFont(titleFont);
                titWidth = g.stringExtent(title).x;
                titHeight = g.stringExtent(title).y;
                g.setFont(getFont());
                
                // Create Image Buffer for title
                titleImage = new Image(getDisplay(), titWidth, titHeight);
                gTitle = new GC(titleImage);

                gTitle.setBackground(backgroundColor);
                
                gTitle.fillRectangle(0, 0, titWidth, titHeight);
                
                gTitle.setForeground(titleColor);

				gTitle.setFont(titleFont);
				
                gTitle.drawString(title, 0, 0);
                
                g.setForeground(foregroundColor);
                
                if (orientation == VERTICAL)
                {
        	        titleImage = rotateImage();
                }
            }    
        }    
    }    
    
    Image rotateImage()
    {
    	ImageData titleData = titleImage.getImageData();
    	
    	Image temp = new Image(getDisplay(), titleData.height, titleData.width);
    	
    	ImageData tempData = temp.getImageData();
    	
    	for (int h=0; h<titleData.height; h++)
    	{
    		for (int w=0; w<titleData.width; w++)
    		{
    			tempData.setPixel(h, tempData.height - 1 - w, titleData.getPixel(w,h));
    		}
    	}
    	
    	return new Image(getDisplay(), tempData);
    }
    
    
    // computeSize
    /** The preferred size of the component is dependent on the maximum possible string
        length out of all the ticks */
	public Point computeSize(int wHint, int hHint, boolean changed)
	{
		Point standard = super.computeSize(wHint, hHint, changed);
		Point size = new Point(standard.x, standard.y); 
		
		GC gcTemp = new GC(this);
		
   	    if (orientation == VERTICAL)
	    {
			if (wHint == SWT.DEFAULT)
			{
	            // Find out whether the max or min value is the "longest"
	            String maxTick = (new ZoomSliderTick(this, limit[MAX] + resolution)).representation;
	            String minTick = (new ZoomSliderTick(this, limit[MIN] - resolution)).representation;
	
	            int biggestMax = gcTemp.stringExtent(maxTick).x;
				int biggestMin = gcTemp.stringExtent(minTick).x;            
	
	            int biggest = biggestMax;
	            if (biggestMin > biggest)
	            {
	                biggest = biggestMin;
	            }
	
		        size.x = biggest + 20;
			}
			else
			{
				size.x = wHint;
			}
			
			if (hHint != SWT.DEFAULT)
			{
				size.y = hHint;
			}
	    }
        else
	    {
	    	if (hHint == SWT.DEFAULT)
	    	{
	    		size.y = gcTemp.getFontMetrics().getAscent() + 20;
	    	}
	    	else
	    	{
	    		size.y = hHint;
	    	}
	    	
	    	if (wHint != SWT.DEFAULT)
	    	{
	    		size.x = wHint;
	    	}
	    }
	    
		gcTemp.dispose();
		
	    return size;
	}
		
	public void configure(double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    checkConfiguration(minLimit, maxLimit, minVisible, maxVisible, resolution);
		
	    limit[MIN] = minLimit;
	    limit[MAX] = maxLimit;
	    visible[MIN] = minVisible;
	    visible[MAX] = maxVisible;
	    this.resolution = resolution;
	    
	    try
	    {
	        recreateScale();
	    }
	    catch (NullPointerException e){}
	} 
	
	void checkConfiguration(double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    if (minLimit >= maxLimit)
	    {
	    	throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_MAX_TOO_SMALL"));
	    }
	        
	    if (minVisible >= maxVisible)
	    {
	    	throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_MAXVIS_TOO_SMALL"));
	    }
	    
	    if (minVisible < minLimit)
	    {
	    	throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_MIN_TOO_LARGE"));
	    }
	    
	    if (maxVisible > maxLimit)
	    {
	    	throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_MINVIS_TOO_LARGE"));
	    }
	    
		if (resolution > (maxLimit - minLimit))
		{
			throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_RES_TOO_LARGE"));
		}
		
		if (resolution <= 0)
		{
			throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_RES_NONPOS"));
		}

		double oomZeroVersion = resolution * Math.pow(10, -1 * ZoomSliderUtilities.calculateOOM(resolution));
		if (oomZeroVersion != 1 && oomZeroVersion != 2 && oomZeroVersion != 5)
		{
			throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_RES_BADPOW"));
		}
	}
	
	public void setTitle(String title)
	{
	    this.title = title;
	    
	    titleImage = null;
	}
	
	public String getTitle()
	{
	    return title;
	}    
	
	public void setTitleColor(Color color)
	{
	    titleColor = color;
	    
	    titleImage = null;
	}
	
	public Color getTitleColor()
	{
	    return titleColor;
	}
	
	public void setTitleFont(Font font)
	{
		titleFont = font;

	    titleImage = null;
	}
	
	public Font getTitleFont()
	{
		return titleFont;
	}    
	
	public void setForeground(Color color)
	{
		super.setForeground(color);
		
		if (color != null)
		{
		    foregroundColor = color;
		}

	    titleImage = null;
	}

	public Color getForeground()
	{
	    return foregroundColor;
	}

	public void setBackground(Color color)
	{
		super.setBackground(color);
		
		if (color != null)
		{
		    backgroundColor = color;
		    indicatorsCollection.setXORColors();

		}

	    titleImage = null;
	}

	public Color getBackground()
	{
		if (backgroundColor != null) {
	    	return backgroundColor;
		} else {
			return getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		}
	}

	public void setMaxVisible(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], visible[MIN], d, resolution);

	    visible[MAX] = d;
        
        recreateScale();
    }
    
    public double getMaxVisible()
    {
        return visible[MAX];
    }    
	
	public void setMinVisible(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], d, visible[MAX], resolution);

        visible[MIN] = d;
        
        recreateScale();
    }    

    public double getMinVisible()
    {
        return visible[MIN];
    }    


	public void setMaxLimit(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], d, visible[MIN], visible[MAX], resolution);

	    limit[MAX] = d;
        
        recreateScale();
    }    

    public double getMaxLimit()
    {
        return limit[MAX];
    }    
	
	public void setMinLimit(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(d, limit[MAX], visible[MIN], visible[MAX], resolution);

	    limit[MIN] = d;

        recreateScale();    
    }    

    public double getMinLimit()
    {
        return limit[MIN];
    }    

	public void setResolution(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], visible[MIN], visible[MAX], d);		

	    resolution = d;
        scaleChanged = true;
        findDecimalPointPosition();
	    redraw();       
    }
    
    public double getResolution()
    {
        return resolution;
    }    
	
	public void setDirection(int direction) throws ZoomSliderConfigurationException
	{
		if (direction == INCREASING || direction == DECREASING)
		{
	    	this.direction = direction;

	        recreateScale();
		}
		else
		{
			throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_BADDIRECTION"));
		}
	}
	
	public int getDirection()
	{
	    return direction;
	}    

	public void setOrientation(int orientation) throws ZoomSliderConfigurationException
	{
		if (orientation == HORIZONTAL || orientation == VERTICAL)
		{
	        this.orientation = orientation;
        
        	recreateScale();
		}
		else
		{
			throw new ZoomSliderConfigurationException(EditorPlugin.getString("ZOOMSLIDER_CONFIG_BADORIENTATION"));
		}			
	}    
	
    
    public int getOrientation()
	{
	    return orientation;
	}
	
	
	public void lock(int end, boolean lock)
	{
	    locked[end] = lock;
	}
	
	public boolean isLocked(int end)
	{
	    return locked[end];
	}
	
	public void setZoomable(boolean b)
	{
	    zoomable = b;
	}
	
	public boolean isZoomable()
	{
	    return zoomable;
	}
	
	public void setTransposable(boolean b)
	{
	    transposable = b;
	}
	
	public boolean isTransposable()
	{
	    return transposable;
	}    

	public void setLastVisible(ZoomSliderTick tick)
	{
	    lastVisible = tick;
	}

	public ZoomSliderTick getLastVisible()
	{
	    return lastVisible;
	}
	
	
	private boolean isMaxTickWidthExceeded()
	{
	    if (getVisibleValueRange()/resolution < getPixelRange()/MAX_TICKWIDTH)
	    {
	        return true;
	    }
	    return false;
	}
	
	public double getMinRange()
	{
	    return getPixelRange()/MAX_TICKWIDTH * resolution;
	}    
	    


    // COMPLEMENT
    /** Convenience method, converts from zero to one and visa-versa */
    private int complement(int i)
    {
        if (i == 0) return 1;
        return 0;
    }
    
	private void primeZoom(MouseEvent event)
	{
	    if (orientation == VERTICAL)
	    {
	        zoomStart = event.y;
	    }
	    else
	    {
	        zoomStart = event.x;
	    }
	    
	    zoomValue = pixel2Value(zoomStart);
    	    
	    zoomPrimed = true;	    
	}
	
	private void configureZoom(MouseEvent event)
	{
        int zoomPosition = 0;
        
        if (orientation == VERTICAL)
        {
            zoomPosition = event.y;
        }
        else
        {
            zoomPosition = event.x;
        }
        
        if (Math.abs(zoomPosition - zoomStart) > 2)
        {
            if (!locked[MIN] && !locked[MAX])
            {
                if (zoomPosition > zoomStart)
                {
    	            if (direction == INCREASING)
    	            {
    		            zoomFixed = MAX;
    	            }
    	            else
    	            {
    		            zoomFixed = MIN;
    	            }
                }
                else
                {
    	            if (direction == INCREASING)
    	            {
    		            zoomFixed = MIN;
    	            }
    	            else
    	            {
    		            zoomFixed = MAX;
    	            }
                }
                
                zoomPrimed = false;

                zoomConfigured = true;
            }
            else
            {
                if (locked[MAX] && locked[MIN])
                {
                    zoomPrimed = false;
                    
                    zoomConfigured = false;
                }
                else
                {
                    if (locked[MAX])
                    {
                        zoomFixed = MAX;
                    }
                    else
                    {
                        zoomFixed = MIN;
                    }
                    
                    zoomPrimed = false;
                    
                    zoomConfigured = true;
                }
            }    
        }    
    }
	    

	private void zoom(MouseEvent event)
	{
        int pixel = event.y;

	    if (orientation == HORIZONTAL)
	    {
	        pixel = event.x;
	    }    
	    
	    double diff = Math.abs(pixel - value2Pixel(visible[zoomFixed]));
	    
	    if (diff > 0)
	    {
	        double percentage = getPixelRange()/diff;
	        
	        double temp = visible[complement(zoomFixed)];
	        
	        if (zoomFixed == MIN)
	        {
                visible[complement(zoomFixed)] = visible[zoomFixed] + (percentage * Math.abs(visible[zoomFixed] - zoomValue));
	            
	            if (visible[complement(zoomFixed)] > limit[complement(zoomFixed)])
	            {
	                visible[complement(zoomFixed)] = limit[complement(zoomFixed)];
	            }    
	        }
	        else
	        {
                visible[complement(zoomFixed)] = visible[zoomFixed] - (percentage * Math.abs(visible[zoomFixed] - zoomValue));
	            
	            if (visible[complement(zoomFixed)] < limit[complement(zoomFixed)])
	            {
	                visible[complement(zoomFixed)] = limit[complement(zoomFixed)];
	            }    
	        }
	        
	        
	        if (isMaxTickWidthExceeded())
	        {
	            visible[complement(zoomFixed)] = temp;
	        } 
	        
	        if (visible[complement(zoomFixed)] != temp)
	        {
	        	zoomed = true;
	        	
	        	recreateScale();
	        	
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.ZOOMED, false));
	            }
	        }
        }    
	}    
	
	
    private void primeTranspose(MouseEvent event)
    {
	    if (!(locked[MIN] || locked[MAX]))
	    {
	        if (orientation == VERTICAL)
	        {
	            transposeStart = event.y;
	        }
	        else
	        {
	            transposeStart = event.x;
	        }
    	    
	        transposePrimed = true;
	    }    
    }    
    
    private void configureTranspose(MouseEvent event)
    {
        int transposePosition = 0;
        
   	    if (orientation == VERTICAL)
	    {
	        transposePosition = event.y;
	    }
	    else
	    {
	        transposePosition = event.x;
	    }

        if (Math.abs(transposePosition - transposeStart) > 2)
        {
            transposeValue = pixel2Value(transposePosition);
            
            transposeRange = getVisibleValueRange();

            transposePrimed = false;

            transposeConfigured = true;
        }    
    }
    
   	private void transpose(MouseEvent event)
	{
		double oldVisibleMin = visible[MIN];

        int pixel = event.y;

	    if (orientation == HORIZONTAL)
	    {
	        pixel = event.x;
	    }    
	    
        double diff = Math.abs(pixel - value2Pixel(visible[MIN]));
        
	    visible[MIN] = transposeValue - diff * getSinglePixelRange();
	    
	    if (visible[MIN] < limit[MIN])
	    {
	        visible[MIN] = limit[MIN];
	    }
	    
	    visible[MAX] = visible[MIN] + transposeRange;

        if (visible[MAX] > limit[MAX])
	    {
	        visible[MAX] = limit[MAX];
	        visible[MIN] = visible[MAX] - transposeRange;
	    }
	    
	    if (visible[MIN] != oldVisibleMin)
	    {
	    	transposed = true;
	    	
		    recreateScale();
		    
            for (int i=0; i<listeners.size(); i++)
            {
                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.TRANSPOSED, false));
            }
	    }
	}   

	public void addZoomSliderListener(ZoomSliderListener listener)
	{
		listeners.add(listener);
	}
	
	public void removeZoomSliderListener(ZoomSliderListener listener)
	{
		listeners.remove(listener);
	}
	
	class SliderPaintListener implements PaintListener
	{
		public void paintControl(PaintEvent e)
		{
			ZoomSlider.this.paint(e);
		}
	}
	
	// MOUSE LISTENER
	/** Listens for the mouse button being pressed, released and double-clicked */
	class SliderMouseListener implements MouseListener
	{
		public void mouseDown(MouseEvent e)
		{
			Slider_MouseDown(e);
		}
		
		public void mouseUp(MouseEvent e)
		{
			Slider_MouseUp(e);
		}

		public void mouseDoubleClick(MouseEvent e)
		{
			// Do nothing
		}
	}
	
	//MOUSE MOTION LISTENER
	/** Listens for the mouse being moved */
    class SliderMouseMoveListener implements MouseMoveListener
	{
		public void mouseMove(MouseEvent event)
		{
		    Slider_MouseMove(event);
		}
	}
	
	class SliderControlListener extends ControlAdapter 
	{
		public void controlResized(ControlEvent e)
		{
			ZoomSlider.this.resizeScale();
		}
    } 

	void Slider_MouseDown(MouseEvent event)
	{
	    if (!barsCollection.mouseDown(event))
	    {
	        if (event.button > 1)
	        {
				if (transposable) primeTranspose(event);
	        }    
	        else
	        {
                if (zoomable) primeZoom(event);
	        } 
	    }
	}
	
	void Slider_MouseUp(MouseEvent event)
	{
        if (!barsCollection.mouseUp(event))
        {
	        if (zoomPrimed) zoomPrimed = false;
    	    
	        if (zoomConfigured)
	        {
	            zoomConfigured = false;
	        }    
    	    
	        if (transposePrimed)
	        {
	            transposePrimed = false;
	        }
	        
	        if (transposeConfigured)
	        {
	            transposeConfigured = false;
	        }
	        
			if (zoomed)
			{
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.ZOOMED, true));
	            }
	            
	            zoomed = false;
			}
			
			if (transposed)
			{
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.TRANSPOSED, true));
	            }
	            
	            transposed = false;
			}
	    }
    }

    void Slider_MouseMove(MouseEvent event)
    {
    	String ntooltip;
    	
   	    ZoomControlBar bar = barsCollection.getZoomControlBarContaining(event.x, event.y);
   	    if (bar != null) {
   	    	ntooltip = bar.getToolTipText();
   	    	if (getToolTipText() == null || !getToolTipText().equals(ntooltip)) {
   	    		setToolTipText(ntooltip);
   	    	}
   	    } else {
  	    	ntooltip = null;	
   	    	if (getToolTipText() != null) {
   	    		setToolTipText(ntooltip);
   	    	}
   	    }

		if (!barsCollection.mouseMove(event))
		{
		    if (zoomPrimed) configureZoom(event);
    		
		    if (zoomConfigured) zoom(event);
    		
		    if (transposePrimed) configureTranspose(event);
		    
		    if (transposeConfigured) transpose(event);
		}    
    }
    
    class SliderDisposeListener implements DisposeListener
    {
    	public void widgetDisposed(DisposeEvent e)
    	{
			if (image != null) image.dispose();
			if (gImage != null) gImage.dispose();
			
			if (plainScale != null) plainScale.dispose();
			if (gPlain != null) gPlain.dispose();
			
			if (titleImage != null) titleImage.dispose();
			if (gTitle != null) gTitle.dispose();
			
//			if (backgroundColor != null) backgroundColor.dispose();
//			if (foregroundColor != null) foregroundColor.dispose();
//			if (titleColor != null) titleColor.dispose();
			
			if (titleFont != null) titleFont.dispose();
			
			barsCollection.dispose();
			indicatorsCollection.dispose();
    	}
    }    
}
