/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.PartInitException;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;

/**
 * Abstract extension of the Eclipse's <code>NewWizard</code> that defines a location 
 * page and a "name and description" page.  The subclasses are supposed to add the pages
 * in the appropriate order and using the correct values for their titles and descriptions.
 * 
 * <p>This implementation is based on the 
 * <code>org.eclipse.ui.wizards.newresource.BasicNewResourceWizard</code> class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public abstract class HyadesNewWizard 
extends HyadesWizard implements INewWizard, Listener
{
	/*
	 * Dialog Settings constants
	 */
	protected static final String SET_LAST_SELECTION = "LAST_SELECTION";
	
	private LocationPage locationPage;
	private AttributeWizardPage attributeWizardPage;
	private boolean firstNameDescriptionPageActivation = true;
	private IFile newFile;

	/**
	 * Constructor for HyadesNewWizard
	 */
	public HyadesNewWizard()
	{
		super();
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#dispose()
	 */
	public void dispose()
	{
		attributeWizardPage.removeActivationListener(this);
		super.dispose();
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
	 */
	protected void initPages()
	{
		IStructuredSelection structuredSelection = adjustSeletion();
		
		locationPage = new LocationPage("location", structuredSelection);
		String fileExtension = getFileExtension();
		if(fileExtension != null)
		{
			locationPage.setFileExtension(fileExtension);
			locationPage.setFileName("*." + fileExtension);
		}

		attributeWizardPage = new AttributeWizardPage("nameDescription");
		attributeWizardPage.addActivationListener(this);
	}
	
	/**
	 * Adjusts the selection passed to this wizard.
	 * @return IStructuredSelection
	 */
	protected IStructuredSelection adjustSeletion()
	{
		IStructuredSelection structuredSelection = getSelection();
		if(getDialogSettings() != null)
		{
			if((structuredSelection == null) || (structuredSelection.isEmpty()))
			{
				String lastSelection = getDialogSettings().get(SET_LAST_SELECTION);
				if(lastSelection != null)
				{
					IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(lastSelection);
					if((resource != null) && (resource.exists()))
						structuredSelection = new StructuredSelection(resource);
				}
			}
		}
		
		return structuredSelection;
	}
		
	/**
	 * Returns the location page used by this wizard.
	 * @return LocationPage
	 */
	protected LocationPage getLocationPage()
	{
		return locationPage;
	}
	
	/**
	 * Returns the "name and description page" used by this wizard.
	 * @return NameDescriptionWizardPage
	 */
	protected AttributeWizardPage getAttributeWizardPage()
	{
		return attributeWizardPage;
	}

	/**
	 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
	 */
	public void handleEvent(Event event)
	{
		if((event.type == AttributeWizardPage.EVENT_BEFORE_ACTIVATION) && (event.data == getAttributeWizardPage()))
		{
			if(firstNameDescriptionPageActivation)
			{
				firstNameDescriptionPageActivation = false;
				adjustAttributePage();
			}
		}
	}
	
	/**
	 * Adjusts the attribute page by copying values from previous pages to it.
	 */
	protected void adjustAttributePage()
	{
		if((getAttributeWizardPage() == null) || (getLocationPage() == null))
			return;
			
		if((getAttributeWizardPage().getItemName() == null) || ("".equals(getAttributeWizardPage().getItemName())))
		{
			String name = getLocationPage().getFileName();
			int index = name.lastIndexOf('.');
			if(index >= 0)
				name = name.substring(0, index);
			getAttributeWizardPage().setItemName(name);
		}
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */
	public boolean performFinish()
	{
		//todo> Add Progress bar
		adjustAttributePage();
		
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();	
		IPath path = getLocationPage().getContainerFullPath();
		if(root.findMember(path) == null)
		{
			try
			{
				ResourceUtil.createContainer(root.getFolder(path), null);
			}
			catch (CoreException e)
			{
				handleException(path.toString(), e);
				return false;
			}
		}
		
		path = path.append(getLocationPage().getFileName());
		IFile file = root.getFile(path);
		
		try
		{
			if(!createObject(file))
				return false;
		}
		catch(Exception e)
		{
			handleException(file.getFullPath().toString(), e);
			return false;
		}
			
		this.newFile = file;
		selectAndReveal(file);
		try
		{
			openEditor(file);
		}
		catch(PartInitException e)
		{
			handleException(file.getFullPath().toString(), e);
		}
		
		return true;
	}
	
	/**
	 * Selects and reveals the newly added resource in all parts
	 * of the active workbench window's active page.
	 * @param newResource
	 */
	protected void selectAndReveal(IResource newResource)
	{
		ResourceUtil.selectAndReveal(newResource, getWorkbench().getActiveWorkbenchWindow());
	}

	/**
	 * Returns the the required extension of the file to be created or 
	 * null if there is no such constraint.
	 * @return int
	 */
	protected String getFileExtension()
	{
		return null;
	}
	
	/**
	 * Opens the editor associated to a file.  Clents can overwrite.
	 * @param file
	 * @throws PartInitException
	 */
	protected void openEditor(IFile file)
	throws PartInitException
	{
		UIUtil.openEditor(file, null, false);
	}
	
	/**
	 * Handles any exception thrown during the 
	 * {@link #performFinish()} execution.
	 * @param exception
	 */
	protected void handleException(String filePath, Exception exception)
	{
		HyadesUIPlugin.logError(exception);
		UIUtil.openSaveFileErrorDialog(getShell(), filePath, exception);
	}
	
	/**
	 * Creates the object.  The file argument indicates the location and the
	 * file name selected in the location page.
	 * @param file
	 * @return <code>true</code> if the object was properly created or 
	 * <code>false</code> otherwise.
	 * @throws Exception any exception thrown by this method is handled
	 * by the {@link #handleException(Exception)} method.
	 */
	abstract protected boolean createObject(IFile file)
	throws Exception;
	
	/**
	 * Returns the newly created file if this wizard is completed successfully, otherwise
	 * returns <code>null</code>.
	 * 
	 * @return the new file object or <code>null</code>
	 * @author bjiang
	 */
	public IFile getNewFile()
	{
		return this.newFile;
	}
}
