/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.ContextIds;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCallStack;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUtils;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifelineDraft;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcesses;
import org.eclipse.hyades.uml2sd.trace.selection.IEObjectSelection;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFindProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDPagingProvider;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.hyades.uml2sd.ui.core.BasicExecutionOccurrence;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.load.BackgroundLoader;
import org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader;
import org.eclipse.hyades.uml2sd.ui.view.SDView;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.help.WorkbenchHelp;

/**
 * The abstract trace loader.
 * It is based on the parsing of MethodInvocations from the model.
 * Lifelines attribution is made by the concrete classes.
 */
public abstract class BaseTraceInteractions
	implements IUml2SDLoader, ISelectionListener, ISDFindProvider {

	/**
	 * With this we know how to implement IUML2SDLoader.setViewer()
	 */
	protected SDView view;
	
	/**
	 * The frame containing the current page
	 */
	protected Frame frame;
	
	protected TraceProcesses currentProcesses;

	/**
	 * Implementation of IUml2SDLoader
	 */
	public void setViewer(SDView view_) {
		view = view_;
	    //listen to selection from other part.
	    view.getSite().getWorkbenchWindow().getSelectionService().addSelectionListener(this);
		view.setSDFindProvider(this);
	    onSetViewer();
		WorkbenchHelp.setHelp(this.view.getSDWidget(), ContextIds.INTERACTION_VIEW);
		parseModelBackground();
	}
	
	abstract public void onSetViewer();
	
	public void aboutToBeReplaced() {
		UIPlugin.getDefault().removeSelectionListener(this);
		clearInternals();
		view = null;
	}
	
	protected EObject currentMofObjectFromPDProjectExplorer;
	private Cursor waitCursor;
	
	/**
	 * 
	 */
	protected void clearInternals() {
		currentMofObjectFromPDProjectExplorer = null;
		currentProcesses = null;
		frame = null;
		if (waitCursor != null && !waitCursor.isDisposed()) {
			waitCursor.dispose();
		}
		waitCursor = null;
	}

	/**
	 * Let's parse the model for the translation
	 * @param eObject
	 */
	protected synchronized void parseModel(EObject eObject) {
//long start = System.currentTimeMillis();
//System.err.println("Start parseModel()");
		clearInternals();
		currentMofObjectFromPDProjectExplorer = eObject;
		createFrame();
		if (currentProcesses == null) {
			if (view != null && !view.getSDWidget().isDisposed()) {
				setFrameName();
				view.setFrameSync(frame);
			}
//System.err.println("Abort parseModel()");
			return;
		}
		computeModel();
		try {
			fillPage(new TraceInteractionUpdate());
			if (view != null && !view.getSDWidget().isDisposed() && frame != null && !monitor.isCanceled()) {
				setFrameName();
				view.setFrameSync(frame);
			}
		} catch (Throwable e) {
			e.printStackTrace();
		}
//if (monitor.isCanceled()) {
//	System.err.println("Cancelled parseModel()"+(System.currentTimeMillis()-start));
//} else {
//	System.err.println("End parseModel()"+(System.currentTimeMillis()-start));
//}
	}
	
	abstract protected void computeModel();
	
	abstract protected void updateSD(TraceInteractionUpdate update);
	
	/**
	 * monitor is the IProgressMonitor to be checked not cancelled during long tasks
	 */
	protected IProgressMonitor monitor;
	
	/**
	 * @param monitor_
	 * @param taskName
	 */
	private void beforeLongTask(IProgressMonitor monitor_, String taskName) {
		if (view == null || view.getSDWidget().isDisposed()) {
			monitor_.setCanceled(true);
			return;
		}
		view.toggleWaitCursorSync(true);
		monitor = monitor_;
		monitor.beginTask(taskName, 100/*???*/);
	}
	
	/**
	 * 
	 */
	private void afterLongTask() {
		monitor.done();
		monitor = null;
		if (view != null && !view.getSDWidget().isDisposed()) {
			view.toggleWaitCursorSync(false);
		}
	}
	
	/**
	 * parseModel() in a background task
	 */
	protected void parseModelBackground() {
		final EObject eObject = HyadesUtil.getMofObject(); // Must be taken in the main loop thread
		if (eObject == null) {
			return;
		}
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor_) throws InterruptedException {
				beforeLongTask(monitor_, TraceSDPlugin.getResourceString("TASK_NAME_CREATING_SD")); //$NON-NLS-1$
				if (!monitor.isCanceled()) {
					parseModel(eObject);
				}
				afterLongTask();
			}
		});
	}
	
	/**
	 * updateSD() in a background task
	 */
	protected void updateSDBackground(final TraceInteractionUpdate update) {
		BackgroundLoader.getInstance().newTask(new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor_) throws InterruptedException {
				beforeLongTask(monitor_, TraceSDPlugin.getResourceString("TASK_NAME_UPDATING_SD")); //$NON-NLS-1$
				if (!monitor.isCanceled()) {
					updateSD(update);
				}
				afterLongTask();
			}
		});
	}
	
	/**
	 * Ensures that any items in this list are selected and that the first one is visible
	 * @param list
	 */
	protected void ensureSelectedAndMoveToFirst(List list) {
		if (list != null && list.size() > 0) {
			view.getSDWidget().addSelection(list);
			view.ensureVisibleSync((GraphNode)list.get(0));
		}
	}

	/**
	 * @param eObject
	 * @return
	 */
	protected void createFrame() {
		if (currentProcesses == null) {
			currentProcesses = TraceProcesses.getTraceProcesses(currentMofObjectFromPDProjectExplorer, monitor);
		}
		frame = new Frame();
		if (currentProcesses == null) {
			frame.setName(TraceSDPlugin.getResourceString("STR_NO_PROCESSES_HERE")); //$NON-NLS-1$
		} else {
			frame.setTimeUnitName(TraceSDPlugin.getResourceString("STR_TIME_UNIT_IS_SECOND")); //$NON-NLS-1$
		}
	}

	/**
	 * @param process
	 */
	abstract public void setFrameName();

	/**
	 *
	 */
	abstract public void fillPage(TraceInteractionUpdate update);
	
	protected HashMap traceThreadMap;
	protected ArrayList traceThreadList;

	/**
	 * @param eThread
	 * @return
	 */
	protected TraceCallStack getTraceCallStack(TRCThread eThread) {
		TraceCallStack ret = (TraceCallStack)traceThreadMap.get(eThread);
		if (ret == null) {
			ret = new TraceCallStack(this, eThread);
			traceThreadMap.put(eThread, ret);
			traceThreadList.add(ret);
		}
		return ret;
	}
	
	/**
	 * @param tcs
	 */
	public void sortTraceThreadList(TraceCallStack tcs) {
		traceThreadList.remove(tcs);
		for (int i = 0; i < traceThreadList.size(); i++) {
			TraceCallStack tcs2 = (TraceCallStack)traceThreadList.get(i);
			if (tcs.getCurrentExitTime() < tcs2.getCurrentExitTime()) {
				traceThreadList.add(i, tcs);
				return;
			}
		}
		traceThreadList.add(tcs);
	}
	
	/**
	 * @return
	 */
	protected double getTraceThreadListFirstTime() {
		double ret = (traceThreadList != null && traceThreadList.size() > 0) ?
				((TraceCallStack)traceThreadList.get(0)).getCurrentExitTime() :
				Double.MAX_VALUE;
		return ret;
	}

	/**
	 * Place to create the execution occurence for the lifeline
	 * @param current
	 */
	protected void setExecutionOccurence(TraceLifelineDraft draft) {
		if (draft != null) {
			BasicExecutionOccurrence occ = new BasicExecutionOccurrence();
			occ.setStartOccurrence(draft.getStart());
			occ.setEndOccurrence(draft.getEnd());
			draft.getLifeline().addExecution(occ);
		}
	}

	/**
	 * Get the lifeline to be used for this message
	 * @param mi
	 * @return
	 */
	protected abstract EObject getLifelineEObjectFromMethodInvocation(TRCFullMethodInvocation mi);

	/**
	 * Get the name of the class
	 * @param mi
	 * @return
	 */
	protected abstract String getLifeLineTitle(EObject to, boolean long_);

	/**
	 * Get the category of the lifeline corresponding to this message
	 * @param mi
	 * @return
	 */
	protected abstract int getLifeLineCategory(EObject to);

	
	/**
	 * Implementation of ISelectionListener
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
//System.err.println("selectionChanged("+part+")="+selection);
	    if( part instanceof PDProjectExplorer ) {
			if (!((PDProjectExplorer)part).isLinkingEnabled()) {
				return;
			}
			if (view != null &&
				view.getSDWidget() != null &&
				!view.getSDWidget().isDisposed()) {
				if (currentMofObjectFromPDProjectExplorer == HyadesUtil.getMofObject()) {
					return;
				}
				parseModelBackground();
			}
	    } else if (part instanceof SDView) {
    		//showInternalSelection(selection);
	    } else {
			if (frame == null) {
				return;
			}
			TraceInteractionUpdate update;
			update = new TraceInteractionUpdate();
    		externalSelectionChanged(part, selection, update);
    		if (update.needsUpdate()) {
        		updateSDBackground(update);
    		}
	    }
	}
	
	/**
	 * @param selection
	 */
//	private void showInternalSelection(Object selection) {
//System.err.println("showInternalSelection("+selection+")");
//		// Do nothing
//		if (selection instanceof IDateSelection) {
//			IDateSelection ds = (IDateSelection)selection;
//System.err.println(ds.getStartDate());
//		} else if (selection instanceof IStructuredSelection) {
//			IStructuredSelection iss = (IStructuredSelection)selection;
//			for (Iterator i = iss.iterator(); i.hasNext(); ) {
//				Object o = i.next();
//System.err.println("IStructuredSelection:showInternalSelection("+o+")");
//				showInternalSelection(o);
//			}
//		}
//	}

	/**
	 * @param selection
	 * @param update
	 */
	protected void externalEObjectSelectionChanged(IEObjectSelection selection,
												   TraceInteractionUpdate update) {
//System.err.println("External EObject selection is "+selection);
		IEObjectSelection ieos = (IEObjectSelection)selection;
		EObject eo = ieos.getEObject();
		if (eo instanceof TRCThread) {
			update.setThreadSelectionChanged(true);
			update.addSelectedThread((TRCThread)eo);
		}
	}
	
	/**
	 * @param part
	 * @param selection
	 * @param update
	 */
	protected void externalSelectionChanged(IWorkbenchPart part,
										    Object selection,
										    TraceInteractionUpdate update) {
//System.err.println("External selection from "+part+" is "+selection);
		if (selection instanceof IStructuredSelection) {
//System.err.println("IStructuredSelection");
			IStructuredSelection iss = (IStructuredSelection)selection;
			for (Iterator i = iss.iterator(); i.hasNext(); ) {
				Object o = i.next();
				externalSelectionChanged(part, o, update);
			}
    	} else if (externalExtendedSelectionChanged(part, selection, update)) {
    		// done by concrete class
    	} else if (selection instanceof IEObjectSelection) {
//System.err.println("IEObjectSelection");
    		    		externalEObjectSelectionChanged((IEObjectSelection)selection, update);
    	}
	}

	abstract protected boolean externalExtendedSelectionChanged(IWorkbenchPart part,
															    Object selection,
															    TraceInteractionUpdate update);

		/**
	 * Implementation of ISDGraphNodeSupporter
	 */
	public boolean isLifelineSupported() {
		return true;
	}
	public boolean isSyncMessageSupported() {
		return false;
	}
	public boolean isSyncMessageReturnSupported() {
		return false;
	}
	public boolean isAsyncMessageSupported() {
		return false;
	}
	public boolean isAsyncMessageReturnSupported() {
		return false;
	}
	public boolean isStopSupported() {
		return false;
	}
	
	/**
	 * implementation of ISDFindProvider
	 */
	public boolean find(Criteria toSearch) {
		if (frame == null) {
			return false;
		}
		ArrayList results = new ArrayList();
		// search in the current page:
		if (toSearch.isLifeLineSelected()) {
			for (int i = 0; i < frame.lifeLinesCount(); i++) {
				if (TraceInteractionUtils.matchCriteria(frame.getLifeline(i).getName(), toSearch)) {
					results.add(frame.getLifeline(i));
				}
			}
		}
		if (toSearch.isSyncMessageSelected()) {
			for (int i = 0; i < frame.syncMessageCount(); i++) {
				if (TraceInteractionUtils.matchCriteria(frame.getSyncMessage(i).getName(), toSearch)) {
					results.add(frame.getSyncMessage(i));
				}
			}
		}
		view.getSDWidget().clearSelection();
		ensureSelectedAndMoveToFirst(results);
		if (results.size() > 0) {
			return true;
		}
		if (this instanceof ISDPagingProvider) {
			// looking in other pages
			TraceInteractionUpdate update = new TraceInteractionUpdate();
			update.setFindRequired(true);
			update.setFindCriteria(toSearch);
			Frame oldFrame = frame;
			updateSDBackground(update);
			boolean ret = update.getFindResults() != null && update.getFindResults().size() > 0;
			if (! ret) {
				frame = oldFrame;
			}
			return ret;
		}
		return false;
	}

}
