/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCTraceObject;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.FoundInPage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCallStack;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsableSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsableSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedLifelines;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUtils;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifeline;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifelineDraft;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceMessageAndCallAssociation;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TracePage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcess;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.preferences.ITraceInteractionPreferenceListener;
import org.eclipse.hyades.uml2sd.trace.selection.IDateSelection;
import org.eclipse.hyades.uml2sd.trace.selection.IEObjectSelection;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDAdvancedPagingProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDInternalMesFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.FilterCriteria;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.core.LifelineCategories;
import org.eclipse.hyades.uml2sd.ui.drawings.impl.ImageImpl;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPart;

/**
 * The abstract trace loader.
 * It is based on the parsing of MethodInvocations from the model.
 * Lifelines attribution is made by the concrete classes.
 */
public abstract class TraceInteractions extends BaseTraceInteractions
	implements ISDFilterProvider, ISDAdvancedPagingProvider,
			   ISDInternalMesFilterProvider, ITraceInteractionPreferenceListener {

	/**
	 * Categories:
	 * - first one for classes
	 * - second one for instances of classes
	 * - third one for collapsed classes and/or objects
	 */
	public static final int CATEGORY_CLASSES = 0;
	public static final int CATEGORY_OBJECTS = 1;
	public static final int CATEGORY_COLLAPSED = 2;
	private static LifelineCategories[] traceCategories;
	static {
		traceCategories = new LifelineCategories[3];
		traceCategories[CATEGORY_CLASSES] = new LifelineCategories();
		traceCategories[CATEGORY_CLASSES].setName(TraceSDPlugin.getResourceString("STR_CLASS")); //$NON-NLS-1$
		traceCategories[CATEGORY_CLASSES].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("class.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_OBJECTS] = new LifelineCategories();
		traceCategories[CATEGORY_OBJECTS].setName(TraceSDPlugin.getResourceString("STR_OBJECT")); //$NON-NLS-1$
		traceCategories[CATEGORY_OBJECTS].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("classobject_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_COLLAPSED] = new LifelineCategories();
		traceCategories[CATEGORY_COLLAPSED].setName(TraceSDPlugin.getResourceString("STR_COLLAPSED_LIFELINE")); //$NON-NLS-1$
		traceCategories[CATEGORY_COLLAPSED].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("collapsed_obj.gif"))); //$NON-NLS-1$
	}
	
	/**
	 * Internal message filtering:
	 * internal means that the message destination is the same as the origin (same lifeline)
	 * see also ISDInternalMesFilterProvider
	 */
	private boolean internalMessagesFiltered = false;
	
	/**
	 * Paging related attributes
	 */
	private int maximumMessagesByPage = (int)TraceSDPlugin.getDefault().getPreferenceStore().
		getLong(ITraceInteractionPreferenceListener.PAGE_SIZE);
	private int nbMessages, nbMessagesInThisPage, pageNumber = 0;
	private boolean hasNextPage = false;

	/**
	 * Implementation of IUml2SDLoader
	 */
	public void onSetViewer() {
		view.setSDFilterProvider(this);
		view.setSDPagingProvider(this);
		view.setInternalMessageFilterProvider(this);
	}
	
	/**
	 * 
	 */
	public void clearInternals() {
		super.clearInternals();
		pageNumber = 0;
		tracePages = null;
		collapsedLifelines = null;
		collapsedMessages = null;
		currentFilters = null;
	}

	/**
	 * Parse the model again for a new translation on the same eObject (page changes, ...)
	 * @param update
	 */
	protected synchronized void updateSD(TraceInteractionUpdate update) {
		// Prepare for showing same date after update
		double visibleDate = 0;
		if (frame != null &&
			!update.isDateSelectionChanged() && // This test avoid infinite recursivity
			!update.isPageChanged() &&
			!update.isFindRequired()) {
			currentProcesses.init();
			TraceSyncMessage firstSM = (TraceSyncMessage)(frame.getSyncMessage(frame.getFirstVisibleSyncMessage()));
			if (firstSM != null) {
				visibleDate = firstSM.getStartDate() + currentProcesses.getStartTime();
			} else {
				TraceSyncMessageReturn firstSMR = (TraceSyncMessageReturn)(frame.getSyncMessageReturn(frame.getFirstVisibleSyncMessageReturn()));
				if (firstSMR != null) {
					visibleDate = firstSMR.getStartDate() + currentProcesses.getStartTime();
				}
			}
		}
//System.err.println("visibleDate="+visibleDate+" "+update.isPageChanged()+" "+update.isDateSelectionChanged());
		createFrame();
		if (currentProcesses == null) {
			return;
		}
		if ((!update.isPageChanged()) &&
			(update.isInternalMessageFilteringChanged() ||
			 update.isMaxMessagesByPageChanged() ||
			 update.isFilteringChanged() ||
			 update.isHorizontalCollapsingChanged() ||
			 update.isVerticalCollapsingChanged())) {
			computeModel();
		}
		try {
			fillPage(update);
		} catch (FoundInPage f) {
//f.printStackTrace();
			if (update.isDateSelectionChanged()) {
				update.setPageChanged(true);
				pageNumber = f.getPage();
				updateSD(update);
				return;
			} else if (update.isFindRequired()) {
				update.setPageChanged(true);
				pageNumber = f.getPage();
				updateSD(update);
				ensureSelectedAndMoveToFirst(update.getFindResults());
				return;
			}
		} catch (Throwable e) {
			e.printStackTrace();
		}
		if (update.getVisibleGraphNode() != null) {
			if (view != null && !view.getSDWidget().isDisposed()) {
				setFrameName();
				//view.setFrameSync(frame);
				view.setFrameAndEnsureVisibleSync(frame, update.getVisibleGraphNode());
				return;
			}
		}
		if (visibleDate != 0) {
			TraceInteractionUpdate updateDate = new TraceInteractionUpdate();
			updateDate.setDateSelectionChanged(true);
			updateDate.setDateToShow(visibleDate);
			updateSD(updateDate); // recursive call
			return;
		}
		if (view != null && !view.getSDWidget().isDisposed() && !monitor.isCanceled()) {
			setFrameName();
			view.setFrameSync(frame);
		}
	}
	
	/**
	 * @return
	 */
	protected void createFrame() {
		super.createFrame();
		frame.setLifelineCategories(traceCategories);
	}

	/**
	 * @param process
	 */
	public void setFrameName() {
		if (frame == null) {
			return;
		}
		if (currentProcesses != null && currentProcesses.size() == 1) {
			TRCProcess process = ((TraceProcess)currentProcesses.get(0)).getProcess();
			frame.setName(process.getAgent().getAgentProxy().getProcessProxy().getName()+
					  " [Pid "+process.getPid()+"]"+pageMessage()); //$NON-NLS-1$ //$NON-NLS-2$
		} else {
			frame.setName(TraceSDPlugin.getResourceString("STR_SEVERAL_PROCESSES")+pageMessage()); //$NON-NLS-1$
		}
//System.err.println("setFrameName() "+frame.getName());
	}

	/**
	 * @return
	 */
	private String pageMessage() {
		String m = ""; //$NON-NLS-1$
		if (pages == 1) {
			return m;
		}
		m = " [" + (1+pageNumber) + "/" + pages + " " + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			TraceSDPlugin.getResourceString("STR_PAGES") + //$NON-NLS-1$
			"]"; //$NON-NLS-1$
		return m;
	}
	
	private int totalNbMessages;
	private int pages;
	private ArrayList tracePages;
	
	/**
	 * Compute the number of pages with and also without internal messages
	 * @param invocations
	 */
	protected void computeModel() {
//System.err.println("computeModel() "+maximumMessagesByPage);
		totalNbMessages = 0;
		instancesMap = new HashMap();
		currentProcesses.init();
		for (TRCFullMethodInvocation methodInvocation = currentProcesses.consumeMethodInvocation();
		 	 methodInvocation != null &&
			 !monitor.isCanceled();
		 	 methodInvocation = currentProcesses.consumeMethodInvocation()) {
			TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
			if (isMessageCollapsed(invokedBy)) {
				continue;
			}
			EObject instanceFrom = null;
			EObject instanceTo = null;
			if (invokedBy != null) {
				instanceFrom = getLifelineEObjectFromMethodInvocation(invokedBy);
				instanceTo = getLifelineEObjectFromMethodInvocation(methodInvocation);
			}
			if (internalMessagesFiltered &&	instanceFrom == instanceTo) {
				continue;
			}
			TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(null, methodInvocation);
			TraceLifelineDraft traceLifelineFrom = null;
			if (invokedBy != null) {
				traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(null, invokedBy);
			}
			if (traceLifelineTo.isFiltered() ||
				(traceLifelineFrom != null &&
				 traceLifelineFrom.isFiltered())) {
				continue;
			}
			String name = methodInvocation.getMethod().getName();
			if (isSyncMessageFiltered(name)) {
				collapseCalledMessage(methodInvocation, instancesMap);
				continue;
			}
			totalNbMessages += 2; // Count one way and return
		}
		pages = totalNbMessages / maximumMessagesByPage;
		if (totalNbMessages % maximumMessagesByPage > 0) {
			pages++;
		}
		cleanCollapsedMessage(instancesMap);
		instancesMap = null;
//System.err.println("computeModel() interrupted("+monitor.isCanceled()+") has counted "+pages+" pages");
	}
	
	private HashMap instancesMap;
	private double lastValidTime;
	private int currentMethodInvocationIndex;
	
	/**
	 * @param update
	 */
	public void fillPage(TraceInteractionUpdate update) {
		tracePages = new ArrayList();
		instancesMap = new HashMap();
		traceThreadMap = new HashMap();
		traceThreadList = new ArrayList();
		lastValidTime = 0.0;
		nbMessages = 0;
		nbMessagesInThisPage = 0;
		hasNextPage = false;
		currentMethodInvocationIndex = 0;
		currentProcesses.init(); // init again
		for (TRCFullMethodInvocation methodInvocation = currentProcesses.consumeMethodInvocation();
	 	 	 methodInvocation != null &&
			 !monitor.isCanceled() &&
			 nbMessagesInThisPage < maximumMessagesByPage;
	 	 	 methodInvocation = currentProcesses.consumeMethodInvocation()) {
//System.err.println(nbMessages+":"+nbMessages/maximumMessagesByPage+" "+methodInvocation);			

			// Let's continue with pending message returns
			finishPendingReturns(update, TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation));
			
			TraceSyncMessage syncMessage = createSyncMessage(methodInvocation, update);
			
			TraceCallStack callStack = getTraceCallStack(methodInvocation.getThread());
			callStack.pushCall(syncMessage, methodInvocation);
		}
		// Let's continue with pending message returns
		finishPendingReturns(update, Double.MAX_VALUE);

		// Let's do last execution occurrences
		Collection c = instancesMap.entrySet();
		for (Iterator i = c.iterator(); i.hasNext(); ) {
			Map.Entry e = (Map.Entry)i.next();
			TraceLifelineDraft draft = (TraceLifelineDraft)e.getValue();
//System.err.println(draft.getLifeline()+":"+draft.getNbUser());
			if (draft.getNbUser() > 0) {
				draft.setEnd(nbMessagesInThisPage+1);
				if (draft.getStart() == Integer.MAX_VALUE) {
					draft.setStart(0);
				}
				setExecutionOccurence(draft);
			}
		}
		instancesMap = null;
		traceThreadMap = null;
		traceThreadList = null;
		cleanCollapsedMessage(frame);
	}
	
	/**
	 * @param update
	 * @param uptodate
	 */
	protected void finishPendingReturns(TraceInteractionUpdate update,
									    double uptodate) {
		while (getTraceThreadListFirstTime() < uptodate) {
			TraceCallStack callStack = (TraceCallStack)traceThreadList.get(0);
			TraceMessageAndCallAssociation tmaca = callStack.popCall();
			TraceSyncMessageReturn syncMessageReturn = createSyncMessageReturn(tmaca.getMethodInvocation(), update);
			if (tmaca.getMessage() != null && syncMessageReturn != null) {
				syncMessageReturn.setMessage(tmaca.getMessage());
			}
		}
	}

	/**
	 * @param methodInvocation
	 * @param update
	 */
	private TraceSyncMessage
			     createSyncMessage(TRCFullMethodInvocation methodInvocation,
								   TraceInteractionUpdate update) {
		TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
		if (isMessageCollapsed(invokedBy)) {
			return null;
		}
		TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(frame, methodInvocation);
		TraceLifelineDraft traceLifelineFrom = null;
		if (invokedBy != null) {
			traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(frame, invokedBy);
		}
		if (traceLifelineTo.isFiltered() ||
			(traceLifelineFrom != null &&
			 traceLifelineFrom.isFiltered())) {
			return null;
		}
		if (internalMessagesFiltered && traceLifelineFrom == traceLifelineTo) {
			return null;
		}
		String name = methodInvocation.getMethod().getName();
		if (isSyncMessageFiltered(name)) {
			collapseCalledMessage(methodInvocation, frame);
			return null;
		}
		nbMessages++;
//System.err.println("in:"+nbMessages+":"+nbMessages/maximumMessagesByPage+" "+name+" "+TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation));
		if (nbMessagesInThisPage == maximumMessagesByPage) {
			hasNextPage = true;
			return null;
		}
		traceLifelineTo.addAUser();
		if ((nbMessages-1) % maximumMessagesByPage == 0) {
			tracePages.add(new TracePage(TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation)));
		} else if ((nbMessages-1) % maximumMessagesByPage == maximumMessagesByPage-1) {
			((TracePage)tracePages.get(tracePages.size()-1)).setLastTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		}
		if (update.isPageChanged() &&
			(nbMessages-1) < pageNumber*maximumMessagesByPage) {
			return null;
		}
		if (!update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation) < update.getDateToShow()) {
					return null;
				} else {
					throw new FoundInPage(tracePages.size()-1);
				}
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					throw new FoundInPage(tracePages.size()-1);
				} else {
					return null;
				}
			}
		}
		if (traceLifelineTo.getNbUser() == 1) {
			traceLifelineTo.setStart(nbMessagesInThisPage+1);
		}
		TraceSyncMessage syncMessage;
		String collapsedMark = ""; //$NON-NLS-1$
		if (isMessageCollapsed(methodInvocation)) {
			collapsedMark = "[+] "; //$NON-NLS-1$
			syncMessage = new TraceCollapsedSyncMessage();
		} else if (methodInvocation.getInvokes() != null &&
				   methodInvocation.getInvokes().size() > 0) {
			syncMessage = new TraceCollapsableSyncMessage();
		} else {
			syncMessage = new TraceSyncMessage();
		}
		if (update.isPageChanged()) {
			if (update.isDateSelectionChanged() &&
				TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation) >= update.getDateToShow() &&
				update.getVisibleGraphNode() == null) {
				update.setVisibleGraphNode(syncMessage);
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					update.addFindResult(syncMessage);
				}
			}
		}
		syncMessage.model = methodInvocation;
		if (traceLifelineFrom != null) {
			traceLifelineFrom.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessage.setStartLifeline(traceLifelineFrom.getLifeline());
			syncMessage.setName(collapsedMark+name.intern());
		} else { 
			traceLifelineTo.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessage.setName(collapsedMark+name+" ("+methodInvocation.getThread().getName()+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		syncMessage.setEndLifeline(traceLifelineTo.getLifeline());
		syncMessage.setTime(TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation));
		lastValidTime = TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation);
		if (monitor.isCanceled()) {
			return null;
		}
		frame.addMessage(syncMessage);
		nbMessagesInThisPage++;
		return syncMessage;
	}
	
	/**
	 * @param methodInvocation
	 * @param update
	 */
	private TraceSyncMessageReturn
				 createSyncMessageReturn(TRCFullMethodInvocation methodInvocation,
										 TraceInteractionUpdate update) {
		TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
		if (isMessageCollapsed(invokedBy)) {
			return null;
		}
		TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(frame, methodInvocation);
		TraceLifelineDraft traceLifelineFrom = null;
		if (invokedBy != null) {
			traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(frame, invokedBy);
		}
		if (traceLifelineTo.isFiltered() ||
			(traceLifelineFrom != null &&
			 traceLifelineFrom.isFiltered())) {
			return null;
		}
		if (internalMessagesFiltered && traceLifelineFrom == traceLifelineTo) {
			return null;
		}
		String name = methodInvocation.getMethod().getName();
		if (isSyncMessageFiltered(name)) {
			collapseCalledMessage(methodInvocation, frame);
			return null;
		}
		nbMessages++;
//System.err.println("out:"+nbMessages+":"+nbMessages/maximumMessagesByPage+" "+name+" "+TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		if (nbMessagesInThisPage == maximumMessagesByPage) {
			hasNextPage = true;
			return null;
		}
		traceLifelineTo.removeAUser();
		if ((nbMessages-1) % maximumMessagesByPage == 0) {
			tracePages.add(new TracePage(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation)));
		} else if ((nbMessages-1) % maximumMessagesByPage == maximumMessagesByPage-1) {
			((TracePage)tracePages.get(tracePages.size()-1)).setLastTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		}
		if (update.isPageChanged() &&
			(nbMessages-1) < pageNumber*maximumMessagesByPage) {
			return null;
		}
		if (!update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteExitTime(methodInvocation) < update.getDateToShow()) {
					return null;
				} else {
					throw new FoundInPage(tracePages.size()-1);
				}
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageReturnSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					throw new FoundInPage(tracePages.size()-1);
				} else {
					return null;
				}
			}
		}
		if (traceLifelineTo.getNbUser() == 0) {
			traceLifelineTo.setEnd(nbMessagesInThisPage+1);
			if (traceLifelineTo.getStart() == Integer.MAX_VALUE) {
				traceLifelineTo.setStart(0);
			}
			setExecutionOccurence(traceLifelineTo);
		}
		TraceSyncMessageReturn syncMessageReturn;
		String collapsedMark = ""; //$NON-NLS-1$
		if (isMessageCollapsed(methodInvocation)) {
			collapsedMark = "[+] "; //$NON-NLS-1$
			syncMessageReturn = new TraceCollapsedSyncMessageReturn();
		} else if (methodInvocation.getInvokes() != null &&
				   methodInvocation.getInvokes().size() > 0) {
			syncMessageReturn = new TraceCollapsableSyncMessageReturn();
		} else {
			syncMessageReturn = new TraceSyncMessageReturn();
		}
		if (update.isPageChanged()) {
			if (update.isDateSelectionChanged() &&
				TraceInteractionUtils.getAbsoluteExitTime(methodInvocation) >= update.getDateToShow() &&
				update.getVisibleGraphNode() == null) {
				update.setVisibleGraphNode(syncMessageReturn);
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageReturnSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					update.addFindResult(syncMessageReturn);
				}
			}
		}
		syncMessageReturn.model = methodInvocation;
		if (traceLifelineFrom != null) {
			traceLifelineFrom.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessageReturn.setEndLifeline(traceLifelineFrom.getLifeline());
			syncMessageReturn.setName(collapsedMark+name.intern());
		} else { 
			traceLifelineTo.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessageReturn.setName(collapsedMark+name+" ("+methodInvocation.getThread().getName()+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		syncMessageReturn.setStartLifeline(traceLifelineTo.getLifeline());
		// Sometimes, getExitTime() is 0.0
		if (methodInvocation.getExitTime() > 0.0) {
			syncMessageReturn.setTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
			lastValidTime = TraceInteractionUtils.getAbsoluteExitTime(methodInvocation);
		} else {
			syncMessageReturn.setTime(lastValidTime);
		}
		if (monitor.isCanceled()) {
			return null;
		}
		frame.addMessage(syncMessageReturn);
		nbMessagesInThisPage++;
		return syncMessageReturn;
	}
	
	/**
	 * Get the instance corresponding to this method invocation or create it
	 * @param f
	 * @param mi
	 * @return
	 */
	protected TraceLifelineDraft 
				getTraceLifelineDraftForMethodInvocation
					(Frame f, TRCFullMethodInvocation mi) {
		EObject obj = getLifelineEObjectFromMethodInvocation(mi);
		TRCTraceObject o = mi.getOwningObject();
		TraceLifelineDraft draft = (TraceLifelineDraft)instancesMap.get(obj);
		if (draft == null) {
			draft = new TraceLifelineDraft();
			String name = getLifeLineTitle(obj, true);
			if (isLifelineFiltered(name)) {
				draft.setFiltered(true);
				return draft;
			}
			ArrayList list = collapsedLifelines != null ?
							 (ArrayList)collapsedLifelines.get(obj) : 
							 null;
			if (list != null) {
				for (ArrayList l = (ArrayList)collapsedLifelines.get(list);
					 l != null;
					 list = l, l = (ArrayList)collapsedLifelines.get(l));
				name = getCollapsedLifelineName(list, ""); //$NON-NLS-1$
				if (name.length() > 80) {
					name = name.substring(0, 77)+"..."; //$NON-NLS-1$
				}
				if (isLifelineFiltered(name)) {
					draft.setFiltered(true);
					return draft;
				}
				if (f == null) {
					return draft;
				}
				TraceCollapsedLifelines traceCollapsedLifelines = new TraceCollapsedLifelines();
				draft.setLifeline(traceCollapsedLifelines);
				traceCollapsedLifelines.setCollapsedLifelines(list);
				traceCollapsedLifelines.setName(name);
				putInstancesMap(list, draft);
				traceCollapsedLifelines.setCategory(CATEGORY_COLLAPSED);
			} else {
				if (f == null) {
					return draft;
				}
				TraceLifeline traceLifeline = new TraceLifeline();
				draft.setLifeline(traceLifeline);
				traceLifeline.setEObject(obj);
				instancesMap.put(obj, draft);
				if (o.getIsA() != null) {
					traceLifeline.setName(name);
					traceLifeline.setCategory(getLifeLineCategory(obj));
				}
			}
			f.addLifeLine(draft.getLifeline());
			draft.setStart(Integer.MAX_VALUE);
			draft.setEnd(0);
		}
		return draft;
	}
	
	/**
	 * @param draft
	 * @param list
	 */
	private void putInstancesMap(ArrayList list, TraceLifelineDraft draft) {
		for (Iterator i = list.iterator(); i.hasNext(); ) {
			Object oo = (Object)i.next();
			if (oo instanceof EObject) {
				instancesMap.put(oo, draft);
			} else if (oo instanceof ArrayList) {
				putInstancesMap((ArrayList)oo, draft);
			} // else ignored
		}
	}

	/**
	 * @param draft
	 * @param list
	 */
	private String getCollapsedLifelineName(ArrayList list, String indent) {
		String ret = "["; //$NON-NLS-1$
		int j = 0;
		for (Iterator i = list.iterator(); i.hasNext(); j++) {
			Object oo = (Object)i.next();
			if (oo instanceof EObject) {
				if (j > 0) {
					ret += ","+indent; //$NON-NLS-1$
				}
				ret += getLifeLineTitle((EObject)oo, false);
			} else if (oo instanceof ArrayList) {
				if (j > 0) {
					ret += ","+indent; //$NON-NLS-1$
				}
				ret += getCollapsedLifelineName((ArrayList)oo, indent+" "); //$NON-NLS-1$
			} // else ignored
		}
		return ret + "]"; //$NON-NLS-1$
	}

	
	/**
	 * @param part
	 * @param selection
	 */
	private void externalDateSelectionChanged(IDateSelection selection,
											  TraceInteractionUpdate update) {
		if (frame == null) {
			return;
		}
//System.err.println("External Date selection from "+part+" is "+selection);
		IDateSelection ids = (IDateSelection)selection;
		EObject eo = selection.getEObject();
		if (!(eo instanceof TRCProcess) || currentProcesses == null || !currentProcesses.contains((TRCProcess)eo)) {
			return;
		}
		double dateSelection = ids.getStartDate();
		TRCProcess process = (TRCProcess)eo;
		
		// Is it in the current page? (this question is answered with date relative to the process)
		if (ids.getMeaning() == IDateSelection.NOW) {
			if (frame.syncMessageCount() > 0) {
				TraceSyncMessage first = (TraceSyncMessage)frame.getSyncMessage(0);
				TraceSyncMessage last = (TraceSyncMessage)frame.getSyncMessage(frame.syncMessageCount()-1);
				if (first.getStartDate() <= dateSelection &&
					last.getStartDate() >= dateSelection) {
					// This date can be found in the current page
					for (int i = 0; i < frame.syncMessageCount(); i++) {
						TraceSyncMessage sm = (TraceSyncMessage)frame.getSyncMessage(i);
						if (sm.getStartDate() >= ids.getStartDate()) {
							view.getSDWidget().ensureVisible(sm);
							return;
						}
					}
				}
			}
			if (frame.syncMessageReturnCount() > 0) {
				TraceSyncMessageReturn first = (TraceSyncMessageReturn)frame.getSyncMessageReturn(0);
				TraceSyncMessageReturn last = (TraceSyncMessageReturn)frame.getSyncMessageReturn(frame.syncMessageReturnCount()-1);
				if (first.getStartDate() <= dateSelection &&
					last.getStartDate() >= dateSelection) {
					// This date can be found in the current page
					for (int i = 0; i < frame.syncMessageReturnCount(); i++) {
						TraceSyncMessageReturn sm = (TraceSyncMessageReturn)frame.getSyncMessageReturn(i);
						if (sm.getStartDate() >= ids.getStartDate()) {
							view.getSDWidget().ensureVisible(sm);
							return;
						}
					}
				}
			}
			// Look in previous pages
			if (tracePages != null &&
				tracePages.size() > 0 &&
				dateSelection <= ((TracePage)tracePages.get(tracePages.size()-1)).getLastTime()) {
				for (int i = 0; i < tracePages.size(); i++) {
					TracePage tp = (TracePage)tracePages.get(i);
					if (dateSelection >= tp.getFirstTime() &&
						dateSelection <= tp.getLastTime()) {
						update.setDateSelectionChanged(true);
						update.setDateToShow(dateSelection);
						update.setPageChanged(true);
						pageNumber = i;
						return;
					}
				}
			}
			if (dateSelection < process.getAgent().getStartTime()+process.getStartTime()) {
				// determine if we received an absolute or a process relative date
				dateSelection += process.getAgent().getStartTime()+process.getStartTime();
			}
			//System.err.println("dateSelection="+dateSelection+" process.getAgent().getStartTime()+process.getBaseTime()="+(process.getAgent().getStartTime()+process.getBaseTime()));
			// Worst case: must look in all pages for an absolute date (no longer relative)
			update.setDateSelectionChanged(true);
			update.setDateToShow(dateSelection);
		}
	}

	protected Timer dateSelectionTimer;

	/**
	 * @param selection
	 */
	private void subscribeForDateSelectionChanged(final IDateSelection selection) {
		if (dateSelectionTimer != null) {
			dateSelectionTimer.cancel();
		}
		dateSelectionTimer = new Timer();
		dateSelectionTimer.schedule(new TimerTask() {
			public void run() {
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						TraceInteractionUpdate update = new TraceInteractionUpdate();
						externalDateSelectionChanged(selection, update);
			    		if (update.needsUpdate()) {
			        		updateSDBackground(update);
			    		}
					}
				});
				dateSelectionTimer = null;
			}
		}, 50);
	}

	/**
	 * @param part
	 * @param selection
	 * @param update
	 */
	protected boolean externalExtendedSelectionChanged(IWorkbenchPart part,
										    		   Object selection,
													   TraceInteractionUpdate update) {
//System.err.println("TraceInteraction.External extended selection from "+part+" is "+selection);
		if (selection instanceof IDateSelection) {
//System.err.println("TraceInteraction.IDateSelection: "+((IDateSelection)selection).getStartDate());
    		subscribeForDateSelectionChanged((IDateSelection)selection);
    		return true;
    	}
		return false;
	}


	/**
	 * Implementation of ISDGraphNodeSupporter
	 */
	public boolean isSyncMessageSupported() {
		return true;
	}
	
	/**
	 * implementation of ISDPagingProvider
	 */
	public boolean hasNextPage() {
		return hasNextPage;
	}
	public boolean hasPrevPage() {
		return pageNumber > 0;
	}
	public void nextPage() {
		pageNumber++;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(true);
		updateSDBackground(update);
	}
	public void prevPage() {
		pageNumber--;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * Implementation of ISDAdvancedPagingProvider
	 */
	public int maxItemsByPageCount() {
		return maximumMessagesByPage;
	}
	public int itemsTotalCount() {
		return totalNbMessages;
	}
	public String noItemsText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_NO_ITEMS"); //$NON-NLS-1$
	}
	public String oneItemText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_ONE_ITEM"); //$NON-NLS-1$
	}
	public String itemsText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_SEVERAL_ITEMS"); //$NON-NLS-1$
	}
	public int currentPage() {
		return pageNumber;
	}
	public int pagesCount() {
		return pages;
	}
	public void pageSettingsChanged(int maxItemsByPage_, int pageNumber_) {
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setMaxMessagesByPageChanged(maximumMessagesByPage != maxItemsByPage_);
		maximumMessagesByPage = maxItemsByPage_;
		update.setPageChanged(pageNumber != pageNumber_);
		pageNumber = pageNumber_;
		if (update.isMaxMessagesByPageChanged() ||
			update.isPageChanged()) {
			updateSDBackground(update);
		}
	}
	public void pageNumberChanged(int pageNumber_) {
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(pageNumber != pageNumber_);
		pageNumber = pageNumber_;
		if (update.isPageChanged()) {
			updateSDBackground(update);
		}
	}

	/**
	 * Implementation of ISDInternalMesFilterProvider
	 */
	public void setFiltered(boolean value) {
		if (internalMessagesFiltered != value) {
			internalMessagesFiltered = value;
			TraceInteractionUpdate update = new TraceInteractionUpdate();
			update.setInternalMessageFilteringChanged(true);
			updateSDBackground(update);
		}
	}
	
	private ArrayList currentFilters;

	public ArrayList getCurrentFilters() {
		return currentFilters;
	}
	
	private boolean isLifelineFiltered(String name) {
		if (currentFilters != null) {
			for (Iterator i = currentFilters.iterator(); i.hasNext(); ) {
				FilterCriteria tf = (FilterCriteria)i.next();
				if (tf.isActive() && tf.getCriteria().isLifeLineSelected()) {
					if (TraceInteractionUtils.matchCriteria(name, tf.getCriteria())) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	private boolean isSyncMessageFiltered(String name) {
		if (currentFilters != null) {
			for (Iterator i = currentFilters.iterator(); i.hasNext(); ) {
				FilterCriteria tf = (FilterCriteria)i.next();
				if (tf.isActive() && tf.getCriteria().isSyncMessageSelected()) {
					if (TraceInteractionUtils.matchCriteria(name, tf.getCriteria())) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	/**
	 * 
	 */
	public void filterSelectedGraphNodes(ArrayList graphNodes) {
//System.err.println("filterGraphNodes("+graphNodes.size()+" selected graph nodes)");
		if (currentFilters == null) {
			currentFilters = new ArrayList();
		}
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			Criteria c = new Criteria();
			if (gn instanceof IEObjectSelection) {
				EObject eo = (EObject)((IEObjectSelection)gn).getEObject();
				if (eo instanceof TRCFullMethodInvocation) {
					c.setExpression(((TRCFullMethodInvocation)eo).getMethod().getName());
				}
			}
			c.setCaseSenstiveSelected(true);
			if (gn instanceof TraceLifeline) {
				c.setExpression(gn.getName());
				c.setLifeLineSelected(true);
			} else if (gn instanceof TraceSyncMessage) {
				c.setExpression(((TRCFullMethodInvocation)((TraceSyncMessage)gn).getEObject()).getMethod().getName());
				c.setSyncMessageSelected(true);
			} else if (gn instanceof TraceSyncMessageReturn) {
				c.setExpression(((TRCFullMethodInvocation)((TraceSyncMessageReturn)gn).getEObject()).getMethod().getName());
				c.setSyncMessageSelected(true);
			}
			currentFilters.add(new FilterCriteria(c, true));
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setFilteringChanged(true);
		updateSDBackground(update);
	}

	/**
	 * Come back from the filters dialog
	 */
	public boolean filter(ArrayList filters) {
//System.err.println("filter("+filters.size()+" filters)");
		currentFilters = filters;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setFilteringChanged(true);
		updateSDBackground(update);
		return true;
	}

	private HashMap collapsedLifelines;

	/**
	 * 
	 */
	public void collapseLifelines(ArrayList graphNodes) {
//System.err.println("collapseLifelines("+graphNodes.size()+")");
		if (collapsedLifelines == null) {
			collapsedLifelines = new HashMap();
		}
		ArrayList list = new ArrayList();
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			Object key = null;
			if (gn instanceof TraceCollapsedLifelines) {
				key = ((TraceCollapsedLifelines)gn).getCollapsedLifelines();
			} else if (gn instanceof TraceLifeline) {
				key = ((TraceLifeline)gn).getEObject();
			} else {
				continue; // Error ignored
			}
			list.add(key);
			collapsedLifelines.put(key, list);
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setHorizontalCollapsingChanged(true);
		updateSDBackground(update);
	}

	/**
	 * 
	 */
	public void expandLifelines(ArrayList graphNodes) {
//System.err.println("expandLifelines("+graphNodes.size()+")");
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			if (gn instanceof TraceCollapsedLifelines) {
				ArrayList sublist = ((TraceCollapsedLifelines)gn).getCollapsedLifelines();
				for (Iterator j = sublist.iterator(); j.hasNext(); ) {
					Object key = j.next();
					collapsedLifelines.put(key, null);
				}
			} else  {
				continue; // Error ignored
			}
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setHorizontalCollapsingChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * Map of collapsed messages: vertical collapsing
	 */
	private HashMap collapsedMessages;

	/**
	 * @param methodInvocation
	 * @return
	 */
	private boolean isMessageCollapsed(TRCFullMethodInvocation methodInvocation) {
		if (methodInvocation == null || collapsedMessages == null) {
			return false;
		}
		if (collapsedMessages.get(methodInvocation) != null) {
			return true;
		}
		for (TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
			 invokedBy != null;
			 invokedBy = (TRCFullMethodInvocation)invokedBy.getInvokedBy()) {
			if (collapsedMessages.get(invokedBy) != null) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 
	 */
	public void collapseCalledMessages(ArrayList graphNodes) {
//System.err.println("collapseCalledMessages("+graphNodes.size()+")");
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			Object gn = i.next();
			TRCFullMethodInvocation methodInvocation = null;
			if (gn instanceof TraceSyncMessage) {
				TraceSyncMessage sm = (TraceSyncMessage)gn;
				methodInvocation = (TRCFullMethodInvocation)sm.getEObject();
			} else if (gn instanceof TraceSyncMessageReturn) {
				TraceSyncMessageReturn smr = (TraceSyncMessageReturn)gn;
				methodInvocation = (TRCFullMethodInvocation)smr.getEObject();
			} // else ignored
			collapseCalledMessage(methodInvocation, currentMofObjectFromPDProjectExplorer);
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setVerticalCollapsingChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * @param methodInvocation
	 */
	private void collapseCalledMessage(TRCFullMethodInvocation methodInvocation,
									   Object marker) {
		if (methodInvocation != null) {
			if (collapsedMessages == null) {
				collapsedMessages = new HashMap();
			}
			collapsedMessages.put(methodInvocation, marker);
		}
	}

	/**
	 * Remove filtered messages from the collapsedMessages map.<br>
	 * Their marker is a given as parameter.
	 */
	private void cleanCollapsedMessage(Object marker) {
		if (collapsedMessages == null) {
			return;
		}
		Set s = collapsedMessages.entrySet();
		for (Iterator i = s.iterator(); i.hasNext(); ) {
			Map.Entry e = (Map.Entry)i.next();
			if (e.getValue() == marker) {
				i.remove();
			}
		}
	}

	/**
	 * 
	 */
	public void expandCalledMessages(ArrayList graphNodes) {
//System.err.println("expandCalledMessages("+graphNodes.size()+")");
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			Object gn = i.next();
			TRCFullMethodInvocation methodInvocation = null;
			if (gn instanceof TraceSyncMessage) {
				TraceSyncMessage sm = (TraceSyncMessage)gn;
				methodInvocation = (TRCFullMethodInvocation)sm.getEObject();
			} else if (gn instanceof TraceSyncMessageReturn) {
				TraceSyncMessageReturn smr = (TraceSyncMessageReturn)gn;
				methodInvocation = (TRCFullMethodInvocation)smr.getEObject();
			} // else ignored
			if (methodInvocation != null) {
				collapsedMessages.put(methodInvocation, null);
			}
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setVerticalCollapsingChanged(true);
		updateSDBackground(update);
	}

	/**
	 * Implementation of ITraceInteractionPreferenceListener
	 * @param event
	 */
	public void applyPreferences() {
		int max = (int)TraceSDPlugin.getDefault().getPreferenceStore().
							getLong(ITraceInteractionPreferenceListener.PAGE_SIZE);
		if (maximumMessagesByPage == max) {
			return;
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setMaxMessagesByPageChanged(true);
		maximumMessagesByPage = max;
		updateSDBackground(update);
	}

}
