/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.testgen.ui;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.hyades.execution.testgen.TestgenPlugin;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ISetSelectionTarget;

/**
 * @author ejessee
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class TestgenUIUtility
{
	
	/**
		 * exception thrown by the RecorderHarnessUIUtility class when problems occur while createing or laying out various controls.
		 */
		public static class TestGenUIException extends Exception
		{
	
			/**
			 * 
			 */
			public TestGenUIException()
			{
				super();
			}

			/**
			 * @param message
			 */
			public TestGenUIException(String message)
			{
				super(message);
			}

			

	}
	
	
	
	/**
	 * Utility method for laying out a control in a form layout by specifying percentage-of-parent-width and border regions
	 *  If the parent of the control is not a FormLayout, an exception will be thrown
	 * @param Control control - the control to be layed out
	 * @param intpixelsToTop - pixels from top of the parent control to the parent of the control of focus
	 * @param int verticalSpanPixels - the vertical span of the control being layed out
	 * @param int leftPercentWidth - percentage of total parent width from left for left of target
	 * @param int rightPercentWidth - percentage of total parent width from left for right of target 
	 * @param int leftBorder - pixels for left border region
	 * @param int bottomBorder - pixels for bottom border region
	 * @param int topBorder - pixels for top border region
	 * @param int rightBorder - pixels for bottom border region
	 * @throws TestGenUIException
	 */
	public static void layoutCompositelInFormLayout(Control composite,
								 int pixelsToTop,
								 int verticalSpanPixels,
								 int leftPercentWidth, 
								 int rightPercentWidth, 
								 int leftBorder, 
								 int bottomBorder,
								 int topBorder, 
								 int rightBorder) throws TestGenUIException
	{
		if(!FormLayout.class.isInstance(composite.getParent().getLayout()))
		{
			String[] layoutName = {"org.eclipse.swt.layout.FormLayout"}; //$NON-NLS-1$
			String errorMessage = MessageFormat.format(TestgenPlugin.getResourceString("TestGenUIUtility.INCORRECT_LAYOUT_MESSAGE_FORMAT_STRING"),layoutName); //$NON-NLS-1$
			throw new TestGenUIException(errorMessage);
		}
		
		FormData formData = new FormData();
		formData.left = new FormAttachment(leftPercentWidth,leftBorder);
		formData.right = new FormAttachment(rightPercentWidth,-rightBorder);
		formData.top = new FormAttachment(0,pixelsToTop+topBorder);
		formData.bottom = new FormAttachment(0,pixelsToTop+verticalSpanPixels-bottomBorder);
		composite.setLayoutData(formData);
	}
	
	
	
		/** 
		 * Positions a control on it's parent FormLayout.
		 * @param Control control - the Control to be positioned
		 * @param int topBorder - the top border for this control in pixels
		 * @param int leftBorder - the left border for this control in pixels.
		 * @param int bottomBorder - the bottom border for this control in pixels
		 * @param int rightBorder - the right border for this control in pixels
		 */
		public static void layoutControlInFormLayout(Control control,
												 int topBorder, 
												 int leftBorder, 
												 int bottomBorder, 
												 int rightBorder) throws TestGenUIException
		{
			if(!FormLayout.class.isInstance(control.getParent().getLayout()))
			{
				String[] layoutName = {"org.eclipse.swt.layout.GridLayout"}; //$NON-NLS-1$
				String errorMessage = MessageFormat.format(TestgenPlugin.getResourceString("TestGenUIUtility.INCORRECT_LAYOUT_MESSAGE_FORMAT_STRING"),layoutName); //$NON-NLS-1$
				throw new TestGenUIException(errorMessage);
			}
	
			FormData formData = new FormData();
			formData.left = new FormAttachment(0,leftBorder);
			formData.right = new FormAttachment(100,-rightBorder);
			formData.top = new FormAttachment(0,topBorder);
			formData.bottom = new FormAttachment(100,-bottomBorder);
				
			control.setLayoutData(formData);
		}
	


	
		/**
		 * method for laying out a control in a grid layout by providing columnSpan and any "or'ed" combination of GridData options
		 * @param Control control
		 * @param int columnSpan
		 * @param int gridDataOptions
		 * @throws RecorderHarnessUIException
		 */
		public static void layoutControlInGrid(Control control, int columnSpan, int gridDataOptions) throws TestGenUIException
		{
//			if(!GridLayout.class.isInstance(control.getParent().getLayout()))
//			{
//				throw new TestGenUIException("Parent Composite does not have an assigned grid Layout (org.eclipse.swt.layout.GridLayout)");
//			}
			GridData gridData = new GridData(gridDataOptions);
			gridData.horizontalSpan=columnSpan;
			control.setLayoutData(gridData);	
		}



		/**
		 * method for assigning a FormLayoout to a composite
		 * @param Composite parent
		 */
		public static void assignFormLayout(Composite parent)
		{
			FormLayout layout = new FormLayout();
			layout.marginHeight=0;
			layout.marginWidth=0;
			parent.setLayout(layout);
		}



		/**
		 * method for assigning a GridLayout to a Composite
		 * @param Composite compsite
		 * @param int numberColumns
		 * @param int columnsEqual	 
		 */
		public static void assignGridLayout(Composite compsite, int numberColumns)
		{
			GridLayout gridLayout = new GridLayout(numberColumns,false);
			compsite.setLayout(gridLayout);
		}



		/**
		 * method for creating a label
		 * @param Composite parent
		 * @param String labelText
		 * @param Font font
		 * @return Label
		 */
		public static Label createLabel(Composite composite, String labelText, Font font)
		{
			Label label = new Label(composite,SWT.WRAP);
			label.setText(labelText);
		
			if(font!=null)
				label.setFont(font);
		
			return label;
		}



		/**
		 * method for creating an Edit Control
		 * @param Composite parent
		 * @param String defaultText
		 * @param Font font
		 * @return Text
		 */
		public static Text createEditControl(Composite composite, String textValue, Font font)
		{
			Text text = new Text(composite,SWT.BORDER);
			text.setText(textValue);
		
			if(font!=null)
				text.setFont(font);
			
			return text;
		}
	
	
		/**
		 * method for creating a TreeViewer
		 * @param composite
		 * @param font
		 * @return Tree
		 */
		public static TreeViewer createTreeViewer(Composite composite, Font font)
		{
			TreeViewer tree = new TreeViewer(composite);
			tree.setUseHashlookup(true);
			tree.getControl().setFont(font);
			return tree;
		}
	
		/**
		 * method for creating a control group
		 * @param controlArea
		 * @param text
		 * @return Group
		 */
		public static Group createGroup(Composite controlArea, String text)
		{
			Group recorderHostGroup = new Group(controlArea,SWT.SHADOW_ETCHED_IN);
	
			assignFormLayout(controlArea);
			
			recorderHostGroup.setText(text);
	
			return recorderHostGroup;
		}
	
		/**
		 * method for creating a Button
		 * @param parent
		 * @param text
		 * @param swtOptions
		 * @return Button
		 */
		public static Button createButton(Composite parent, String text, int swtOptions)
		{
			Button button = new Button(parent,swtOptions);
			button.setText(text);
			return button;
	
		}
	/**
	 * Opens the editor for the specified file.
	 * 
	 * <p>If <code>editorId</code> is null the default editor is opened.  
	 * If substring is true then the <code>editorId</code> value is considered to be 
	 * first part of the editor id.
	 * 
	 * @param file;
	 * @param editorId;
	 * @param substring;
	 * @return The opened editor part or <code>null</code> if this method was not
	 * able to open the editor.
	 */
	public static IEditorPart openEditor(IFile file, String editorId, boolean substring)
	{
		if(file == null)
			return null;
		IWorkbenchPage activePage = null;	
		try {	
			activePage = TestgenPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage();
		}
		catch (NullPointerException e)
		{
			return null;
		}
		if(activePage == null)
			return null;
		
		if("".equals(editorId))
			editorId = null;
		
		try
		{
			if(editorId == null)
			{
				return activePage.openEditor(file);
			}
			else
			{
				IWorkbench workbench = PlatformUI.getWorkbench();
				IEditorDescriptor[] editors = workbench.getEditorRegistry().getEditors(file);
				if(editors != null)
				{
					boolean found = false;
					for(int i=0, length=editors.length; i<length; i++)
					{
						if(substring)
							found = editors[i].getId().startsWith(editorId);
						else
							found = editorId.equals(editors[i].getId());
							
						if(found)
							return activePage.openEditor(file, editors[i].getId());
					}
				}				
			}
		}
		catch(Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
	
		return null;
	}

	/**
	 * Attempts to select and reveal the specified resource in all
	 * parts within the supplied workbench window's active page.
	 * 
	 * <p>Checks all parts in the active page to see if they implement 
	 * <code>ISetSelectionTarget</code>, either directly or as an adapter. 
	 * If so, tells the part to select and reveal the specified resource.
	 *
	 * @param resource the resource to be selected and revealed
	 * @param window the workbench window to select and reveal the resource
	 * @see org.eclipse.ui.wizards.newresource.BasicNewResourceWizard
	 * @see ISetSelectionTarget
	 */
	public static void selectAndReveal(IResource resource, IWorkbenchWindow window)
	{
		// validate the input
		if (window == null || resource == null)
			return;
		IWorkbenchPage page = window.getActivePage();
		if (page == null)
			return;

		// get all the view and editor parts
		List parts = new ArrayList();
		IWorkbenchPartReference refs[] = page.getViewReferences();
		for (int i = 0; i < refs.length; i++)
		{
			IWorkbenchPart part = refs[i].getPart(false);
			if(part != null)
				parts.add(part);
		}	
		refs = page.getEditorReferences();
		for (int i = 0; i < refs.length; i++)
		{
			if(refs[i].getPart(false) != null)
				parts.add(refs[i].getPart(false));
		}
	
		final ISelection selection = new StructuredSelection(resource);
		Iterator enum = parts.iterator();
		while (enum.hasNext())
		{
			IWorkbenchPart part = (IWorkbenchPart) enum.next();
		
			// get the part's ISetSelectionTarget implementation
			ISetSelectionTarget target = null;
			if (part instanceof ISetSelectionTarget)
				target = (ISetSelectionTarget) part;
			else
				target = (ISetSelectionTarget) part.getAdapter(ISetSelectionTarget.class);
			
			if (target != null)
			{
				// select and reveal resource
				final ISetSelectionTarget finalTarget = target;
				window.getShell().getDisplay().asyncExec(new Runnable()
				{
					public void run()
					{
						finalTarget.selectReveal(selection);
					}
				});
			}
		}
	}

}
