package org.eclipse.hyades.logging.events.cbe.util;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ConfigureSituation;
import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.CreateSituation;
import org.eclipse.hyades.logging.events.cbe.DependencySituation;
import org.eclipse.hyades.logging.events.cbe.DestroySituation;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FeatureSituation;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.StopSituation;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The EventFormatter class defines a utility object used for 
 * converting, serializing and deserializing CommonBaseEvent
 * and related objects.  It is mostly used by other objects
 * in this API to facilitate shared functionality, but can
 * be used by anyone using the API.
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @author John K. Gerken
 * @version 1.0.1
 * @since 1.0
 * @see org.eclipse.hyades.logging.events.cbe.impl.CommonBaseEventImpl
 * @see org.eclipse.hyades.logging.events.cbe.impl.AssociationEngineImpl
 */
public final class EventFormatter {

    private final static String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * Converts an CommonBaseEvent object into a non-formatted serialized
     * XML fragment.  The format of the serialized XML fragment is
     * based on the parameter CommonBaseEvent's version.
     * 
     * @param commonBaseEvent The CommonBaseEvent to serialize.
     * @return An XML fragment String representing the parameter CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(CommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLString(commonBaseEvent, false));
    }

    /**
     * Converts an CommonBaseEvent object into a serialized
     * XML fragment.  Formatting is based on the format parameter.
     * 
     * @param commonBaseEvent The CommonBaseEvent to serialize.
     * @param format If the serialized XML fragment is formatted with platform-dependent new line(s) and tab(s).
     * @return An XML fragment String representing the parameter CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLString(CommonBaseEvent commonBaseEvent, boolean format) {

        /* Start with a 1k buffer to load the XML String into */
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvent");

        if (commonBaseEvent.isSetCreationTime()) {
            buffer.append(" creationTime=\"");
            buffer.append(XmlUtility.normalize(commonBaseEvent.getCreationTime()));
            buffer.append("\"");
        }

        String attributeValue = commonBaseEvent.getExtensionName();

        if (attributeValue != null) {
            buffer.append(" extensionName=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getGlobalInstanceId();

        if (attributeValue != null) {
            buffer.append(" globalInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getLocalInstanceId();

        if (attributeValue != null) {
            buffer.append(" localInstanceId=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getMsg();

        if (attributeValue != null) {
            buffer.append(" msg=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetElapsedTime()) {
            buffer.append(" elapsedTime=\"");
            buffer.append(commonBaseEvent.getElapsedTime());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetPriority()) {
            buffer.append(" priority=\"");
            buffer.append(commonBaseEvent.getPriority());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetRepeatCount()) {
            buffer.append(" repeatCount=\"");
            buffer.append(commonBaseEvent.getRepeatCount());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSequenceNumber()) {
            buffer.append(" sequenceNumber=\"");
            buffer.append(commonBaseEvent.getSequenceNumber());
            buffer.append("\"");
        }

        if (commonBaseEvent.isSetSeverity()) {
            buffer.append(" severity=\"");
            buffer.append(commonBaseEvent.getSeverity());
            buffer.append("\"");
        }

        attributeValue = commonBaseEvent.getVersion();

        if (attributeValue != null) {
            buffer.append(" version=\"");
            buffer.append(XmlUtility.normalize(attributeValue));
            buffer.append("\"");
        }

        buffer.append(">");

        List contextDataElements = commonBaseEvent.getContextDataElements();

        if (contextDataElements != null) {

            for (int i = 0; i < contextDataElements.size(); i++) {

                if (contextDataElements.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<contextDataElements");

                    attributeValue = ((ContextDataElement) (contextDataElements.get(i))).getName();

                    if (attributeValue != null) {
                        buffer.append(" name=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ContextDataElement) (contextDataElements.get(i))).getType();

                    if (attributeValue != null) {
                        buffer.append(" type=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    attributeValue = ((ContextDataElement) (contextDataElements.get(i))).getContextId();

                    if (attributeValue != null) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<contextId>");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("</contextId>");
                    }
                    else {

                        attributeValue = ((ContextDataElement) (contextDataElements.get(i))).getContextValue();

                        if (attributeValue != null) {

                            if (format) {
                                buffer.append(LINE_SEPARATOR);
                                buffer.append("\t\t");
                            }

                            buffer.append("<contextValue>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</contextValue>");
                        }
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("</contextDataElements>");
                }
            }
        }

        List extendedDataElements = commonBaseEvent.getExtendedDataElements();

        if (extendedDataElements != null) {

            for (int i = 0; i < extendedDataElements.size(); i++) {

                buffer.append(getExtendedDataElementXML(((ExtendedDataElement) (extendedDataElements.get(i))), "extendedDataElements", format, (format ? 1 : 0)));
            }
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();

        if (associatedEvents != null) {

            for (int i = 0; i < associatedEvents.size(); i++) {

                if (associatedEvents.get(i) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("<associatedEvents");

                    String resolvedEvents = ((AssociatedEvent) (associatedEvents.get(i))).getResolvedEvents();

                    if (resolvedEvents != null) {

                        buffer.append(" resolvedEvents=\"");
                        buffer.append(resolvedEvents);
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    AssociationEngine associationEngineInfo = ((AssociatedEvent) (associatedEvents.get(i))).getAssociationEngineInfo();

                    if (associationEngineInfo != null) {

                        buffer.append("<associationEngineInfo");

                        attributeValue = associationEngineInfo.getId();

                        if (attributeValue != null) {
                            buffer.append(" id=\"");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("\"");
                        }

                        attributeValue = associationEngineInfo.getName();

                        if (attributeValue != null) {
                            buffer.append(" name=\"");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("\"");
                        }

                        attributeValue = associationEngineInfo.getType();

                        if (attributeValue != null) {
                            buffer.append(" type=\"");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("\"");
                        }

                        buffer.append("/>");
                    }
                    else {

                        attributeValue = ((AssociatedEvent) (associatedEvents.get(i))).getAssociationEngine();

                        if (attributeValue != null) {

                            buffer.append("<associationEngine>");
                            buffer.append(XmlUtility.normalize(attributeValue));
                            buffer.append("</associationEngine>");
                        }

                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    buffer.append("</associatedEvents>");
                }
            }
        }

        ComponentIdentification componentId = commonBaseEvent.getReporterComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<reporterComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        componentId = commonBaseEvent.getSourceComponentId();

        if (componentId != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<sourceComponentId");

            attributeValue = componentId.getApplication();

            if (attributeValue != null) {
                buffer.append(" application=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponent();

            if (attributeValue != null) {
                buffer.append(" component=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentIdType();

            if (attributeValue != null) {
                buffer.append(" componentIdType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getExecutionEnvironment();

            if (attributeValue != null) {
                buffer.append(" executionEnvironment=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getInstanceId();

            if (attributeValue != null) {
                buffer.append(" instanceId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocation();

            if (attributeValue != null) {
                buffer.append(" location=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getLocationType();

            if (attributeValue != null) {
                buffer.append(" locationType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getProcessId();

            if (attributeValue != null) {
                buffer.append(" processId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getSubComponent();

            if (attributeValue != null) {
                buffer.append(" subComponent=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getThreadId();

            if (attributeValue != null) {
                buffer.append(" threadId=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = componentId.getComponentType();

            if (attributeValue != null) {
                buffer.append(" componentType=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append("/>");
        }

        MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

        if (msgDataElement != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<msgDataElement");

            attributeValue = msgDataElement.getMsgLocale();

            if (attributeValue != null) {
                buffer.append(" msgLocale=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            List tokens = msgDataElement.getMsgCatalogTokens();

            if (tokens != null) {

                for (int i = 0; i < tokens.size(); i++) {

                    if ((tokens.get(i) != null) && (((MsgCatalogToken) (tokens.get(i))).getValue() != null)) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t");
                        }

                        buffer.append("<msgCatalogTokens value=\"");
                        buffer.append(XmlUtility.normalize(((MsgCatalogToken) (tokens.get(i))).getValue()));
                        buffer.append("\"/>");
                    }
                }
            }

            attributeValue = msgDataElement.getMsgId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgId>");
            }

            attributeValue = msgDataElement.getMsgIdType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgIdType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgIdType>");
            }

            attributeValue = msgDataElement.getMsgCatalogId();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogId>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogId>");
            }

            attributeValue = msgDataElement.getMsgCatalogType();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalogType>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalogType>");
            }

            attributeValue = msgDataElement.getMsgCatalog();

            if (attributeValue != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<msgCatalog>");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("</msgCatalog>");
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</msgDataElement>");
        }

        Situation situation = commonBaseEvent.getSituation();

        if (situation != null) {

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("<situation");

            attributeValue = situation.getCategoryName();

            if (attributeValue != null) {
                buffer.append(" categoryName=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            SituationType situationType = situation.getSituationType();

            if (situationType != null) {

                if (format) {
                    buffer.append(LINE_SEPARATOR);
                    buffer.append("\t\t");
                }

                buffer.append("<situationType");

                attributeValue = situationType.getReasoningScope();

                if (situationType instanceof StartSituation) {

                    buffer.append(" xsi:type=\"StartSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StartSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StartSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof StopSituation) {

                    buffer.append(" xsi:type=\"StopSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StopSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((StopSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof ConnectSituation) {

                    buffer.append(" xsi:type=\"ConnectSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConnectSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConnectSituation) (situationType)).getSituationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" situationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof ReportSituation) {

                    buffer.append(" xsi:type=\"ReportSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ReportSituation) (situationType)).getReportCategory();

                    if (attributeValue != null) {

                        buffer.append(" reportCategory=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof FeatureSituation) {

                    buffer.append(" xsi:type=\"FeatureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((FeatureSituation) (situationType)).getFeatureDisposition();

                    if (attributeValue != null) {

                        buffer.append(" featureDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof ConfigureSituation) {

                    buffer.append(" xsi:type=\"ConfigureSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((ConfigureSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof DependencySituation) {

                    buffer.append(" xsi:type=\"DependencySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((DependencySituation) (situationType)).getDependencyDisposition();

                    if (attributeValue != null) {

                        buffer.append(" dependencyDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }
                else if (situationType instanceof CreateSituation) {

                    buffer.append(" xsi:type=\"CreateSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((CreateSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof DestroySituation) {

                    buffer.append(" xsi:type=\"DestroySituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((DestroySituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof AvailableSituation) {

                    buffer.append(" xsi:type=\"AvailableSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getOperationDisposition();

                    if (attributeValue != null) {

                        buffer.append(" operationDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getProcessingDisposition();

                    if (attributeValue != null) {

                        buffer.append(" processingDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((AvailableSituation) (situationType)).getAvailabilityDisposition();

                    if (attributeValue != null) {

                        buffer.append(" availabilityDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof RequestSituation) {

                    buffer.append(" xsi:type=\"RequestSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((RequestSituation) (situationType)).getSuccessDisposition();

                    if (attributeValue != null) {

                        buffer.append(" successDisposition=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = ((RequestSituation) (situationType)).getSituationQualifier();

                    if (attributeValue != null) {

                        buffer.append(" situationQualifier=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");
                }

                else if (situationType instanceof OtherSituation) {

                    buffer.append(" xsi:type=\"OtherSituation\"");

                    if (attributeValue != null) {
                        buffer.append(" reasoningScope=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append(">");

                    String anyData = ((OtherSituation) (situationType)).getAny();

                    if ((anyData != null) && (anyData.length() > 0)) {

                        if (format) {
                            buffer.append(LINE_SEPARATOR);
                            buffer.append("\t\t\t");
                        }

                        //Do NOT normalize since this string is assumed to be a valid XML fragment:
                        buffer.append(anyData);
                    }

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t\t");
                    }

                    buffer.append("</situationType>");
                }
                else {
                    buffer.append("/>");
                }
            }

            if (format) {
                buffer.append(LINE_SEPARATOR);
                buffer.append("\t");
            }

            buffer.append("</situation>");
        }

        List anyElements = commonBaseEvent.getAny();

        if (anyElements != null) {

            for (int counter = 0; counter < anyElements.size(); counter++) {

                if (anyElements.get(counter) != null) {

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                        buffer.append("\t");
                    }

                    //Do NOT normalize since this string is assumed to be a valid XML fragment:
                    buffer.append(((String) (anyElements.get(counter))));
                }
            }
        }

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        buffer.append("</CommonBaseEvent>");

        return (buffer.toString());
    }

    private static String getExtendedDataElementXML(ExtendedDataElement extendedDataElement, String tagName, boolean format, int indent) {

        if (extendedDataElement != null) {

            StringBuffer buffer = new StringBuffer(512);

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++)
                    buffer.append("\t");
            }

            buffer.append("<");
            buffer.append(tagName);

            String attributeValue = extendedDataElement.getName();

            if (attributeValue != null) {
                buffer.append(" name=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            attributeValue = extendedDataElement.getType();

            if (attributeValue != null) {
                buffer.append(" type=\"");
                buffer.append(XmlUtility.normalize(attributeValue));
                buffer.append("\"");
            }

            buffer.append(">");

            if (extendedDataElement.getTypeAsInt() == ExtendedDataElement.TYPE_HEX_BINARY_VALUE) {

                String hexValues = extendedDataElement.getHexValue();

                if ((hexValues != null) && (hexValues.length() > 0)) {

                    if (format) {

                        buffer.append(LINE_SEPARATOR);

                        for (int counter = 0; counter < (indent + 1); counter++)
                            buffer.append("\t");
                    }

                    buffer.append("<hexValue>");
                    buffer.append(hexValues);
                    buffer.append("</hexValue>");
                }
            }
            else {

                List values = extendedDataElement.getValues();

                if ((values != null) && (values.size() > 0)) {

                    for (int counter = 0; counter < values.size(); counter++) {

                        if ((values.get(counter) != null) && (((String) (values.get(counter))).length() > 0)) {

                            if (format) {

                                buffer.append(LINE_SEPARATOR);

                                for (int index = 0; index < (indent + 1); index++)
                                    buffer.append("\t");
                            }

                            buffer.append("<values>");
                            buffer.append(XmlUtility.normalize(((String) (values.get(counter)))));
                            buffer.append("</values>");
                        }
                    }
                }
            }

            List childDataElements = extendedDataElement.getChildren();

            if (childDataElements != null) {

                for (int counter = 0; counter < childDataElements.size(); counter++) {

                    buffer.append(getExtendedDataElementXML(((ExtendedDataElement) childDataElements.get(counter)), "children", format, (format ? (indent + 1) : 0)));
                }
            }

            if (format) {

                buffer.append(LINE_SEPARATOR);

                for (int counter = 0; counter < indent; counter++)
                    buffer.append("\t");
            }

            buffer.append("</");
            buffer.append(tagName);
            buffer.append(">");

            return (buffer.toString());
        }
        else {
            return "";
        }
    }

    /**
     * Converts an CommonBaseEvent object into a non-formatted serialized
     * XML document.  The format of the serialized XML document is
     * based on the parameter CommonBaseEvent's version.
     * 
     * @param commonBaseEvent The CommonBaseEvent to serialize.
     * @return An XML fragment String representing the parameter CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent) {
        return (toCanonicalXMLDocString(commonBaseEvent, false));
    }

    /**
     * Converts an CommonBaseEvent object into a serialized
     * XML document.  Formatting is based on the format parameter.
     * 
     * @param commonBaseEvent The CommonBaseEvent to serialize.
     * @param format If the serialized XML document is formatted with platform-dependent new line(s) and tab(s).
     * @return An XML fragment String representing the parameter CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent, boolean format) {

        // Create a buffer containing the top level element and namespace info...
        StringBuffer buffer = new StringBuffer(1024);

        buffer.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");

        if (format) {
            buffer.append(LINE_SEPARATOR);
        }

        List associatedEvents = commonBaseEvent.getAssociatedEvents();

        // If associatedEvents exist, get the serialized association engine...
        if ((associatedEvents != null) && (associatedEvents.size() > 0)) {

            AssociationEngine associationEngine = null;

            for (int counter = 0; counter < associatedEvents.size(); counter++) {

                associationEngine = ((AssociatedEvent) (associatedEvents.get(counter))).getAssociationEngineInfo();

                if (associationEngine != null) {

                    if (format) {
                        buffer.append("\t");
                    }

                    buffer.append("<AssociationEngine");

                    String attributeValue = associationEngine.getId();

                    if (attributeValue != null) {
                        buffer.append(" id=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = associationEngine.getName();

                    if (attributeValue != null) {
                        buffer.append(" name=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    attributeValue = associationEngine.getType();

                    if (attributeValue != null) {
                        buffer.append(" type=\"");
                        buffer.append(XmlUtility.normalize(attributeValue));
                        buffer.append("\"");
                    }

                    buffer.append("/>");

                    if (format) {
                        buffer.append(LINE_SEPARATOR);
                    }
                }
            }
        }

        if (format) {

            //Add the CommonBaseEvent element fragment with an extra tab after every platform dependent new line character to indent all CommonBaseEvents child elements:
            StringTokenizer tokens = new StringTokenizer(toCanonicalXMLString(commonBaseEvent, format).trim(), LINE_SEPARATOR);

            while (tokens.hasMoreTokens()) {

                buffer.append("\t");
                buffer.append(tokens.nextToken());
                buffer.append(LINE_SEPARATOR);
            }
        }
        else {

            //Add the CommonBaseEvent element fragment:
            buffer.append(toCanonicalXMLString(commonBaseEvent, format));
        }

        // Close and return the top level element:
        buffer.append("</CommonBaseEvents>");

        return buffer.toString();
    }

    /**
     * Converts an XML fragment to an CommonBaseEvent object.
     * 
     * @param commonBaseEvent The CommonBaseEvent to be populated by the XML fragment.
     * @param xmlString The XML fragment to populate the CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized void fromCanonicalXMLString(CommonBaseEvent commonBaseEvent, String xmlString) {

        /* setup the pointers and our initial work area */
        int masterIndex = xmlString.indexOf("<CommonBaseEvent "); // tracks location in passed string
        int index = 0; // tracks location in temp work area
        String tempString = xmlString.substring(masterIndex, xmlString.indexOf(">", masterIndex + 1));
        masterIndex = xmlString.indexOf(">", masterIndex + 1) + 1; // push pointer after '>' character

        /* populate the attributes for the root element of this fragment */
        index = tempString.indexOf(" creationTime=\"");
        if (index != -1) {
            commonBaseEvent.setCreationTime(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " creationTime=\"".
        }

        index = tempString.indexOf(" extensionName=\"");
        if (index != -1) {
            commonBaseEvent.setExtensionName(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " extensionName=\"".
        }

        index = tempString.indexOf(" globalInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setGlobalInstanceId(XmlUtility.denormalize(tempString.substring(index + 19, tempString.indexOf("\"", index + 19)))); //19 is the length of " globalInstanceId=\"".
        }

        index = tempString.indexOf(" localInstanceId=\"");
        if (index != -1) {
            commonBaseEvent.setLocalInstanceId(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " localInstanceId=\"".
        }

        index = tempString.indexOf(" msg=\"");
        if (index != -1) {
            commonBaseEvent.setMsg(XmlUtility.denormalize(tempString.substring(index + 6, tempString.indexOf("\"", index + 6)))); //6 is the length of " msg=\"".
        }

        index = tempString.indexOf(" situationType=\"");
        String situationTypeAttributeValue = null;
        if (index != -1) {
            situationTypeAttributeValue = (XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " situationType=\"".
        }

        index = tempString.indexOf(" elapsedTime=\"");
        if (index != -1) {
            commonBaseEvent.setElapsedTime(Long.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).longValue()); //14 is the length of " elapsedTime=\"".
        }

        index = tempString.indexOf(" priority=\"");
        if (index != -1) {
            commonBaseEvent.setPriority(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " priority=\"".
        }

        index = tempString.indexOf(" repeatCount=\"");
        if (index != -1) {
            commonBaseEvent.setRepeatCount(Short.valueOf(tempString.substring(index + 14, tempString.indexOf("\"", index + 14))).shortValue()); //14 is the length of " repeatCount=\"".
        }

        index = tempString.indexOf(" sequenceNumber=\"");
        if (index != -1) {
            commonBaseEvent.setSequenceNumber(Long.valueOf(tempString.substring(index + 17, tempString.indexOf("\"", index + 17))).longValue()); //17 is the length of " sequenceNumber=\"".
        }

        index = tempString.indexOf(" severity=\"");
        if (index != -1) {
            commonBaseEvent.setSeverity(Short.valueOf(tempString.substring(index + 11, tempString.indexOf("\"", index + 11))).shortValue()); //11 is the length of " severity=\"".
        }

        /* Begin processing of ContextDataEements */
        while (xmlString.indexOf("<contextDataElements ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<contextDataElements ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</contextDataElements>", masterIndex + 1));

            ContextDataElement cde = EventFactory.eINSTANCE.createContextDataElement();
            /* process the attributes for this element */
            index = tempString.indexOf(" name=\"");
            if (index != -1) {
                cde.setName(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
            }

            index = tempString.indexOf(" type=\"");
            if (index != -1) {
                cde.setType(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
            }

            /**
             * contextValue and contextId are mutually exclusive.  Therefore, only 
             * one of these two properties can be defined.  contextValue takes 
             * precidence, such that if contextValue is set then contextId is ignored. 
             */
            index = tempString.indexOf("<contextValue>");
            if (index != -1) {
                cde.setContextValue(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</contextValue>", index + 14))));

            }
            else if (tempString.indexOf("<contextId>") != -1) {
                index = tempString.indexOf("<contextId>");
                cde.setContextId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</contextId>", index + 11))));
            }

            commonBaseEvent.addContextDataElement(cde);
            masterIndex = xmlString.indexOf("</contextDataElements>", masterIndex + 1) + 22;
        }

        int edeBeginIndex = -1;
        int edeEndIndex = masterIndex;
        int edeEndTestIndex = -1;

        /* Begin processing of ExtendedDataEements */
        while ((edeBeginIndex = xmlString.indexOf("<extendedDataElements ", edeEndIndex)) != -1) {

            edeEndTestIndex = xmlString.indexOf("</extendedDataElements>", edeBeginIndex);

            if (edeEndTestIndex != -1) {

                edeEndIndex = edeEndTestIndex + 23; //23 is the length of "</extendedDataElements>".

                edeEndTestIndex = xmlString.indexOf("/>", edeBeginIndex);

                if ((edeEndTestIndex != -1) && (edeEndTestIndex < edeEndIndex)) {
                    edeEndIndex = edeEndTestIndex + 2; //2 is the length of "/>".
                }
            }
            else {
                edeEndIndex = xmlString.indexOf("/>", edeBeginIndex) + 2; //2 is the length of "/>".            	
            }

            commonBaseEvent.addExtendedDataElement(loadExtendedDataElement(xmlString.substring(edeBeginIndex, edeEndIndex)));
        }

        masterIndex = edeEndIndex;

        //Workaround since CommonBaseEventImpl stores associated events by name and not ID:
        StringBuffer spaces = new StringBuffer();

        /* Begin processing of AssociatedEvents */
        while (xmlString.indexOf("<associatedEvents ", masterIndex) != -1) {

            /* update pointers and build work area */
            masterIndex = xmlString.indexOf("<associatedEvents ", masterIndex);
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));

            AssociatedEvent associatedEvent = EventFactory.eINSTANCE.createAssociatedEvent();

            index = tempString.indexOf(" resolvedEvents=\""); //Start from the beginning of the new temporary string.

            if (index != -1) {
                associatedEvent.setResolvedEvents(XmlUtility.denormalize(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)))); //17 is the length of " resolvedEvents=\"".
            }

            // If there is an associationEngine
            index = tempString.indexOf("<associationEngineInfo ", index);

            if (index == -1) {

                //If there is an associationEngine:
                index = tempString.indexOf("<associationEngine>");

                if (index != -1)
                    associatedEvent.setAssociationEngine(XmlUtility.denormalize(tempString.substring(index + 19, tempString.indexOf("</associationEngine>", index + 19)))); //19 is the length of "<associationEngine>".
            }
            else {

                AssociationEngine associationEngine = EventFactory.eINSTANCE.createAssociationEngine();

                index = tempString.indexOf(" id=\"");

                if (index != -1) {
                    associationEngine.setId(XmlUtility.denormalize(tempString.substring(index + 5, tempString.indexOf("\"", index + 5)))); //5 is the length of " id=\"".
                }

                index = tempString.indexOf(" name=\"");

                if (index != -1) {
                    associationEngine.setName(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
                }

                index = tempString.indexOf(" type=\"");

                if (index != -1) {
                    associationEngine.setType(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
                }

                associatedEvent.setAssociationEngineInfo(associationEngine);
            }

            commonBaseEvent.addAssociatedEvent(associatedEvent);

            masterIndex = xmlString.indexOf("/>", masterIndex) + 2; //2 is the length of "/>".
        }

        /* if reporterComponentId is included... */
        if (xmlString.indexOf("<reporterComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<reporterComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            ComponentIdentification rci = EventFactory.eINSTANCE.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                rci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                rci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                rci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                rci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                rci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                rci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                rci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                rci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                rci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                rci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            index = tempString.indexOf(" componentType=\"");
            if (index != -1) {
                rci.setComponentType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " componentType=\"".
            }

            commonBaseEvent.setReporterComponentId(rci);
        }

        /* if sourceComponentId is included... */
        if (xmlString.indexOf("<sourceComponentId") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<sourceComponentId");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("/>", masterIndex + 1));
            masterIndex = xmlString.indexOf("/>", masterIndex + 1) + 2;

            ComponentIdentification sci = EventFactory.eINSTANCE.createComponentIdentification();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" application=\"");
            if (index != -1) {
                sci.setApplication(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("\"", index + 14)))); //14 is the length of " application=\"".
            }

            index = tempString.indexOf(" component=\"");
            if (index != -1) {
                sci.setComponent(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " component=\"".
            }

            index = tempString.indexOf(" componentIdType=\"");
            if (index != -1) {
                sci.setComponentIdType(XmlUtility.denormalize(tempString.substring(index + 18, tempString.indexOf("\"", index + 18)))); //18 is the length of " componentIdType=\"".
            }

            index = tempString.indexOf(" executionEnvironment=\"");
            if (index != -1) {
                sci.setExecutionEnvironment(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " executionEnvironment=\"".
            }

            index = tempString.indexOf(" instanceId=\"");
            if (index != -1) {
                sci.setInstanceId(XmlUtility.denormalize(tempString.substring(index + 13, tempString.indexOf("\"", index + 13)))); //13 is the length of " instanceId=\"".
            }

            index = tempString.indexOf(" location=\"");
            if (index != -1) {
                sci.setLocation(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " location=\"".
            }

            index = tempString.indexOf(" locationType=\"");
            if (index != -1) {
                sci.setLocationType(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " locationType=\"".
            }

            index = tempString.indexOf(" processId=\"");
            if (index != -1) {
                sci.setProcessId(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " processId=\"".
            }

            index = tempString.indexOf(" subComponent=\"");
            if (index != -1) {
                sci.setSubComponent(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " subComponent=\"".
            }

            index = tempString.indexOf(" threadId=\"");
            if (index != -1) {
                sci.setThreadId(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " threadId=\"".
            }

            index = tempString.indexOf(" componentType=\"");
            if (index != -1) {
                sci.setComponentType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("\"", index + 16)))); //16 is the length of " componentType=\"".
            }

            commonBaseEvent.setSourceComponentId(sci);
        }

        /* if msgDataElement is included... */
        if (xmlString.indexOf("<msgDataElement") != -1) {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<msgDataElement");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</msgDataElement>", masterIndex + 1));
            masterIndex = xmlString.indexOf("</msgDataElement>", masterIndex + 1) + 17;

            MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();

            /* populate the attributes for this element... */
            index = tempString.indexOf(" msgLocale=\"");
            if (index != -1) {
                mde.setMsgLocale(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("\"", index + 12)))); //12 is the length of " msgLocale=\"".
            }

            // If there are tokens to process...
            if (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                MsgCatalogToken msgCatalogToken = null;

                // For each token...
                while (tempString.indexOf("<msgCatalogTokens ", index) != -1) {

                    /* process the attribute for this subelement */
                    index = tempString.indexOf(" value=\"", index);

                    if (index != -1) {

                        msgCatalogToken = EventFactory.eINSTANCE.createMsgCatalogToken();
                        msgCatalogToken.setValue(XmlUtility.denormalize(new String(tempString.substring(index + 8, tempString.indexOf("\"", index + 8))))); //8 is the length of " value=\"".
                        mde.addMsgCatalogToken(msgCatalogToken);
                    }

                    index = tempString.indexOf("/>", index + 1) + 2;
                }
            }

            index = tempString.indexOf("<msgCatalogType>");
            if (index != -1) {
                mde.setMsgCatalogType(XmlUtility.denormalize(tempString.substring(index + 16, tempString.indexOf("</msgCatalogType>", index + 16))));
            }

            index = tempString.indexOf("<msgId>");
            if (index != -1) {
                mde.setMsgId(XmlUtility.denormalize(tempString.substring(index + 7, tempString.indexOf("</msgId>", index + 7))));
            }

            index = tempString.indexOf("<msgIdType>");
            if (index != -1) {
                mde.setMsgIdType(XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("</msgIdType>", index + 11))));
            }

            index = tempString.indexOf("<msgCatalogId>");
            if (index != -1) {
                mde.setMsgCatalogId(XmlUtility.denormalize(tempString.substring(index + 14, tempString.indexOf("</msgCatalogId>", index + 14))));
            }

            index = tempString.indexOf("<msgCatalog>");
            if (index != -1) {
                mde.setMsgCatalog(XmlUtility.denormalize(tempString.substring(index + 12, tempString.indexOf("</msgCatalog>", index + 12))));
            }

            commonBaseEvent.setMsgDataElement(mde);
        }

        //Check for the situation element:
        if (xmlString.indexOf("<situation") == -1) {

            if (situationTypeAttributeValue != null) {

                ReportSituation reportSituation = EventFactory.eINSTANCE.createReportSituation();
                reportSituation.setReasoningScope("INTERNAL");
                reportSituation.setReportCategory("LOG");

                Situation situation = EventFactory.eINSTANCE.createSituation();
                situation.setCategoryName(situationTypeAttributeValue);
                situation.setSituationType(reportSituation);

                commonBaseEvent.setSituation(situation);
            }
        }
        else {

            /* setup the pointers and our initial work area */
            masterIndex = xmlString.indexOf("<situation");
            tempString = xmlString.substring(masterIndex, xmlString.indexOf("</situation>", masterIndex));
            masterIndex = xmlString.indexOf("</situation>", masterIndex) + 12; //12 is the length of "</situation>".

            Situation situation = EventFactory.eINSTANCE.createSituation();

            //Find the categoryName attribute:
            index = tempString.indexOf(" categoryName=\"");

            if (index != -1) {
                situation.setCategoryName(XmlUtility.denormalize(tempString.substring(index + 15, tempString.indexOf("\"", index + 15)))); //15 is the length of " categoryName=\"".
            }

            //Check for the SituationType element:
            index = tempString.indexOf("<situationType ");

            if (index != -1) {

                int endIndex = tempString.indexOf("</situationType>", index);

                if (endIndex == -1) {
                    endIndex = tempString.indexOf("/>", index);
                }

                tempString = tempString.substring(index, endIndex);

                //Find the xsi:type attribute:
                index = tempString.indexOf(" xsi:type=\""); //Start from the beginning of the new temporary string.

                if (index != -1) {

                    String situationType = (XmlUtility.denormalize(tempString.substring(index + 11, tempString.indexOf("\"", index + 11)))); //11 is the length of " xsi:type=\"".
                    String reasoningScope = null;

                    //Find the reasoningScope attribute:
                    index = tempString.indexOf(" reasoningScope=\""); //Start from the beginning of the new temporary string.

                    if (index != -1) {
                        reasoningScope = (XmlUtility.denormalize(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)))); //17 is the length of " reasoningScope=\"".
                    }

                    if (situationType.equals("StartSituation")) {

                        StartSituation startSituation = EventFactory.eINSTANCE.createStartSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            startSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            startSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        startSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(startSituation);
                    }
                    else if (situationType.equals("StopSituation")) {

                        StopSituation stopSituation = EventFactory.eINSTANCE.createStopSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            stopSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            stopSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        stopSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(stopSituation);
                    }
                    else if (situationType.equals("ConnectSituation")) {

                        ConnectSituation connectSituation = EventFactory.eINSTANCE.createConnectSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            connectSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationDisposition=\"");

                        if (index != -1)
                            connectSituation.setSituationDisposition(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " situationDisposition=\"".

                        connectSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(connectSituation);
                    }
                    else if (situationType.equals("ReportSituation")) {

                        ReportSituation reportSituation = EventFactory.eINSTANCE.createReportSituation();

                        index = tempString.indexOf(" reportCategory=\"");

                        if (index != -1)
                            reportSituation.setReportCategory(XmlUtility.denormalize(tempString.substring(index + 17, tempString.indexOf("\"", index + 17)))); //17 is the length of " reportCategory=\"".

                        reportSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(reportSituation);
                    }
                    else if (situationType.equals("FeatureSituation")) {

                        FeatureSituation featureSituation = EventFactory.eINSTANCE.createFeatureSituation();

                        index = tempString.indexOf(" featureDisposition=\"");

                        if (index != -1)
                            featureSituation.setFeatureDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " featureDisposition=\"".

                        featureSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(featureSituation);
                    }
                    else if (situationType.equals("ConfigureSituation")) {

                        ConfigureSituation configureSituation = EventFactory.eINSTANCE.createConfigureSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            configureSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        configureSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(configureSituation);
                    }
                    else if (situationType.equals("DependencySituation")) {

                        DependencySituation dependencySituation = EventFactory.eINSTANCE.createDependencySituation();

                        index = tempString.indexOf(" dependencyDisposition=\"");

                        if (index != -1)
                            dependencySituation.setDependencyDisposition(XmlUtility.denormalize(tempString.substring(index + 24, tempString.indexOf("\"", index + 24)))); //24 is the length of " dependencyDisposition=\"".

                        dependencySituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(dependencySituation);
                    }
                    else if (situationType.equals("CreateSituation")) {

                        CreateSituation createSituation = EventFactory.eINSTANCE.createCreateSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            createSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        createSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(createSituation);
                    }
                    else if (situationType.equals("DestroySituation")) {

                        DestroySituation destroySituation = EventFactory.eINSTANCE.createDestroySituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            destroySituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        destroySituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(destroySituation);
                    }
                    else if (situationType.equals("AvailableSituation")) {

                        AvailableSituation availableSituation = EventFactory.eINSTANCE.createAvailableSituation();

                        index = tempString.indexOf(" operationDisposition=\"");

                        if (index != -1)
                            availableSituation.setOperationDisposition(XmlUtility.denormalize(tempString.substring(index + 23, tempString.indexOf("\"", index + 23)))); //23 is the length of " operationDisposition=\"".

                        index = tempString.indexOf(" processingDisposition=\"");

                        if (index != -1)
                            availableSituation.setProcessingDisposition(XmlUtility.denormalize(tempString.substring(index + 24, tempString.indexOf("\"", index + 24)))); //24 is the length of " processingDisposition=\"".

                        index = tempString.indexOf(" availabilityDisposition=\"");

                        if (index != -1)
                            availableSituation.setAvailabilityDisposition(XmlUtility.denormalize(tempString.substring(index + 26, tempString.indexOf("\"", index + 26)))); //26 is the length of " availabilityDisposition=\"".

                        availableSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(availableSituation);
                    }
                    else if (situationType.equals("RequestSituation")) {

                        RequestSituation requestSituation = EventFactory.eINSTANCE.createRequestSituation();

                        index = tempString.indexOf(" successDisposition=\"");

                        if (index != -1)
                            requestSituation.setSuccessDisposition(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " successDisposition=\"".

                        index = tempString.indexOf(" situationQualifier=\"");

                        if (index != -1)
                            requestSituation.setSituationQualifier(XmlUtility.denormalize(tempString.substring(index + 21, tempString.indexOf("\"", index + 21)))); //21 is the length of " situationQualifier=\"".

                        requestSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(requestSituation);
                    }
                    else if (situationType.equals("OtherSituation")) {

                        OtherSituation otherSituation = EventFactory.eINSTANCE.createOtherSituation();

                        String anyData = tempString.substring((tempString.indexOf(">") + 1));

                        if ((anyData.length() > 0)) {
                            otherSituation.setAny(anyData);
                        }

                        otherSituation.setReasoningScope(reasoningScope);
                        situation.setSituationType(otherSituation);
                    }
                }
            }

            commonBaseEvent.setSituation(situation);
        }

        String[] otherData = getXMLFragments(xmlString.substring(masterIndex, xmlString.indexOf("</CommonBaseEvent>", masterIndex)));

        if ((otherData != null) && (otherData.length > 0)) {

            EList anyElements = commonBaseEvent.getAny();

            for (int counter = 0; counter < otherData.length; counter++) {
                anyElements.add(otherData[counter]);
            }
        }
    }

    /**
     * Converts an XML document to an CommonBaseEvent object.  The format of the 
     * XML document is based on the parameter CommonBaseEvent's version.
     * 
     * @param commonBaseEvent The CommonBaseEvent to be populated by the XML document.
     * @param xmlString The XML document to populate the CommonBaseEvent object.
     * @since 1.0
     */
    public static synchronized void fromCanonicalXMLDocString(CommonBaseEvent event, String aXMLDocString) {
        fromCanonicalXMLString(event, aXMLDocString);
    }

    private static ExtendedDataElement loadExtendedDataElement(String xml) {

        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();

        //Retrieve and set the 'name' property of the ExtendedDataElement:
        int index = xml.indexOf(" name=\"");

        if (index != -1) {
            extendedDataElement.setName(XmlUtility.denormalize(xml.substring(index + 7, xml.indexOf("\"", index + 7)))); //7 is the length of " name=\"".
        }

        //Retrieve and set the 'type' property of the ExtendedDataElement:
        index = xml.indexOf(" type=\"");

        if (index != -1) {
            extendedDataElement.setType(XmlUtility.denormalize(xml.substring(index + 7, xml.indexOf("\"", index + 7)))); //7 is the length of " type=\"".
        }

        String[] childFragments = getChildXMLFragments(xml);

        if (childFragments != null) {

            ArrayList values = new ArrayList();

            for (int counter = 0; counter < childFragments.length; counter++) {

                //Process each <values> element(s):
                if (childFragments[counter].startsWith("<values>")) {
                    values.add(XmlUtility.denormalize(childFragments[counter].substring(8, childFragments[counter].indexOf("</values>")))); //8 is the length of "<values>".
                }
                else if (childFragments[counter].startsWith("<hexValue>")) {
                    extendedDataElement.setHexValue(XmlUtility.denormalize(childFragments[counter].substring(10, childFragments[counter].indexOf("</hexValue>")))); //10 is the length of "<hexValue>".
                }
                else if (childFragments[counter].startsWith("<children ")) {
                    extendedDataElement.addChild(loadExtendedDataElement(childFragments[counter]));
                }
            }

            //Retrieve and set the 'values' properties of the ExtendedDataElement:
            int valuesSize = values.size();

            if (valuesSize == 1) {
                extendedDataElement.setValuesAsString(String.valueOf(values.get(0)));
            }
            else if (valuesSize > 1) {
                extendedDataElement.setValues(((String[]) (values.toArray(new String[valuesSize]))));
            }
        }

        return extendedDataElement;
    }

    /**
     * Separates all the child XML fragments in the parameter XML fragment and returns the child 
     * fragments as a String array.  Child non-fragments (e.g. PCDATA), if any, are concatenated and 
     * returned in the last index of the String array. 
     * 
     * @param xmlFragment The root level XML fragment containing child elements. 
     * @return The resultant String array of separated child XML fragments.
     */
    private static String[] getChildXMLFragments(String xmlFragment) {

        int firstGreaterThanIndex = xmlFragment.indexOf('>');
        int lastLessThanIndex = xmlFragment.lastIndexOf('<');

        if ((firstGreaterThanIndex != -1) && (lastLessThanIndex != -1) && (firstGreaterThanIndex < lastLessThanIndex)) {
            return (getXMLFragments(xmlFragment.substring((firstGreaterThanIndex + 1), lastLessThanIndex)));
        }

        return null;
    }

    /**
     * Separates all the root level XML fragments in the parameter XML fragment(s) and returns the 
     * fragments as a String array.  Non-fragments (e.g. PCDATA), if any, are concatenated and 
     * returned in the last index of the String array. 
     * 
     * @param xmlFragments The root level XML fragments to be separated. 
     * @return The resultant String array of separated XML fragments.
     */
    private static String[] getXMLFragments(String xmlFragments) {

        if ((xmlFragments != null) && (xmlFragments.trim().length() > 0)) {

            //String array of separated XML fragment(s) and/or PCDATA:
            ArrayList xmlFragmentsArray = new ArrayList();

            //PCDATA buffer:
            StringBuffer pcdata = new StringBuffer();

            //Properties for locating complete XML fragments in parameter XML fragment(s):
            char previousChar = 0;
            char currentChar = 0;
            char currentXMLMarkupChar = 0;
            int xmlMarkupCharsCount = 0;
            int startTagIndex = -1;

            //NOTE:  XML fragments are determined by maintaining a count of the XML markup characters (e.g. <, > and /).  For every '<' character, the count 
            //in incremented.  For every '/>' set of characters, the count is decremented.  Finally, for every '</' set of characters, the count is decremented 
            //by 2.  The count is checked for every '>' character to determine if it is the end of a complete XML fragment (e.g. count is 0).

            //Check the parameter XML fragment(s) for a complete XML fragment(s) character by character while persisting the previous character and XML markup character: 						
            for (int counter = 0; counter < xmlFragments.length(); counter++) {

                //Retrive the current character:
                currentChar = xmlFragments.charAt(counter);

                //If the current character is '<', increment the count of XML markup characters.  This is the start of a complete XML fragment, if one has not been previously located:   
                if (currentChar == '<') {

                    if (startTagIndex == -1)
                        startTagIndex = counter;

                    xmlMarkupCharsCount++;
                }

                //If the current character is '/', decrement the count of XML markup characters by 2 if the previous character is '<'.  This is a close tag and the end of a nested level, if the previous character is '<':
                else if (currentChar == '/') {

                    if (previousChar == '<')
                        xmlMarkupCharsCount -= 2;
                }

                //If the current character is '>', decrement the count of XML markup characters if the previous character is '/'.  This is a open tag and the start of a nested level, if the previous character is '<', otherwise a complete XML fragment if the start of the complete XML fragment has been located and no nested levels:
                else if (currentChar == '>') {

                    if (previousChar == '/')
                        xmlMarkupCharsCount--;

                    if ((startTagIndex != -1) && (xmlMarkupCharsCount == 0)) {

                        //Add the complete XML fragment to the String array of separated XML fragment(s) and/or PCDATA:
                        xmlFragmentsArray.add(xmlFragments.substring(startTagIndex, (counter + 1)).trim());

                        startTagIndex = -1;
                    }
                }

                //Otherwise the current character is PCDATA and is added to the PCDATA buffer:
                else if (startTagIndex == -1)
                    pcdata.append(currentChar);

                //Presist the previous non-whitespace character:
                if (!Character.isWhitespace(currentChar))
                    previousChar = currentChar;
            }

            //If any PCADTA was found, add it to the last index of the String array of separated XML fragments:
            if (pcdata.length() > 0) {
                xmlFragmentsArray.add(pcdata.toString().trim());
            }

            if (xmlFragmentsArray.size() > 0) {
                //Return the String array of separated XML fragments:
                return (((String[]) (xmlFragmentsArray.toArray(new String[xmlFragmentsArray.size()]))));
            }
        }

        return null;
    }
}
