/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.actions.widgets;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFilterProvider;
import org.eclipse.hyades.uml2sd.ui.view.SDView;
import org.eclipse.hyades.uml2sd.util.ContextIDs;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.help.WorkbenchHelp;

/**
 * This is the filters list dialog.<br>
 * It is associated to an SDView and to a ISDFilterProvider.<br> 
 */
public class FilterListDialog extends Dialog {
	
	/**
	 * viewer and provided are kept here as attributes
	 */
	private IViewPart viewer = null;	
	private ISDFilterProvider provider = null;
	
	/**
	 * filters are the result of editing this list
	 */
	private ArrayList filters;
	
	/**
	 * add, remove and edit buttons
	 */
	private Button add, remove, edit;

	/**
	 * table
	 */
	private Table table;
	
	/**
	 * A class to map TableItems that can be toggled active or inactive and Criterias
	 */
	private class CriteriaTableItem {
		
		private Criteria criteria;
		private TableItem tableItem;
		
		public CriteriaTableItem(Table parent, boolean checked_) {
			tableItem = new TableItem(parent, SWT.NONE);
			tableItem.setData(this);
			tableItem.setChecked(checked_);
		}
		public CriteriaTableItem(Table parent, boolean checked_, int index) {
			tableItem = new TableItem(parent, SWT.NONE, index);
			tableItem.setChecked(checked_);
		}
		public void setCriteria(Criteria criteria_) {
			criteria = criteria_;
			tableItem.setText(criteria.getExpression());
		}
		public Criteria getCriteria() {
			return criteria;
		}
	}
	
	/**
	 * @param c
	 * @param checked
	 */
	private void addCriteria(Criteria c, boolean checked) {
		CriteriaTableItem cti = new CriteriaTableItem(table, checked);
		cti.setCriteria(c);
	}
 
	/**
	 * @param new_
	 */
	private void replaceSelectedCriteria(Criteria new_) {
		CriteriaTableItem cti = (CriteriaTableItem)table.getSelection()[0].getData();
		cti.setCriteria(new_);
	}
 
	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	public Control createDialogArea(Composite parent) {
		
		Group ret = new Group(parent, SWT.NONE);
		ret.setText(SDViewerPlugin.getResourceString("FILTER_LIST_GROUP_TITLE")); //$NON-NLS-1$
		WorkbenchHelp.setHelp(ret,SDViewerPlugin.getId()+ContextIDs.UML2SDFILTER);
		RowLayout rowLayout = new RowLayout();
		rowLayout.wrap = false;
		rowLayout.pack = true;
		rowLayout.justify = false;
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.marginLeft = 4;
		rowLayout.marginTop = 4;
		rowLayout.marginRight = 4;
		rowLayout.marginBottom = 4;
		rowLayout.spacing = 8;
		ret.setLayout(rowLayout);

		table = new Table(ret, SWT.MULTI | SWT.CHECK);
		table.setLayoutData(new RowData(180, 84));
		table.setHeaderVisible(false);
		table.addSelectionListener(new SelectionListener() {
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetDefaultSelected(SelectionEvent e) {
				int count = table.getSelectionCount();
				if (count == 1) {
					Criteria c = openFilterDialog(((CriteriaTableItem)table.getSelection()[0].getData()).getCriteria(),
							SDViewerPlugin.getResourceString("FILTER_LIST_UPDATE")); //$NON-NLS-1$
					if (c != null) {
						replaceSelectedCriteria(c);
					}
				}
			}
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				int count = ((Table)e.getSource()).getSelectionCount();
				edit.setEnabled(count == 1);
				remove.setEnabled(count > 0);
			}
		});
		if (filters != null) {
			for (Iterator i = filters.iterator(); i.hasNext(); ) {
				FilterCriteria tf = (FilterCriteria)i.next();
				addCriteria(tf.getCriteria(), tf.isActive());
			}
		}

		Composite commands = new Composite(ret, SWT.NONE);
		RowLayout rowLayoutCommands = new RowLayout();
		rowLayoutCommands.wrap = false;
		rowLayoutCommands.pack = false;
		rowLayoutCommands.justify = true;
		rowLayoutCommands.type = SWT.VERTICAL;
		rowLayoutCommands.marginLeft = 0;
		rowLayoutCommands.marginTop = 4;
		rowLayoutCommands.marginRight = 0;
		rowLayoutCommands.marginBottom = 4;
		rowLayoutCommands.spacing = 8;
		commands.setLayout(rowLayoutCommands);
		add = new Button(commands, SWT.NONE);
		add.setText(SDViewerPlugin.getResourceString("FILTER_LIST_ADD")); //$NON-NLS-1$
		add.addSelectionListener(new SelectionListener() {
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetDefaultSelected(SelectionEvent e) {
				// Nothing to do
			}
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				Criteria init = new Criteria();
				Criteria c = openFilterDialog(init, SDViewerPlugin.getResourceString("FILTER_LIST_CREATE")); //$NON-NLS-1$
				if (c != null) {
					addCriteria(c, true);
				}
			}
		});
		
		edit = new Button(commands, SWT.NONE);
		edit.setText(SDViewerPlugin.getResourceString("FILTER_LIST_EDIT")); //$NON-NLS-1$
		edit.addSelectionListener(new SelectionListener() {
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetDefaultSelected(SelectionEvent e) {
				// Nothing to do
			}
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				Criteria c = openFilterDialog(((CriteriaTableItem)table.getSelection()[0].getData()).getCriteria(),
								SDViewerPlugin.getResourceString("FILTER_LIST_UPDATE")); //$NON-NLS-1$
				if (c != null) {
					replaceSelectedCriteria(c);
				}
			}
		});
		edit.setEnabled(false);
		
		remove = new Button(commands, SWT.NONE);
		remove.setText(SDViewerPlugin.getResourceString("FILTER_LIST_REMOVE")); //$NON-NLS-1$
		remove.addSelectionListener(new SelectionListener() {
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetDefaultSelected(SelectionEvent e) {
				// Nothing to do
			}
			/**
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				table.remove(table.getSelectionIndices());
			}
		});
		remove.setEnabled(false);
		
		getShell().setText(SDViewerPlugin.getResourceString("FILTER_LIST_TITLE")); //$NON-NLS-1$
		return ret;
	}
	
	/**
	 * @param view_
	 * @param loader_
	 */
	public FilterListDialog (IViewPart view_, ISDFilterProvider loader_) {
		super (view_.getSite().getShell());
		viewer = view_;
		provider = loader_;
		filters = provider.getCurrentFilters();
		setShellStyle(SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
	}
	
	/**
	 * @param criteria
	 * @param action between "Update" and "Create"
	 * @return the criteria that has been updated or created
	 */
	private Criteria openFilterDialog(Criteria criteria, String action) {
		SearchFilterDialog filter = new SearchFilterDialog((SDView)viewer, provider);
		filter.setCriteria(criteria);
		filter.setOkText(action);
		filter.setTitle(SDViewerPlugin.getResourceString("FILTER_LIST_CRITERIA_DIALOG")); //$NON-NLS-1$
		filter.open();
		return filter.getCriteria();
	}
	
	/**
	 * Open the dialog box
	 */
	public int open() {
		create();
		getShell().pack();
		getShell().setLocation(getShell().getDisplay().getCursorLocation());
		return super.open();
	}
	
	/**
	 * Called when the dialog box ok button is pressed
	 */
	public void okPressed()	{
		if (table.getItemCount() > 0) {
			filters = new ArrayList();
		} else {
			filters = null;
		}
		for (int i = 0; i < table.getItemCount(); i++) {
			TableItem item = table.getItem(i);
			filters.add(new FilterCriteria(((CriteriaTableItem)item.getData()).getCriteria(), item.getChecked()));
		}
		super.close();
		provider.filter(filters);
	}
	
	/**
	 * @param filters_
	 */
	public void setFilters(ArrayList filters_) {
		filters = filters_;
	}

	/**
	 * @return the filters list after editing
	 */
	public ArrayList getFilters() {
		return filters;
	}

}

