/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.drawings.IColor;
import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.util.SortAsyncForBackward;
import org.eclipse.hyades.uml2sd.util.SortAsyncMessageComparator;
import org.eclipse.hyades.uml2sd.util.SortSyncMessageComparator;
import org.eclipse.hyades.uml2sd.util.TimeEventComparator;

/**
 * The Frame class is the base sequence diagram graph nodes container. (Also called fragment in UML 2.0)<br>
 * For instance, only one frame can be drawn in the View.<br>
 * Lifelines, Messages and Stop which are supposed to represent a Sequence diagram are
 * drawn in a Frame.<br>
 * Only the graph node added to their representing list will be drawn.
 * 
 * The lifelines are appended along the X axsis when added in a frame.<br>
 * The syncMessages are ordered along the Y axsis depending on the event occurrence they are attached to.<br>
 * 
 * @see org.eclipse.hyades.uml2sd.ui.core.Lifeline Lifeline for more event occurence details
 * @author sveyrier
 * @version 1.0
 */
public class Frame extends GraphNode{
	
	/**
	 * Contains the max enlapsed time between two consecutive messages in the whole frame
	 */
	protected double maxTime = -50;
	/**
	 * Contains the min enlapsed time between two consecutive messages in the whole frame
	 */
	protected double minTime = -50;
	
	/**
	 * Indicate if the min and max enlapsed time between two consecutive 
	 * messages in the whole frame need to be computed
	 */
	private boolean computeMinMax = true;
	/**
	 * The lifelines list contained in the frame
	 */
	protected List lifeLines;
	/**
	 * The synchronous message list contained in the frame
	 */
	protected List syncMessages;
	/**
	 * The synchronous message return list contained in the frame
	 */
	protected List syncMessagesReturn;
	/**
	 * The asynchronous message list contained in the frame
	 */
	protected List asyncMessages;
	/**
	 * The asynchronous message return list contained in the frame
	 */
	protected List asyncMessagesReturn;
	
	protected List asyncMessagesRetForBackward;
	protected List asyncMessagesRetForForward;
	
	protected List asyncMessagesForBackward;
	protected List asyncMessagesForForward;
	
	/**
	 * The greater event occurence created 
	 * on graph nodes drawn in this Frame
	 * This directly impact the Frame height
	 */
	protected int verticalIndex = 0;

	/**
	 * The index along the x axis where the next lifeline will is drawn
	 * This directly impact the Frame width
	 */
	protected int horizontalIndex = 0;
	
	/**
	 * Point the first visible lifeline in the 
	 * lifeline list
	 */
	private int	lifeLineDrawIndex = 0;
	/**
	 * Point the first visible synchronous message
	 * in the syncMessages list
	 */
	private int messageDrawIndex = 0;
	/**
	 * Point the first visible synchronous message return
	 * in the syncMessages return list
	 */
	private int messageReturnDrawIndex = 0;
	
	/**
	 * Point the first visible asynchronous message
	 * in the syncMessages list
	 */
	private int asyncMessageDrawIndex = 0;
	
	/**
	 * Point the first visible asynchronous message return
	 * in the syncMessages return list
	 */
	private int asyncMessageReturnDrawIndex = 0;
	
	
	protected LifelineCategories[] lifelineCategories =null;
	
	/**
	 * Used know if synchnous syncMessages list need to be ordered
	 * before drawing
	 */
	protected boolean sortSyncMessage	= false;
	
	/**
	 * Used know if synchnous syncMessages return list need to be ordered
	 * before drawing
	 */
	

	protected boolean sortSyncMessageReturn	= false;
	
	/**
	 * Used know if asynchnous asyncMessages list need to be ordered
	 * before drawing
	 */
	protected boolean sortAsyncMessage	= false;
	
	/**
	 * Used know if asynchnous asyncMessages return list need to be ordered
	 * before drawing
	 */
	
	protected boolean sortAsyncMessageReturn	= false;
	
	protected boolean sortBackwardAsyncList = false;
	protected boolean  sortBackwardAsyncRetList = false;
	
	protected Lifeline highlightLifeline = null;
	protected int startEvent = 0;
	protected int nbEvent = 0;
	protected IColor highlightColor = null;
	protected boolean timeInfo = false;
	
	protected List executionOccurrencesWithTime;
	
	
	/**
	 * The current Frame visible area
	 */
	private int visibleAreaX;
	private int visibleAreaY;
	private int visibleAreaWidth;
	private int visibleAreaHeight;
	
	protected static ISDPreferences userPref = null;
	
	protected GraphNode focusNode = null;
	
	protected String unitName = null;
	
	protected int forceEventOccurrenceSpacing = -1;
	

	
	/**
	 * Creates an empty frame.
	 *
	 */
	public Frame()
	{
		lifeLines 			= new ArrayList();
		syncMessages 		= new ArrayList();
		syncMessagesReturn 	= new ArrayList();
		asyncMessagesForBackward = new ArrayList();
		asyncMessagesForForward = new ArrayList();
		asyncMessages = asyncMessagesForForward;
		
		asyncMessagesRetForBackward = new ArrayList();
		asyncMessagesRetForForward = new ArrayList();
		asyncMessagesReturn = asyncMessagesRetForForward;
		Metrics.forcedEventSpacing = forceEventOccurrenceSpacing;
	}
	
	/**
	 * Returns a list of all lifelines known by this frame.
	 * Known lifelines are the only one which can be displayed on screen.
	 * @return the lifelines list
	 */
	protected List getLifelines()
	{
		return lifeLines;
	}
	
	/**
	 * Returns an array of all lifelines known by this frame.
	 * This table can be manipulated by the caller, the order is kept.
	 * @return the lifelines array
	 * @deprecated
	 */
	public Object[] getLifelinesArray()
	{
		return lifeLines.toArray();
	}
	
	/**
	 * Returns the number of lifelines stored in the frame
	 * @return the number of lifelines
	 */
	public int lifeLinesCount()
	{
		if (lifeLines != null)
			return lifeLines.size();
		else return 0;
	}
	
	/**
	 * Returns the lifeline at the given index in the lifelines array
	 * @param index the position in the lifeline array
	 * @return the lifeline
	 */
	public Lifeline getLifeline(int index)
	{
		if ((lifeLines !=null)&&(index>=0)&&(index<lifeLines.size()))
			return (Lifeline)lifeLines.get(index);
		return null;
	}
	
	/**
	 * Returns a list of syncMessages known by this frame.
	 * Known syncMessages are the only on which can be displayed on screen
	 * @return the syncMessages list
	 */
	protected List getSyncMessages()
	{
		return syncMessages;
	}
	
	/**
	 * Returns an array of all sync messages known by this frame.
	 * This table can be manipulated by the caller, the order is kept.
	 * @return the sync messages array
	 * @deprecated
	 */
	public Object[] getSyncMessagesArray()
	{
		return syncMessages.toArray();
	}
	
	/**
	 * Returns the number of syncMessages stored in the frame
	 * @return the number of syncMessage
	 */
	public int syncMessageCount()
	{
		if (syncMessages != null)
			return syncMessages.size();
		else return 0;
	}
	
	/**
	 * Returns the syncMessage at the given index in the syncMessages array
	 * @param index the position in the syncMessages array
	 * @return the syncMessage
	 */
	public SyncMessage getSyncMessage(int index)
	{
		if ((syncMessages !=null)&&(index>=0)&&(index<syncMessages.size()))
			return (SyncMessage)syncMessages.get(index);
		return null;
	}
	
	/**
	 * Returns the number of asyncMessage stored in the frame
	 * @return the number of asyncMessage
	 */
	public int asyncMessageCount()
	{
		if (asyncMessages != null)
			return asyncMessages.size();
		else return 0;
	}
	
	/**
	 * Returns the asyncMessage at the given index in the asyncMessage array
	 * @param index the position in the asyncMessage array
	 * @return the asyncMessage
	 */
	public AsyncMessage getAsyncMessage(int index)
	{
		if ((asyncMessages !=null)&&(index>=0)&&(index<asyncMessages.size()))
			return (AsyncMessage)asyncMessages.get(index);
		return null;
	}
	
	/**
	 * Reset the internal index of the first visible graph node for
	 * each ordered graph node lists (lifelines, and
	 * synchronlous message lists)
	 *
	 */
	public void resetIndex()
	{
		lifeLineDrawIndex 		= 0;
		messageDrawIndex 		= 0;
		messageReturnDrawIndex 	= 0;
	}

	/**
	 * Set the lifeline categories which will be use during the lifelines creation
	 * @see Lifeline#setCategory(int) 
	 * @param categories the lifeline categories array
	 */	
	public void setLifelineCategories(LifelineCategories[] categories)
	{
		lifelineCategories = categories;
	}

	/**
	 * Returns the lifeline categories array set for the this frame
	 * @return the lifeline categories array or null if not set
	 */	
	public LifelineCategories[] getLifelineCategories()
	{
		return lifelineCategories;
	}
	

	/**
	 * Returns a list of syncMessages return known by this frame.
	 * Known syncMessages return are the only on which can be displayed on screen
	 * @return the syncMessages return list
	 */	
	protected List getSyncMessagesReturn()
	{
		return syncMessagesReturn;
	}
	
	/**
	 * Returns an array of all sync messages return known by this frame.
	 * This table can be manipulated by the caller, the order is kept.
	 * @return the sync messages return array
	 * @deprecated
	 */
	public Object[] getSyncMessagesReturnArray()
	{
		return syncMessagesReturn.toArray();
	}
	
	/**
	 * Returns the number of syncMessageReturn stored in the frame
	 * @return the number of syncMessageReturn
	 */
	public int syncMessageReturnCount()
	{
		if (syncMessagesReturn != null)
			return syncMessagesReturn.size();
		else return 0;
	}
	
	/**
	 * Returns the syncMessageReturn at the given index in the syncMessageReturn array
	 * @param index the position in the syncMessageReturn array
	 * @return the syncMessageReturn
	 */
	public SyncMessageReturn getSyncMessageReturn(int index)
	{
		if ((syncMessagesReturn !=null)&&(index>=0)&&(index<syncMessagesReturn.size()))
			return (SyncMessageReturn)syncMessagesReturn.get(index);
		return null;
	}
	
	/**
	 * Returns the number of asyncMessageReturn stored in the frame
	 * @return the number of asyncMessageReturn
	 */
	public int asyncMessageReturnCount()
	{
		if (asyncMessagesReturn != null)
			return asyncMessagesReturn.size();
		else return 0;
	}
	
	/**
	 * Returns the asyncMessageReturn at the given index in the asyncMessageReturn array
	 * @param index the position in the asyncMessageReturn array
	 * @return the asyncMessageReturn
	 */
	public AsyncMessageReturn getAsyncMessageReturn(int index)
	{
		if ((asyncMessagesReturn !=null)&&(index>=0)&&(index<asyncMessagesReturn.size()))
			return (AsyncMessageReturn)asyncMessagesReturn.get(index);
		return null;
	}

	/**
	 * 
	 * Returns the greater event occurence 
	 * known by the Frame
	 * @return the greater event occurrence
	 */	
	protected int getMaxEventOccurrence()
	{
		return verticalIndex;
	}
	
	/**
	 * Set the greater event occurrence created
	 * in graph nodes included in the frame
	 * @param eventOccurrence the new greater event occurrence
	 */
	protected void setMaxEventOccurrence(int eventOccurrence)
	{
			verticalIndex=eventOccurrence;
	}
	
	/**
	 * This method increase the lifeline place holder
	 * The return value is usually assign to a lifeline. This can be used to set the lifelines drawing order.
	 * Also, calling this method two times and assigning only the last given index to a lifeline will increase
	 * this lifeline draw spacing (2 times the default spacing) from the last added lifeline.
	 * @return a new lifeline index
	 */
	protected int getNewHorizontalIndex ()
	{
		return ++horizontalIndex;
	}
	
	/**
	 * Returns the current horizontal index
	 * @return the current horizontal index
	 * @see Frame#getNewHorizontalIndex() for horizontal index description
	 */
	protected int getHorizontalIndex ()
	{
		return horizontalIndex;
	}
	
	/**
	 * Adds a lifeline to the frame lifelines list.
	 * The lifeline X drawing order depends on the lifeline addition order into the frame lifelines list.
	 * @param the lifeline to add
	 */
	public void addLifeLine (Lifeline lifeLine)
	{	computeMinMax=true;
		if (lifeLine == null)
			return; 
		//set the lifeline parent frame
		lifeLine.setFrame(this);
		//Increate the frame lifeline counter
		//and set the lifeline drawing order
		lifeLine.setIndex(getNewHorizontalIndex());
		if (lifeLine.hasTimeInfo()) {
			timeInfo = true;
		}
		//add the lifeline to the lifelines list
		lifeLines.add(lifeLine);
	}
	
	/**
	 * Adds a message to the Frame message list.
	 * Four kinds of syncMessages can be added:<br>
	 * 		- synchronous syncMessages<br>
	 * 		- synchronous syncMessages return<br>
	 * 		- asynchonous syncMessages<br>
	 * 		- asynchronous syncMessages return<br>
	 * For drawing performance reason, it is recommanded to add synchronous syncMessages in the same
	 * order they should appear along the Y axis in the Frame.
	 * @param the message to add
	 */
	public void addMessage (BaseMessage message)
	{
		computeMinMax=true;
		//No Message to add
		if (message == null)
			return; 
			
		//Depending on the message kind, we add it to the the right list
		if (message instanceof SyncMessageReturn)
		{
			if (syncMessagesReturn.size()>0)
			{
				//check if the sync message return are added y ordered
				//if not, tag the list to sort it later (during draw)
				BaseMessage lastMessage = (BaseMessage)syncMessagesReturn.get(syncMessagesReturn.size()-1);
				if (lastMessage.getEventOccurrence()>message.getEventOccurrence())
					sortSyncMessageReturn = true;
			}
			syncMessagesReturn.add(message);
		}
		else if (message instanceof SyncMessage)
		{
			if (syncMessages.size()>0)
			{
				//check if the sync message are added y ordered
				//if not, tag the list to sort it later (during draw)
				BaseMessage lastMessage = (BaseMessage)syncMessages.get(syncMessages.size()-1);
				if (lastMessage.getEventOccurrence()>message.getEventOccurrence())
					sortSyncMessage = true;
			}
			syncMessages.add(message);
		}
		else if (message instanceof AsyncMessageReturn)
			{
				if (asyncMessagesReturn.size()>0)
				{
					AsyncMessage m1 = (AsyncMessage)asyncMessagesReturn.get(asyncMessagesReturn.size()-1);
					AsyncMessage m2 = (AsyncMessage)message;
					int m1Min,m2Min;
					SortAsyncMessageComparator comp = new SortAsyncMessageComparator();
					SortAsyncForBackward backComp = new SortAsyncForBackward();
					if (comp.compare(m1,m2)==1)
						sortAsyncMessageReturn = true;
					if (backComp.compare(m1,m2)==1)
						sortBackwardAsyncRetList = true;
				}
				asyncMessagesRetForForward.add(message);
				asyncMessagesRetForBackward.add(message);
				asyncMessagesReturn = asyncMessagesRetForForward;
			}
			else if (message instanceof AsyncMessage)
			{
				if (asyncMessages.size()>0)
				{
					AsyncMessage m1 = (AsyncMessage)asyncMessages.get(asyncMessages.size()-1);
					AsyncMessage m2 = (AsyncMessage)message;

					SortAsyncMessageComparator comp = new SortAsyncMessageComparator();
					SortAsyncForBackward backComp = new SortAsyncForBackward();
					if (comp.compare(m1,m2)==1)
						sortAsyncMessage = true;
					if (backComp.compare(m1,m2)==1)
						sortBackwardAsyncList = true;
				}
				asyncMessagesForForward.add(message);
				asyncMessagesForBackward.add(message);
				asyncMessages = asyncMessagesForForward;
			}
	}
	
	
	/**
	 * Computes the index of the first visible graph node for
	 * each ordered graph node lists depending on the visible 
	 * area given in parameter
	 *
	 * @param x visible area top left corner x coordinate
	 * @param y visible area top left corner y coordinate
	 * @param width visible area width
	 * @param height visible area height
	 */
	public void updateIndex(int x, int y, int width, int height)
	{
		visibleAreaX = x;
		visibleAreaY = y;
		visibleAreaWidth = width;
		visibleAreaHeight = height;
		
		SortSyncMessageComparator syncComp = new SortSyncMessageComparator();
		SortAsyncMessageComparator asyncComp = new SortAsyncMessageComparator();
		SortAsyncForBackward backComp = new SortAsyncForBackward();
		
		int countLi = 0;
		int countM = 0;
		int countAM = 0;
		int countMR = 0;
		int countAMR = 0;
		int countEO = 0;
		
		int directionX = 1;
		if (x==0)
			lifeLineDrawIndex = 0;
		if ((lifeLines == null)||(lifeLines.size()<1))
			return;
		if(((Lifeline)lifeLines.get(lifeLineDrawIndex)).getX()>x)
			directionX=-1;
		for (int i=lifeLineDrawIndex; i<lifeLines.size()&&i>=0;i=i+directionX)
		{
			countLi++;
			lifeLineDrawIndex = i;
			if (directionX ==1)
			{
				if(((Lifeline)lifeLines.get(i)).getX()>
						x - Metrics.swimmingLaneWidth())
					break;
			}
			else 
			{
				if (((Lifeline)lifeLines.get(i)).getX()<x)
					break;
			}
		}
		
		int directionY = 1;
		if (y==0)
		{
			messageDrawIndex = 0;
			messageReturnDrawIndex = 0;
			asyncMessageDrawIndex = 0;
			asyncMessageReturnDrawIndex = 0;
		}
		if ((syncMessages != null)&&(syncMessages.size()>0))
		{
			if(((BaseMessage)syncMessages.get(messageDrawIndex)).getY()>y)
				directionY = -1;
			for (int i=messageDrawIndex; i<syncMessages.size()&&i>=0;i=i+directionY)
			{
				countM++;
				messageDrawIndex = i;
				SyncMessage toDraw= (SyncMessage)syncMessages.get(i);
				if (i<syncMessages.size()-1)
				{
					SyncMessage next= (SyncMessage)syncMessages.get(i+1);
					if (syncComp.compare(toDraw,next)==1)
						sortSyncMessage = true;
				}
				if (directionY ==1)
				{
					if (((BaseMessage)syncMessages.get(i)).getY()>y)
						break;
				}
				else 
				{
					if (((BaseMessage)syncMessages.get(i)).getY()<y)
						break;
				}
			}
		}
		
		if ((asyncMessagesReturn != null)&&(asyncMessagesReturn.size()>0))
		{
			int mY=((BaseMessage)asyncMessagesReturn.get(asyncMessageReturnDrawIndex)).getY();
			int mH=((BaseMessage)asyncMessagesReturn.get(asyncMessageReturnDrawIndex)).getHeight();
			if ((mY>y)||(mY+mH>y))
				directionY = -1;
			if (asyncMessageReturnDrawIndex==0)
				directionY = 1;
			if (directionY == -1)
			{
				asyncMessageReturnDrawIndex = Arrays.binarySearch(asyncMessagesRetForBackward.toArray(),
						asyncMessagesReturn.get(asyncMessageReturnDrawIndex),
						new SortAsyncForBackward());
				asyncMessagesReturn = asyncMessagesRetForBackward;
			}
			AsyncMessage prev=null;
			for (int i=asyncMessageReturnDrawIndex; i<asyncMessagesReturn.size()&&i>=0;i=i+directionY)
			{
				asyncMessageReturnDrawIndex = i;
				countAMR++;
				AsyncMessage toDraw= (AsyncMessage)asyncMessagesReturn.get(i);
				if (prev==null)
					prev = toDraw;
				if (directionY == 1)
				{
					if (i<asyncMessagesReturn.size()-1)
					{
						AsyncMessage next= (AsyncMessage)asyncMessagesReturn.get(i+1);
						if (asyncComp.compare(toDraw,next)==1)
							sortAsyncMessageReturn = true;
					}
					if ((toDraw.getY()>=y)||(toDraw.getY()+ toDraw.getHeight()>=y))
						break;
				}
				else 
				{
					if (i<asyncMessagesReturn.size()-1)
					{
						AsyncMessage next= (AsyncMessage)asyncMessagesReturn.get(i+1);
						if (backComp.compare(toDraw,next)==1)
							sortBackwardAsyncRetList = true;
					}
					if ((toDraw.getY()<=y)&&(toDraw.getY()+ toDraw.getHeight()<=y))
					{
						if (asyncComp.compare(toDraw,prev)<=0)
							break;
					}
					else if (asyncComp.compare(toDraw,prev)<=0)
						prev = toDraw;
				}
			}
			asyncMessagesReturn = asyncMessagesRetForForward;
			if ((directionY ==-1) && (asyncMessageReturnDrawIndex>0))
			{
					asyncMessageReturnDrawIndex = Arrays.binarySearch(asyncMessagesReturn.toArray(),
						asyncMessagesRetForBackward.get(asyncMessageReturnDrawIndex),
					new SortAsyncMessageComparator());
			}
		}
		if ((asyncMessages != null)&&(asyncMessages.size()>0))
		{
			int mY=((BaseMessage)asyncMessages.get(asyncMessageDrawIndex)).getY();
			int mH=((BaseMessage)asyncMessages.get(asyncMessageDrawIndex)).getHeight();
			if ((mY>y)||(mY+mH>y))
				directionY = -1;
			if (asyncMessageDrawIndex==0)
				directionY = 1;
			if (directionY == -1)
			{
				asyncMessages = asyncMessagesForForward;
				asyncMessageDrawIndex = Arrays.binarySearch(asyncMessagesForBackward.toArray(),
					asyncMessagesForForward.get(asyncMessageDrawIndex),
						new SortAsyncForBackward());
				asyncMessages = asyncMessagesForBackward;
			}
			AsyncMessage prev=null;
			for (int i=asyncMessageDrawIndex; i<asyncMessages.size()&&i>=0;i=i+directionY)
			{
				if (SDViewerPlugin.debugIndex())
				{
					System.out.print("Direction = " + directionY + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
				}
				countAM++;
				asyncMessageDrawIndex = i;
				AsyncMessage toDraw= (AsyncMessage)asyncMessages.get(i);
				if (prev==null)
					prev = toDraw;
				if (directionY == 1)
				{
					AsyncMessage next= null;
					if (i<asyncMessages.size()-1)
					{
						next= (AsyncMessage)asyncMessages.get(i+1);
						if (asyncComp.compare(toDraw,next)==1)
							sortAsyncMessage = true;
					}
					if ((toDraw.getY()>=y)||toDraw.getY()+ toDraw.getHeight()>=y)
						break;
				}
				else 
				{
					if (i<asyncMessages.size()-1)
					{
						AsyncMessage next= (AsyncMessage)asyncMessages.get(i+1);
						if (backComp.compare(toDraw,next)==1)
							sortBackwardAsyncList = true;
					}
					if ((toDraw.getY()<=y)&&(toDraw.getY()+ toDraw.getHeight()<=y))
					{
						if (asyncComp.compare(toDraw,prev)<=0)
							break;
					}
					else if (asyncComp.compare(toDraw,prev)<=0)
						prev = toDraw;
				}
			}
			asyncMessages = asyncMessagesForForward;
			if ((directionY ==-1)&&(asyncMessageDrawIndex >0))
			{
				asyncMessageDrawIndex = Arrays.binarySearch(asyncMessages.toArray(),
						asyncMessagesForBackward.get(asyncMessageDrawIndex),
					new SortAsyncMessageComparator());				
			}
			
		}
		
		if ((syncMessagesReturn != null)&&(syncMessagesReturn.size()>0))
		{
			if(((BaseMessage)syncMessagesReturn.get(messageReturnDrawIndex)).getY()>y)
					directionY = -1;
			for (int i=messageReturnDrawIndex; i<syncMessagesReturn.size()&&i>=0;i=i+directionY)
			{
				countMR++;
				messageReturnDrawIndex = i;
				SyncMessage toDraw= (SyncMessage)syncMessagesReturn.get(i);
				if (i<syncMessagesReturn.size()-1)
				{
					SyncMessage next= (SyncMessage)syncMessagesReturn.get(i+1);
					if (syncComp.compare(toDraw,next)==1)
						sortSyncMessageReturn = true;
				}
				if (directionY ==1)
				{ 
					if (((BaseMessage)syncMessagesReturn.get(i)).getY()>y)
						break;
				}
				else 
				{
					if (((BaseMessage)syncMessagesReturn.get(i)).getY()<y)
						break;
				}
			}
		}

		for (int i=lifeLineDrawIndex; i<lifeLines.size();i++)
		{
			Lifeline toDraw = (Lifeline)lifeLines.get(i);
			toDraw.updateIndex(x, y);
			if (toDraw.getX()> x + width)
				break;
		}
		if (SDViewerPlugin.debugIndex())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
			System.out.print("Visible area position in virtual screen (x,y)= "+x+" "+y+"\n\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			System.out.print("First drawn lifeline index = "+lifeLineDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("First drawn sync message index = "+messageDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("First drawn sync message return index = "+messageReturnDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("First drawn async message index = "+asyncMessageDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("First drawn async return message index = "+asyncMessageReturnDrawIndex+"\n\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("Lifeline index found in = "+countLi+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$		
			System.out.print("Sync Message index found in = "+countM+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$		
			System.out.print("Sync Message Return index found in "+countMR+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$		
			System.out.print("Async Message found in "+countAM+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$		
			System.out.print("Async Message Return found in "+countAMR+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Returns the first visible lifeline drawn in the view
	 * @return the first visible lifeline index
	 */
	public int getFirstVisibleLifeline()
	{
		return lifeLineDrawIndex;
	}
	
	/**
	 * Returns the first visible synchronous message drawn in the view
	 * @return the first visible synchronous message index
	 */
	public int getFirstVisibleSyncMessage()
	{
		return messageDrawIndex;
	}

	/**
	 * Returns the first visible synchronous message return drawn in the view
	 * @return the first visible synchronous message return index
	 */
	 public int getFirstVisibleSyncMessageReturn()
	 {
		 return messageReturnDrawIndex;
	 }
	 
	/**
	 * Returns the first visible synchronous message drawn in the view
	 * @return the first visible synchronous message index
	 */
	public int getFirstVisibleAsyncMessage()
	{
		return asyncMessageDrawIndex;
	}
	
	/**
	 * Returns the first visible synchronous message return drawn in the view
	 * @return the first visible synchronous message return index
	 */
	 public int getFirstVisibleAsyncMessageReturn()
	 {
		 return asyncMessageReturnDrawIndex;
	 }
	
	/**
	 * @return the frame x axis value in the containing view
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getX()
	 */
	public int getX()
	{
		return Metrics.FRAME_H_MARGIN;
	}
	
	/**
	 * @return the frame y axis value in the containing view
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getX()
	 */
	public int getY()
	{
		return Metrics.FRAME_V_MARGIN;
	}

	/**
	 * The frame width depends on the number of lifeline added in the frame
	 * @return the frame width  
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getWidth()
	 */
	public int getWidth()
	{
		if (horizontalIndex == 0)
			return Metrics.swimmingLaneWidth() + Metrics.LIFELINE_H_MAGIN*2- Metrics.FRAME_H_MARGIN -
				Metrics.LIFELINE_SPACING/2;
		else return horizontalIndex * Metrics.swimmingLaneWidth() + Metrics.LIFELINE_H_MAGIN*2 -	Metrics.LIFELINE_SPACING;
	}
	
	/**
	 * The frame height depends on the maximum number of messsages added to a lifeline( Taking all lifelines into account)
	 * @return the frame height  
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#getHeight()
	 */
	public int getHeight()
	{
		return  verticalIndex * (Metrics.getMessagesSpacing() + Metrics.getMessageFontHeigth()) +
				Metrics.LIFELINE_NAME_H_MARGIN +Metrics.FRAME_NAME_H_MARGIN + Metrics.getLifelineFontHeigth() + Metrics.LIFELINE_VT_MAGIN +
				Metrics.LIFELINE_VB_MAGIN + Metrics.LIFELINE_NAME_H_MARGIN + Metrics.FRAME_NAME_H_MARGIN +	Metrics.getLifelineFontHeigth()*2;
	}
	
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * for the given graph node list and starting the iteration at the given index<br>
	 * WARNING: Only graph nodes with smaller coordinates
	 * than the current visible area can be returned.<br>
	 * 
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @param list the list to search in
	 * @param fromIndex list browsing starting point
	 * @return the graph node containing the point given in parameter, null otherwise
	 */ 
	protected GraphNode getNodeFromListAt(int x, int y, List list, int fromIndex)
	{
		if (list==null)
			return null;
		for (int i=fromIndex; i<list.size();i++)
		{
			GraphNode node = (GraphNode)list.get(i);
			//only lifeline list is x ordered
			//Stop browsing the list if the node is outside  the visible area
			//all others nodes will be not visibles			
			if ((node instanceof Lifeline)&&
				(node.getX()>visibleAreaX+visibleAreaWidth))
				break;
			if (node.getHeight()<0)
			{
				if (node.getY()+node.getHeight()>visibleAreaY+visibleAreaHeight)
					break;
			}
			else 
			{
				if (node.getY()>visibleAreaY+visibleAreaHeight)
					break;
			}
			if (node.contains(x, y))
				return node;
		}
		return null;
	}
	
	protected GraphNode getCloserLeavingMessage(Lifeline lifeline,BaseMessage message, List list, boolean smallerEvent)
	{
		if (list==null)
			return null;
		if (smallerEvent == false)
		{
			int event =0;
			if (message != null)
				event = message.getEventOccurrence();
			for (int i=0; i<list.size();i++)
			{
				GraphNode node = (GraphNode)list.get(i);
				if (node instanceof SyncMessage)
				{
					if ((((SyncMessage)node).getEventOccurrence()>=event)&&
						(((SyncMessage)node).getStartLifeline() == lifeline)&&(node!=message))
						return node;
				}
				else if (node instanceof AsyncMessage)
					if ((((AsyncMessage)node).getStartOccurrence()>=event)&&
						(((AsyncMessage)node).getStartLifeline() == lifeline)&&(node!=message))
						return node;
			}
		}
		else 
		{
			int event = getMaxEventOccurrence();
			if (message != null)
				event = message.getEventOccurrence();
			for (int i=list.size()-1; i>=0;i--)
			{
				GraphNode node = (GraphNode)list.get(i);
				if (node instanceof SyncMessage)
				{
					if ((((SyncMessage)node).getEventOccurrence()<event)&&
						(((SyncMessage)node).getStartLifeline() == lifeline)&&(node!=message))
						return node;
				}
				else if (node instanceof AsyncMessage)
					if ((((AsyncMessage)node).getStartOccurrence()<event)&&
						(((AsyncMessage)node).getStartLifeline() == lifeline)&&(node!=message))
						return node;
			}
		}
		return null;
	}
	
	protected GraphNode getCloserEnteringMessage( Lifeline lifeline,BaseMessage message, List list, boolean smallerEvent)
	{
		if (list==null)
			return null;
		if (smallerEvent == false)
		{
			int event =0;
			if (message != null)
				event = message.getEventOccurrence();
			for (int i=0; i<list.size();i++)
			{
				GraphNode node = (GraphNode)list.get(i);
				if (node instanceof SyncMessage)
				{
					if ((((SyncMessage)node).getEventOccurrence()<event)&&
						(((SyncMessage)node).getEndLifeline() == lifeline)&&(node!=message))
						return node;
				}
				else if (node instanceof AsyncMessage)
					if ((((AsyncMessage)node).getStartOccurrence()<event)&&
						(((AsyncMessage)node).getEndLifeline() == lifeline)&&(node!=message))
						return node;
			}
		}
		else
		{
			int event = getMaxEventOccurrence();
			if (message != null)
			{
				event = message.getEventOccurrence();
			}
			for (int i=list.size()-1; i>=0;i--)
			{
				GraphNode node = (GraphNode)list.get(i);
				if (node instanceof SyncMessage)
				{
					if ((((SyncMessage)node).getEventOccurrence()<event)&&
						(((SyncMessage)node).getEndLifeline() == lifeline)&&(node!=message))
						return node;
				}
				else if (node instanceof AsyncMessage)
					if ((((AsyncMessage)node).getStartOccurrence()<event)&&
						(((AsyncMessage)node).getEndLifeline() == lifeline)&&(node!=message))
						return node;
			}
		}
		return null;
	}
	
	private int distanceFromEvent(GraphNode node, int event)
	{
		int distance = 0;
		if (node instanceof SyncMessage)
			distance = ((SyncMessage)node).getEventOccurrence()-event;
		else if (node instanceof AsyncMessage)
		{
			int start = ((AsyncMessage)node).getStartOccurrence();
			int end = ((AsyncMessage)node).getEndOccurrence();
			if ((start-event)<(end-event))
				distance=start-event;
			else distance=start-event;
		}
		return Math.abs(distance);	
	}
	
	private GraphNode getCloserToEvent(GraphNode node1, GraphNode node2, int event)
	{
		if ((node1!=null)&&(node2!=null))
		{
			if (distanceFromEvent(node1,event)<distanceFromEvent(node2,event))
				return node1;
			else return node2;
		}
		else if (node1 != null)
			return node1;
		else if (node2 != null)
			return node2;
		else return null;
	}
	
	public GraphNode getCalledMessage(BaseMessage StartMessage)
	{
		int event = 0;
		GraphNode result = null;
		Lifeline lifeline = null;
		if (StartMessage != null)
		{
			event = ((BaseMessage)StartMessage).getEventOccurrence();
			lifeline = ((BaseMessage)StartMessage).getEndLifeline();
			if (lifeline == null)
				lifeline = ((BaseMessage)StartMessage).getStartLifeline();
		}
		if (lifeline == null)
			return null;
		GraphNode message=getCloserLeavingMessage(lifeline, StartMessage,syncMessages,false);
		GraphNode messageReturn=getCloserLeavingMessage(lifeline,StartMessage,syncMessagesReturn,false);
		result = getCloserToEvent(message,messageReturn, event);
		message=getCloserLeavingMessage(lifeline,StartMessage,asyncMessages,false);
		result = getCloserToEvent(result,message, event);
		messageReturn=getCloserLeavingMessage(lifeline,StartMessage,asyncMessagesReturn,false);
		result = getCloserToEvent(result,messageReturn, event);
		return result;
	}
	
	public GraphNode getCallerMessage(BaseMessage StartMessage)
	{
		int event = getMaxEventOccurrence();
		GraphNode result = null;
		Lifeline lifeline = null;
		if (StartMessage != null)
		{
			event = ((BaseMessage)StartMessage).getEventOccurrence();
			lifeline = ((BaseMessage)StartMessage).getStartLifeline();
			if (lifeline == null)
				lifeline = ((BaseMessage)StartMessage).getEndLifeline();
		}
		if (lifeline == null)
			return null;
		GraphNode message=getCloserEnteringMessage(lifeline,StartMessage,syncMessages,true);
		GraphNode messageReturn=getCloserEnteringMessage(lifeline,StartMessage,syncMessagesReturn,true);
		result = getCloserToEvent(message,messageReturn, event);
		message=getCloserEnteringMessage(lifeline,StartMessage,asyncMessages,true);
		result = getCloserToEvent(result,message, event);
		messageReturn=getCloserEnteringMessage(lifeline,StartMessage,asyncMessagesReturn,true);
		result = getCloserToEvent(result,messageReturn, event);
		return result;
	}
	
	public GraphNode getNextLifelineMessage(Lifeline lifeline, BaseMessage StartMessage)
	{
		int event = 0;
		if (StartMessage != null)
			event = ((BaseMessage)StartMessage).getEventOccurrence();
		if (lifeline == null)
			return null;
		GraphNode message=getCloserLeavingMessage(lifeline, StartMessage,syncMessages,false);
		GraphNode messageReturn=getCloserLeavingMessage(lifeline,StartMessage,syncMessagesReturn,false);
		GraphNode result = getCloserToEvent(message,messageReturn, event);
		message=getCloserLeavingMessage(lifeline,StartMessage,asyncMessages,false);
		result = getCloserToEvent(result,message, event);
		messageReturn=getCloserLeavingMessage(lifeline,StartMessage,asyncMessagesReturn,false);
		result = getCloserToEvent(result,messageReturn, event);
		return result;
	}
	
	public GraphNode getPrevLifelineMessage(Lifeline lifeline, BaseMessage StartMessage)
	{	
		int event = getMaxEventOccurrence();
		if (StartMessage != null)
			event = StartMessage.getEventOccurrence();
		if (lifeline == null)
			return null;
		GraphNode message=getCloserLeavingMessage(lifeline, StartMessage,syncMessages,true);
		GraphNode messageReturn=getCloserLeavingMessage(lifeline,StartMessage,syncMessagesReturn,true);
		GraphNode result = getCloserToEvent(message,messageReturn, event);
		message=getCloserLeavingMessage(lifeline,StartMessage,asyncMessages,true);
		result = getCloserToEvent(result,message, event);
		messageReturn=getCloserLeavingMessage(lifeline,StartMessage,asyncMessagesReturn,true);
		result = getCloserToEvent(result,messageReturn, event);
		return result;
	}
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * WARNING: Only graph nodes in the current visible area can be returned
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @return the graph node containing the point given in parameter, null otherwise
	 */
	public GraphNode getNodeAt(int x, int y)
	{
		//search in the lifelines list starting at the first visible lifeline
		GraphNode node = getNodeFromListAt(x, y, lifeLines, lifeLineDrawIndex);
		if (node != null)
		{
			//return the execution occurrence first if any
			GraphNode exec = getNodeFromListAt(x, y, ((Lifeline)node).executionOccurrences, 0);
			if (exec != null)
				return exec;
			else return node;
		}
		//search in the syncMessages list starting at the first visible syncMessages
		node = getNodeFromListAt(x, y, syncMessages, messageDrawIndex);
		if (node != null)
			return node;
		//search in the syncMessagesReturn list starting at the first visible syncMessagesReturn
		node = getNodeFromListAt(x, y, syncMessagesReturn, messageReturnDrawIndex);
		if (node != null)
			return node;	
		//search in the asyncMessages list starting at from the begining
		//asynch syncMessages are not optimized
		node = getNodeFromListAt(x, y,  asyncMessages, 0);
		if (node != null)
			return node;
		//search in the asyncMessages list starting at from the begining
		//asynch syncMessages are not optimized
	    node = getNodeFromListAt(x, y,  asyncMessagesReturn, 0);
		if (node != null)
			return node;
		return null;
	}


	/**
	 * Draws the Frame on the given context.<br>
	 * This method start to order synchronous message if neeeded.<br>
	 * After, depending on the visible area, only visible graph nodes are drawn.<br>
	 * If the messages are not added in the increasing event occurrence order, they are ordered before drawing
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#draw(IGC)
	 */
	public void draw (IGC context)
	{
		visibleAreaHeight = context.getVisibleHeight();
		visibleAreaWidth = context.getVisibleWidth();
		visibleAreaX= context.getContentsX();
		visibleAreaY= context.getContentsY();
		
		if (forceEventOccurrenceSpacing>=0)
			Metrics.forcedEventSpacing = forceEventOccurrenceSpacing;
		if (userPref == null)
			return;
		//If the Messages have not been added ordered, the array is ordered
		if (sortSyncMessage)
		{
			Object[] temp=syncMessages.toArray();
			Arrays.sort(temp,new SortSyncMessageComparator());
			sortSyncMessage = false;
			syncMessages = Arrays.asList(temp);
			if (SDViewerPlugin.debugSorting())
				System.out.print("Syn messages array sorted\n"); //$NON-NLS-1$
		}
		if (sortSyncMessageReturn)
		{
			Object[] temp=syncMessagesReturn.toArray();
			Arrays.sort(temp,new SortSyncMessageComparator());
			sortSyncMessageReturn = false;
			syncMessagesReturn = Arrays.asList(temp);
			if (SDViewerPlugin.debugSorting())
				System.out.print("Syn messages return array sorted\n"); //$NON-NLS-1$
		}
		if (sortAsyncMessage)
		{
			Object[] temp=asyncMessagesForForward.toArray();
			Arrays.sort(temp,new SortAsyncMessageComparator());
			sortAsyncMessage = false;
			asyncMessagesForForward = Arrays.asList(temp);
			asyncMessages=asyncMessagesForForward;
			if (SDViewerPlugin.debugSorting())
				System.out.print("Asyn messages array sorted\n"); //$NON-NLS-1$
		}
		if (sortBackwardAsyncList)
		{
			Object[] temp=asyncMessagesForBackward.toArray();
			Arrays.sort(temp,new SortAsyncForBackward());
			sortBackwardAsyncList = false;
			asyncMessagesForBackward = Arrays.asList(temp);
			if (SDViewerPlugin.debugSorting())
				System.out.print("Backward asyn messages array sorted\n"); //$NON-NLS-1$
		}
		if (sortAsyncMessageReturn)
		{
			Object[] temp=asyncMessagesRetForForward.toArray();
			Arrays.sort(temp,new SortAsyncMessageComparator());
			sortAsyncMessageReturn = false;
			asyncMessagesRetForForward = Arrays.asList(temp);
			asyncMessagesReturn = asyncMessagesRetForForward;
			if (SDViewerPlugin.debugSorting())
  				System.out.print("Asyn messages return array sorted\n"); //$NON-NLS-1$
		}
		if (sortBackwardAsyncRetList)
		{
			Object[] temp=asyncMessagesRetForBackward.toArray();
			Arrays.sort(temp,new SortAsyncForBackward());
			sortBackwardAsyncRetList = false;
			asyncMessagesRetForBackward = Arrays.asList(temp);
			if (SDViewerPlugin.debugSorting())
				System.out.print("Backward asyn messages return array sorted\n"); //$NON-NLS-1$
		}
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME));

		//Draw the frame main rectangle		
		context.fillRectangle(getX(), getY(), getWidth(), getHeight());
		context.drawRectangle(getX(), getY(), getWidth(), getHeight());
		int nameWidth = context.textExtent(getName()) + 2*Metrics.FRAME_NAME_V_MARGIN;
		int nameHeight = Metrics.getLifelineFontHeigth() + Metrics.FRAME_NAME_H_MARGIN*2;
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME_NAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME_NAME));
		context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_FRAME_NAME));
		
	
		//Draw the frame name area	
		int[] points={getX()+1,getY()+1,
			getX()+1 + nameWidth, getY()+1,
			getX()+1 + nameWidth, getY()+1 -11 + nameHeight,
			getX()+1 - 11 + nameWidth, getY()+1 + nameHeight,
			getX()+1, getY()+1 + nameHeight,
			getX()+1, getY()+1 + nameHeight};
		context.fillPolygon(points);
		context.drawPolygon(points);
		context.drawLine(getX()+1, getY()+1, getX()+1, getY()+1 + nameHeight);
//		if (SDViewerPlugin.debugDisplay())
//			System.out.print("Frame height: "+getHeight()+"\n"); //$NON-NLS-1$ //$NON-NLS-2$	

		context.drawTextTruncatedCentred(getName(),getX(), getY(),nameWidth , nameHeight ,false);
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_FRAME));
		int lifeLineCount=0;
		int lifelineArryStep = 1;
		int messageArrayStep =1;
		if (Metrics.swimmingLaneWidth()*context.getZoom()<Metrics.LIFELINE_SIGNIFICANT_HSPACING)
			lifelineArryStep = Math.round(Metrics.LIFELINE_SIGNIFICANT_HSPACING/(Metrics.swimmingLaneWidth()
						*context.getZoom()));		
		if ((Metrics.getMessageFontHeigth()+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()<Metrics.MESSAGE_SIGNIFICANT_VSPACING)
			messageArrayStep = Math.round(Metrics.MESSAGE_SIGNIFICANT_VSPACING/((Metrics.getMessageFontHeigth()
				+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()));
/*		if (SDViewerPlugin.debugDisplay())
		{
			System.out.print("Lifelines step = "+lifelineArryStep+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.print("Messages step = "+messageArrayStep+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}*/
		
		//without somme lifelines, there is nothing to see
		if ((lifeLines == null)||(lifeLines.size()<1))
			return;
			
//		TODO Comment here			
		if (highlightLifeline != null)
		{
			IColor backupColor = context.getBackground();
			context.setBackground(Frame.getUserPref().getTimeCompressionSelectionColor());
			int gy=highlightLifeline.getY() + highlightLifeline.getHeight() + 
				(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * startEvent;
			context.fillRectangle(Metrics.FRAME_H_MARGIN+1,gy,
				highlightLifeline.getX()+Metrics.getLifelineWidth()/2-Metrics.FRAME_H_MARGIN,
				(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing())*nbEvent);
			context.setBackground(backupColor);
		}
		
		//Draw the lifeline starting at the first visible
		//stop as soon as the next lifeline in the array is not X visible
		//This works because lifelines are ordered along the X axis
		context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_LIFELINE));
		for (int i=lifeLineDrawIndex; i<lifeLines.size();i=i+lifelineArryStep)
		{
			Lifeline toDraw = (Lifeline)lifeLines.get(i);
			if (toDraw.getX()>context.getContentsX() + context.getVisibleWidth())
				break;
			toDraw.draw(context);
			
//TODO Comment here
			if (highlightLifeline != null)
			{
				if (toDraw == highlightLifeline)
					toDraw.highlightExecOccurrenceRegion(context,startEvent,
						nbEvent, highlightColor);
				else if ((toDraw.getIndex()<highlightLifeline.getIndex())||
						((toDraw.getIndex()<highlightLifeline.getIndex())))
				{
//TODO Comment this (transparency simulation)					
				
					int acIndex=toDraw.getExecOccurrenceDrawIndex();
					//acIndex = first visible execution occurrence
					//for drawing speed reason with only search on the visivle subset
					for (int h=acIndex; h<toDraw.getExecutions().size();h++)
					{
						BasicExecutionOccurrence exec = (BasicExecutionOccurrence)toDraw.getExecutions().get(h);
						int tempEvent=startEvent;
						for (int j=0; j<nbEvent;j++)
						{
							if (((tempEvent>=exec.startOccurrence)&&
								(tempEvent<=exec.endOccurrence)&&
							    (tempEvent+1>=exec.startOccurrence)&&
								(tempEvent+1<=exec.endOccurrence)))
								{
									toDraw.highlightExecOccurrenceRegion(context,tempEvent,
										1, Frame.getUserPref().getTimeCompressionSelectionColor());
								}
								tempEvent = tempEvent+1;
						}
						//if we are ouside the visble area we stop right now
						//This works because execution occurrences are ordered along the Y axis
						if (exec.getY()>getY())
							break;
					}
				}
			}
			lifeLineCount++;
		}
		
		if (SDViewerPlugin.debugDisplay())
		{
			System.out.print("\n");//$NON-NLS-1$
			System.out.print(lifeLineCount +" Lifeline drawn, starting from index "+lifeLineDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		int messageCount=0;
		if ((syncMessages != null)&&(syncMessages.size()>0))
		{
			SortSyncMessageComparator comparator = new SortSyncMessageComparator();
			context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_SYNC_MESS));
			//Draw the syncMessages starting at the first visible
			//stop as soon as the next message in the array is not Y visible
			//This works because syncMessages are ordered along the Y axis
			for (int i=messageDrawIndex; i<syncMessages.size(); i=i+messageArrayStep)
			{
				SyncMessage toDraw= (SyncMessage)syncMessages.get(i);
				if (i<syncMessages.size()-1)
				{
					SyncMessage next= (SyncMessage)syncMessages.get(i+1);
					if (comparator.compare(toDraw,next)==1)
						sortSyncMessage = true;
				}
				if (toDraw.getY()>context.getContentsY() + context.getVisibleHeight()+
				//take into account the message name drawn abobe the arrow
							Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth())
					break;
					
				//UML2 lost message visibility special case
				//Others visibility cases are perfrom in the ***common*** threatment
				if (toDraw.endLifeline==null && toDraw.startLifeline!=null)	 
				{
					if (context.getContentsX() + context.getVisibleWidth()
								> toDraw.getX() + toDraw.getWidth() &&
						context.getContentsX() < toDraw.getX() + toDraw.getWidth())
					{
						toDraw.draw(context);
						messageCount++;
					}
				}
				//UML2 found message visibility special case
				//Others visibility cases are perfrom in the ***common*** threatment
				else if (toDraw.endLifeline!=null && toDraw.startLifeline==null)	 
				{
					if (context.getContentsX() + context.getVisibleWidth()
								> toDraw.getX() + toDraw.getWidth() &&
						context.getContentsX() < toDraw.getX() + toDraw.getWidth())
					{
						toDraw.draw(context);
						messageCount++;
					}
				}
				//***Common*** syncMessages visibility
				if (lifelineDependantDrawing(context,toDraw))
					messageCount++;
			}
		}
		
		if (SDViewerPlugin.debugDisplay())
			System.out.print(messageCount +" Message drawn, starting from index "+messageDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
			
		messageCount = 0;
		if ((syncMessagesReturn != null)&&(syncMessagesReturn.size()>0))
		{
			SortSyncMessageComparator comparator = new SortSyncMessageComparator();
			context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_SYNC_MESS_RET));
			for (int i=messageReturnDrawIndex; i<syncMessagesReturn.size(); i=i+messageArrayStep)
			{
				SyncMessage toDraw= (SyncMessage)syncMessagesReturn.get(i);
				if (i<syncMessagesReturn.size()-1)
				{
					SyncMessage next= (SyncMessage)syncMessagesReturn.get(i+1);
					if (comparator.compare(toDraw,next)==1)
						sortSyncMessageReturn = true;
				}
				if (toDraw.getY()>context.getContentsY() + context.getVisibleHeight()+
				//take into account the message name drawn abobe the arrow
						Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth())
					break;
				if (lifelineDependantDrawing(context,toDraw))
					messageCount++;
			}
		}
		
		if (SDViewerPlugin.debugDisplay())
			System.out.print(messageCount +" Message return drawn, starting from index "+messageReturnDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
		
		messageCount = 0;
		if ((asyncMessages != null)&&(asyncMessages.size()>0))
		{
			SortAsyncMessageComparator comparator = new SortAsyncMessageComparator();
			context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_ASYNC_MESS));
			for (int i=asyncMessageDrawIndex; i<asyncMessages.size(); i=i+messageArrayStep)
			{
				AsyncMessage toDraw= (AsyncMessage)asyncMessages.get(i);
				if (i<asyncMessages.size()-1)
				{
					AsyncMessage next= (AsyncMessage)asyncMessages.get(i+1);
					if (comparator.compare(toDraw,next)==1)
						sortAsyncMessage = true;
				}
				int toDrawY = toDraw.getY();
				int toDrawHeight = toDraw.getHeight();
				if ((toDrawY>context.getContentsY() + context.getVisibleHeight()+
				//take into account the message name drawn abobe the arrow
						Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth())&&
				(toDrawY+ toDrawHeight >context.getContentsY() + context.getVisibleHeight()+
				Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth()))
					break;
				if (lifelineDependantDrawing(context,toDraw))
					messageCount++;
			}
		}
		if (SDViewerPlugin.debugDisplay())
			System.out.print(messageCount +" Async message drawn, starting from index "+asyncMessageDrawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$
		
		messageCount = 0;
		if ((asyncMessagesReturn != null)&&(asyncMessagesReturn.size()>0))
		{
			SortAsyncMessageComparator comparator = new SortAsyncMessageComparator();
			context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_ASYNC_MESS_RET));
			for (int i=asyncMessageReturnDrawIndex; i<asyncMessagesReturn.size(); i=i+messageArrayStep)
			{
				AsyncMessage toDraw= (AsyncMessage)asyncMessagesReturn.get(i);
				if (i<asyncMessagesReturn.size()-1)
				{
					AsyncMessage next= (AsyncMessage)asyncMessagesReturn.get(i+1);
					if (comparator.compare(toDraw,next)==1)
						sortAsyncMessageReturn = true;
				}
				int toDrawY = toDraw.getY();
				int toDrawHeight = toDraw.getHeight();
				if ((toDrawY>context.getContentsY() + context.getVisibleHeight()+
				//take into account the message name drawn abobe the arrow
						Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth())&&
				  	(toDrawY+toDrawHeight>context.getContentsY() + context.getVisibleHeight()+
					Metrics.MESSAGES_NAME_SPACING+Metrics.getMessageFontHeigth()))
					break;
				if (lifelineDependantDrawing(context,toDraw))
					messageCount++;
			}
		}
		if (SDViewerPlugin.debugDisplay())
			System.out.print(messageCount +" Async message return drawn, starting from index "+asyncMessageReturnDrawIndex+"\n\n"); //$NON-NLS-1$ //$NON-NLS-2$
		
/*		if (focusNode == null)
				return;
		else 
		{
			int x = focusNode.getX();
			int y = focusNode.getY();
			int width = focusNode.getWidth();
			int height = focusNode.getHeight();
			if (width<0)
			{
				x = x + width;
				width = -width;
			}
			
			if (height<0)
			{
				y = y + height;
				height = -height;
			}
			if (focusNode instanceof BaseMessage)
			{
				Lifeline startLifeline = null;
				Lifeline endLifeline = null;
				int startOcc=0;
				int endOcc = 0;
				if (focusNode instanceof SyncMessage)
				{
					startLifeline = ((SyncMessage)focusNode).getStartLifeline();
					endLifeline = ((SyncMessage)focusNode).getEndLifeline();
					startOcc = ((SyncMessage)focusNode).getEventOccurrence();
					endOcc = ((SyncMessage)focusNode).getEventOccurrence();
				}
				else if (focusNode instanceof AsyncMessage)
				{
					startLifeline = ((AsyncMessage)focusNode).getStartLifeline();
					endLifeline = ((AsyncMessage)focusNode).getEndLifeline();
					startOcc = ((AsyncMessage)focusNode).getStartOccurrence();
					endOcc = ((AsyncMessage)focusNode).getEndOccurrence();
				}
				if (startLifeline !=null && endLifeline != null && startLifeline == endLifeline &&(startOcc==endOcc))
				{
					context.drawFocus(x-Metrics.FOCUS_DRAWING_MARGIN,
					y-Metrics.FOCUS_DRAWING_MARGIN,
					width+Metrics.INTERNAL_MESSAGE_WIDTH+Metrics.FOCUS_DRAWING_MARGIN*2,
					height+Metrics.FOCUS_DRAWING_MARGIN+Metrics.INTERNAL_MESSAGE_WIDTH);
				}
				else 
				{
					context.drawFocus(x-Metrics.MESSAGES_NAME_SPACING,
					y-Metrics.FOCUS_DRAWING_MARGIN,
					width+Metrics.FOCUS_DRAWING_MARGIN*2,
					height+Metrics.FOCUS_DRAWING_MARGIN*2);
				}
			}
			else if (focusNode instanceof Lifeline)
			{
				context.drawFocus(x-Metrics.FOCUS_DRAWING_MARGIN,
									y-Metrics.FOCUS_DRAWING_MARGIN,
									width+Metrics.FOCUS_DRAWING_MARGIN*2,
									height+Metrics.FOCUS_DRAWING_MARGIN*2);
				int len = verticalIndex * (Metrics.getMessageSpacing() + Metrics.getMessageFontHeigth());
				context.drawFocus(x+(width-Metrics.FOCUS_DRAWING_MARGIN*2-Metrics.EXECUTION_OCCURRENCE_WIDTH)/2,
								y+height+Metrics.FOCUS_DRAWING_MARGIN,
								Metrics.EXECUTION_OCCURRENCE_WIDTH+Metrics.FOCUS_DRAWING_MARGIN*2,
								len+Metrics.FOCUS_DRAWING_MARGIN*2);
			}
			else if (focusNode instanceof ExecutionOccurrence)
			{
				context.drawFocus(x-Metrics.MESSAGES_NAME_SPACING,
					y-Metrics.MESSAGES_NAME_SPACING,
					width+Metrics.MESSAGES_NAME_SPACING*2,
					height+Metrics.MESSAGES_NAME_SPACING*2);
			}
		}*/		
	}

	public void setFocusNode(GraphNode node)
	{
		focusNode = node;
	}
	
	public GraphNode getFocusNode()
	{
		return focusNode;
	}

	
	private boolean lifelineDependantDrawing(IGC context, BaseMessage toDraw)
	{
		//		***Common*** syncMessages visibility
		//draw the message only if at least one end is visible			
		if (toDraw.endLifeline!=null&&(toDraw.endLifeline.isVisible(context.getContentsX(),0,
				context.getVisibleWidth(),0))||(toDraw.startLifeline!=null&&
				toDraw.startLifeline.isVisible(context.getContentsX(),0,
				context.getVisibleWidth(),0)))
		{
			toDraw.draw(context);
			return true;
		}
		//In this case it can be a message which cross the whole visible area
		else if (toDraw.endLifeline!=null&&(!toDraw.endLifeline.isVisible(context.getContentsX(),0,
				  context.getVisibleWidth(),0))&&(toDraw.startLifeline!=null&&
				  !toDraw.startLifeline.isVisible(context.getContentsX(),0,
				  context.getVisibleWidth(),0)))
		  {
			  if (toDraw.endLifeline.getX()>context.getContentsX()+context.getVisibleWidth()&&
				  toDraw.startLifeline.getX()<context.getContentsX())
			  {
				  toDraw.draw(context);
				return true;
			  }
			  else if (toDraw.startLifeline.getX()>context.getContentsX()+context.getVisibleWidth()&&
					  toDraw.endLifeline.getX()<context.getContentsX())
				  {
					  toDraw.draw(context);
					  return true;
				  }
		  }
		return false;
	}

//TODO Comment here
	public void highlightTimeCompression(Lifeline lifeline, int startEvent, int nbEvent, IColor color)
	{
		highlightLifeline = lifeline;
		this.startEvent = startEvent;
		this.nbEvent = nbEvent;
		highlightColor = color;
	}
	
	public void resetTimeCompression()
	{
		highlightLifeline = null;
		this.startEvent = 0;
		this.nbEvent = 0;
		highlightColor = null;
	}
	
	public double getMinTime()
	{
		if (computeMinMax)
		{
			computeMinMax();
			computeMinMax = false;
		}
		return minTime;
	}
	
	public double getMaxTime()
	{
		if (computeMinMax)
		{
			computeMinMax();
			computeMinMax = false;
		}
		return maxTime;
	}
	
	private void computeMinMax()
	{
		List timeArray = new ArrayList();
		for (int i=0;i<syncMessages.size();i++)
		{
			if (((ITimeRange)syncMessages.get(i)).hasTimeInfo())
			{
				int event = ((BaseMessage)syncMessages.get(i)).getEventOccurrence();
				double time = ((ITimeRange)syncMessages.get(i)).getFirstTime();
				TimeEvent f = new TimeEvent(time, event, (ITimeRange)syncMessages.get(i));
				timeArray.add(f);
			}
		}
		
		for (int i=0;i<syncMessagesReturn.size();i++)
		{
			if (((ITimeRange)syncMessagesReturn.get(i)).hasTimeInfo())
			{
				int event = ((BaseMessage)syncMessagesReturn.get(i)).getEventOccurrence();
				double time = ((ITimeRange)syncMessagesReturn.get(i)).getFirstTime();
				TimeEvent f = new TimeEvent(time, event, (ITimeRange)syncMessagesReturn.get(i));
				timeArray.add(f);
			}
		}
				
		for (int i=0;i<asyncMessages.size();i++)
		{
			if (((ITimeRange)asyncMessages.get(i)).hasTimeInfo())
			{
				int event = ((AsyncMessage)asyncMessages.get(i)).getStartOccurrence();
				double time = ((ITimeRange)asyncMessages.get(i)).getFirstTime();
				TimeEvent f = new TimeEvent(time, event, (ITimeRange)asyncMessages.get(i));
				timeArray.add(f);
				event = ((AsyncMessage)asyncMessages.get(i)).getEndOccurrence();
				time = ((ITimeRange)asyncMessages.get(i)).getLastTime();
				f = new TimeEvent(time, event, (ITimeRange)asyncMessages.get(i));
				timeArray.add(f);
			}
		}
		
		for (int i=0;i<asyncMessagesReturn.size();i++)
		{
			if (((ITimeRange)asyncMessagesReturn.get(i)).hasTimeInfo())
			{
				int event = ((AsyncMessage)asyncMessagesReturn.get(i)).getStartOccurrence();
				double time = ((ITimeRange)asyncMessagesReturn.get(i)).getFirstTime();
				TimeEvent f = new TimeEvent(time, event, (ITimeRange)asyncMessagesReturn.get(i));
				timeArray.add(f);
				event = ((AsyncMessage)asyncMessagesReturn.get(i)).getEndOccurrence();
				time = ((ITimeRange)asyncMessagesReturn.get(i)).getLastTime();
				f = new TimeEvent(time, event, (ITimeRange)asyncMessagesReturn.get(i));
				timeArray.add(f);
			}
		}
		
		executionOccurrencesWithTime = null;
		for (int i=0;i<lifeLines.size();i++) {
			Lifeline l = getLifeline(i);
			if (l.hasTimeInfo()) {
				for (Iterator j=l.executionOccurrences.iterator();j.hasNext();) {
					Object o = j.next();
					if (o instanceof ExecutionOccurrence) {
						ExecutionOccurrence eo = (ExecutionOccurrence)o;
						if (eo.hasTimeInfo()) {
							int event = eo.getStartOccurrence();
							double time = eo.getFirstTime();
							TimeEvent f = new TimeEvent(time, event, eo);
							timeArray.add(f);
							if (executionOccurrencesWithTime == null) {
								executionOccurrencesWithTime = new ArrayList();
							}
							executionOccurrencesWithTime.add(f);
							event = eo.getEndOccurrence();
							time = eo.getLastTime();
							f = new TimeEvent(time, event, eo);
							timeArray.add(f);
							executionOccurrencesWithTime.add(f);
						}
					}
				}
			}
		}
		
		if (executionOccurrencesWithTime != null) {
			Object[] temp=executionOccurrencesWithTime.toArray();
			Arrays.sort(temp,new TimeEventComparator());
			executionOccurrencesWithTime = Arrays.asList(temp);
		}

		Object[] temp=timeArray.toArray();
		Arrays.sort(temp,new TimeEventComparator());
		timeArray = Arrays.asList(temp);
		for (int i=0; i<timeArray.size()-1;i++)
		{
			TimeEvent m1 = (TimeEvent)timeArray.get(i);
			TimeEvent m2 = (TimeEvent)timeArray.get(i+1);
			if (computeMinMax)
			{
				minTime = m2.getTime()-m1.getTime();
				computeMinMax = false;
			}
			if ((m2.getTime()-m1.getTime())<minTime)
				minTime = m2.getTime()-m1.getTime();
			if ((m2.getTime()-m1.getTime())>maxTime)
				maxTime = m2.getTime()-m1.getTime();
		}
	}
	
	public List getExecutionOccurrencesWithTime() {
		return executionOccurrencesWithTime;
	}
	
	protected void setHasTimeInfo(boolean value)
	{
		timeInfo = value;
	}
	
	public boolean hasTimeInfo()
	{
		return timeInfo;
	}
	
	public static boolean contains(int x,int y, int width, int height, int px, int py)
	{
		int locX = x;
		int locY = y;
		int locWidth = width;
		int locHeight = height;
		
		if (width<0)
		{
			locX = locX + width;
			locWidth = -locWidth;
		}
		
		if (height<0)
		{
			locY = locY + height;
			locHeight = -locHeight;
		}
		return (px >= locX) && (py >= locY) && ((px - locX) < locWidth) && ((py - locY) < locHeight);
	}
	
	public static void setUserPref(ISDPreferences pref)
	{
		userPref = pref;
	}
	
	public static ISDPreferences getUserPref()
	{
		return userPref;
	}
	
	public void setTimeUnitName(String name)
	{
		unitName = name;
	}
	
	public String getTimeUnitName()
	{
		return unitName;
	}
	
	public void forceEventOccurrenceSpacing(int space)
	{
		forceEventOccurrenceSpacing = space;
	}
}
