/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.preferences;

import java.util.Hashtable;

import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.hyades.uml2sd.ui.drawings.IColor;
import org.eclipse.hyades.uml2sd.ui.drawings.IFont;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.ui.drawings.impl.ColorImpl;
import org.eclipse.hyades.uml2sd.ui.drawings.impl.FontImpl;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;


/**
 * This is the Sequence Diagram preference handler
 * This class is reponsible for accessing the current user preferences selection
 * This class also provider getters for each modifiable preferences
 * @author sveyrier
 */
public class SDViewPref implements ISDPreferences{
	
	public static final String  BACK_COLOR_POSTFIX = "_BACK_COLOR";//$NON-NLS-1$
	public static final String  FORE_COLOR_POSTFIX = "_FORE_COLOR";//$NON-NLS-1$
	
	public static final String[] fontList={
		PREF_LIFELINE,
		PREF_EXEC,
		PREF_SYNC_MESS,
		PREF_SYNC_MESS_RET,
		PREF_ASYNC_MESS,
		PREF_ASYNC_MESS_RET,
		PREF_FRAME,
		PREF_FRAME_NAME};
		
	public static final String[] prefBackColorList={
		PREF_LIFELINE,
		PREF_EXEC,
		PREF_FRAME,
		PREF_FRAME_NAME};
	
	public static final String[] prefForeColorList={
		PREF_LIFELINE,
		PREF_EXEC,
		PREF_SYNC_MESS,
		PREF_SYNC_MESS_RET,
		PREF_ASYNC_MESS,
		PREF_ASYNC_MESS_RET,
		PREF_FRAME,
		PREF_FRAME_NAME};
		
	protected Hashtable fontPref;
	protected Hashtable foreColorPref;
	protected Hashtable backColorPref;
	
	private static SDViewPref handle = null;
	private IPreferenceStore prefStore = null;
	private IColor timeCompressionSelectionColor = null;
	private Font lifeLineFont = null;
	private Font synMessageFont = null;
	private Font synMessageReturnFont = null;
	private Font asynMessageFont = null;
	private Font asynMessageReturnFont = null;
	
	/**
	 * Builds the Sequence Diagram preference handler:
	 * - Define the preference default values.
	 * - Load the currently used preferences setting 
	 *
	 */
	private SDViewPref()
	{		
		prefStore = SDViewerPlugin.getDefault().getPreferenceStore();

		prefStore.setDefault(PREF_LINK_FONT,true);
		prefStore.setDefault(PREF_LIFELINE_WIDTH,120);
		prefStore.setDefault(PREF_USE_GRADIENT,true);
					
		fontPref = new Hashtable();	
		foreColorPref = new Hashtable();	
		backColorPref = new Hashtable();		
							
		Display display = Display.getDefault();
		for (int i=0; i<fontList.length;i++)
		{
			PreferenceConverter.setDefault(prefStore,fontList[i],
								Display.getDefault().getSystemFont().getFontData());
		}

		for (int i=0; i<prefBackColorList.length;i++)
		{
			IColor color;
			if ((prefBackColorList[i].equals(PREF_EXEC))||prefBackColorList[i].equals(PREF_FRAME_NAME))
				color = new ColorImpl(Display.getDefault(),201,222,233);
			else if (prefBackColorList[i].equals(PREF_LIFELINE))
				color =  new ColorImpl(Display.getDefault(),220,220,220);
			else color = new ColorImpl(Display.getDefault(),255,255,255);
			PreferenceConverter.setDefault(prefStore,prefBackColorList[i]+BACK_COLOR_POSTFIX,
					((Color)color.getColor()).getRGB());
			color.dispose();
		}

		for (int i=0; i<prefForeColorList.length;i++)
		{
			IColor color;
			if (prefForeColorList[i].equals(PREF_LIFELINE))
				color = new ColorImpl(Display.getDefault(),129,129,129);
			else if (prefForeColorList[i].equals(PREF_FRAME_NAME))
				color =  new ColorImpl(Display.getDefault(),81,153,200);
			else color = new ColorImpl(Display.getDefault(),134,176,212);
			PreferenceConverter.setDefault(prefStore,prefForeColorList[i]+FORE_COLOR_POSTFIX,
				((Color)color.getColor()).getRGB());
			color.dispose();
		}
		
		IColor color = new ColorImpl(Display.getDefault(),218,232,238);
		PreferenceConverter.setDefault(prefStore,PREF_TIME_COMP,((Color)color.getColor()).getRGB());
		color.dispose();
		
									
		buildFontsAndColors();
	}
	
	/**
	 * Returns the PreferenceStore
	 * @return the PreferenceStore
	 */
	public IPreferenceStore getPreferenceStore()
	{
		return prefStore;
	}
	
	/**
	 * Apply the preferences in the preferences handler
	 */
	public void apply()
	{
		buildFontsAndColors();
		prefStore.firePropertyChangeEvent("PREFOK",null,null); //$NON-NLS-1$
	}
	
	
	/**
	 * Returns an unique instance of the Sequence Diagram preference handler
	 * @return the preference handler instance
	 */
	static public SDViewPref getInstance()
	{
		if (handle == null)
			handle = new SDViewPref();
		return handle;
	}
	
	/**
	 * Returns the chosen foreground color 
	 * @return the forgroud color
	 */
	public IColor getForeGroundColor(String prefName)
	{
		if ((foreColorPref.get(prefName+FORE_COLOR_POSTFIX)!=null)&&(foreColorPref.get(prefName+FORE_COLOR_POSTFIX)instanceof ColorImpl))
			return (IColor)foreColorPref.get(prefName+FORE_COLOR_POSTFIX);
		else return ColorImpl.getSystemColor(SWT.COLOR_BLACK);
	}
	
	/**
	 * Returns the chosen backgrounf color
	 * @return the background color
	 */
	public IColor getBackGroundColor(String prefName)
	{
		if ((backColorPref.get(prefName+BACK_COLOR_POSTFIX)!=null)&&(backColorPref.get(prefName+BACK_COLOR_POSTFIX)instanceof ColorImpl))
			return (IColor)backColorPref.get(prefName+BACK_COLOR_POSTFIX);
		else return ColorImpl.getSystemColor(SWT.COLOR_WHITE);
	}
	
	/**
	 * Returns the forground color to use for widget selection.
	 * This color is system dependant and not set using the prefrence page
	 * @return the forground color
	 */
	public IColor getForeGroundColorSelection()
	{
		Display display = Display.getDefault();
		return ColorImpl.getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT);
	}

	/**
	 * Returns the background color to use for widget selection.
	 * This color is system dependant and not set using the prefrence page
	 * @return the background color
	 */
	public IColor getBackGroundColorSelection()
	{
		Display display = Display.getDefault();
		return ColorImpl.getSystemColor(SWT.COLOR_LIST_SELECTION);
	}
	
	/**
	 * Returns the chosen font 
	 * @return the font
	 */
	public IFont getFont(String prefName)
	{
		if ((fontPref.get(prefName)!=null)&&(fontPref.get(prefName)instanceof IFont))
			return (IFont)fontPref.get(prefName);
		else return FontImpl.getSystemFont();
	}
	
	/**
	 * Returns the SwimLane width chosen
	 * @return the SwimLane width
	 */
	public int getLifelineWidth()
	{
		return prefStore.getInt(PREF_LIFELINE_WIDTH);
	}
	
	/**
	 * Returns if font linkage with zoom has been chosen
	 * @return true if cheched false otherwise
	 */
	public boolean fontLinked()
	{
		return prefStore.getBoolean(PREF_LINK_FONT);
	}
	
	/**
	 * Returns if the use gradient color has been chosen
	 * @return true if cheched false otherwise
	 */
	public boolean useGradienColor()
	{
		return prefStore.getBoolean(PREF_USE_GRADIENT);
	}
	
	/**
	 * Returns the color used to connect the time compression bar to the diagram graph node
	 * @return the selection color
	 */
	public IColor getTimeCompressionSelectionColor()
	{
		return timeCompressionSelectionColor;
	}

	/**
	 * Builds the new colors and fonts according the current user selection
	 * when the OK or Aplly button is clicked
	 *
	 */
	private void buildFontsAndColors()
	{
		
		Display display = Display.getDefault();

		for (int i=0; i<fontList.length;i++)
		{
			FontData fontData = PreferenceConverter.getFontData(prefStore,fontList[i]);
			if ((fontPref.get(fontList[i])!=null)&&(fontPref.get(fontList[i])instanceof IFont))
				((IFont)fontPref.get(fontList[i])).dispose();
			fontPref.put(fontList[i],new FontImpl(display,fontData));
		}
		
		for (int i=0; i<prefBackColorList.length;i++)
		{
			RGB rgb = PreferenceConverter.getColor(prefStore,prefBackColorList[i]+BACK_COLOR_POSTFIX);
			if ((backColorPref.get(prefBackColorList[i]+BACK_COLOR_POSTFIX)!=null)&&
					(backColorPref.get(prefBackColorList[i]+BACK_COLOR_POSTFIX)instanceof IColor))
				((IColor)backColorPref.get(prefBackColorList[i]+BACK_COLOR_POSTFIX)).dispose();
			backColorPref.put(prefBackColorList[i]+BACK_COLOR_POSTFIX,new ColorImpl(display,rgb.red,rgb.green,rgb.blue));
		}
		
		for (int i=0; i<prefForeColorList.length;i++)
		{
			RGB rgb = PreferenceConverter.getColor(prefStore,prefForeColorList[i]+FORE_COLOR_POSTFIX);
			if ((foreColorPref.get(prefForeColorList[i]+FORE_COLOR_POSTFIX)!=null)&&
					(foreColorPref.get(prefForeColorList[i]+FORE_COLOR_POSTFIX)instanceof IColor))
				((IColor)foreColorPref.get(prefForeColorList[i]+FORE_COLOR_POSTFIX)).dispose();
			foreColorPref.put(prefForeColorList[i]+FORE_COLOR_POSTFIX,new ColorImpl(display,rgb.red,rgb.green,rgb.blue));
		}
		
		RGB rgb = PreferenceConverter.getColor(prefStore,PREF_TIME_COMP); //$NON-NLS-1$
		if(timeCompressionSelectionColor != null)
			timeCompressionSelectionColor.dispose();
		timeCompressionSelectionColor = new ColorImpl(display,rgb.red,rgb.green,rgb.blue);
	}
	
	public void addPropertyChangeListener(IPropertyChangeListener listener)
	{
		prefStore.addPropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(IPropertyChangeListener listener)
	{
		prefStore.removePropertyChangeListener(listener);
	}
}
