/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.test.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Arrays;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.hyades.ui.internal.util.FileUtil;

/**
 * Constains the methods to test the 
 * {@link org.eclipse.hyades.ui.internal.util.FileUtil} class.
 *  
 * @author marcelop
 * @since 0.0.1
 */
public class FileUtilTest 
extends TestCase
{
	public final static String TEST_ARTIFACTS_PATH = "artifacts/util/";
	
	/**
	 * Constructor for FileUtilTest.
	 * @param name
	 */
	public FileUtilTest(String name)
	{
		super(name);
	}

	/**
	 * Returns a suite with the tests defined in this class.
	 * @return Test
	 */
	public static Test suite()
	{
		TestSuite testSuite = new TestSuite("Hyades UI File Util");
		testSuite.addTestSuite(FileUtilTest.class);
		return testSuite;
	}
		
	/**
	 * Test for {@link BaseFileUtil#appendPath(String, String)}
	 */
	public void testAppendPath()
	{
		String path1 = "c:\\temp";
		String path2 = "download\\eclipse.zip";
		assertEquals("c:/temp/download/eclipse.zip", FileUtil.appendPath(path1, path2));
		
		path1 = "c:\\temp\\";
		path2 = "\\download/eclipse.zip";
		assertEquals("c:/temp/download/eclipse.zip", FileUtil.appendPath(path1, path2));
		
		path1 = "\\temp";
		path2 = "/download/eclipse.zip";
		assertEquals("/temp/download/eclipse.zip", FileUtil.appendPath(path1, path2));

		path1 = "temp/";
		path2 = "/download/eclipse.zip";
		assertEquals("temp/download/eclipse.zip", FileUtil.appendPath(path1, path2));			

		path1 = "http://www.eclipse.org";
		path2 = "hyades";
		assertEquals("http://www.eclipse.org/hyades", FileUtil.appendPath(path1, path2));			

		path1 = "";
		path2 = "\\hyades";
		assertEquals("/hyades", FileUtil.appendPath(path1, path2));			
	}
	
	/**
	 * Test for {@link FileUtil#delete(File)}
	 */
	public void testDelete()
	throws IOException
	{
		File directory = getUnexistingDirectoryHandle(); //File#: 1
		String msg = "Base directory:" + directory.getAbsolutePath();
		assertTrue(msg, directory.mkdir());
		
		File file1 = new File(directory, "file1.tmp");
		assertTrue(msg, file1.createNewFile());
		File file2 = new File(directory, "file2.tmp"); //File#: 2
		assertTrue(msg, file2.createNewFile());
		
		assertEquals(msg, 1, FileUtil.delete(file1));
		assertFalse(msg, file1.exists());
		assertTrue(msg, file2.exists());
		
		File file = new File(directory, "dir1/f1.tmp"); //File#: 4
		assertTrue(msg, file.getParentFile().mkdir());
		assertTrue(msg, file.createNewFile());
			
		file = new File(directory, "dir1/f2.tmp");	//File#: 5
		assertTrue(msg, file.createNewFile());
			
		file = new File(directory, "dir2/f1.tmp"); //File#: 7
		assertTrue(msg, file.getParentFile().mkdir());
		assertTrue(msg, file.createNewFile());
			
		file = new File(directory, "dir2/dir21/f1.tmp"); //File#: 9
		assertTrue(msg, file.getParentFile().mkdir());
		assertTrue(msg, file.createNewFile());
		
		assertEquals(msg, 9, FileUtil.delete(directory));
		assertFalse(msg, directory.exists());
	}
	
	/**
	 * Test for {@link FileUtil#getFileExtension(File)}
	 */
	public void testGetFileExtension()
	{
		assertEquals("txt", FileUtil.getFileExtension(new File(TEST_ARTIFACTS_PATH + "test.txt")));
		assertEquals("log", FileUtil.getFileExtension(new File(".log")));
		assertEquals("", FileUtil.getFileExtension(new File("test")));
	}
	
	/**
	 * Test for {@link FileUtil#getLastSegment(String)}
	 */
	public void testGetLastSegment()
	{
		assertEquals("test", FileUtil.getLastSegment("x:\\temp\\test"));
		assertEquals("temp", FileUtil.getLastSegment("/temp"));
		assertEquals("test", FileUtil.getLastSegment("test"));
		assertEquals("test", FileUtil.getLastSegment("/test/"));
	}
	
	/**
	 * Test for {@link FileUtil#getTempDir()}
	 */
	public void testGetTempDir()
	{
		// Not tested
	}

	/**
	 * Test for {@link FileUtil#getValidFileOrDirectory(File))}
	 */
	public void testGetValidFileOrDirectory()
	throws IOException
	{
		File directoryHandle = getUnexistingDirectoryHandle();
		assertTrue(directoryHandle.toString(), directoryHandle.mkdir());
		File fileHandle = new File(directoryHandle, "test.tmp");
		assertTrue(fileHandle.toString(), fileHandle.createNewFile());
		
		assertEquals(fileHandle, FileUtil.getValidFileOrDirectory(fileHandle));
		assertEquals(directoryHandle, FileUtil.getValidFileOrDirectory(directoryHandle));
		
		fileHandle.delete();
		assertNull(FileUtil.getValidFileOrDirectory(fileHandle));
		assertEquals(directoryHandle, FileUtil.getValidFileOrDirectory(directoryHandle));

		directoryHandle.delete();
		assertNull(FileUtil.getValidFileOrDirectory(fileHandle));
		assertNull(FileUtil.getValidFileOrDirectory(directoryHandle));
	}	
	
	/**
	 * Test for {@link FileUtil#getZipEntryDataByZipEntryName(String))}
	 */
	public void testGetZipEntryDataByZipEntryName()
	{
		URL url = getClass().getClassLoader().getResource(TEST_ARTIFACTS_PATH  + "test.zip");
		assertNotNull(url);
		File zipFile = new File(url.getFile());
		assertTrue(zipFile.getAbsolutePath(), zipFile.exists());
		
		Map map = FileUtil.getZipEntryDataByZipEntryName(zipFile.getAbsolutePath());
		assertEquals(6, map.size());
		
		assertNotNull(map.get("test/text1.txt"));
		assertNotNull(map.get("test/text2.txt"));
		assertNotNull(map.get("test/folder1/text1.txt"));
		assertNotNull(map.get("test/folder1/text2.txt"));
		assertNotNull(map.get("test/folder2/text1.txt"));
		assertNotNull(map.get("test/folder2/text2.txt"));
	}
	
	/**
	 * Test for {@link FileUtil#isParent(File, File))}
	 */
	public void testIsParent()
	{
		File parentCandidate = new File("/home/usr/public");
		File fileToAnalyse = new File(parentCandidate, "xyz/temp/test.tst");
		assertTrue(FileUtil.isParent(parentCandidate, fileToAnalyse));
		
		fileToAnalyse = new File("xyz/temp/test.tst");
		assertFalse(FileUtil.isParent(parentCandidate, fileToAnalyse));
		
		assertTrue(FileUtil.isParent(fileToAnalyse, fileToAnalyse));
	}

	/**
	 * Test for {@link FileUtil#normalizePath(String))}
	 */
	public void testNormalizePath()
	{
		assertEquals("c:/temp", FileUtil.normalizePath("c:\\temp\\"));
		assertEquals("/home/temp", FileUtil.normalizePath("/home/temp"));
		assertEquals("//home/temp", FileUtil.normalizePath(" //home\\temp/ "));
	}
	
	/**
	 * Test for {@link FileUtil#readFromFile(String, File, boolean))}
	 */
	public void testReadFromFile()
	throws IOException
	{
		URL url = getClass().getClassLoader().getResource(TEST_ARTIFACTS_PATH + "test.txt");
		File textFile = new File(url.getFile());
		assertTrue(textFile.getAbsolutePath(), textFile.exists());
		assertTrue(textFile.getAbsolutePath(), textFile.canRead());
		
		String expected = "This is the line 1\r\nThis is the line 2\r\nThis is the line 3\r\n\r\nThis is the line 5";
		assertEquals(expected, FileUtil.readFromFile("UTF-8", textFile, false));
		assertTrue(textFile.getAbsolutePath(), textFile.canWrite());		
	}
	
	/**
	 * Test for {@link FileUtil#readFromInputStream(String, InputStream, boolean))}
	 */
	public void testReadFromInputStream()
	throws IOException
	{
		URL url = getClass().getClassLoader().getResource(TEST_ARTIFACTS_PATH  + "test.txt");
		assertNotNull(url);
		File textFile = new File(url.getFile());
		assertTrue(textFile.getAbsolutePath(), textFile.exists());
		assertTrue(textFile.getAbsolutePath(), textFile.canRead());
		
		InputStream inputStream = new FileInputStream(textFile);
		
		String expected = "This is the line 1\r\nThis is the line 2\r\nThis is the line 3\r\n\r\nThis is the line 5";
		assertEquals(expected, FileUtil.readFromInputStream("UTF-8", inputStream, false));
		assertTrue(textFile.getAbsolutePath(), textFile.canWrite());
		
		String text = "\u4E13a\u4E13\u4E13";
		inputStream = new ByteArrayInputStream(text.getBytes("UTF-16"));
		assertEquals(text, FileUtil.readFromInputStream("UTF-16", inputStream, false));
	}
	
	/**
	 * Test for {@link FileUtil#removeLastSegment(String))}
	 */
	public void testRemoveLastSegment()
	{
		assertEquals("c:/temp", FileUtil.removeLastSegment("c:\\temp\\test.tst"));
		assertEquals("", FileUtil.removeLastSegment("test.tst"));
		assertEquals("//home/temp", FileUtil.removeLastSegment("//home/temp\\test.tst"));
	}
	
	/**
	 * Test for {@link FileUtil#writeToFile(String, File, boolean, String))}
	 * 
	 * <p>The test {@link #testReadFromFile()} must be executed before this test.
	 */
	public void testWriteToFile()
	throws IOException
	{
		File directory = getUnexistingDirectoryHandle();
		assertTrue(directory.getAbsolutePath(), directory.mkdir());
		
		File file = new File(directory, "writeToFile.test");
		String text = "This is a\r\ntest of the\r\n\t\"WriteToFile\"\r\nmethod of the class\r\n\tFileUtil";
		assertTrue(file.getAbsolutePath(), FileUtil.writeToFile("UTF-8", file, false, text));
		
		String textFromFile = FileUtil.readFromFile("UTF-8", file, false);
		assertEquals(file.getAbsolutePath(), text, textFromFile);
		
		assertTrue(file.getAbsolutePath(), FileUtil.writeToFile("UTF-8", file, true, text));
		textFromFile = FileUtil.readFromFile("UTF-8", file, false);
		assertEquals(file.getAbsolutePath(), text+text, textFromFile);
		
		assertTrue(file.getAbsolutePath(), file.delete());
		
		//Appending to an inexistent file
		assertTrue(file.getAbsolutePath(), FileUtil.writeToFile("UTF-8", file, true, text));
		textFromFile = FileUtil.readFromFile("UTF-8", file, false);
		assertEquals(file.getAbsolutePath(), text, textFromFile);
		
		assertTrue(file.getAbsolutePath(), file.delete());
		assertTrue(directory.getAbsolutePath(), directory.delete());
	}
	
	/**
	 * Test for {@link FileUtil#writeToOutputStream(String, OutputStream, String))}
	 */
	public void testWriteToOutputStream()
	throws IOException
	{
		String text = "\u4E13a\u4E13\u4E13";
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		assertTrue(FileUtil.writeToOutputStream("UTF-16", outputStream, text));
		assertTrue(Arrays.equals(text.getBytes("UTF-16"), outputStream.toByteArray()));
	}

	/**
	 * Test for {@link FileUtil#copyFiles(File[], File, boolean))}
	 * 
	 * <p>The test {@link #testDelete()} must be executed before this test.
	 */
	public void testCopyFiles()
	throws IOException
	{
		File[] files = new File[5];
		
		URL url = getClass().getClassLoader().getResource(TEST_ARTIFACTS_PATH + "test.txt");
		files[0] = new File(url.getFile()).getAbsoluteFile();
		assertTrue(files[0].getAbsolutePath(), files[0].exists());
		
		url = getClass().getClassLoader().getResource(TEST_ARTIFACTS_PATH + "test.zip");
		files[1] = new File(url.getFile()).getAbsoluteFile();
		assertTrue(files[1].getAbsolutePath(), files[1].exists());
		
		files[2] = files[1].getParentFile();

		File toDirectory = getUnexistingDirectoryHandle();
		
		//Creates a directory with a file name
		new File(toDirectory, files[0].getName()).mkdir();
		
		assertEquals(2, FileUtil.copyFiles(files, toDirectory, false));
		
		File[] newFiles = toDirectory.listFiles();
		assertEquals(2, newFiles.length);
		for(int i=0, max=newFiles.length; i<max; i++)
		{
			for(int j=0,maxJ=files.length; j<maxJ; j++)
			{
				if((files[j] != null) && (newFiles[i].getName().equals(files[j].getName())))
					assertEquals(files[j].length(), newFiles[i].length());
			}
		}
		

		assertEquals(0, FileUtil.copyFiles(files, toDirectory, false));
		
		assertEquals(2, FileUtil.copyFiles(files, toDirectory, true));
		newFiles = toDirectory.listFiles();
		assertEquals(2, newFiles.length);
		for(int i=0, max=newFiles.length; i<max; i++)
		{
			for(int j=0,maxJ=files.length; j<maxJ; j++)
			{
				if((files[j] != null) && (newFiles[i].getName().equals(files[j].getName())))
					assertEquals(files[j].length(), newFiles[i].length());
			}
		}
		
		FileUtil.delete(toDirectory);
	}

	/**
	 * Returns a file handle to a directory that doesn't exist.
	 * @return File
	 */
	private synchronized File getUnexistingDirectoryHandle()
	{
		String baseName = "baseTestDir";
		
		for(int i=0;i<200;i++)
		{
			File directory = new File(baseName + i);
			if(!directory.exists())
				return directory.getAbsoluteFile();
		}
		
		fail("Unable to create inexistent directory handle.");
		return null;
	}	
}