/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.text.MessageFormat;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.DatapoolUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

/**
 * Dialog to collect information about a record/row that is being
 * inserted or edited by the datapool table.  
 * 
 * @author psun
 *
 */
public class DatapoolRowDialog extends Dialog 
{

	private static final String TAG_EQUIVALENCE_CLASSES = "equivalenceClasses"; //$NON-NLS-1$
	private static final String TAG_EQUIVALENCE_CLASS = "equivalenceClass"; //$NON-NLS-1$
	private static final String TAG_RECORD = "record"; //$NON-NLS-1$

	private static final int HEIGHT = 40;
	private static final int WIDTH = 380;
	private Shell shell = null;
	private IDatapool datapool = null;
	private String title = null;
	private boolean insertionMode = false;


	private Table table = null;
	private TableItem previousTableItem = null;
	private TableItem selectedTableItem = null;

	private Combo equivalenceClassCombo = null;
	private Combo recordIndexCombo = null;

	private int insertionEquivalenceClassIndex = -1;
	private int insertionRecordIndex = -1;
	private TableItem insertionTableItem = null;

	public DatapoolRowDialog(Shell parentShell, 
								IDatapool datapool,
								Table table,
								TableItem selectedTableItem,
								TableItem previousTableItem,
								String title)
	{
		super(parentShell);
		this.shell = parentShell;
		this.datapool = datapool;
		this.title = title; 
		
		this.table = table;
		this.selectedTableItem = selectedTableItem;
		this.previousTableItem = previousTableItem;
		if(title.equals(DatapoolPlugin.getResourceString("DATA_ROW_DLG_TITLE_INS"))) //$NON-NLS-1$
			this.insertionMode = true;

	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		enableOK();
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		getShell().setText(title);
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = HEIGHT;
		gridData.widthHint = WIDTH;
		superComposite.setLayoutData(gridData);		

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		superComposite.setLayout(gridLayout);

		Label equivalenceClassLabel = new Label(superComposite, SWT.NONE);
		if(insertionMode)
			equivalenceClassLabel.setText(DatapoolPlugin.getResourceString("DATA_ROW_DLG_INS_INTO")); //$NON-NLS-1$
		else
			equivalenceClassLabel.setText(DatapoolPlugin.getResourceString("DATA_ROW_DLG_MOVE_INTO")); //$NON-NLS-1$

		
		equivalenceClassCombo = new Combo(superComposite, SWT.READ_ONLY | SWT.DROP_DOWN);
		IDatapoolEquivalenceClass[] equivalenceClasses = DatapoolUtil.getInstance().getEquivalenceClasses(datapool, null);
		String[] equivalenceClassLabels = getEquivalenceClassLabels(equivalenceClasses);

		equivalenceClassCombo.setItems(equivalenceClassLabels);
		equivalenceClassCombo.setData(TAG_EQUIVALENCE_CLASSES, equivalenceClasses);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		gridData.widthHint = 150;
		equivalenceClassCombo.setLayoutData(gridData);
		equivalenceClassCombo.addSelectionListener(equivalenceClassListener);

		Label recordIndexLabel = new Label(superComposite, SWT.NONE);
		recordIndexLabel.setText(DatapoolPlugin.getResourceString("DATA_ROW_DLG_INDEX")); //$NON-NLS-1$

		IDatapoolEquivalenceClass selectedEquivalenceClass = null;
		IDatapoolRecord selectedRecord = null;
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		IDatapoolRecord previousRecord = null;

		if(selectedTableItem != null)
		{
			selectedEquivalenceClass = (IDatapoolEquivalenceClass)selectedTableItem.getData(TAG_EQUIVALENCE_CLASS);
			selectedRecord = (IDatapoolRecord)selectedTableItem.getData(TAG_RECORD);			
		}

		if(previousTableItem != null)
		{
			previousEquivalenceClass = (IDatapoolEquivalenceClass)previousTableItem.getData(TAG_EQUIVALENCE_CLASS);
			previousRecord = (IDatapoolRecord)previousTableItem.getData(TAG_RECORD);
		}
				
		String[] recordIndexLabels = null;
		if(insertionMode)
			recordIndexLabels = getRecordIndexLabels(previousEquivalenceClass);
		else
			recordIndexLabels = getRecordIndexLabels(selectedEquivalenceClass);

		recordIndexCombo = new Combo(superComposite, SWT.READ_ONLY | SWT.DROP_DOWN);
		if(recordIndexLabels != null)
			recordIndexCombo.setItems(recordIndexLabels);
		recordIndexCombo.setData(TAG_EQUIVALENCE_CLASS, previousEquivalenceClass);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 1;
		gridData.widthHint = 55;
		recordIndexCombo.setLayoutData(gridData);
		recordIndexCombo.addSelectionListener(recordIndexListener);
		if(!insertionMode && (previousEquivalenceClass == null || !previousEquivalenceClass.equals(selectedEquivalenceClass)))
		{
			int selectedEquivalenceClassIndex = datapool.getEquivalenceClassIndexById(selectedEquivalenceClass.getId());
			if(selectedEquivalenceClassIndex != -1)
			{
				equivalenceClassCombo.select(selectedEquivalenceClassIndex);
				insertionEquivalenceClassIndex = selectedEquivalenceClassIndex;	
			}			
			recordIndexCombo.select(0);
			insertionRecordIndex = -1;						
		}
		else
		{
			int previousEquivalenceClassIndex = datapool.getEquivalenceClassIndexById(previousEquivalenceClass.getId());
			int previousRecordIndex = getRecordIndex(previousEquivalenceClass, previousRecord);
			if(previousRecordIndex != -1)
			{	
				recordIndexCombo.select(previousRecordIndex + 1);
				insertionRecordIndex = previousRecordIndex;
			}
			if(previousEquivalenceClassIndex != -1)
			{
				equivalenceClassCombo.select(previousEquivalenceClassIndex);
				insertionEquivalenceClassIndex = previousEquivalenceClassIndex;
			}			
		}
		return superComposite;
	}

	private boolean enableOK()
	{
		getButton(IDialogConstants.OK_ID).setEnabled(true);
		return true;
	}

	private String[] getEquivalenceClassLabels(IDatapoolEquivalenceClass[] equivalenceClasses)
	{
		if(equivalenceClasses == null)
			return null;
		int listSize = equivalenceClasses.length;
		String[] returnList = new String[listSize];
		for(int i = 0; i < listSize; i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = equivalenceClasses[i];
			returnList[i] = equivalenceClass.getName();
		}	
		return returnList;
	}

	private String[] getRecordIndexLabels(IDatapoolEquivalenceClass equivalenceClass)
	{
		int listSize = 0;
		if(equivalenceClass != null)
			listSize = equivalenceClass.getRecordCount();
		String[] returnList = new String[listSize + 1];
		for(int i = 0; i < listSize; i++)
		{
			String recordIndex = String.valueOf(i); 
			if(i == 0)
			{
				Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_BEFORE"), new Integer(recordIndex)}; //$NON-NLS-1$
				returnList[i] = MessageFormat.format("{0} {1}", messageElements); //$NON-NLS-1$
			}
			Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_AFTER"), new Integer(recordIndex)};  //$NON-NLS-1$
			returnList[i + 1] = MessageFormat.format("{0} {1}", messageElements);  //$NON-NLS-1$
		}
		if(listSize == 0)
		{
			returnList[0] = DatapoolPlugin.getResourceString("DATA_DLG_FIRST_ELEMENT"); //$NON-NLS-1$
		}		
		return returnList;
	}


	private int findEquivalenceClassIndexInCombo(IDatapoolEquivalenceClass[] equivalenceClasses, IDatapoolEquivalenceClass equivalenceClass)
	{
		if(equivalenceClasses == null || equivalenceClass == null)
			return -1;
		for(int i = 0; i < equivalenceClasses.length; i++)
		{
			if(equivalenceClasses[i].equals(equivalenceClass))
				return i;	
		}
		return -1;
	}

	public int getInsertionEquivalenceClassIndex()
	{
		return insertionEquivalenceClassIndex;
	}

	public int getInsertionRecordIndex()
	{
		return insertionRecordIndex;
	}

	private int getRecordIndex(IDatapoolEquivalenceClass equivalenceClass, IDatapoolRecord record)
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolRecord compRecord = equivalenceClass.getRecord(i);
			if(compRecord.equals(record))
				return i;
		}
		return -1;
	}

	private SelectionListener equivalenceClassListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent e)
		{
			IDatapoolEquivalenceClass equivalenceClass = null; 
			Combo combo = (Combo)e.widget;
			insertionEquivalenceClassIndex = combo.getSelectionIndex();
			IDatapoolEquivalenceClass[] equivalenceClasses = (IDatapoolEquivalenceClass[])combo.getData(TAG_EQUIVALENCE_CLASSES);
			equivalenceClass = equivalenceClasses[insertionEquivalenceClassIndex];

			String[] recordIndexLabels = getRecordIndexLabels(equivalenceClass);
			recordIndexCombo.setItems(recordIndexLabels);
			recordIndexCombo.select(0);
			insertionRecordIndex = -1;
			recordIndexCombo.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
			enableOK();			
		}
	};

	private SelectionListener recordIndexListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent e)
		{
			Combo combo = (Combo)e.widget;
			insertionRecordIndex = combo.getSelectionIndex() - 1;
			enableOK();
		}
	};
}
