/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder.local;

import java.net.InetAddress;
import java.util.Calendar;
import java.util.Date;

import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.recorder.remote.IRecorderMessageTypes;
import org.eclipse.hyades.internal.execution.recorder.ui.views.RecorderControlView;

/**
 * Object which contains data management and direction functionality.  It receives incoming messages from the RAC and sends them to the
 * appropriate API in the contained IRecorderDataProcessor
 * @see org.eclipse.hyades.internal.execution.recorder.local.IRecorderDataProcessor
 * @author Ernest Jessee
 */
public class RecorderDataProcessorHelper implements DataProcessor, IRecorderMessageTypes
{

	/**
	 * the target IRecorderDataProcessor which is the object which will ultimately handle the data.
	 */
	private IRecorderDataProcessor loadTestDataProcessor = null;
	
		
	/**
	 *boolean indicating whether or not a recording session is complete
	 */
	private boolean recordingComplete;

	/**
	 * bytes received counter
	 */
	private int bytesReceived=0;
	
	/**
	 * Time object to hold the time of the last bytes received report.  This is used to verify that reporting is not done to frequently which could
	 * hurt performance
	 */
	private Date lastReceivedReportTime = null;
	
	
	/**
	 * constructor
	 * @param IRecorderDataProcessor theProcessor
	 */
	public RecorderDataProcessorHelper(IRecorderDataProcessor theProcessor)
	{
		loadTestDataProcessor = theProcessor;		
	}
	
	/**
	 * main method for handling incomingData.  It deciphers the control byte and determines which API of the IRecorderDataProcessor should receive
	 * the data.
	 * @param byte[] data
	 * @param int lengthOfData
	 * @param InetAddress peer
	 */
	public void incommingData(byte[] data, int bufferLength, InetAddress peer)
	{
			
		int controlByte = decipherContolByte(data);
		
		byte[] buffer = trimBuffer(data, bufferLength);
				
		 
		switch(controlByte)
		{
			case MESSAGE_TYPE_DATA:
				loadTestDataProcessor.processData(buffer);				
				updateBytesRecievedControl(bufferLength, false);
				break;			
			case MESSAGE_TYPE_CONTROL_STRING:
				loadTestDataProcessor.processControlString(new String(buffer));
				break;
			case MESSAGE_TYPE_COMPLETE_NOTIFICATION:
				complete();
				break;
			case MESSAGE_TYPE_DEBUG_STRING:
			default:
				String debugString = new String(buffer);
//				StringTokenizer debugTokens = new StringTokenizer(debugString,"\n");
//				while(debugTokens.hasMoreTokens())
//				{
//					RecorderControlView.getInstance().addMessage("===>"+debugTokens.nextToken());
//				}				
				loadTestDataProcessor.processDebugString(debugString);
				break;	
				
		}
		
		
	}

	/**
	 * updates the bytes received control if at least 1/2 seconds have past sine the last update
	 * @param int bufferLength
	 */
	private void updateBytesRecievedControl(int bufferLength, boolean ignoreTiming)
	{
		bytesReceived+=bufferLength;
		Date currentTime = Calendar.getInstance().getTime();
		
		if(ignoreTiming || lastReceivedReportTime == null || currentTime.getTime() - lastReceivedReportTime.getTime()>500)
		{
			RecorderControlView.getInstance().setKBytesReceived(bytesReceived/1000);
			lastReceivedReportTime = currentTime;
		}
	}

	/**
	 * removes the control byte from the beginning of the data byte array
	 */
	private byte[] trimBuffer(byte[] data, int length)
	{
		
		byte[] buffer = new byte[length-1];
		
		for(int i=1; i<length; i++)
		{
			buffer[i-1] = data[i];
		}
		return buffer;
	}

	
	/**
	 * reads the control byte from the beginning of the data byte array.
	 */
	private int decipherContolByte(byte[] buffer)
	{		
		return new Integer(buffer[0]).intValue();
		
	}

	

	/**
	 * overload of incommingData
	 */
	public void incommingData(char[] buffer, int length, InetAddress peer)
	{		

	}

	
	/**
	 * 
	 */
	public void invalidDataType(byte[] data, int length, InetAddress peer)
	{
		
			
	}

	
	public void waitingForData()
	{
		

	}

	/**
	 * method called when MESSAGE_TYPE_COMPLETE_NOTIFICATION is received.  As the cleanup thread is watching, setting this boolean to true will
	 * allow the clean up thread to do it's business and kick off scriptgen.
	 */
	private void complete()
	{
		//the first thing we need to do is update the recorder control view to show the actual amt of data received.
		//since we only update every 1/2 seconds, we may be significantly off.
		updateBytesRecievedControl(0,true);
		loadTestDataProcessor.complete();
		recordingComplete = true;
	}

	/**
	 * called to find out whether or not the current recording is complete
	 * @return boolean
	 */
	public boolean isRecordingComplete()
	{
		return recordingComplete;
	}

}
