/**********************************************************************
Copyright (c) 2004 Hyades project.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.internal.config.generator;

import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.Hashtable;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

public class ConfigFile extends DefaultHandler {
	protected static String TAG = ConfigUtility.getString("Config.AgentControllerConfig.Tag");
	private static String lineSeparator = System.getProperty("line.separator");

	protected DocumentBuilderFactory docBuildFactory = null;
	protected DocumentBuilder docBuild = null;
	protected Document doc = null;
	protected Element acConfig = null;
	protected Element holder = null;

	private int size = 100;
	private Hashtable hash = new Hashtable(size);

	private String sr = PlatformObject.sr;

	public ConfigFile() {
		docBuildFactory = DocumentBuilderFactory.newInstance();
		try {
			docBuild = docBuildFactory.newDocumentBuilder();
			createNewDocument();
		} catch (ParserConfigurationException e) {
			Logger.err("Cannot create document builder");
		}
	}

	public ConfigFile(String fileName) {
		this();
		loadExistingDocument(fileName);
	}

	public void saveToFile() {
		String filename = new String(getValue("RASERVER_HOME") + sr + "config" + sr + "serviceconfig.xml");
		saveToFile(filename);
	}

	public void saveToFile(String fileName) {
		PrintWriter writer = null;

		try {
			writer = new PrintWriter(new FileWriter(fileName));
		}
		catch(Exception e) {
			Logger.err(ConfigUtility.getString("Config.Writer.Error.CannotWriteFile") + " " + fileName);
			Logger.err(e.getMessage());
		}

		if(writer != null) {
			writer.print(toString());
			writer.flush();
			writer.close();
		}
	}

	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		String encoding = PlatformObject.encoding; // Set a default value
		String version = "1.0"; // Set a default value

		//
		// XML header information
		//
		strbuf.append("<?xml");

		//
		// Version and encoding
		//
		strbuf.append(" version=\"" + version + "\"");
		strbuf.append(" encoding=\"" + encoding + "\"");

		//
		// Closing the header
		//
		strbuf.append("?>" + ConfigFile.lineSeparator);

		//
		// Print each child node
		//
		NodeList nodes = holder.getChildNodes();
		if(nodes != null) {
			for(int i = 0; i < nodes.getLength(); i++) {
				strbuf.append(ConfigUtility.print(nodes.item(i)));
			}
		}

		return strbuf.toString();
	}

	public void generateConfiguration() {
		generateHyadesConfiguration();
	}

	/**
	 * Generate the basic Hyades Data Collection Engine configuration
	 */
	public void generateHyadesConfiguration() {
		Element acEnv;
		Element app;
		Element var;
		Element param;
		Element hosts;
		Element allow;
		Element plugin;
		String sr = PlatformObject.sr;

		//
		// Set the version to current
		//
		AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));

		//
		// Set the Jvm location
		//
		AgentControllerConfig.setJvm(acConfig, resolveJvmDll(getValue("JAVA_PATH")));

		//
		// Agent Controller Environment
		//
		acEnv = doc.createElement(AgentControllerEnvironment.TAG);
		AgentControllerEnvironment.setConfiguration(acEnv, "default");
		acConfig.appendChild(acEnv);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "JAVA_PATH");
		Variable.setValue(var, getValue("JAVA_PATH"));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "RASERVER_HOME");
		Variable.setValue(var, getValue("RASERVER_HOME"));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, PlatformObject.libEnv);
		Variable.setValue(var, "%RASERVER_HOME%" + sr + PlatformObject.libPath);
		Variable.setPosition(var, "prepend");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "LOCAL_AGENT_TEMP_DIR"); // Bug 59544
		Variable.setValue(var, "%TMP%");
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "SYS_TEMP_DIR");
		Variable.setValue(var, PlatformObject.tempDir);
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "commons-logging.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hcframe.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexcore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexl.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexr.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hl14.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcommons.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlevents.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		/* Bug 58037 begins */
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcbe101.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "ecore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "common.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);
		/* Bug 58037 ends */

		//
		// Application: java.exe
		//
		app = doc.createElement(Application.TAG);
		Application.setConfiguration(app, "default");
		Application.setExecutable(app, "java.exe");
		Application.setPath(app, "%JAVA_PATH%");
		Application.setLocation(app, "%SYS_TEMP_DIR%");
		acConfig.appendChild(app);

		//
		// Hosts
		//
		hosts = doc.createElement(Hosts.TAG);
		Hosts.setConfiguration(hosts, "default");
		acConfig.appendChild(hosts);

		if(getValue("ALLOW") != null) {
			if(getValue("ALLOW").toLowerCase().equals("custom")) {
				String[] allowedHostList = ConfigUtility.stringToArray(getValue("HOSTS"));
				if(allowedHostList.length > 0) {
					for(int i = 0; i < allowedHostList.length; i++) {
						allow = doc.createElement(Allow.TAG);
						hosts.appendChild(allow);
						if(allowedHostList[i].equalsIgnoreCase("all") || allowedHostList[i].equalsIgnoreCase("local")) {
							Allow.setHost(allow, allowedHostList[i].toUpperCase());
						}
						else {
							Allow.setHost(allow, allowedHostList[i]);
						}
					}
				}
				else {
					allow = doc.createElement(Allow.TAG);
					Allow.setHost(allow, "LOCAL");
					hosts.appendChild(allow);
				}
			}
			else {
				allow = doc.createElement(Allow.TAG);
				Allow.setHost(allow, getValue("ALLOW").toUpperCase());
				hosts.appendChild(allow);
			}
		}
		else {
			allow = doc.createElement(Allow.TAG);
			Allow.setHost(allow, "LOCAL");
			hosts.appendChild(allow);
		}

		//
		// Plugins
		//
		plugin = doc.createElement(Plugin.TAG);
		Plugin.setPath(plugin, "%RASERVER_HOME%" + sr + "plugins");
		acConfig.appendChild(plugin);
	}

	public void createNewDocument() {
		DOMImplementation dom = docBuild.getDOMImplementation();

		doc = dom.createDocument(null, TAG, null);

		holder = doc.createElement("ElementHolder");

		//
		// There can be only one AgentControllerConfiguration element
		//
		acConfig = doc.createElement(AgentControllerConfig.TAG);
		AgentControllerConfig.setActiveConfiguration(acConfig, "default");
		AgentControllerConfig.setLoggingDetail(acConfig, "LOW");
		AgentControllerConfig.setLoggingLevel(acConfig, "INFORMATION");
		AgentControllerConfig.setPort(acConfig, "10002");
		AgentControllerConfig.setSecuredPort(acConfig, "10003");
		AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));
		AgentControllerConfig.setFilePort(acConfig, "10005"); // Bug 60082
		AgentControllerConfig.setProcessPolling(acConfig, "true"); // Bug 59316
		holder.appendChild(acConfig); // Add to the root holder
	}

	public void loadExistingDocument(String file) {
		SAXParser saxp = null;
		XMLReader parser = null;

		//
		// Create the parser
		//
		try {
			saxp = SAXParserFactory.newInstance().newSAXParser();
			parser = saxp.getXMLReader();
			parser.setContentHandler(this);
			parser.setErrorHandler(this);
		} catch (Exception e) {
			Logger.err(ConfigUtility.getString("Config.Parser.Error.CreateParser") + " " + parser);
			Logger.err(e.getMessage());
		}

		if(parser != null) {
			try {
				parser.parse(file);
			} catch (Exception e) {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.ParseFile") + " " + file);
				Logger.err(e.getMessage());
				if(e instanceof SAXException) {
					Exception e1 = ((SAXException)e).getException();
					if(e1 != null) {
						Logger.err(e1.getMessage());
					}
				}
			}
		}
	}

	public void startElement(String uri, String local, String raw, Attributes attrs) throws SAXException {
		//
		// UserDefinition
		//
		if(raw.equals(UserDefinition.TAG)) {
			String authenticationType = null;
			String name = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.AuthenticationType.Tag"))) {
					authenticationType = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.Name.Tag"))) {
					name = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}
			if((authenticationType != null) && (name != null)) {
				setValue("SECURITY", "true");
				if(getValue("USERS") == null) {
					setValue("USERS", name);
				}
				else {
					setValue("USERS", getValue("USERS") + "," + name);
				}

			}
		}

		//
		// Variable
		//
		else if(raw.equals(Variable.TAG)) {
			String name = null;
			String value = null;
			String position = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Name.Tag"))) {
					name = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Value.Tag"))) {
					value = attrs.getValue(i);
				}
				else if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Position.Tag"))) {
					position = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}

			if((name != null) && (value != null) && (position != null)) {
				//
				// Populate the hash table, do not add CLASSPATH or PATH
				//
				if(!name.equals("CLASSPATH") && !name.equals("PATH")) {
					setValue(name, value);
				}
			}
		}

		//
		// Allow
		//
		else if(raw.equals(Allow.TAG)) {
			String name = null;

			for(int i = 0; i < attrs.getLength(); i++) {
				if(attrs.getQName(i).equals(ConfigUtility.getString("Config.Allow.Host.Tag"))) {
					name = attrs.getValue(i);
				}
				else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}

			if(name != null) {
				if(name.toLowerCase().equals("all") || name.toLowerCase().equals("local")) {
					setValue("ALLOW", name.toUpperCase());
					removeKey("HOSTS");
				}
				else {
					setValue("ALLOW", "CUSTOM");
					if(getValue("HOSTS") == null) {
						setValue("HOSTS", name);
					}
					else {
						setValue("HOSTS", getValue("HOSTS") + "," + name);
					}
				}
			}
		}

		//
		// Unrecognized element name
		//
		else {
			Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedElement") + " " + local);
		}
	}

	public String resolveJvmDll(String string) {
		if(PlatformObject.name.equals("OS/400")) {
			return new String("QSYS/QJVAJNI"); // Bug 58152
		}
		else {
			int index;

			index = string.lastIndexOf(sr + "java" + PlatformObject.exeExt);
			if(index == -1) {
				index = string.lastIndexOf(sr + "javaw" + PlatformObject.exeExt);
				if(index == -1) {
					return new String("%JAVA_PATH%");
				}
			}

			String str = string.substring(0, index);
			String rc = null;

			try {
				rc = ConfigUtility.getJvmLib(str);
			}
			catch(Exception e) {
				Logger.out(ConfigUtility.getString("Config.Jvm.Warning.CannotFindJvmLibrary") + ": " + PlatformObject.jvmLib);
			}

			if(rc == null) {
				return new String("%JAVA_PATH%");
			}
			else {
				return rc;
			}
		}
	}

	/**
	 * @return Returns the acConfig.
	 */
	public Element getAcConfig() {
		return acConfig;
	}

	public String getValue(String tag) {
		return (String)hash.get(tag);
	}

	public void setValue(String tag, String value) {
		hash.put(tag, value);
	}

	public void removeKey(String tag) {
		hash.remove(tag);
	}

	public boolean isExist(String tag) {
		return hash.containsKey(tag);
	}
}
