/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui;
import java.io.*;
import java.util.*;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.plugin.AbstractUIPlugin;


public class Debugger {

public String plugin_name = "";
AbstractUIPlugin plugin = null;	

HashMap specials = new HashMap();

	public Debugger(String plugin_name) {
		this(null,plugin_name);
	}

	private void addSpecials(String s) {
		if (s==null) return;
		StringTokenizer tok = new StringTokenizer(s,",");
		while (tok.hasMoreTokens()) {
			String token = tok.nextToken();
			System.out.println("DEBUGGER SPECIAL TOKEN:"+token);
			specials.put(token,new Boolean(true));	
		}
	}

	public Debugger(AbstractUIPlugin plugin, String plugin_name) {

		String special_debug_info = System.getProperty("SPECIAL_DEBUG_INFO");
		addSpecials(special_debug_info);	

		String special_debug = System.getProperty("SPECIAL_DEBUG");
		addSpecials(special_debug);	

		String special_debug_all = System.getProperty("SPECIAL_DEBUG_ALL");
		if (special_debug_all != null) {
			special_debug_info = special_debug_all;
			special_debug = special_debug_all;	
		}

//		System.err.println("NEW DEBUGGER "+plugin_name);

		this.plugin = plugin;
		this.plugin_name = plugin_name;
		
		GENERATE_STACKTRACE = false;
		PRINT_STACKTRACE = true;
		
		ALL_TO_STDERR = false;
		ALL_TO_STDOUT = false;
	
		try {
			String debug;
	
			debug = Platform.getDebugOption(plugin_name+"/debug/error");
			if (debug != null) {
				PRINT_ERROR = debug.equalsIgnoreCase("true");	
//			} else {
//				PRINT_ERROR = true;
//				System.err.println("Debugger / "+plugin_name+": Error debug = null");	
			}
		
			debug = Platform.getDebugOption(plugin_name+"/debug/warning");
			if (debug != null) {
				PRINT_WARNING = debug.equalsIgnoreCase("true");	
//			} else {
//				PRINT_WARNING = true;
//				System.err.println("Debugger / "+plugin_name+": Warning debug = null");	
			}
		
			debug = Platform.getDebugOption(plugin_name+"/debug/info");
			if (debug != null) {
				PRINT_INFO = debug.equalsIgnoreCase("true");	
//			} else {
//				PRINT_INFO = true;
//				System.err.println("Debugger / "+plugin_name+": Info debug = null");	
			}
			
			boolean MASTER = false;
			
			debug = Platform.getDebugOption(plugin_name+"/debug");
			if (debug != null) {
				MASTER = debug.equalsIgnoreCase("true");
//			} else {
//				MASTER = true;
//				System.err.println("Debugger / "+plugin_name+": Debug = null");	
			}

			PRINT_ERROR = PRINT_ERROR && MASTER;	
			PRINT_WARNING = PRINT_WARNING && MASTER;	
			PRINT_INFO = PRINT_INFO && MASTER;	

			debug = Platform.getDebugOption(plugin_name+"/debug/showclass");
			if (debug != null) {
				SHOW_CLASS = debug.equalsIgnoreCase("true");	
			}

			debug = Platform.getDebugOption(plugin_name+"/debug/showthread");
			if (debug != null) {
				SHOW_THREAD = debug.equalsIgnoreCase("true");	
			}

		} catch (Throwable e) {
			e.printStackTrace();
		}

		if (special_debug_info != null) {
			PRINT_INFO = true;	

			PRINT_WARNING = true;
			PRINT_ERROR = true;	
		}

		if (special_debug != null) {
			PRINT_WARNING = true;
			PRINT_ERROR = true;	
		}			
	
		info(plugin_name+" TEST INFORMATION MESSAGE");
		warning(plugin_name+" TEST WARNING MESSAGE");
		error(plugin_name+" TEST ERROR MESSAGE");
	
	}

	public void setPlugin(AbstractUIPlugin plugin) {
		this.plugin = plugin;
//		logVisibleError(new Throwable("DEBUGGER GENERATED TEST ERROR - no dialog"),"THIS ALSO IS A TEST ERROR",false);
//		logVisibleError(new Throwable("DEBUGGER GENERATED TEST ERROR - with dialog"),"THIS IS A TEST ERROR",true);
	}
	
	public boolean check(String debug_option) {

		//check if we are showing all specials
		String special_debug = System.getProperty("SPECIAL_DEBUG_ALL");
		if (special_debug != null) {
			return true;	
		}

		//check if this particular special has been turned on
		if (specials.get(debug_option) != null) return true;

		String debug = Platform.getDebugOption(plugin_name+"/debug/"+debug_option);
		if (debug != null) {
			return debug.equalsIgnoreCase("true");	
		}
		return false;
	}

///////////////////////////////////////////////////////////////////////
// Settings
//
	// Print information messages
	// Print warning messages
	// Print error messages
	public boolean PRINT_INFO = false;
	public boolean PRINT_WARNING = false;
	public boolean PRINT_ERROR = false;

	public boolean SHOW_THREAD = true;
	public boolean SHOW_CLASS = true;

	// Log everything to the cli
	// Create a log file per thread
	public boolean ALL_TO_STDOUT = false;
	public boolean ALL_TO_STDERR = false;

	// Print the stack trace if it is given
	// Generate a stack trace for all warnings and errors
	public boolean PRINT_STACKTRACE = true;
	public boolean GENERATE_STACKTRACE = false;

//
///////////////////////////////////////////////////////////////////////

	private HashMap mark_map = new HashMap();

	private void logStdOut (String message, String thread) {

		if (ALL_TO_STDERR) {
			System.err.print(message);
		} else {
			System.out.print(message);
		}//end if
	
	}//end method

	private void logStdErr (String message, String thread) {

		if (ALL_TO_STDOUT) {
			System.out.print(message);
		} else {
			System.err.print(message);
		}//end if
	
	}//end method
/*
	private void log (String message, String thread) {

		if (LOG_TO_STDOUT) {
			System.out.print(message);
		}//end if
		if (LOG_TO_STDERR) {
			System.err.print(message);
		}//end if
	
	}//end method
*/
	/**
	 * Print some text - this method is used for timing
	 * @param textToPrint the text to print
	 */
	public void mark( String textToPrint ) {
		if (PRINT_INFO) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			
			long tmp = 0;
			try {
				long now = System.currentTimeMillis();
				Long mark = (Long) mark_map.get(thread);
				if (mark == null) {
					mark = new Long(System.currentTimeMillis());
				}//end if
				tmp = now - mark.longValue();
				mark_map.put( thread , new Long(now) );
				
			} catch (Exception e) {}
			
			String message = "M:("+thread+"):("+tmp+"ms):"+caller+":"+textToPrint+"\n";
			logStdOut(message,thread);
		}//end if
	}//end method



	/**
	 * Print something at the information level
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) to print the stacktrace of
	 */
	public void info( String textToPrint, Throwable t ) {
		if (PRINT_INFO) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "I:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			logStdOut(message,thread);
		}//end if
	}//end method

	/**
	 * Print something at the information level
	 * @param textToPrint the text to print
	 */
	public void info( String textToPrint ) {
		if (PRINT_INFO) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "I:("+thread+"):"+caller+":"+textToPrint+"\n";
			logStdOut(message,thread);
		}//end if
	}//end method

	/**
	 * Print some text as a warning
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) which pertains to this warning
	 */
	public void warning( String textToPrint, Throwable t ) {
		if (PRINT_WARNING) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "W:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			logStdOut(message,thread);
		}//end if
	}//end method
	
	/**
	 * Print some text as a warning
	 * @param textToPrint the text to print
	 */
	public void warning( String textToPrint ) {
		if (PRINT_WARNING) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "W:("+thread+"):"+caller+":"+textToPrint+"\n";
			if (GENERATE_STACKTRACE) {
				message = message + getStackTrace();
			}//end if
			logStdOut(message,thread);
		}//end if
	}//end method

	/**
	 * Print some text as an error
	 * @param textToPrint the text to print
	 * @param t the Throwable (Exception) which pertains to this error
	 */
	public void error( String textToPrint, Throwable t ) {
		if (PRINT_ERROR) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "E:("+thread+"):"+caller+":"+textToPrint+": "+t+"\n";
			if (PRINT_STACKTRACE) {
				message = message + getStackTrace(t);
			}//end if
			logStdErr(message,thread);
		}//end if
	}//end method

	/**
	 * Print some text as an error
	 * @param textToPrint the text to print
	 */
	public void error( String textToPrint ) {
		if (PRINT_ERROR) {
			String caller = "";
			if (SHOW_CLASS) caller = getCallerStackTrace();
			String thread = "";
			if (SHOW_THREAD) thread = Thread.currentThread().getName()+"/"+Thread.currentThread().hashCode();
			String message = "E:("+thread+"):"+caller+":"+textToPrint+"\n";
			if (GENERATE_STACKTRACE) {
				message = message + getStackTrace();
			}//end if
			logStdErr(message,thread);
		}//end if
	}//end method

	/**
	 * log an error to the hyades PDE error log - the user will be able to see this but their workflow will not
	 * be disrupted by a dialog
	 * @param t
	 */
	public void logVisibleError(Throwable t) {
		error("Error: "+t.getMessage(),t);
		logVisibleError(t,"Error: "+t.getMessage(),false);
	}
	
	/**
	 * log an error to the hyades PDE error log and optionally show a dialog
	 * @param t the exception to log
	 * @param error_msg the error message to log (may be null but it is recommended that it is not)
	 * @param show_dialog whether to show a dialog to the user
	 */
	public void logVisibleError(Throwable t, String error_msg, boolean show_dialog) {
		error(error_msg,t);
		if (plugin != null) {
			
			String plugin_id = plugin.getDescriptor().getUniqueIdentifier();
			String throwable_str = t.toString();
			
			Status status = new Status(IStatus.ERROR, plugin_id, IStatus.OK, throwable_str, t);
			if (error_msg != null) {
				status = new Status(IStatus.ERROR, plugin_id, IStatus.OK, error_msg, t);
			}
			plugin.getLog().log(status);
			
			if (show_dialog) {
				MultiStatus mstatus = new MultiStatus(plugin_id, IStatus.ERROR, throwable_str, t);
		
				StringBuffer sb = new StringBuffer();
				sb.append("Plugin ID...................").append(plugin_id).append("\n");
				sb.append("Plugin Provider.............").append(plugin.getDescriptor().getProviderName()).append("\n");
				sb.append("Plugin Version..............").append(plugin.getDescriptor().getVersionIdentifier()).append("\n");
				sb.append("Plugin Install URL..........").append(plugin.getDescriptor().getInstallURL()).append("\n");
				sb.append(getStackTrace(t));
		
				buildStatus(mstatus,sb.toString());		
				
				Display display = plugin.getWorkbench().getActiveWorkbenchWindow().getShell().getDisplay();
				if (display == null) display = Display.getCurrent();
				if (display == null) display = Display.getDefault();
				
				display.syncExec(new DialogShower(mstatus,error_msg,throwable_str));
			}
			
		} else {
			error("failed to log visible error - plugin is NULL",t);	
		}
	}
	
	class DialogShower implements Runnable {
		MultiStatus mstatus;
		String error_msg;
		String throwable_str;
		public DialogShower(MultiStatus mstatus, String error_msg, String throwable_str) {
			this.mstatus = mstatus;
			this.error_msg = error_msg;
			this.throwable_str = throwable_str;
		}
		public void run() {
			try {
				IWorkbenchWindow def_window = plugin.getWorkbench().getActiveWorkbenchWindow();
				if (def_window == null) {
					def_window = plugin.getWorkbench().getWorkbenchWindows()[0];
				} 

				DialogOpener dialog = new DialogOpener();
				dialog.shell = def_window.getShell();
				dialog.status = mstatus;
				if (error_msg != null) {
					dialog.throwable_str = error_msg;
				} else {
					dialog.throwable_str = throwable_str;
				}
				dialog.msg = "Error in "+plugin_name;
				
				dialog.shell.getDisplay().asyncExec(dialog);

			} catch (Throwable x) {
				error("failed to show error dialog",x);
			} 
		}
	}
	
	public void buildStatus(MultiStatus mstatus, String s) {
	
		s = s.replace('\t',' ');

		StringTokenizer stok = new StringTokenizer(s,"\r\n");
		while (stok.hasMoreTokens()) {
			String tok = stok.nextToken();
			Status status = new Status(mstatus.getCode(),mstatus.getPlugin(),mstatus.getSeverity(),tok+"\r\n",mstatus.getException());
			mstatus.add(status);
		}
	}

	/**
  	 * Get the line of the stack trace pertaining to the object calling this Debug method
	 */
	private String getCallerStackTrace() {
		Throwable t = new Throwable();
		return(getLineOfStackTrace( t, getStackTrace(t), 4));
	}//end method

	/**
  	 * Get a specific line of a stack trace without the "   at " and the newline after it
	 */
	private String getLineOfStackTrace( Throwable t, String stackTrace, int lineNumber ) {
		int beginIndex = 0;
		int endIndex = 0;

		String cname = t.getClass().getName();

		//jump one line down from the last occurrence of the exception class
		int lastindex = stackTrace.lastIndexOf(cname);
		if (lastindex == -1) {
			lastindex = 0;	
		} else {
			lastindex += cname.length();	
		}

		//find string based on newline chars
		int index = lastindex;
		for (int i = 0; i < lineNumber; i++) {
			beginIndex = index+5; // get rid of the "    at "
			index = stackTrace.indexOf("\n",index+1);
			endIndex = index-1;  // get rid of the newline
		}//end for

		return stackTrace.substring( beginIndex, endIndex );
	}//end method

	/**
	 * Returns a stack trace for the Throwable (Exception)
	 * @param t the Throwable (Exception) to get the stack trace for
	 * @return a string representation of a stack trace for the Throwable
	 */
	public String getStackTrace(Throwable t) {
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method

	/**
	 * Returns a stack trace for the Throwable (Exception)
	 * @param t the Throwable (Exception) to get the stack trace for
	 * @return a string representation of a stack trace for the Throwable
	 */
	public String getStackTrace() {
		Throwable t= new Throwable("DEBUGGER GENERATED STACKTRACE");
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		PrintStream ps = new PrintStream(os);   // printing destination
		t.printStackTrace(ps);
		return os.toString();
	}//end method

	class DialogOpener implements Runnable {
		Shell shell;
		String msg;
		String throwable_str;
		Status status;
		public void run() {
			ErrorDialog.openError(shell, msg, throwable_str, status);
		}	
	}

private static Object timers_lock = new Object();
private static HashMap timers = new HashMap();

	public static void startAccumulator(String timer) {
		
		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				thread_timers = new HashMap();	
			}
			thread_timers.put("ACC:"+timer,new Long(System.currentTimeMillis()));	
			timers.put(Thread.currentThread(),thread_timers);
		}
	}
	public static void resetAccumulator(String timer) {
		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				return;
			}
			thread_timers.put("ACC_TOT:"+timer,new Long(0));
		}		
	}
	public static String getAccumulator(String timer) {
		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				return "ACCUMULATOR NOT FOUND";
			}
			Long t = (Long)thread_timers.get("ACC_TOT:"+timer);

			if (t != null) {
				return timer+":"+t;
			} else {
				return "ACCUMULATOR NOT FOUND";	
			}	
		}		
	}
	public static String finishAccumulator(String timer) {
		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				return "ACCUMULATOR NOT FOUND";
			}
	
			Long t = (Long)thread_timers.get("ACC:"+timer);
			long add = System.currentTimeMillis()-t.longValue();

			Long total = (Long)thread_timers.get("ACC_TOT:"+timer);
			if (total != null) {
				add += total.longValue();
			}
			
			thread_timers.put("ACC_TOT:"+timer,new Long(add));
			
			if (t != null) {
				return timer+":"+add;
			} else {
				return "ACCUMULATOR NOT FOUND";	
			}	
		}
	}

	public static void startTimer(String timer) {
		
		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				thread_timers = new HashMap();	
			}
			thread_timers.put("TIM:"+timer,new Long(System.currentTimeMillis()));	
			timers.put(Thread.currentThread(),thread_timers);
		}
	}
	public static String finishTimer(String timer) {

		synchronized(timers_lock) {
			HashMap thread_timers = (HashMap)timers.get(Thread.currentThread());
			if (thread_timers == null) {
				return "TIMER NOT FOUND";
			}
	
			Long t = (Long)thread_timers.get("TIM:"+timer);
			if (t != null) {
				return timer+":"+(System.currentTimeMillis()-t.longValue());
			} else {
				return "TIMER NOT FOUND";	
			}	
		}
	}


}//end class
