/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.Dirtiable;

import java.io.*;
import java.util.ArrayList;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.URI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.part.EditorPart;

public class StatisticalModelGraphEditor extends EditorPart implements Dirtiable, IResourceChangeListener {

long last_modified = 0;

IEditorSite site;
IFileEditorInput input;	
IFile file;
String filename;

/* the statistical model (assumed to contain graphable aggregations) */
ArrayList configs = new ArrayList();
StatisticalModelGraphViewer agg_editor;

ArrayList disposables = new ArrayList();

	public StatConInterface getStatCon() {
		return agg_editor;
	}

	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		this.setSite(site);
		this.setInput(input);

		this.site = site;
		this.input = (IFileEditorInput)input;

		EditorPlugin.DBG.info("StatCon Editor got editor input");
		
		file = this.input.getFile();
		filename = file.getName();

		this.setTitle("StatCon - "+file.getName());
		EditorPlugin.DBG.info("set title - "+this.getTitle());

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		workspace.addResourceChangeListener(this);
	
		try {
			EditorPlugin.DBG.info("loading file");
			load(file);
			last_modified = file.getModificationStamp();	
		} catch (CoreException e) {
			e.printStackTrace();
			throw new PartInitException(EditorPlugin.getString("ERROR_OCCURRED")+":"+e.getMessage());
		} catch (IOException e) {
			e.printStackTrace();
			throw new PartInitException(EditorPlugin.getString("IO_ERROR_OCCURRED")+":"+e.getMessage());
		}
		if (configs.size() == 0) {
			throw new PartInitException(EditorPlugin.getString("NO_STAT_MODEL_FOUND"));
		}
	}
	
	
	
	public void load(IFile file) throws CoreException, IOException {

		if (file == null) {
			throw new IOException(EditorPlugin.getString("NULL_FILE"));
		}

		configs.add(file);

		EditorPlugin.DBG.info("asked to load file URI "+URI.createPlatformResourceURI(file.getFullPath().toString()));
	}

	
	public void createPartControl(Composite parent) {
		System.gc();

		long free= Runtime.getRuntime().freeMemory();
//		long max = Runtime.getRuntime().maxMemory();
		long total = Runtime.getRuntime().totalMemory();
		
		double d = total-free;
		d = d / (double)total;
		d = d * 100.0d;
		
		EditorPlugin.DBG.info("MEMORY USED = "+((total-free)/1024)+"k ("+d+"%)");

			
		URI curr_project = null;
		try {
			curr_project = URI.createPlatformResourceURI(file.getParent().getFullPath().toString());
		} catch (Exception e) {
		}
		agg_editor = new StatisticalModelGraphViewer(parent,0,false,this,curr_project);

		for (int i = 0; i < configs.size(); i++) {

			try {
				IFile file = (IFile)configs.get(i);
				InputStream filecontents = file.getContents();
	
				agg_editor.applyConfig(filecontents);
			
			} catch (Throwable e) {
				e.printStackTrace();
			}			
		}

		setDirty(false);
		
		disposables.add(agg_editor);		
		
	}
		
	public void dispose() {
		agg_editor.unloadModules();
		
		super.dispose();
		for (int i = 0; i < disposables.size(); i++) {
			try {
			Object o = disposables.get(i);
			if (o != null) {
				if (o instanceof Widget) {
					((Widget)o).dispose();	
				} else if (o instanceof Color) {
					((Color)o).dispose();	
				} else if (o instanceof Image) {
					((Image)o).dispose();	
				} else {
					try {
						EditorPlugin.disposeObject(o);
					} catch (Throwable e) {
						EditorPlugin.DBG.warning("failed to dispose object "+o);
					}
				}
			}
			} catch (Throwable e) {
				EditorPlugin.DBG.error("failed to dispose properly",e);
			}
		}		
	}
	
	public class EditorClose implements Runnable {
		public void run() {
			EditorPlugin.DBG.info("closing Editor");
			//close this editor
			if (!isDirty()) {
				EditorPlugin.DBG.info("editor is not dirty - will close now");
				site.getPage().closeEditor(StatisticalModelGraphEditor.this,false);
			} else {
				EditorPlugin.DBG.info("file has been deleted, editor is dirty - asking user for input");
				MessageBox mbox = new MessageBox(agg_editor.getGraphWindow().getShell(),SWT.ICON_WARNING|SWT.YES|SWT.NO);
				mbox.setMessage(EditorPlugin.getString("EDITOR_FILE_DELETED")+" ("+filename+"), "+EditorPlugin.getString("EDITOR_FILE_DELETED_2")); 
				mbox.setText(EditorPlugin.getString("EDITOR_FILE_DELETED_TITLE")+" - "+getTitle());
				int result = mbox.open();
				if (result == SWT.YES) {
					last_modified = IResource.NULL_STAMP;
					doSaveAs();
				}
				site.getPage().closeEditor(StatisticalModelGraphEditor.this,false);
			}

		}	
	}
	
	public void resourceChanged(IResourceChangeEvent e) {

		if (e == null) return;
		if (e.getDelta() == null) return;

		try {
			if (agg_editor.getGraphWindow().isDisposed()) return;
			
		if ((e.getDelta().getKind() & IResourceDelta.REMOVED) != 0
			|| (e.getDelta().getKind() & IResourceDelta.CHANGED) != 0) {
			if (!file.exists()) {
				agg_editor.getGraphWindow().getDisplay().asyncExec(new EditorClose());
			}
		}	
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}
	
	public boolean requestOverwrite() {
		try {
			file.refreshLocal(IResource.DEPTH_ZERO,null);
		} catch (Throwable e) {}
		if (last_modified != file.getModificationStamp()) {
			MessageBox mbox = new MessageBox(agg_editor.getGraphWindow().getShell(),SWT.ICON_WARNING|SWT.YES|SWT.NO|SWT.CANCEL);	
			mbox.setMessage(EditorPlugin.getString("EDITOR_FILE_CHANGED"));
			mbox.setText(EditorPlugin.getString("EDITOR_FILE_CHANGED_TITLE"));
			int result = mbox.open();
			if (result != SWT.YES) {
				return false;
			}
		}
		return true;
	}
	
	public void doSave(IProgressMonitor monitor) {

		EditorPlugin.DBG.info("saving file");

		if (!requestOverwrite()) return;

		//save
		try {
			String xmlconfig = agg_editor.generateConfig();
			
			ByteArrayInputStream bin = new ByteArrayInputStream(xmlconfig.getBytes());
			
//			file.setContents(bin,true,false,monitor);
			if (file.exists()) {
				file.setContents(bin,true,false,null);
			} else {
				file.create(bin,true,null);
			}

			last_modified = file.getModificationStamp();	
						
			setDirty(false);
		} catch (Throwable e) {
			EditorPlugin.DBG.error("problem saving file",e);
		}
	}

	public void doSaveAs() {

		EditorPlugin.DBG.info("saving file As");

		if (!requestOverwrite()) return;
		
		try {
			
			IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
			
			FileDialog dialog = new FileDialog(agg_editor.getGraphWindow().getShell(),SWT.SAVE);
			
			dialog.setFileName("new.statcon");
			dialog.setFilterExtensions(new String[]{".statcon"});
			dialog.setFilterPath(workspaceRoot.getLocation().toString());
//			dialog.setText("Save As");

			String file = dialog.open();
			EditorPlugin.DBG.info("user selected file "+file);
			
			if (file != null) {
				//get an IFile reference and save stuff to it
				IPath path = new Path(file); 
				EditorPlugin.DBG.info("user selected path "+path);

				IFile ifile = workspaceRoot.getFileForLocation(path);

				String xmlconfig = agg_editor.generateConfig();
			
				ByteArrayInputStream bin = new ByteArrayInputStream(xmlconfig.getBytes());

				EditorPlugin.DBG.info("trying to set contents of "+ifile);
	
				if (ifile.exists()) {
					ifile.setContents(bin,true,false,null);
				} else {
					ifile.create(bin,true,null);
				}
				EditorPlugin.DBG.info("new file saved ok");

				last_modified = ifile.getModificationStamp();	

			} else {
				EditorPlugin.DBG.info("user cancelled Save As");
			}
			
		} catch (Throwable e) {
			EditorPlugin.DBG.error("problem saving file As",e);
		}
	}

	public void gotoMarker(IMarker marker) {
	}
		
	public boolean isSaveAsAllowed() {
		return true;
	}

boolean dirty = false;

	public void setDirty(boolean b) {
		dirty = b;
		firePropertyChange(PROP_DIRTY);
	}
	
	public boolean isDirty() {
		return dirty;
	}
	
	public void setFocus() {
		//do nothing
	}
}