/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.perfmon;

import java.util.*;
import org.eclipse.hyades.perfmon.xml.*;

import org.eclipse.emf.common.util.*;

import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.model.statistical.*;

public class StatisticalXMLTranslator implements XMLParserListener {

StatisticalFactory factory = StatisticalFactory.eINSTANCE;

SDDescriptor root;

HashMap idmap = new HashMap();
HashMap descriptormap = new HashMap();

SDDescriptor sdparent = null;
SDDescriptor sd = null;

SDSampleWindow window;

boolean inside_description = false;

ArrayList listeners = new ArrayList();

	public String getID(SDDescriptor counter) {
		return (String)descriptormap.get(counter);
	}
	
	public SDDescriptor getDescriptor(String id) {
		return (SDDescriptor)idmap.get(id);	
	}

	public void addLoaderListener(StatisticalLoaderListener listener) {
		listeners.add(listener);
	}

	public void removeLoaderListener(StatisticalLoaderListener listener) {
		listeners.remove(listener);	
	}

	public StatisticalXMLTranslator(TRCAgent agent, String name) {
		root = factory.createSDDescriptor();
		root.setName(name);
		root.setAgent(agent);
		
		window = getOrCreateView(agent);
	}

	public StatisticalXMLTranslator(SDDescriptor model) {
		root = model;

		window = getOrCreateView(model.getAgent());
	}

	private SDSampleWindow getOrCreateView(TRCAgent trcagent) {
		
		SDView view = (SDView)trcagent.getView();

		//create a view if none exists
		if (view == null) {
			view = factory.createSDView();
			view.setAgent(trcagent);
			view.setName("view");
		}
		
		//create a sample window if none exists
		if (view.eContents().size() == 0) {
			SDSampleWindow window = factory.createSDSampleWindow();
			window.setView(view);
			
			return window;
		}

		return (SDSampleWindow)view.eContents().get(0);
	}
	
	public SDDescriptor getModel() {
		return root;	
	}
	
	private void modelChanged(SDDescriptor parent) {
		for (int i = 0; i < listeners.size(); i++) {
			StatisticalLoaderListener listener = (StatisticalLoaderListener)listeners.get(i);
			listener.modelChanged(parent);
		}	
	}
	
	public void startElement(String tag,HashMap h) throws Exception {
		if (tag.equals("description")) {
			inside_description = true;
			return;	
		}

		if (tag.equals("RangeRepresentation")) return;
		if (tag.equals("GaugeRepresentation")) return;

		//if parent == null then add at root
		
		//Descriptor information
		String id = null;
		String name = null;
		String parent = null;
		
		//Observation information
		double time = 0;
		String value = null;
		String descriptor = null;

		if (tag.endsWith("Observation")) {
			time = Double.parseDouble((String)h.get("time"));
			value = (String)h.get("value");
			descriptor = (String)h.get("memberDescriptor");

			sd = (SDDescriptor)idmap.get(descriptor);
		}

		//some kind of descriptor - get it and its parent
		if (tag.endsWith("Descriptor")) {
			
			id = (String)h.get("id");
			name = (String)h.get("name");
			parent = (String)h.get("parent");

			if (id != null) {
				sd = (SDDescriptor)idmap.get(id);
				if (sd != null) return;
			}
		
			if (parent != null) {
				sdparent = (SDDescriptor)idmap.get(parent);
			}
			if (parent == null || sdparent == null) {
				sdparent = root;
			}
		}

		if (tag.equals("Descriptor")) {
	
			if (sd == null) {		
				sd = factory.createSDDescriptor();
				sd.setParent(sdparent);
				sd.setName(name);

				idmap.put(id,sd);
				descriptormap.put(sd,id);
				
				modelChanged(sdparent);
			}
			
		} else if (tag.equals("CounterDescriptor")) {
			
			if (sd == null) {		
				sd = factory.createSDCounterDescriptor();
				sd.setParent(sdparent);
				sd.setName(name+" - "+id);

				idmap.put(id,sd);
				descriptormap.put(sd,id);

				modelChanged(sdparent);
			}

		} else if (tag.equals("SampleDescriptor")) {
			
			if (sd == null) {		
				sd = factory.createSDSampleDescriptor();
				sd.setParent(sdparent);
				sd.setName(name);

				idmap.put(id,sd);
				descriptormap.put(sd,id);

				modelChanged(sdparent);
			}
			
		} else if (tag.equals("ContiguousObservation")) {
			
			SDMemberDescriptor mem = (SDMemberDescriptor)sd;
			List list = mem.getSnapshotObservation();

			//check to see if we have a contiguous observation already			
			SDContiguousObservation obs = null;
			for (int i = 0; i < list.size(); i++) {
				SDSnapshotObservation tmp = (SDSnapshotObservation)list.get(i);
				if (tmp instanceof SDContiguousObservation) {
					obs = (SDContiguousObservation)tmp;	
					break;	
				}
			}
			//if there isnt one then add one			
			if (obs == null) {
				obs = factory.createSDContiguousObservation();
				obs.setMemberDescriptor(mem);
				obs.setWindow(window);
				list.add(obs);	//?

				modelChanged(mem);
			}

			//get the data lists
			BasicEList times = (BasicEList)obs.getCreationTime();
			BasicEList values = (BasicEList)obs.getValue();

			times.addUnique(new Double(time));						
			values.addUnique(new Double(value));						
			
		} else if (tag.equals("DescreteObservation")
					|| tag.equals("DiscreteObservation")) {

			SDMemberDescriptor mem = (SDMemberDescriptor)sd;
			List list = mem.getSnapshotObservation();

			//check to see if we have a contiguous observation already			
			SDDiscreteObservation obs = null;
			for (int i = 0; i < list.size(); i++) {
				SDSnapshotObservation tmp = (SDSnapshotObservation)list.get(i);
				if (tmp instanceof SDDiscreteObservation) {
					obs = (SDDiscreteObservation)tmp;
					break;	
				}
			}
			//if there isnt one then add one			
			if (obs == null) {
				obs = factory.createSDDiscreteObservation();
				obs.setMemberDescriptor(mem);
				obs.setWindow(window);
				list.add(obs);	//?

				modelChanged(mem);
			}

			//get the data lists
			BasicEList times = (BasicEList)obs.getCreationTime();
			BasicEList values = (BasicEList)obs.getValue();

			times.addUnique(new Double(time));						
			values.addUnique(new Integer((int)Double.parseDouble(value)));						
			
		} else if (tag.equals("TextObservation")) {
			
			SDMemberDescriptor mem = (SDMemberDescriptor)sd;
			List list = mem.getSnapshotObservation();

			//check to see if we have a contiguous observation already			
			SDTextObservation obs = null;
			for (int i = 0; i < list.size(); i++) {
				SDSnapshotObservation tmp = (SDSnapshotObservation)list.get(i);
				if (tmp instanceof SDTextObservation) {
					obs = (SDTextObservation)tmp;
					break;	
				}
			}
			//if there isnt one then add one			
			if (obs == null) {
				obs = factory.createSDTextObservation();
				obs.setMemberDescriptor(mem);
				obs.setWindow(window);
				list.add(obs);	//?

				modelChanged(mem);
			}

			//get the data lists
			BasicEList times = (BasicEList)obs.getCreationTime();
			BasicEList values = (BasicEList)obs.getTextValue();

			if (value == null) throw new NullPointerException("null value for SDTextObservation");

			times.addUnique(new Double(time));						
			values.addUnique(value);						
			
		} 
	}
	public void endElement(String tag) throws Exception {

		if (tag.equals("description")) {
			inside_description = false;
			return;	
		}

		if (tag.equals("RangeRepresentation")) return;
		if (tag.equals("GaugeRepresentation")) return;
	}
	public void startDocument() throws Exception {
		//do nothing
	}
	public void endDocument() throws Exception {
		//do nothing
	}
	public void text(String str) throws Exception {
		if (inside_description) {
			sd.setDescription(str);
		}
	}
	
}