/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.extension;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;

/**
 * An internal utility class to facilitate dealing with
 * navigator extensions. Allows getting all the extensions,
 * as well as various convenience methods.
 *  
 * @author curtispd
 * @since 3.0
 */
public class NavigatorExtensionUtil {
	private static final String ALL = "all";
	private static final Map _contributions = new HashMap();
	private static final Map _filterSets = new HashMap();
	
	/**
	 * Delegates the filling of the context menu to all
	 * navigator extensions.
	 * 
	 * @param menu the menu to populate.
	 * @param selection the selected items.
	 * @param navigatorID the navigator's unique ID
	 */
	public static void fillContextMenu(IMenuManager menu, IStructuredSelection selection, String navigatorID) {
		INavigatorContribution[] e = NavigatorExtensionUtil.getContributions(navigatorID);
		for (int i=0;i<e.length;++i)
			e[i].fillContextMenu(menu, selection);
	}
	
	/**
	 * Returns <code>true</code> if and only if the given item
	 * should be filtered. This method covers all filters.
	 * 
	 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public static boolean select(Viewer viewer, Object parent, Object element, String navigatorID) {
		return select(viewer, parent, element, navigatorID, false) || select(viewer, parent, element, navigatorID, true);
	}

	/**
	 * Returns <code>true</code> if and only if the given item
	 * should be filtered. If recursive is <code>true</code>,
	 * will use only recursive filters, otherwise will use only
	 * non-recursive filters.
	 * 
	 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public static boolean select(Viewer viewer, Object parent, Object element, String navigatorID, boolean recursive) {
		NavigatorFilterSet[] sets = getFilterSetElements(navigatorID);
		for (int i=0;i<sets.length;++i)
			if (sets[i].isEnabled() && (sets[i].isRecursive() == recursive) && sets[i].getViewerFilter().select(viewer, parent, element))
				return true;
		return false;
	}
	
	/**
	 * Returns all navigator contributions for a specific navigator.
	 * 
	 * @param navigatorID the navigator's unique ID
	 * @return the contributions for the given navigator.
	 */
	public static INavigatorContribution[] getContributions(String navigatorID) {
		INavigatorContribution[] cached = (INavigatorContribution[])_contributions.get(navigatorID == null ? ALL : navigatorID);
		
		if (cached == null) {
			IExtension[] exts = HyadesUIPlugin.getInstance().getDescriptor().getExtensionPoint(HyadesUIPlugin.EP_NAVIGATOR_EXTENSIONS).getExtensions();
			List list = new ArrayList();
			for (int i=0;i<exts.length;++i) {
				IConfigurationElement[] elements = exts[i].getConfigurationElements();
				for (int j=0;j<elements.length;++j) {
					if ("navigatorContribution".equals(elements[j].getName())) {
						String nav = elements[j].getAttribute("navigatorID");
						if (navigatorID == null || navigatorID.equals(nav)) {
							try {
								String className = elements[j].getAttribute("class");
								Class classObj = exts[i].getDeclaringPluginDescriptor().getPluginClassLoader().loadClass(className);
								list.add(classObj.newInstance());
							}
							catch (Exception e) {
								HyadesUIPlugin.logError(e);
							}
						}
					}
				}
			}
			cached = new INavigatorContribution[list.size()];
			list.toArray(cached);
			_contributions.put(navigatorID == null ? ALL : navigatorID, cached);
		}
		return cached;
	}

	/**
	 * Returns the filter set element with the given id.
	 * 
	 * @param filterID the filter set element's unique id.
	 * @return the filter set element with the given id.
	 */
	public static NavigatorFilterSet getFilterSetElement(String filterID) {
		if (filterID == null)
			return null;
		
		NavigatorFilterSet[] e = getFilterSetElements(null);
		for (int i=0;i<e.length;++i)
			if (filterID.equals(e[i].getID()))
				return e[i];
		return null;
	}
	
	/**
	 * Returns all the filter sets associated with the given
	 * navigator.
	 * 
	 * @param navigatorID the navigator's unique id
	 * @return the filter sets for the navigator
	 */
	public static NavigatorFilterSet[] getFilterSetElements(String navigatorID) {
		NavigatorFilterSet[] cached = (NavigatorFilterSet[])_filterSets.get(navigatorID == null ? ALL : navigatorID);
		
		if (cached == null) {
			IExtension[] exts = HyadesUIPlugin.getInstance().getDescriptor().getExtensionPoint(HyadesUIPlugin.EP_NAVIGATOR_EXTENSIONS).getExtensions();
			List list = new ArrayList();
			for (int i=0;i<exts.length;++i) {
				IConfigurationElement[] elements = exts[i].getConfigurationElements();
				for (int j=0;j<elements.length;++j) {
					if ("navigatorFilterSet".equals(elements[j].getName())) {
						String nav = elements[j].getAttribute("navigatorID");
						if (navigatorID == null || navigatorID.equals(nav)) {
							try {
								list.add(new NavigatorFilterSet(elements[j]));
							}
							catch (Exception e) {
								HyadesUIPlugin.logError(e);
							}
						}
					}
				}
			}
			cached = new NavigatorFilterSet[list.size()];
			list.toArray(cached);
			_filterSets.put(navigatorID == null ? ALL : navigatorID, cached);
		}
		return cached;
	}
	
	/**
	 * Returns <code>true</code> if the item has any children
	 * contributed by extensions.
	 * 
	 * @param item the parent item.
	 * @param navigatorID the navigator's unique ID
	 * @return true if there are any children.
	 */
	public static boolean hasChildren(Object item, String navigatorID) {
		INavigatorContribution[] e = NavigatorExtensionUtil.getContributions(navigatorID);
		for (int i=0;i<e.length;++i)
			if (e[i].hasChildren(item))
				return true;
		return false;
	}
	
	/**
	 * Returns all children contributed by all extensions,
	 * for the specific <code>Object</code>.
	 * 
	 * @param item the item whose children we want.
	 * @param navigatorID the navigator's unique ID
	 * @return the item's children.
	 */
	public static List getAllChildren(Object item, String navigatorID) {
		INavigatorContribution[] e = NavigatorExtensionUtil.getContributions(navigatorID);
		List list = new ArrayList();
		for (int i=0;i<e.length;++i)
			list.addAll(e[i].getChildren(item));
		return list;
	}

	/**
	 * Returns all <code>INavigatorItem</code>s that need
	 * to be saved. This is used to determine which items
	 * should be saved when the workbench is shut down.
	 * 
	 * @param navigatorID the navigator's unique ID
	 * @return all external items that need to be saved.
	 */
	public static Collection getAllModifiedItems(String navigatorID) {
		INavigatorContribution[] e = NavigatorExtensionUtil.getContributions(navigatorID);
		Collection c = new HashSet();
		for (int i=0;i<e.length;++i)
			c.addAll(e[i].getModifiedItems());
		return c;
	}
	
	/**
	 * Disposes all navigator extensions.
	 */
	public static void disposeAll() {
		Iterator iter = _contributions.values().iterator();
		while (iter.hasNext()) {
			INavigatorContribution[] e = (INavigatorContribution[])iter.next();
			for (int i=0;i<e.length;++i) {
				e[i].dispose();
				e[i] = null;
			}
		}
		_contributions.clear();
	}
}
