/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.tabs;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.internal.launch.providers.ExecutionHistoryLocationProvider;
import org.eclipse.hyades.test.ui.launch.configurations.ExecutionHistoryLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider2;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPage;

/**
 * Provides the Test Execution History Tab. This tab allows the user to choose the name
 * and the location of the Test Execution History file.
 * @author jcanches
 */
public class ExecutionHistoryTab extends AbstractLaunchConfigurationTab {
	
	private TreeViewer tree;
	private Text nameText;
	private Button override;
	protected boolean userHasChosenName;
	protected boolean userHasChosenOverrideState;
	protected boolean userHasChosenLocation;
	
	public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);
        GridLayout gl = new GridLayout();        
        composite.setLayout(gl);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        composite.setFont(parent.getFont());

        Label lblName = new Label(composite, SWT.NONE);
        GridData gdl = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl.horizontalSpan = 2;
        lblName.setLayoutData(gdl);
        lblName.setText(TestUIPlugin.getString("ExecutionHistoryTab.name")); //$NON-NLS-1$

        nameText = new Text(composite, SWT.BORDER);
        GridData gdt = new GridData(GridData.FILL_HORIZONTAL);
        gdt.horizontalSpan = 2;
        nameText.setLayoutData(gdt);
        nameText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
				userHasChosenName = true;
			}
		});
        
		override = new Button(composite, SWT.CHECK);
		override.setText(TestUIPlugin.getString("ExecutionHistoryTab.override")); //$NON-NLS-1$
		override.setToolTipText(TestUIPlugin.getString("ExecutionHistoryTab.overrideTooltip")); //$NON-NLS-1$
		override.setSelection(false); // By default this check is left unselected
		override.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				userHasChosenOverrideState = true;
			}
		});
        GridData gdl3 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl3.horizontalSpan = 2;
        override.setLayoutData(gdl3);
        
        Label lblLocation = new Label(composite, SWT.NONE);
        GridData gdl2 = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gdl2.horizontalSpan = 2;
        lblLocation.setLayoutData(gdl2);
        lblLocation.setText(TestUIPlugin.getString("ExecutionHistoryTab.location")); //$NON-NLS-1$

        createLocationSelector(composite);

        setControl(composite);
	}
	
    private void createLocationSelector(Composite composite) {
    	GridData grid = new GridData(GridData.FILL_BOTH);
		grid.widthHint	= composite.getClientArea().width/2 ;
		grid.heightHint	= composite.getClientArea().height/2;

        tree = new TreeViewer(composite);
        tree.getControl().setLayoutData(grid);
        ExecutionHistoryLocationProvider provider = new ExecutionHistoryLocationProvider();
        tree.setContentProvider(provider);
        tree.setLabelProvider(provider);
        tree.setInput(ResourcesPlugin.getWorkspace().getRoot());
        
        tree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateLaunchConfigurationDialog();
				userHasChosenLocation = true;
			}
		});
    }
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, computeDefaultName(configuration));
		ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryOverriden(configuration, computeDefaultOverrideState(configuration));
        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, computeDefaultLocation(configuration));
	}
	
	private IExecutionHistoryDefaultsProvider2 getExecutionHistoryDefaultsProvider2(ILaunchConfiguration configuration) {
		String launchConfigurationType = null;
		try {
			launchConfigurationType = configuration.getType().getIdentifier();
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
		}
		// Search an IExecutionHistoryDefaultsProvider2 for the launch configuration first
		if (launchConfigurationType != null) {
			IExecutionHistoryDefaultsProvider2 defaultsProvider2 = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider2(launchConfigurationType);
			if (defaultsProvider2 != null) {
				return defaultsProvider2;
			}
		}
		return null;
	}
	
	/**
	 * @return
	 */
	private String computeDefaultName(ILaunchConfiguration configuration) {
		String name = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			name = defaultsProvider2.getDefaultName(configuration);
		}
		
		if (name == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration);
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
			}
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			if (defaultsProvider != null) {
				name = defaultsProvider.getDefaultName(test);
			}
			if (name == null) {
				// Use the DefaultDeploymentsProvider suggestion
				name = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultName(test);
			}
		}
		return name;
	}
	
	private boolean computeDefaultOverrideState(ILaunchConfiguration configuration) {
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			return defaultsProvider2.getDefaultOverrideState(configuration);
		}
		
		TPFTest test = null;
		try {
			test = TestLaunchConfigurationFacade.getTest(configuration);
		} catch (CoreException e) {
			TestUIPlugin.logError(e);
		}
		
		IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
		// The return value is never null
		return defaultsProvider.getDefaultOverrideState(test);
	}
	
	private IContainer computeDefaultLocation(ILaunchConfiguration configuration) {
		IContainer location = null;
		IExecutionHistoryDefaultsProvider2 defaultsProvider2 = getExecutionHistoryDefaultsProvider2(configuration);
		if (defaultsProvider2 != null) {
			location = defaultsProvider2.getDefaultLocation(configuration);
		}
		
		if (location == null) {
			TPFTest test = null;
			try {
				test = TestLaunchConfigurationFacade.getTest(configuration);
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
			}
			
			IExecutionHistoryDefaultsProvider defaultsProvider = LaunchConfigurationExtensionsManager.getInstance().getExecutionHistoryDefaultsProvider(test);
			// The return value (above) is never null
			location = defaultsProvider.getDefaultLocation(test);
			if (location == null) {
				// Use the DefaultDeploymentsProvider suggestion
				location = LaunchConfigurationExtensionsManager.getInstance().getDefaultExecutionHistoryDefaultsProvider().getDefaultLocation(test);
			}
			if (location == null) {
				// Use the location derived from the current selection in the Workbench
				location = getContext();
			}
		}
	
		return location;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
        try {
    		userHasChosenName = getUserHasChosenName(configuration);
    		userHasChosenOverrideState = getUserHasChosenOverrideState(configuration);
    		userHasChosenLocation = getUserHasChosenLocation(configuration);
    		
    		String name = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryName(configuration);
    		if (name == null || !userHasChosenName) {
    			nameText.setText(computeDefaultName(configuration));
    			userHasChosenName = false;
    		} else {
    			nameText.setText(name);
    		}
    		
    		boolean overrideState;
    		if (!userHasChosenOverrideState) {
    			overrideState = computeDefaultOverrideState(configuration);
    		} else {
    			overrideState = ExecutionHistoryLaunchConfigurationFacade.isExecutionHistoryOverriden(configuration);
    		}
    		override.setSelection(overrideState);
    		
    		IContainer location = ExecutionHistoryLaunchConfigurationFacade.getExecutionHistoryLocation(configuration);
    		if (location == null || !userHasChosenLocation) {
    			location = computeDefaultLocation(configuration);
    		}
    		if (location != null) {
    			StructuredSelection sel = new StructuredSelection(location);
    			tree.setSelection(sel, /*reveal*/true);
    		}
    		
        } catch (CoreException e) {
            TestUIPlugin.logError(e);
        }
	}
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
        String name = getExecutionHistoryName();
        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryName(configuration, name);
        
        boolean overrideState = getOverrideState();
        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryOverriden(configuration, overrideState);
        
        IContainer location = getSelectedLocation();
        ExecutionHistoryLaunchConfigurationFacade.setExecutionHistoryLocation(configuration, location);
        
        setUserHasChosenName(configuration, userHasChosenName);
        setUserHasChosenOverrideState(configuration, userHasChosenOverrideState);
        setUserHasChosenLocation(configuration, userHasChosenLocation);
	}

	public String getName() {
		return TestUIPlugin.getString("ExecutionHistoryTab.tabName"); //$NON-NLS-1$
	}
	
	public Image getImage() {
		return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_LCONFIG_EXECRESULTS_TAB);
	}

	private String getExecutionHistoryName() {
		return nameText.getText();
	}
	
	public boolean getOverrideState() {
		return override.getSelection();
	}
	
	private IContainer getSelectedLocation() {
		ISelection selection = tree.getSelection();
		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {
			Object obj = ((IStructuredSelection)selection).getFirstElement();
			if (obj instanceof IContainer) {
				return (IContainer)obj;
			}
		}
		return null;
	}
	
    /*
     * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
     */
    public boolean isValid(ILaunchConfiguration launchConfig) {
        //- Sets the state of the Run button
        setErrorMessage(null);
        String name = getExecutionHistoryName();
        if (name == null || name.equals("")) { //$NON-NLS-1$
        	setErrorMessage(TestUIPlugin.getString("_ERROR_ExecutionHistoryTab.invalidName")); //$NON-NLS-1$
        	return false;
        }
        IContainer location = getSelectedLocation();
        if (location == null) {
        	setErrorMessage(TestUIPlugin.getString("_ERROR_ExecutionHistoryTab.invalidLocation")); //$NON-NLS-1$
        	return false;
        }
        return true;      		
    }

    /**
     * @return the first test element from the workbench selection
     */
    public static IContainer getContext() {
        try {
            IWorkbenchPage page = TestUIPlugin.getInstance().getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (page != null) {
                ISelection selection = page.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection ss = (IStructuredSelection) selection;
                    Object obj = ss.getFirstElement();
                    if (obj instanceof IResource) {
                    	IProject project = ((IResource)obj).getProject();
                    	if (project != null) {
                    		return project;
                    	}
                    }
                }
            }
        } catch (Throwable t) {
            TestUIPlugin.logError(t);
        }
        return null;
    }
    
    private static final String PROPERTY_USER_HAS_CHOSEN_NAME = TestUIPlugin.getID() + "internal.launch.history.name_chosen"; //$NON-NLS-1$
    private static final String PROPERTY_USER_HAS_CHOSEN_OVERRIDE_STATE = TestUIPlugin.getID() + "internal.launch.history.override_chosen"; //$NON-NLS-1$
    private static final String PROPERTY_USER_HAS_CHOSEN_LOCATION = TestUIPlugin.getID() + "internal.launch.history.location_chosen"; //$NON-NLS-1$
    
    private boolean getUserHasChosenName(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_USER_HAS_CHOSEN_NAME, false);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }

    private boolean getUserHasChosenOverrideState(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_USER_HAS_CHOSEN_OVERRIDE_STATE, false);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }

    private boolean getUserHasChosenLocation(ILaunchConfiguration configuration) {
    	try {
    		return configuration.getAttribute(PROPERTY_USER_HAS_CHOSEN_LOCATION, false);
    	} catch (CoreException e) {
    		TestUIPlugin.logError(e);
    		return true;
    	}
    }
    
    private void setUserHasChosenName(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_USER_HAS_CHOSEN_NAME, value);
    }

    private void setUserHasChosenOverrideState(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_USER_HAS_CHOSEN_OVERRIDE_STATE, value);
    }

    private void setUserHasChosenLocation(ILaunchConfigurationWorkingCopy configuration, boolean value) {
    	configuration.setAttribute(PROPERTY_USER_HAS_CHOSEN_LOCATION, value);
    }

}
