/**********************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.event;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.hyades.test.common.util.XMLUtil;

/**
 * This is a flat class packaged in the execution environment and
 * it mirrors the org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent
 * 
 * @author kobylin
 * @author marcelop
 * @author amathur 
 */
public class ExecutionEvent implements IEventConstants
{
	private String id;
	private String parentId;
	private String ownerId;
	private long timestamp;
	private String text;
	private String name;
	private String eventType;
	private String sortBy;
	private int conflict = CONFLICT_NONE;
	private ArrayList properties = new ArrayList();
	
	/**
	 * Returns the id of this event.
	 * @return String
	 */
	public String getId()
	{
		return id;
	}

	/**
	 * Sets the id of this event.
	 * @param value
	 */
	public void setId(String value)
	{
		id = value;
	}

	/**
	 * Returns the ID of the parent event
	 * @return String
	 */
	public String getParentId() {
		return parentId;
	}

	/**
	 * Sets the ID of the parent event
	 * @param parentId
	 */
	public void setParentId(String parentId) {
		this.parentId = parentId;
	}

	/**
	 * Returns the owner id of this event.
	 * @return String
	 */
	public String getOwnerId()
	{
		return ownerId;
	}

	/**
	 * Sets the owner id of this event.
	 * @param value
	 */
	public void setOwnerId(String value)
	{
		ownerId = value;
	}

	/**
	 * Returns the timestamp of this event.
	 * @return String
	 */
	public long getTimestamp()
	{
		return timestamp;
	}

	/**
	 * Sets the timestamp of this event.  The timestamp
	 * is set by the {@link #toString()} method if this
	 * {@link #getTimestamp()} returns 0.
	 * @param value
	 */
	public void setTimestamp(long value)
	{
		timestamp = value;
	}

	/**
	 * Returns the text of this event.
	 * @return String
	 */
	public String getText()
	{
		return text;
	}

	/**
	 * Sets the text of this event.
	 * @return String
	 */
	public void setText(String value)
	{
		text = value;
	}

	/**
	 * @return Returns the eventType.
	 */
	public String getEventType() {
		return eventType;
	}
	/**
	 * This attribute is optional. If provided, the execution history viewer, will query the 
	 * extension point that will need to be provided, for the label of the tree element 
	 * for this event, in the execution history viewer.
	 * @param eventType The eventType to set.
	 */
	public void setEventType(String eventType) {
		this.eventType = eventType;
	}
	/**
	 * @return Returns the name.
	 */
	public String getName() {
		return name;
	}
	/**
	 * If this attribute is speicified, the execution history viewer will use it
	 * as the label of the tree element for this event. However, if the eventType is
	 * specified, the it will query the corresponding extension point for the label 
	 * text. 
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * @return Returns the sortBy.
	 */
	public String getSortBy() {
		return sortBy;
	}
	
	/**
	 * Set this attribute if this event needs to be sorted within the scope of its
	 * immediate parent in the execution history. This attribute does not make it 
	 * all the way to the execution history model but is used by the loaders to 
	 * figure out sorting.
	 * @param sortBy The sortBy to set.
	 */
	public void setSortBy(String sortBy) {
		this.sortBy = sortBy;
	}

	/**
	 * @return Returns the conflict.
	 */
	public int getConflict() {
		return conflict;
	}
	
	/**
	 * This attribute indicates what should be done by the execution history loader
	 * if it finds an existing event with the same credentials (hierarchical id).
	 * Default behavior isto not worry about conflicts and just insert this event
	 * into the execution history
	 * 
	 * @param conflict The conflict to set.
	 */
	public void setConflict(int conflict) {
		this.conflict = conflict;
	}

	/**
	 * @return Returns the properties.
	 */
	public ArrayList getProperties() {
		return properties;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer();
		sb.append("<").append(getXMLRoot());
		addXMLAttributes(sb);
		sb.append(">");
		addXMLProperties(sb);
		sb.append("</").append(getXMLRoot());
		sb.append(">");
		
		return sb.toString();
	}
	
	/**
	 * Returns the string that is used as the root of the
	 * xml fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 */
	protected String getXMLRoot()
	{
		return "executionEvent";
	}
	
	/**
	 * Returns the string that is used as the tag for property
	 * xml fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 */
	protected String getXMLPropertyRoot()
	{
		return "property";
	}
	
	/**
	 * Add the attributes to the xml <code>StringBuffer</code> that
	 * is used by the {@link #toString()} method. 
	 * @param xml
	 */
	protected void addXMLAttributes(StringBuffer xml)
	{
		xml.append((XMLUtil.createXMLAttribute("id", getId(), false)).toString());
		xml.append((XMLUtil.createXMLAttribute("ownerId", getOwnerId(), false)).toString());
		xml.append((XMLUtil.createXMLAttribute("text", getText(), false)).toString());
		if ( parentId != null && !parentId.equals(""))
			xml.append((XMLUtil.createXMLAttribute("parentId", getParentId(), false)).toString());
		
		long ts = getTimestamp();
		if(ts == 0)
			ts = System.currentTimeMillis();
		xml.append((XMLUtil.createXMLAttribute("timestamp", Long.toString(ts), false)).toString());
		xml.append((XMLUtil.createXMLAttribute("name", getName(), false)).toString());
		xml.append((XMLUtil.createXMLAttribute("eventType", getEventType(), false)).toString());
		xml.append((XMLUtil.createXMLAttribute("sortBy", getSortBy(), false)).toString());
		if (getConflict() != IEventConstants.CONFLICT_NONE) {
			xml.append((XMLUtil.createXMLAttribute("conflict", Integer.toString(getConflict()), false)).toString());			
		}
	}	
	
	/**
	 * Add the property attributes to the xml <code>StringBuffer</code> that
	 * is used by the {@link #toString()} method. 
	 * @param xml
	 */
	protected void addXMLProperties(StringBuffer xml) {
		Iterator iter = properties.iterator();
		while (iter.hasNext()) {
			EventProperty property = (EventProperty) iter.next();
			xml.append("<").append(getXMLPropertyRoot());
			xml.append((XMLUtil.createXMLAttribute("pname", property.getName(), false)).toString());
			xml.append((XMLUtil.createXMLAttribute("ptype", property.getType(), false)).toString());
			xml.append((XMLUtil.createXMLAttribute("pvalue", property.getValue(), false)).toString());
			xml.append("/>");
		}
	}
}
