/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.internal.reports;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.test.http.TestHttpPlugin;
import org.eclipse.hyades.ui.sample.svg.generator.GraphicDocumentProperties;
import org.eclipse.hyades.ui.sample.svg.generator.SVGBarChart;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author amathur
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class ReportGraph 
{
	IFile outputFile = null;
	ArrayList list = null;
	ReportGraphConfiguration config = null;
	
	public ReportGraph(ArrayList list, ReportGraphConfiguration configuration)
	{
		this.list = list;
		this.config=configuration;		
	}
	
	
	public InputStream generate(IPath svgPath)
		throws Exception {
			
	
		// Defines a new color palette and initializes it with white in all the positions
		ArrayList colourPalette = new ArrayList(); 

		// creates a Document by collecting information from the Component Test objects
		Document newDoc = generateDataDocument(list, colourPalette);
		if(newDoc == null)
			return null;

		// turn the DOM into an SVG chart
		GraphicDocumentProperties properties = new GraphicDocumentProperties();
		properties.setDataDocument(newDoc);
		
		//TODO: make width dynamic
		properties.setGraphicWidth(new Integer(config.getMinWidth()).toString());
		
		properties.setGraphicHeight(new Integer(config.getHeight()).toString());
		//properties.setTimestampSuppressed("true");
		properties.setLegendSuppressed("true");
		
		//Turn off preferences
		properties.setUserPreferencesSuppressed("true");

		//CR - Add custom colour palettes 
		String[] newPalette = {"red", "green", "blue", "yellow", "black", "cyan", "#996699", "#FF00CC", "#000099"};
		properties.addGraphicColorPalette("Test Status", newPalette);

		//		create the bar chart SVG
		SVGBarChart svgBarChart = new SVGBarChart();
		Document generatedDocument = svgBarChart.generateGraphicDocument(properties);
		
		// write the SVG to a file
		IPath svgPathExt = svgPath.removeFileExtension().addFileExtension("svg");
		String svgFileName = createFile(svgPathExt,svgBarChart.serializeGeneratedDocumentToString(generatedDocument)).getName();
		String svgURL = TestHttpPlugin.getString("SVG_URL");
		
		// create the HTML to return to caller
		StringBuffer sb = new StringBuffer();
		sb.append("<html>\n").append("<head>\n").append("<title>").append(getUserText("chart.Title")).append("</title>").append("</head>\n");
		sb.append("<body>\n").append("<embed type=\"image/svg+xml\" src=\"");
		sb.append(svgFileName).append("\" ").append("width=\"").append(config.getMinWidth()).append("\" ");
		sb.append("height=\"").append(config.getHeight()).append("\" ");
		sb.append("name=\"mySvgDocument\"");
		sb.append("/>\n");
		sb.append("<p><font face=\"arial\" font size = \"1\">").append(svgURL).append("</font><p>");
		sb.append("</body>").append("</html>");

		IPath htmlPath = svgPath.removeFileExtension().addFileExtension("html");
		outputFile = createFile(htmlPath, sb.toString());
		
		return new ByteArrayInputStream(sb.toString().getBytes());
	}

	protected Document generateDataDocument(ArrayList list, ArrayList colourPalette)
	throws Exception
	{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document document = builder.newDocument();

		// create the root element, i.e. configuration
		Element root = document.createElement("configuration");
		document.appendChild(root);
		
		root.setAttribute("title", config.getTitle());
		root.setAttribute("legendtitle", config.getLegendTitle());
		root.setAttribute("timestampPrefix", config.getTimeStampPrefix());
		

		// Create the element dataUpdate
		Element dataUpdateElement = document.createElement("dataUpdate");
		root.appendChild(dataUpdateElement);
		Date dateNow = new Date();
		SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");	
		dataUpdateElement.setAttribute("timestamp", dateFormatter.format(dateNow));

		/* Create a list of dataset elements (each dataset correponds to a line in the graph) inside the 
		 * dataUpdate element.  If the line is added then the colourPalette will be changed so it can be
		 * drawn in a specific color,
		 */
		int position = 0;			
		//CR Temp for TEST	
		if(addBarData(document, dataUpdateElement, position, getUserText("chart.expectedAttempt"), list))
		{
			colourPalette.add("#D3D3D3"); //grey
			position++;
		}

		//CR Temp for TEST			
		createXData(document, dataUpdateElement, list);
		
		// Create the categorization element which contains the information
		// related Y axis
		//CR Temp for TEST		
		createYData(document, dataUpdateElement, list);

		return document;
	}
	
	/**
	 * Adds a line data to the document.  The line will represent one of the values
	 * of the execution statistics.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param int the position of the line
	 * @param String the name of the line
	 * @param int the type of statistic value will be represented on the line
	 * @param DayStatistic[] the sorted daily statistics
	 * @return boolean if the line data was added
	 */
	protected boolean addBarData(Document document, Element dataUpdateElement, int position, String name, ArrayList list)
	{
			
		Element dataset = document.createElement("dataSet");
		dataUpdateElement.appendChild(dataset);
		dataset.setAttribute("label",name);
		dataset.setAttribute("position", new Integer(position).toString());
		dataset.setAttribute("dataRange1Ref", "day");
		dataset.setAttribute("dataRange2Ref", "count");
		
		Element data = document.createElement("dataPoint");
		dataset.appendChild(data);	
		data.setAttribute("value1", "0");
		data.setAttribute("value2", "0");

		Iterator iter = list.iterator();
		int i = 1;
		
		//
		while (iter.hasNext()) {
			
			IReportGraphDataProvider provider = (IReportGraphDataProvider) iter.next();
			data = document.createElement("dataPoint");
			dataset.appendChild(data);
			
			data.setAttribute("value1", Integer.toString(i++));
			data.setAttribute("value2", provider.getValue());			
		}
		return true;
	}
	
	/**
	 * Creates the data that is presented at the X axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param DayStatistic[] the sorted daily statistics
	 */
	protected void createXData(Document document, Element dataUpdateElement, ArrayList list)
	{
		Element dataRangeElement = document.createElement("dataRange");
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Title information
		StringBuffer xTitle = new StringBuffer();
		///xTitle.append(xTitle);
		dataRangeElement.setAttribute("id", "day");
		dataRangeElement.setAttribute("location", "S");
		//CR- Fix categorization attribute
		dataRangeElement.setAttribute("type", "categorization");
		dataRangeElement.setAttribute("label", xTitle.toString());
		
		// Adds a "0"  day
		Element segmentMarkerElement = document.createElement("segmentMarker");

		// Add the X axis points
		Iterator iter = list.iterator();
		int i = 1;

		while (iter.hasNext()) {
			
			//get each url and store it in a string
			IReportGraphDataProvider provider = (IReportGraphDataProvider) iter.next();
			segmentMarkerElement = document.createElement("segmentMarker");
			dataRangeElement.appendChild(segmentMarkerElement);
			String name = provider.getId();
			
			StringBuffer buf = new StringBuffer();	
			
			//Search for all slashes and add a white space to cause name to wrap
			if (name.indexOf('/') != -1) {
				StringTokenizer st = new StringTokenizer(name, "/");
				
				while (st.hasMoreTokens()) {
					buf.append(st.nextToken() + "/ ");
					}
					String str = buf.toString();
					
					//Remove last instance of slash
					if (str.endsWith("/ ") == true) {
						int b = str.lastIndexOf("/ ");
					 	str = str.substring(0, b);
					}
					//Search for queries and delete them to help limit length of URLs
					int index = 0;
					index = str.indexOf('?', index);
						if (index != -1) {
							String substring = str.substring(0, index);
							System.out.println(substring);
							segmentMarkerElement.setAttribute("label", substring);
						}
						else {
							segmentMarkerElement.setAttribute("label", str);
						}
			}
			else {
				segmentMarkerElement.setAttribute("label", name);
			}
			segmentMarkerElement.setAttribute("value", Integer.toString(i++));
			
			if (i > 6) {
				int m = config.getMinWidth() + 80;
				config.setMinWidth(m);
			}
		}
		
	}

	/**
	 * Creates the data that is presented at the Y axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 */
	protected void createYData(Document document, Element dataUpdateElement, ArrayList list)
	{
		Element dataRangeElement = document.createElement("dataRange");
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Define the Y axis range
		dataRangeElement.setAttribute("id", "count");
		dataRangeElement.setAttribute("location", "W");
		dataRangeElement.setAttribute("label", config.getYTitle());

		// First figure out the min & max
		// Assume 10 steps
		int number = list.size();
		Iterator iter = list.iterator();
		// Figure out the min & max
		double min = 1000000;
		double max = 0;
		while (iter.hasNext()) {
			IReportGraphDataProvider dataProvider = (IReportGraphDataProvider) iter.next();
			double val=-1;
			String dataString =dataProvider.getValue(); 
			String dataName = dataProvider.getId();
			try {
				val = Double.parseDouble(dataString);
			}
			catch (NumberFormatException e)
			{
				val = 0;
			}
 
			if (val < min) {
				min = val;
			}
			if (val > max) {
				max = val;
			}
		}
		
		// Figure out the multiplication factor
		
		double factor = 1;
		
		if (max < .1) {
			factor = .01;		
		}
		else if (max < .5) {
			factor = .05;
		}
		else if (max < 1) {
			factor = .1;
		}
		else if (max < 5) {
			factor = .5;
		}

		else if (max < 10) {
			factor = 1;
		}
		else if (max < 20) {
			factor = 2;
		}
		else if (max > 20) {
			factor = (double)(int)(max / 9); //change to integer
		
		}

		for ( int m = 0; m < 11; m++) {
			Element segmentMarkerElement = document.createElement("segmentMarker");
			dataRangeElement.appendChild(segmentMarkerElement);
			
			String val = Double.toString(m*factor);
			 
			//due to floating point issues, we need to verify that this has only a set # of decimal place
			int nDecimals = 1;
			if (factor < .1)
			 	nDecimals = 2;
			
			if (val.indexOf('.')>=1)
			{
				val += "00"; //ensure string is long enough
				val = val.substring(0, val.indexOf('.')+1+nDecimals);
			}
			
			//fix this
			segmentMarkerElement.setAttribute("value", val);
			
			//segmentMarkerElement.setAttribute("label", Integer.toString((int)metric.getAvgResponse()));
			segmentMarkerElement.setAttribute("label", val);
			
		}
	}


	/**
	 * Returns the values that will be presented for the user.
	 * 
	 * @param String key The key for the value
	 * @return String the value
	 */
	protected String getUserText(String key)
	{
		if("chart.Title".equals(key))
			return "Report";
		
		if("chart.Legend".equals(key))
			return "Status";
			
		if("chart.timestampPrefix".equals(key))
			return "Last update:";
		
		if("chart.Xtitle".equals(key))	
			return "Date starting at:";
			
		if("chart.Ytitle".equals(key))
			return "Tests";


		if("chart.attempt".equals(key))
			return "Attempt";
			
		if("chart.error".equals(key))
			return "Error";
		
		if("chart.pass".equals(key))
			return "Pass";
		
		if("chart.fail".equals(key))	
			return "Fail";
			
		if("chart.inconclusive".equals(key))
			return "Inconclusive";


		if("chart.expectedAttempt".equals(key))
			return "Expected Attempt";
			
		if("chart.expectedError".equals(key))
			return "Expected Error";
		
		if("chart.expectedPass".equals(key))
			return "Expected Pass";
		
		if("chart.expectedFail".equals(key))	
			return "Expected Fail";
			
		if("chart.expectedInconclusive".equals(key))
			return "Expected Inconclusive";
		
		return key;
	}

	/**
	 * Creates a file in the same location and name of the <i>originalFile</i> but with a different
	 * extension.
	 * 
	 * @param String IPath the IPath for the generated svg file
	 * @param String extension The extension of the new file
	 * @param String content The content of the new file.
	 * @return IFile The created file;
	 */
	protected IFile createFile(IPath svgPath, String content) throws CoreException
	{
				
		IFile file = null;
		
		try 
		{
			file =	ResourcesPlugin.getWorkspace().getRoot().getFile(svgPath);
			ByteArrayInputStream inputStream = new ByteArrayInputStream(content.getBytes());
			if (file.exists()) 
			{
				try 
				{
					file.setContents(
						inputStream,
						true,
						false,
						new NullProgressMonitor());
				}
				catch (CoreException e1) {
					e1.printStackTrace();
				}
			}
			else
			{
				file.create(inputStream,true,new NullProgressMonitor());
			}
					
		}
		catch (RuntimeException e) {
			e.printStackTrace();
		}

		return file;
	}
	


}
