/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.analysis.engine;

public class Directive {


	/**
	 * Unique identifier.
	 */
	private String id = null;

	/**
	 * Associated action to be executed.
	 */
	private String directiveString = null;

	/**
	 * English description.
	 */
	private String description = null;

	/**
	 * Convenience constructor for creating a <code>Directive</code>
	 * with the parameter unique identifier but no associated action 
	 * to be executed and no English description. 
	 * 
	 * @param id the unique identifier
	 * @see   #Directive(String,String,String)
	 */
	public Directive(String id) {
		this(id, null, null);
	}

	/**
	 * Convenience constructor for creating a <code>Directive</code>
	 * with the parameter unique identifier and associated action to 
	 * be executed but no English description.
	 * 
	 * @param id the unique identifier
	 * @param directiveString the associated action to be executed
	 * @see   #Directive(String,String,String)
	 */
	public Directive(String id, String directiveString) {
		this(id, directiveString, null);
	}

	/**
	 * Constructor for creating a <code>Directive</code>
	 * with the parameter unique identifier, associated action to be 
	 * executed and English description.
	 * 
	 * @param id the unique identifier
	 * @param directiveString the associated action to be executed
	 * @param description the English description
	 */
	public Directive(String id, String directiveString, String description) {
		this.id = id;
		this.directiveString = directiveString;
		this.description = description;
	}

	/**
	 * Sets the unique identifier.
	 * 
	 * @param id the unique identifier
	 * @see   #getId()
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * Gets the unique identifier.
	 * 
	 * @return the unique identifier
	 * @see    #setId(String)
	 */
	public String getId() {
		return id;
	}

	/**
	 * Sets the associated action to be executed.
	 * 
	 * @param directiveString the associated action to be executed
	 * @see   #getDirectiveString()
	 */
	public void setDirectiveString(String directiveString) {
		this.directiveString = directiveString;
	}

	/**
	 * Gets the associated action to be executed.
	 * 
	 * @return the associated action to be executed
	 * @see    #setDirectiveString(String)
	 */
	public String getDirectiveString() {
		return directiveString;
	}

	/**
	 * Sets the English description.
	 * 
	 * @param description the English description
	 * @see   #getDescription()
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Gets the English description.
	 * 
	 * @return the English description
	 * @see    #setDescription(String)
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Overrides java.lang.Object.equals() to indicate 
	 * whether some other object is "equal to" this Directive
	 * object.
	 * 
	 * @param object  the reference object with which to compare.
	 * @return true if this object is the same as the object argument otherwise false.
	 */
	public boolean equals(Object object) {

		if ((object == null) || (!(object instanceof Directive)))
			return false;

		Directive testDirective = ((Directive) (object));
		String variable;

		//Test id:
		variable = testDirective.getId();

		if (((id != null) && (!id.equals(variable))) || ((id == null) && (variable != null)))
			return false;

		//Test directiveString:
		variable = testDirective.getDirectiveString();

		if (((directiveString != null) && (!directiveString.equals(variable))) || ((directiveString == null) && (variable != null)))
			return false;

		//Test description:
		variable = testDirective.getDescription();

		if (((description != null) && (!description.equals(variable))) || ((description == null) && (variable != null)))
			return false;

		return true;
	}

	/**
	 * Overrides java.lang.Object.toString() to return 
	 * a string representation of this Directive object.
	 * 
	 * @return the string representation of this object.
	 */
	public String toString() {
		StringBuffer toStringBuffer = new StringBuffer("com.ibm.etools.analysis.engine.Directive [");

		toStringBuffer.append("id = ");
		toStringBuffer.append(id);
		toStringBuffer.append(", directiveString = ");
		toStringBuffer.append(directiveString);
		toStringBuffer.append(", description = ");
		toStringBuffer.append(description);
		toStringBuffer.append("]");

		return (toStringBuffer.toString());
	}
}