/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.common.internal.dialog;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.adapter.TestWorkbenchAdapter;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.model.ui.EMFContentProvider;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.ITypeValidator;
import org.eclipse.hyades.ui.internal.provider.WorkbenchAdapterLabelProvider;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class InvokedTestSelectionDialog
extends Dialog implements IDisposable, IDoubleClickListener, ISelectionChangedListener
{
	protected static final int HELP_DLG_CONTROL = 1;
	protected static final int HELP_LOCAL_TEST_CASES_CONTROL = 2;
	protected static final int HELP_TESTS_CONTROL = 3;

	private final static int HEIGHT = 400;
	private final static int WIDTH = 400;
	
	private ITestSuite testSuite;
	private Collection tests;
	
	private TableViewer localTestCasesTableViewer;
	private TreeViewer testsTreeViewer;
	private ResourceSet testsResourceSet;
	private List loadedEObjects;
	
	private String title;
	private String tableLabel;
	private String treeLabel;
	private boolean singleSelection;
	
	private ITypeValidator typeValidator;
	
	/**
	 * Constructor for InvokedTestSelectionDialog
	 * @param parentShell
	 */
	public InvokedTestSelectionDialog(Shell parentShell, ITestSuite testSuite, String title, String tableLabel, String treeLabel, boolean singleSelection)
	{
		super(parentShell);
		
		this.testSuite = testSuite;
		this.title = title;
		this.tableLabel = tableLabel;
		this.treeLabel = treeLabel;
		this.singleSelection = singleSelection;
		tests = new ArrayList();
		
		String type = testSuite.getType();
		if(type != null)
		{
			IAssociationDescriptor associationDescriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_VALIDATORS).getDefaultAssociationDescriptor(type);
			if(associationDescriptor != null)
				typeValidator = (ITypeValidator)associationDescriptor.createImplementationClassInstance();			
		}
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testsResourceSet = null;
		if(localTestCasesTableViewer != null)
		{
			localTestCasesTableViewer.removeSelectionChangedListener(this);
			localTestCasesTableViewer.removeDoubleClickListener(this);
			localTestCasesTableViewer = null;
		}
		if(testsTreeViewer != null)
		{
			testsTreeViewer.removeSelectionChangedListener(this);
			testsTreeViewer.removeDoubleClickListener(this);			
			testsTreeViewer = null;
		}	
		testSuite = null;
		typeValidator = null;
		
		if(loadedEObjects != null)
		{
			for(Iterator i = loadedEObjects.iterator(); i.hasNext();)
				((EObject)i.next()).eResource().unload();
			loadedEObjects.clear();
			loadedEObjects = null;
		}
		System.gc();
	}
	
	public void setTest(ITest test)
	{
		tests.clear();
		tests.add(test);
		selectTest(tests);
	}
	
	public Collection getTests()
	{
		return tests;
	}
	
	public ITestSuite getTestSuite()
	{
		return testSuite;
	}
	
	protected ITypeValidator getTypeValidator()
	{
		return typeValidator;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		
		getButton(IDialogConstants.OK_ID).setEnabled(false);
		localTestCasesTableViewer.setInput(testSuite);
		if(testsTreeViewer != null)
			testsTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
		selectTest(getTests());
						
		registerHelp(HELP_DLG_CONTROL, control);	
		registerHelp(HELP_LOCAL_TEST_CASES_CONTROL, localTestCasesTableViewer.getControl());
		if(testsTreeViewer != null)				
			registerHelp(HELP_TESTS_CONTROL, testsTreeViewer.getControl());				

		return control;
	}
	
	protected void registerHelp(int flag, Object object)
	{
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = HEIGHT;
		gridData.widthHint = WIDTH;
		superComposite.setLayoutData(gridData);
		
		createLocalTestCasesControl(superComposite);
		if(getTypeValidator() != null)
			createTestControl(superComposite);
			
		if(title != null)
			getShell().setText(title);
		return superComposite;
	}
	
	protected void selectTest(Collection tests)
	{
		if(localTestCasesTableViewer == null)
			return;
		
		if(tests != null)
		{
			for (Iterator i = tests.iterator(); i.hasNext();)
			{
				ITest test = (ITest)i.next();
				if(test instanceof ITestCase)
				{
					if(((ITestCase)test).getOwner() == testSuite)
					{
						localTestCasesTableViewer.setSelection(new StructuredSelection(test), true);
						continue;
					}
				}
				
				if(testsTreeViewer != null)
				{
					test = (ITest)EMFUtil.getEObject(testsResourceSet, (EObject)test, true);			
					testsTreeViewer.setSelection(new StructuredSelection(test), true);					
				}
			}
		}
		else
		{
			localTestCasesTableViewer.setSelection(StructuredSelection.EMPTY);
			if(testsTreeViewer != null)
				testsTreeViewer.setSelection(StructuredSelection.EMPTY);
		}
	}

	protected void createLocalTestCasesControl(Composite parent)
	{
		if(tableLabel != null)
		{
			Label label = new Label(parent, SWT.NONE);
			label.setText(tableLabel);
		}
		
		int selection = singleSelection?SWT.SINGLE:SWT.MULTI;
		localTestCasesTableViewer = new TableViewer(parent,  selection | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		localTestCasesTableViewer.addDoubleClickListener(this);
		localTestCasesTableViewer.addSelectionChangedListener(this);
		
		if(getTypeValidator() != null)
		{
			GridData gridData = GridDataUtil.createHorizontalFill();
			gridData.grabExcessVerticalSpace = false;
			gridData.heightHint = 100;
			localTestCasesTableViewer.getControl().setLayoutData(gridData);
		}
		else
		{
			localTestCasesTableViewer.getControl().setLayoutData(GridDataUtil.createFill());
		}
		
		IStructuredContentProvider tableViewContentProvider = new IStructuredContentProvider()
		{
			public void dispose()
			{

			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
			{
			
			}
			
			public Object[] getElements(Object inputElement)
			{
				if(getTestSuite() != null)
					return getTestSuite().getITestCases().toArray();
				return new Object[0];
			}
		};
		localTestCasesTableViewer.setContentProvider(tableViewContentProvider);
		localTestCasesTableViewer.setLabelProvider(new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class));
	}
	
	protected void createTestControl(Composite parent)
	{
		if(treeLabel != null)
		{
			Label label = new Label(parent, SWT.NONE);
			label.setText(treeLabel);
		}
		
		int selection = singleSelection?SWT.SINGLE:SWT.MULTI;
		testsTreeViewer =	new TreeViewer(parent, selection | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 220;
		testsTreeViewer.getControl().setLayoutData(gridData);
		testsTreeViewer.addDoubleClickListener(this);
		testsTreeViewer.addSelectionChangedListener(this);

		ViewerFilter viewerFilter = new ViewerFilter()
		{
			public boolean select(Viewer viewer, Object parentElement, Object element)
			{
				if(element instanceof ITest)
				{
					ITest test = (ITest)element;
					if(getTestSuite().getId().equals(test.getId()))
						return false;
						
					return getTypeValidator().isValidType(test.getType());
				}
			
				return true;
			}
		};
		testsTreeViewer.addFilter(viewerFilter);
		
		testsResourceSet = new ResourceSetImpl();
		EMFContentProvider modelContentProvider = new EMFContentProvider(testsResourceSet);
		
		EMFContentProvider.ElementDescription elementDescription = new EMFContentProvider.ElementDescription(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		elementDescription.getEClassifiers().add(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite());
		modelContentProvider.addElementDescription(elementDescription); 
		
		EMFContentProvider.EnabledChildren enabledChildren = new EMFContentProvider.EnabledChildren(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite());
		enabledChildren.getChildrenEStructuralFeatures().add(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestCases());
		modelContentProvider.addEnabledChildren(enabledChildren);

		testsTreeViewer.setContentProvider(modelContentProvider);				
		ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
		testsTreeViewer.setLabelProvider(new DecoratingLabelProvider(new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class), decorator));
		
		modelContentProvider.setShowEnabledChildren(true);
	}

	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event)
	{
		if(processSelection((IStructuredSelection)event.getSelection()))
			okPressed();
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		processSelection((IStructuredSelection)event.getSelection());
	}
	
	protected boolean processSelection(IStructuredSelection structuredSelection)
	{		
		if(!structuredSelection.isEmpty())
		{
			Collection selectedTests = new ArrayList(); 
			for (Iterator i = structuredSelection.iterator(); i.hasNext();)
			{
				Object object = i.next();
				if(object instanceof ITest)
					selectedTests.add(object);
			}

			if(!selectedTests.isEmpty())
			{
				tests = selectedTests;
				getButton(IDialogConstants.OK_ID).setEnabled(true);
				return true;
			}
		}

		getButton(IDialogConstants.OK_ID).setEnabled(false);
		return false;		
	}
	
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#buttonPressed(int)
	 */
	protected void buttonPressed(int buttonId) 
	{
		setWaitCursor();
		// update a list of loaded objects to be unloaded during clean up.
		if(testsTreeViewer != null)
		{
			IContentProvider contentProvider = testsTreeViewer.getContentProvider();
			if(contentProvider instanceof EMFContentProvider)
			{
				loadedEObjects = ((EMFContentProvider)contentProvider).getLoadedEObjects();
			}
		}
		if(buttonId == IDialogConstants.OK_ID)
		{
			// fileter out selected objects.  Only unloaded unselected during clean up.
			if(loadedEObjects != null && !loadedEObjects.isEmpty())
			{
				for(Iterator testIterator = tests.iterator(); testIterator.hasNext();)
				{
					if(loadedEObjects.isEmpty())
						break;
					
					ITest test = (ITest)testIterator.next();
					if(test instanceof EObject)
					{
						for(Iterator it = loadedEObjects.iterator(); it.hasNext();)
						{
							EObject eObj = (EObject)it.next();
							if(test == eObj || ((EObject)test).eContainer() == eObj)
							{
								loadedEObjects.remove(eObj);
								break;
							}
							
						}
					}
				}
			}
		}
		dispose();
		unsetWaitCursor();
		
		super.buttonPressed(buttonId);		
	}
	
	private void setWaitCursor()
	{
		Display display = getShell().getDisplay();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);
		getShell().setCursor(waitCursor);
	}
	
	/*
	 * Cancels the display of the wait cursor.
	 */
	private void unsetWaitCursor()
	{
		getShell().setCursor(null);
	}
}
