/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.edit.datapool;


/**
 * This class defines the primary interface to datapools.  A datapool
 * is an ordered collection of data.  The structure of data is in the
 * form of uniformly sized records that are collected into logical groups.
 * <p>
 * <code>datapool --> (0..n) EquivalenceClasses --> (0..n) Records</code>
 * <p>
 * Each record contains a uniform set of {@link IDatapoolVariable variables}
 * (logically columns) with appropriate name and id properties associated 
 * with each.
 *
 * @see	IDatapoolEquivalenceClass
 * @see	IDatapoolRecord
 * @see	IDatapoolVariable
 * @see IDatapoolListener
 * @see org.eclipse.hyades.execution.runtime.datapool.DatapoolException
 */
public interface IDatapool extends 
	org.eclipse.hyades.execution.runtime.datapool.IDatapool,
	INamedElement
{
	
	//************************************************************
	// These methods provide support for "variables" (logically 
	// columns) in the datapool.
	//************************************************************
	
	/**
	 * Update the variable at a specified zero based variable index.
	 * 
	 * @param	variable		The variable for the cells at <i>variableIndex</i>.
	 * @param	variableIndex	A zero based variable index.
	 * 
	 * @see #getVariableIndex(String)
	 * @see #getVariableIndexById(String)
	 * @see #getVariable(int)
	 */
	public void updateVariable(IDatapoolVariable variable, int variableIndex);
	
	/**
	 * Insert a variable before the variable at the specified zero based
	 * variable index.  This action will cause an empty cell to be 
	 * automatically inserted in each row of the datapool.
	 * 
	 * @param	variable		The new variable for the cells at <i>beforeVariableIndex</i>.
	 * @param	beforeVariableIndex	A zero based variable index.
	 * 
	 * @see #appendVariable(IDatapoolVariable)
	 * @see #moveVariable(int,int)
	 * @see #removeVariable(int)
	 */
	public void insertVariable(IDatapoolVariable variable, int beforeVariableIndex);
	
	/**
	 * Append a variable to the end of the datapool ordered variable collection.
	 * This action will cause an empty cell to be automatically appended to each record 
	 * of the datapool.
	 * 
	 * @param	variable		The variable instance to append.
	 * 
	 * @see #insertVariable(IDatapoolVariable,int)
	 * @see #moveVariable(int,int)
	 * @see #removeVariable(int)
	 */
	public void appendVariable(IDatapoolVariable variable);
	
	/**
	 * Move a variable to immediately before another variable.  The variables are
	 * denoted by a zero based variable index.  This action will cause the
	 * associated cells in each row of the datapool to also be moved.
	 * 
	 * @param	sourceVariableIndex		A zero based variable index for the 
	 * 									variable to be moved.
	 * @param	targetVariableIndex		The variable index marking the target 
	 * 									location for the move action.
	 * 
	 * @see #insertVariable(IDatapoolVariable,int)
	 * @see #appendVariable(IDatapoolVariable)
	 * @see #removeVariable(int)
	 */
	public void moveVariable(int sourceVariableIndex, int targetVariableIndex);
	
	/**
	 * Remove the specified variable from the datapool.  The specified variable
	 * index is zero based.  Each record in the datapool is automatically 
	 * updated to remove the cell that coincides with the variable.
	 * 
	 * @param	variableIndex	A zero based variable index for the variable to 
	 *							be moved.
	 * 
	 * @see #insertVariable(IDatapoolVariable,int)
	 * @see #appendVariable(IDatapoolVariable)
	 * @see #moveVariable(int,int)
	 */
	public void removeVariable(int variableIndex);
	
	
	//************************************************************
	// These methods provide support for the logical equivalence classes
	// associated with an instance of a datapool.
	//************************************************************
	
	/**
	 * Define the default {@link IDatapoolEquivalenceClass equivalence class}
	 * in a datapool.
	 * 
	 * @param equivalenceClassIndex The zero-based index for the default equivalence class.
	 * 
	 * @see #getDefaultEquivalenceClassIndex()
	 */
	public void setDefaultEquivalenceClassIndex(int equivalenceClassIndex);
	
	/**
	 * Replace the {@link IDatapoolEquivalenceClass equivalence class} at a 
	 * supplied index with a new value.  This method is essentially a 
	 * shortcut for a call to {@link #removeEquivalenceClass(int) removeEquivalenceClass}
	 * followed by a call to 
	 * {@link #insertEquivalenceClass(IDatapoolEquivalenceClass,int) insertEquivalenceClass}.
	 * 
	 * @param	equivalenceClass	The equivalence class to insert.
	 * @param	equivalenceClassIndex	The equivalence class indsex to remove and insert at.
	 * 
	 * @see #insertEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #appendEquivalenceClass(IDatapoolEquivalenceClass)
	 * @see #moveEquivalenceClass(int,int)
	 * @see #removeEquivalenceClass(int)
	 */
	public void updateEquivalenceClass(IDatapoolEquivalenceClass equivalenceClass, int equivalenceClassIndex);
	
	/**
	 * Insert the {@link IDatapoolEquivalenceClass equivalence class} at a 
	 * supplied index.  The {@link IDatapoolRecord records} in the equivalence 
	 * class must match the layout of all other records in the datapool.
	 * 
	 * @param	equivalenceClass	The equivalence class to insert.
	 * @param	beforeEquivalenceClassIndex	The equivalence class index to insert before.
	 * 
	 * @see #updateEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #appendEquivalenceClass(IDatapoolEquivalenceClass)
	 * @see #moveEquivalenceClass(int,int)
	 * @see #removeEquivalenceClass(int)
	 */
	public void insertEquivalenceClass(IDatapoolEquivalenceClass equivalenceClass, int beforeEquivalenceClassIndex);
	
	/**
	 * Append the supplied {@link IDatapoolEquivalenceClass equivalence class} to the end 
	 * of the set of available equivalence classes.  The {@link IDatapoolRecord records} in 
	 * the equivalence class must match the layout of all other records in the datapool.
	 * 
	 * @param	equivalenceClass	The equivalence class to append.
	 * 
	 * @see #updateEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #insertEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #moveEquivalenceClass(int,int)
	 * @see #removeEquivalenceClass(int)
	 */
	public void appendEquivalenceClass(IDatapoolEquivalenceClass equivalenceClass);
	
	/**
	 * Move the {@link IDatapoolEquivalenceClass equivalence class} at a 
	 * supplied index to a new position within the set of available equivalence classes.
	 * 
	 * @param	sourceEquivalenceClassIndex	The equivalence class index to move.
	 * @param	targetEquivalenceClassIndex	The equivalence class destination index.
	 * 
	 * @see #updateEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #insertEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #appendEquivalenceClass(IDatapoolEquivalenceClass)
	 * @see #removeEquivalenceClass(int)
	 */
	public void moveEquivalenceClass(int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex);
	
	/**
	 * Remove the {@link IDatapoolEquivalenceClass equivalence class} at a 
	 * supplied index.
	 * 
	 * @param	equivalenceClassIndex	The equivalence class indsex to remove.
	 * 
	 * @see #updateEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #insertEquivalenceClass(IDatapoolEquivalenceClass,int)
	 * @see #appendEquivalenceClass(IDatapoolEquivalenceClass)
	 * @see #moveEquivalenceClass(int,int)
	 */
	public void removeEquivalenceClass(int equivalenceClassIndex);
	
	
	
	//************************************************************
	// These methods provide support for listeners monitoring actions
	// being performed against a datapool.
	//************************************************************
	
	/**
	 * Register a listener for actions against the datapool that modify
	 * the data or structure of the data model.  The listener is informed
	 * when variables and records are added or deleted, and when the contents 
	 * of a cell are changed.
	 * 
	 * @see IDatapoolListener
	 */
	public void addDatapoolListener(IDatapoolListener listener);
	
	/**
	 * Removes a listener from the datapool instance.
	 * 
	 * @see IDatapoolListener
	 */
	public void removeDatapoolListener(IDatapoolListener listener);
	
	
	/**
	 * Constructs an empty equivalence class 
	 */
	public IDatapoolEquivalenceClass constructEquivalenceClass();	
	
	
	/**
	 * Constructs an empty variable 
	 */
	public IDatapoolVariable constructVariable();	
			
}
