package org.eclipse.hyades.logging.events.cbe;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Base for all exceptions in CBE package. 
 * 
 * The <code>EventException</code> class is a subclass
 * of <code>java.lang.Exception</code> that provides support for
 * exception chaining semantics that will be available in the 1.4
 * timeframe. Exceptions that desire to inherit these semantics
 * while running on pre 1.4 JDKs should extend this class.
 * 
 * <p>
 * It should be noted that this class will provide only the exception
 * chaining semantics available in 1.4. This is a subset of all the
 * new functionality. introduced in 1.4. Most of that functionality
 * requires support from the JVM that is not available on earlier
 * editions.
 * </p>
 * 
 * <p>
 * Exceptions that subclass this exception only need to add the
 * various constructors and inherit everything else from this class.
 * </p>
 * 
 * <p>
 * See the javadoc for the JDK 1.4 java.lang.Throwable for a full
 * description of the exception chaining functionality.
 * </p>
 * 
 * <p>
 * <code>EventException</code> alse provides support
 * for localized (language-specific) messages.
 * </p>
 * 
 * <p>
 * A <code>EventException</code> contains the following
 * components:
 * 
 * <ul>
 * <li>
 * A "key" which selects the text to be displayed from the message
 * file. (Separating the text from the classes which use it allows
 * the text to be translated into different national languages.)
 * </li>
 * <li>
 * A message file, which contains a set of messages that might be
 * displayed. Entries in the message file are stored as "key=value"
 * pairs.
 * </li>
 * <li>
 * A set of <code>Objects</code> which can be inserted into the
 * message text. These parameters are required only if the message
 * expects them. These objects are processed according to the rules
 * of the <code>java.text.MessageFormat</code> class.
 * </li>
 * <li>
 * A nested <code>Exception</code> whose message text will be
 * displayed along with that of this
 * <code>EventException</code>.
 * </li>
 * </ul>
 * </p>
 *
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1 
 */
public class EventException extends Exception {
    //~ Instance variables ...........................................

    /** Base name of the ResourceBundle in which the text is found. */
    protected String file;

    /** The key of the message associated with this text. */
    protected String key = null;

    /** Parameters to be included in the message. */
    protected Object parms[] = null;

    /**
     * The throwable that caused this throwable to get thrown, or
     * null if this throwable was not caused by another throwable,
     * or if the causative throwable is unknown.  If this field is
     * equal to this throwable itself, it indicates that the cause
     * of this throwable has not yet been initialized.
     */
    private Throwable ivCause = this;

    //~ Constructors .................................................

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and an uninitialized cause filed. The
     * cause may subsequently be initialized via calling the {@link
     * #initCause} method.
     */
    public EventException() {
        super();
    }

    /**
     * Construct a new <code>EventException</code> with
     * the specified detail message and an uninitialized cause
     * field. The cause may subsequently be initialized via calling
     * the {@link #initCause} method.
     *
     * @param message a detail message. Null is tolerated.
     */
    public EventException(String message) {
        super(message);
    }

    /**
     * Creates a <code>EventException</code> with a detail message, message
     * translation information (resource bundle, message key and message
     * substitutions).
     *
     * @param key The message key.
     * @param file The name of the message file to use.
     */
    public EventException(String key, String file) {
        this(key, file, null);
    }

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and a cause field set to the specified
     * Throwable. Subsequent calls to the {@link #initCause} method
     * on this instance will result in an exception. The value of
     * the cause field may be retrieved at any time via the {@link
     * #getCause()} method.
     *
     * @param cause the Throwable that was caught and is considered
     *        the root cause of this exception. Null is tolerated.
     */
    public EventException(Throwable cause) {
        this(cause.getMessage());
        ivCause = cause;
    }

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and a cause field set to the specified
     * Throwable. Subsequent calls to the {@link #initCause} method
     * on this instance will result in an exception. The value of
     * the cause field may be retrieved at any time via the {@link
     * #getCause()} method.
     *
     * @param message the detail message. Null is tolerated.
     * @param cause the Throwable that was caught and is considered
     *        the root cause of this exception. Null is tolerated.
     */
    public EventException(String message, Throwable cause) {
        this(message);
        ivCause = cause;
    }

    /**
     * Creates a <code>EventException</code> with a
     * detail message, message translation information (resource
     * bundle, message key and message substitutions).
     *
     * @param key The message key.
     * @param file The name of the message file to use.
     * @param parms An array of elements to be inserted into the
     *        message.
     */
    public EventException(String key, String file, Object parms[]) {
        
        this.key = key;
        this.file = file;
        this.parms = parms;

        this.ivCause = null;
    }

    /**
     * Creates a <code>EventException</code> with a
     * detail message, message translation information (resource
     * bundle, message key and message substitutions) and a nested
     * exception.
     *
     * @param key The message key.
     * @param file The name of the message file to use.
     * @param parms An array of elements to be inserted into the
     *        message.
     * @param cause The nested exception, or <code>null</code>, if a
     *        nested exception is not appropriate.
     */
    public EventException(String key, String file, Object parms[], Throwable cause) {
        this.key = key;
        this.file = file;
        this.parms = parms;

        if (cause != null) {
            ivCause = cause;
        }
    }

    //~ Methods ......................................................

    /**
     * Return the Throwable that is considered the root cause of this
     * <code>EventException</code>. Null is returned
     * if the root cause is nonexistent or unknown. The root cause
     * is the throwable that caused this
     * <code>EventException</code> to get thrown.
     * 
     * <p>
     * The Throwable that is returned is either the Throwable
     * supplied via one of the appropriate constructors, or that set
     * via the {@link #initCause(Throwable)} method. While it is
     * typically unnecessary to override this method, a subclass can
     * override it to return a cause set by some other means, such
     * as a legacy exception chaining infrastructure.
     * </p>
     *
     * @return the Throwable that is the cause of this
     *         <code>EventException</code>, or null if
     *         the cause is nonexistent or unknown.
     */
    public Throwable getCause() {
        if (ivCause == this) {
            return null;
        }
        else {
            return ivCause;
        }
    }

    /**
     * Gets the message key.  The result is the key of the message
     * associated with this text.
     * 
     * <P></p>
     *
     * @return String   The message key .
     */
    public String getKey() {
        return key;
    }

    /**
     * Gets the text of the exception message, translated into the
     * current locale.
     *
     * @return The localized exception message.
     */
    public String getLocalizedMessage() {
        return getLocalizedMessage(Locale.getDefault());
    }

    /**
     * Gets the text of the exception message, translated into the
     * specified locale.
     *
     * @param locale The locale to use for translation of the
     *        exception message, or <code>null</code> to use the
     *        default locale.
     *
     * @return The localized exception message.
     */
    public String getLocalizedMessage(Locale locale) {
        String msg = null;

        if (key == null) {
            msg = super.getMessage();
        }
        else {
            try {
                ResourceBundle bundle = ResourceBundle.getBundle(file, locale);
                String resource = bundle.getString(key);

                if (parms == null) {
                    msg = resource;
                }
                else {
                    MessageFormat messageFormat = new MessageFormat(resource);
                    messageFormat.setLocale(locale);
                    msg = messageFormat.format(parms);
                }
            }
            catch (MissingResourceException mre) {
                StringBuffer sb = new StringBuffer(100);
                sb.append(mre.getLocalizedMessage());

                if (parms != null) {
                    sb.append(": ");

                    for (int i = 0; i < (parms.length - 1); ++i) {
                        sb.append((parms[i] == null) ? "null" : parms[i].toString());
                        sb.append(", ");
                    }

                    sb.append((parms[parms.length - 1] == null) ? "null" : parms[parms.length - 1].toString());
                }

                msg = sb.toString();
            }

            if (ivCause != null) {
                String nestedMsg;

                if (ivCause instanceof EventException) {
                    nestedMsg = ((EventException) ivCause).getLocalizedMessage(locale);
                }
                else {
                    nestedMsg = ivCause.getLocalizedMessage();
                }

                if (nestedMsg != null) {
                    msg = msg + " : " + nestedMsg;
                }
            }
        }

        return msg;
    }

    /**
     * Gets the text of the exception message.  The result is the
     * same as that returned by
     * <code>getLocalizedMessage(Locale.ENGLISH)</code>.
     *
     * @return The exception message.
     */
    public String getMessage() {
        return getLocalizedMessage(Locale.ENGLISH);
    }

    /**
     * Initialize the cause field for this
     * <code>EventException</code> to the specified
     * value. The cause is the Throwable that caused this
     * <code>EventException</code> to get thrown.
     * 
     * <p>
     * This method can be called at most once. It is generally called
     * from within a constructor that takes a Throwable, or
     * immediately after constructing this object with a constructor
     * that does not accept a Throwable. Thus, if a constructor that
     * takes Throwable as a parameter is used to construct this
     * object, it cannot be called at all.
     * </p>
     *
     * @param cause the Throwable which caused this
     *        <code>EventException</code> to be
     *        thrown. Null is tolerated.
     *
     * @return a reference to this <code>Throwable</code> instance.
     *
     * @exception IllegalStateException if this
     *            <code>EventException</code> was
     *            created with a constructor that specified a cause,
     *            or this method has already been called on this
     *            object.
     * @exception IllegalArgumentException if the specified cause is
     *            this <code>EventException</code>. An
     *            exception cannot be its own cause.
     */
    public synchronized Throwable initCause(Throwable cause) throws IllegalStateException, IllegalArgumentException {
        if (ivCause != this) {
            throw new IllegalStateException("Can't overwrite cause");
        }

        if (cause == this) {
            throw new IllegalArgumentException("Self-causation not permitted");
        }

        ivCause = cause;

        return this;
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to System.err.
     */
    public void printStackTrace() {
        printStackTrace(System.err);
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to the specified print stream.
     *
     * @param stream the PrintStream to use for output
     */
    public void printStackTrace(PrintStream stream) {
        // first print out the stack trace for this exception frame
        super.printStackTrace(stream);

        // Then print the nested exception, if it exists
        if ((ivCause != null) && (ivCause != this)) {
            stream.println("---- Begin backtrace for nested exception");
            ivCause.printStackTrace(stream);
        }
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to the specified print writer.
     *
     * @param writer the PrintWriter to use for output
     */
    public void printStackTrace(PrintWriter writer) {
        // first print out the stack trace for this exception frame
        super.printStackTrace(writer);

        // Then print the nested exception, if it exists
        if ((ivCause != null) && (ivCause != this)) {
            writer.println("---- Begin backtrace for nested exception");
            ivCause.printStackTrace(writer);
        }
    }
}