/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.hyades.statistical.ui.*;

import java.io.ByteArrayInputStream;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.ui.internal.wizard.*;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.model.*;
import org.eclipse.ui.part.FileEditorInput;

public class StatisticalModelGraphEditorWizard extends HyadesWizard implements INewWizard {

LocationPage location_page;
ResourceSelectionPage select_page;

ModulePage[] module_pages;

	public StatisticalModelGraphEditorWizard() {
		super();
	}

	public void initPages() {
		EditorPlugin.DBG.info("initialising pages");
		
		IStructuredSelection structuredSelection = getSelection();
		location_page = new LocationPage("location", structuredSelection);
		location_page.setTitle(EditorPlugin.getString("CREATE_NEW_STATCON_FILE"));
		location_page.setImageDescriptor(EditorPlugin.img.getImageDescriptor(ImageManager.IMG_NEW_STATCON));
		location_page.setDescription(EditorPlugin.getString("CREATE_NEW_STATCON_FILE_DESC"));
		location_page.setFileExtension("statcon");
		location_page.setFileName("newfile.statcon");
	}
	
	public void addPages() {
		EditorPlugin.DBG.info("adding pages");

		addPage(location_page);	

		select_page = new ResourceSelectionPage(
							EditorPlugin.getString("CHOOSE_MODELS_TO_GRAPH"),
							new String[]{"trcmxmi","trcnxmi","trcpxmi","trcaxmi"},
							EditorPlugin.getString("CHOOSE_MODELS_TO_GRAPH_DESC")
							);
		select_page.setImageDescriptor(EditorPlugin.img.getImageDescriptor(ImageManager.IMG_NEW_STATCON));
		addPage(select_page);

		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(EditorPlugin.PACKAGE_STATCON_FILE);
		IExtension[] extensions = point.getExtensions();
		
//		IPluginRegistry registry = Platform.getPluginRegistry();
//		IExtensionPoint point = registry.getExtensionPoint(EditorPlugin.PACKAGE_STATCON_FILE);
//		IExtension[] extensions = point.getExtensions();
		
		module_pages = new ModulePage[extensions.length];

		EditorPlugin.DBG.info("found "+module_pages+" module pages");
		
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] configs = extensions[i].getConfigurationElements();
//			IPluginDescriptor pdescriptor = extensions[i].getDeclaringPluginDescriptor();
			
			for (int z = 0; z < configs.length; z++) {
				String file_extension = configs[z].getAttribute("file_extension");
				String element_name = configs[z].getAttribute("element_name");
				String path_attribute = configs[z].getAttribute("path_attribute");
				String page_title = configs[z].getAttribute("page_title");
				String page_description = configs[z].getAttribute("page_description");

				EditorPlugin.DBG.info("added module page for "+file_extension+" files");
				
				ModulePage page = new ModulePage();
				page.element = element_name;
				page.attribute = path_attribute;
				page.page.setImageDescriptor(EditorPlugin.img.getImageDescriptor(ImageManager.IMG_NEW_STATCON));
				
				page.page = new ResourceSelectionPage(page_title,new String[]{file_extension},page_description);
				addPage(page.page);
				
				module_pages[i] = page;

			}	
		}
		
	}


	public boolean performFinish() {
		
		EditorPlugin.DBG.info("finish - creating new StatCon file");
		
		//create the new file
		IFile file = location_page.createNewFile();

		StringBuffer xml = new StringBuffer();
		xml.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
		xml.append("<!-- Statistical Model Graph Viewer configuration file -->\n");
		xml.append("<!-- Not intended to be human editable - edit at your own risk -->\n");
		xml.append("<smgconfig>\n");

		//set the contents of the file to include the specified models
		ArrayList models = select_page.getSelectedFiles();
		for (int i = 0; i < models.size(); i++) {
			IFile model = (IFile)models.get(i);

			URI fileURI = URI.createPlatformResourceURI(model.getFullPath().toString());

			xml.append("  <model file=\"").append(fileURI).append("\" />\n");
		}
		
		for (int i = 0; i < module_pages.length; i++) {
			if (module_pages[i] != null) {
				ArrayList files = module_pages[i].page.getSelectedFiles();
				for (int k = 0; k < files.size(); k++) {
					IFile selected_file = (IFile)files.get(k);
					URI fileURI = URI.createPlatformResourceURI(selected_file.getFullPath().toString());
					xml.append("  <");
					xml.append(module_pages[i].element);
					xml.append(" ");
					xml.append(module_pages[i].attribute);
					xml.append("=\"");
					xml.append(fileURI);
					xml.append("\" />\n");
				}	
			}	
		}

		xml.append("</smgconfig>\n");
		
		ByteArrayInputStream bin = new ByteArrayInputStream(xml.toString().getBytes());
		try {
			EditorPlugin.DBG.info("trying to set file contents");
			file.setContents(bin,true,false,null);
		} catch (Exception e) {
			EditorPlugin.DBG.error("unable to set file contents",e);
			return false;
		}
		
		//open the statistical model grapher over the file
		openEditor(file, null, false);

		return true;	
	}

	/**
	 * Returns an active workbench page.
	 * @return IWorkbenchPage
	 */
	public static IWorkbenchPage getActiveWorkbenchPage()
	{
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = null;
		if(workbench.getActiveWorkbenchWindow() != null)
		{
			activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		}
		else if(workbench.getWorkbenchWindows().length > 0)
		{
			activePage = workbench.getWorkbenchWindows()[0].getActivePage();
		}
		
		return activePage;
	}


	/**
	 * Opens the editor for the specified file.
	 * 
	 * <p>If <code>editorId</code> is null the default editor is opened.  
	 * If substring is true then the <code>editorId</code> value is considered to be 
	 * first part of the editor id.
	 * 
	 * @param file;
	 * @param editorId;
	 * @param substring;
	 * @return The opened editor part or <code>null</code> if this method was not
	 * able to open the editor.
	 */
	public static IEditorPart openEditor(IFile file, String editorId, boolean substring)
	{
		if(file == null)
			return null;
			
		IWorkbenchPage activePage = getActiveWorkbenchPage();
		if(activePage == null)
			return null;
		
		if("".equals(editorId))
			editorId = null;
		
		try
		{
			if(editorId == null)
			{
				return IDE.openEditor(activePage, file, true);
			}
			else
			{
				IWorkbench workbench = PlatformUI.getWorkbench();
				IEditorDescriptor[] editors = workbench.getEditorRegistry().getEditors(file.getName());
				if(editors != null)
				{
					boolean found = false;
					for(int i=0, length=editors.length; i<length; i++)
					{
						if(substring)
							found = editors[i].getId().startsWith(editorId);
						else
							found = editorId.equals(editors[i].getId());
							
						if(found)
							return activePage.openEditor(new FileEditorInput(file), editors[i].getId());
					}
				}				
			}
		}
		catch(Exception e)
		{
			EditorPlugin.DBG.error("couldnt open editor",e);
			EditorPlugin.DBG.logVisibleError(e,EditorPlugin.getString("ERROR_OPEN_STATCON"),false);
		}
	
		return null;
	}

	class ModulePage {
		ResourceSelectionPage page;
		String element;
		String attribute;	
	}

	public class ResourceSelectionPage extends WizardPage {
		// the root element to populate the viewer with
		private IAdaptable root;
	
		// the visual selection widget group
		private CheckboxTreeAndListGroup main_widget;
	
		private Control page_control;
	
		private final static int SIZING_SELECTION_WIDGET_WIDTH = 400;
		private final static int SIZING_SELECTION_WIDGET_HEIGHT = 300;

		String[] extensions;

		public Control getControl() {
			return page_control;
		}

		public ResourceSelectionPage(String title, String[] extensions, String description) {
			super(title); 

			setTitle(title);
			setDescription(description);

			this.extensions = extensions;

			root = ResourcesPlugin.getWorkspace().getRoot();
		}
		
		public void createControl(Composite parent) {

			Composite composite = new Composite(parent,0);

			composite.setLayout(new FillLayout());

			//create the input element, which has the root resource
			//as its only child
			ArrayList input = new ArrayList();
			input.add(root);
		
			main_widget =
				new CheckboxTreeAndListGroup(
					composite,
					input,
					new FolderProvider(),
					WorkbenchLabelProvider.getDecoratingWorkbenchLabelProvider(),
					new SMGFileProvider(extensions),
					WorkbenchLabelProvider.getDecoratingWorkbenchLabelProvider(),
					SWT.NONE,
					// since this page has no other significantly-sized
					// widgets we need to hardcode the combined widget's
					// size, otherwise it will open too small
					SIZING_SELECTION_WIDGET_WIDTH,
					SIZING_SELECTION_WIDGET_HEIGHT);
		
			composite.addControlListener(new ControlListener() {
				public void controlMoved(ControlEvent e) {};
				public void controlResized(ControlEvent e) {
					//Also try and reset the size of the columns as appropriate
					TableColumn[] columns = main_widget.getListTable().getColumns();
					for (int i = 0; i < columns.length; i++) {
						columns[i].pack();
					}
				}
			});
		
			page_control = composite;
			setControl(page_control);
		}

		public ArrayList getSelectedFiles() {		
			Iterator resultEnum = main_widget.getAllCheckedListItems();
			ArrayList list = new ArrayList();
			while (resultEnum.hasNext()) {
				list.add(resultEnum.next());
			}
			return list;
		}
	
	}
	
	public class FolderProvider extends WorkbenchContentProvider {
		public Object[] getChildren(Object o) {
			
			int resourceType = IResource.FOLDER | IResource.PROJECT | IResource.ROOT;
			
			if (o instanceof IContainer) {
				IResource[] members = null;
				try {
					members = ((IContainer)o).members();
				} catch (CoreException e) {
					//just return an empty set of children
					return new Object[0];
				}

				//filter out the desired resource types
				ArrayList results = new ArrayList();
				for (int i = 0; i < members.length; i++) {
					//And the test bits with the resource types to see if they are what we want
					if ((members[i].getType() & resourceType) > 0) {
						results.add(members[i]);
					}
				}
				return results.toArray();
			} else {
				//input element case
				if (o instanceof ArrayList) {
					return ((ArrayList)o).toArray();
				} else {
					return new Object[0];
				}
			}
		}
	}

	public class SMGFileProvider extends WorkbenchContentProvider {
		String[] extensions;
		public SMGFileProvider(String[] extensions) {
			this.extensions = extensions;
		}
		public Object[] getChildren(Object o) {
			if (o instanceof IContainer) {
				IResource[] members = null;
				try {
					members = ((IContainer)o).members();
				} catch (CoreException e) {
					//just return an empty set of children
					return new Object[0];
				}

				//filter out the desired resource types
				ArrayList results = new ArrayList();
				for (int i = 0; i < members.length; i++) {
					//And the test bits with the resource types to see if they are what we want
					if ((members[i].getType() & IResource.FILE) > 0) {
						String name = members[i].getName();
						name = name.substring(name.lastIndexOf(".")+1);
						for (int k = 0; k < extensions.length; k++) {
							if (name.equalsIgnoreCase(extensions[k])) {
								results.add(members[i]);
							}
						}
					}
				}
				return results.toArray();
			} else {
				//input element case
				if (o instanceof ArrayList) {
					return ((ArrayList)o).toArray();
				} else {
					return new Object[0];
				}
			}
		}
	}

}