/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.form;

import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.extension.TestSuiteEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementSection;
import org.eclipse.hyades.test.ui.editor.form.util.TestObjectiveSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.help.WorkbenchHelp;

/**
 * This class represents the first page of the test context editor  : page OVERVIEW.
 * <pre>
 *    _______________________  
 *   | Test Context Editor   |
 *   |_______________________|________________________________________________________ 
 *   |                                                                                |
 *   |  _ General Information __________      _ form2 overview _____________________  |
 *   | |                                |    |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |______________________________________| |
 *   | |                                |                                             |
 *   | |________________________________|     _ form4 overview _____________________  |
 *   |                                       |                                      | |
 *   |  _ Test Objective _______________     |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |                                      | |
 *   | |                                |    |______________________________________| |
 *   | |________________________________|                                             |
 *   |                                                                                |
 *   |  _ form1 overview _______________                                              |
 *   | |                                |                                             |
 *   | |                                |                                             |
 *   | |________________________________|                                             |
 *   |                                                                                |
 *   |________________________________________________________________________________|
 *   |_Overview_|form1| form2 | form3 | form4 |                                       | 
 * </pre>
 *  
 * This class should be used in conjonction with <a href=TestContextOverviewContribution.html>
 * org.eclipse.hyades.test.ui.editor.form.TestContextOverviewContribution</a>.
 * 
 * 
 * 
 * @author bcormier
 * @since 3.0.0
 *
 */

public class TestContextOverview extends EditorForm {

    /**
     * This field stores the instance of the test context overview contributor.
     */
    protected TestContextOverviewContribution tsoContributor;
    
    private TestSuiteEditorExtension editor;
    private NamedElementSection genericInfoSection;
    private TestObjectiveSection testObjectiveSection;
	private Object[] leftSections;
	private Object[] rightSections;
    private boolean hideGenericInformation = false;
    private boolean hideTestObjective = false;
    
    /**
     * The constructor for the class.
     * @param baseEditorExtension the base editor extension
     * @param widgetFactory the widget factory
     */
    public TestContextOverview(BaseEditorExtension baseEditorExtension,
            WidgetFactory widgetFactory) {
        super(baseEditorExtension, widgetFactory);
        
        editor = (TestSuiteEditorExtension)baseEditorExtension;
        setHeadingText(TestUIPlugin.getString("W_OVERVIEW")); //$NON-NLS-1$
        
    }
    
    /**
     * The constructor for the class.
     * @param baseEditorExtension the base editor extension
     * @param widgetFactory the widget factory
     * @param testContextOverviewContributor the contributor that allows to customize the overview page 
     */
    public TestContextOverview(BaseEditorExtension baseEditorExtension,
            WidgetFactory widgetFactory,
            TestContextOverviewContribution testContextOverviewContributor) {
        super(baseEditorExtension, widgetFactory);
        
        editor = (TestSuiteEditorExtension)baseEditorExtension;
        tsoContributor = testContextOverviewContributor;
        tsoContributor.setOverviewForm(this);
        setHeadingText(TestUIPlugin.getString("W_OVERVIEW")); //$NON-NLS-1$
    }
    
    
	/**
	 * This method frees the widgets of this form.
	 */
	public void dispose() {
	    
		for (int i=0; leftSections!=null && i<leftSections.length; i++) {
			if (leftSections[i]!=null && (leftSections[i] instanceof IDisposable)) {
			    ((IDisposable)leftSections[i]).dispose();
			    leftSections[i] = null;
			}
		}
		for (int i=0; rightSections!=null && i<rightSections.length; i++) {
			if (rightSections[i]!=null && (rightSections[i] instanceof IDisposable)) {
			    ((IDisposable)rightSections[i]).dispose();
			    rightSections[i] = null;
			}
		}
		if (genericInfoSection!=null && (genericInfoSection instanceof IDisposable)) {
		    genericInfoSection.dispose();
		    genericInfoSection = null;
		}
		if (testObjectiveSection!=null && (testObjectiveSection instanceof IDisposable)) {
		    testObjectiveSection.dispose();
		    testObjectiveSection = null;
		}
		super.dispose();
	}
	
	/**
	 * Returns the test context manipulated by this form.
	 * @return the test context.
	 */
	protected TPFTestSuite getTestContext()
	{
		return editor.getTestSuite();
	}
    
	/**
	 * Creates the control and adds a help handler to the control.
	 * @return Control the created control
	 * @see org.eclipse.hyades.test.ui.editor.form.util.EditorForm#createControl()
	 */
	public Control createControl()
	{
		Control control = super.createControl();
		WorkbenchHelp.setHelp(getControl(), TestUIPlugin.getID() + ContextIds.TESTSUITE_FORM);
		return control;
	}

	/**
	 * Creates the contents of this editor form.
	 * @param parent the parent composite
	 * @see org.eclipse.hyades.test.ui.editor.form.util.EditorForm#createEditorFormContents(Composite parent)
	 */
    protected void createEditorFormContents(Composite parent) {
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);
		
		if (tsoContributor==null || (tsoContributor!=null && tsoContributor.hideGenericInformation())) {
		    hideGenericInformation = true;
		} else {
			Control control = createGenericInformation(leftColumn);
			control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING));
			WorkbenchHelp.setHelp(control, TestUIPlugin.getID() + ContextIds.TESTSUITE_GEN_FORM);
		}
		if (tsoContributor==null || (tsoContributor!=null && tsoContributor.hideTestObjectiveSection())) {
		    hideTestObjective = true;
		} else {
		    createTestObjectiveSection(leftColumn);
		}

		if (tsoContributor!=null) {
			leftSections = tsoContributor.createLeftSections(editor);
			rightSections = tsoContributor.createRightSections(editor);
	
			for (int i=0; i<leftSections.length; i++) {
				if (!(leftSections[i] instanceof FormSection))
					continue;
			    registerSection((FormSection)leftSections[i]);
			    Control control = ((FormSection)leftSections[i]).createControl(leftColumn, factory);
			    control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING));
			}
			
			for (int i=0; i<rightSections.length; i++) {
				if (!(rightSections[i] instanceof FormSection))
					continue;
			    registerSection((FormSection)rightSections[i]);
			    Control control = ((FormSection)rightSections[i]).createControl(rightColumn, factory);
			    control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING));
			}
		}

    }

	/**
	 * Creates the test objective section of this editor form.
	 * @param arg the parent
	 */
    protected Control createTestObjectiveSection(Composite arg) {
        Control retControl = null;
        testObjectiveSection = new TestObjectiveSection(getBaseEditorExtension(), getWidgetFactory());
        testObjectiveSection.setCollapsable(true);
        testObjectiveSection.setHeaderText(TestUIPlugin.getString("L_TEST_OBJECTIVE")); //$NON-NLS-1$
        testObjectiveSection.setDescription(TestUIPlugin.getString("L_TEST_OBJECTIVE_DSC")); //$NON-NLS-1$
        retControl = testObjectiveSection.createControl(arg, getWidgetFactory());
        return retControl;
    }
 
	/**
	 * Creates the generic information section of this editor form.
	 * @param arg the parent
	 */
	protected Control createGenericInformation(Composite arg)
	{
		genericInfoSection = new NamedElementSection(this)
		{
			private StyledText typeText;
			private StyledText fileText;

			protected void addCenterControls(Composite parent, FormWidgetFactory formWidgetFactory) {
			    if (tsoContributor != null)
				    tsoContributor.addCenterGenericInformation(parent, getWidgetFactory());  
			}
			
			protected void addSouthControls(Composite parent, FormWidgetFactory formWidgetFactory)
			{
				Composite detailComposite = getWidgetFactory().createComposite(parent);
				GridLayout gridLayout = new GridLayout(2, false);
				gridLayout.marginHeight = 0;
				gridLayout.marginWidth = 0;
				detailComposite.setLayout(gridLayout);
				detailComposite.setLayoutData(GridDataUtil.createHorizontalFill());
				WidgetFactory detailWidgetFactory = getWidgetFactory();
				detailWidgetFactory.createLabel(detailComposite, TestUIPlugin.getString("L_TYPE")); //$NON-NLS-1$
				typeText = detailWidgetFactory.createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
				typeText.setEnabled(false);
				typeText.setLayoutData(GridDataUtil.createHorizontalFill());

				detailWidgetFactory.createLabel(detailComposite, TestUIPlugin.getString("L_FILE")); //$NON-NLS-1$
				fileText = detailWidgetFactory.createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
				fileText.setEnabled(false);
				fileText.setLayoutData(GridDataUtil.createHorizontalFill());

				WorkbenchHelp.setHelp(typeText, TestUIPlugin.getID() + ContextIds.TESTSUITE_TYPE_FORM);
				WorkbenchHelp.setHelp(fileText, TestUIPlugin.getID() + ContextIds.TESTSUITE_FILE_FORM);
				
				if (tsoContributor != null)
				    tsoContributor.addSouthGenericInformation(detailComposite, getWidgetFactory());

			}

			public void setInput(Object object)
			{
				super.setInput(object);

				if (getNameTextField()!=null) {
					String name = getNameTextField().getText();
					if (name!=null && !name.equals("")) { //$NON-NLS-$
						getBaseEditorExtension().getHyadesEditorPart().setTitle(name);
					}
				}
				String value = getTestContext().getType();
				if(value != null)
				{
					
					IAssociationDescriptor descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
					if((descriptor != null) && (descriptor.getName() != null))
						value = descriptor.getName();
					typeText.setText(value);
				}

				IFile file = EMFUtil.getWorkspaceFile(getTestContext());
				if(file != null)
					fileText.setText(file.getFullPath().toString());
			}
		};
		

		registerSection(genericInfoSection);
		genericInfoSection.setHeaderText(TestUIPlugin.getString("EDT_GENERAL_INFO")); //$NON-NLS-1$
		genericInfoSection.setDescription(TestUIPlugin.getString("EDT_GENERAL_DSC")); //$NON-NLS-1$
		Control retControl = genericInfoSection.createControl(arg, getWidgetFactory());
		
		return retControl;
	}
    
    
	/**
	 * Initializes the sections of this form with the test context information.
	 * @see org.eclipse.hyades.test.ui.editor.form.util.EditorForm#load()
	 */
	public void load() {
		
		if (!hideGenericInformation) {
		    genericInfoSection.setInput(getTestContext());
		}
		
		if (!hideTestObjective) {
		    testObjectiveSection.setInput(getTestContext());
		}
		
		if (tsoContributor != null) {
		    tsoContributor.setTestContext(getTestContext());
		    if (!hideGenericInformation) {
		        tsoContributor.initializeSouthGenericInformation();
		        tsoContributor.initializeCenterGenericInformation();
		    }
		    tsoContributor.initializeLeftSections(leftSections);
		    tsoContributor.initializeRightSections(rightSections);
		}
	}
	
	/**
	 * Returns whether this form has been activated yet.
	 * @return the activation state
	 */
	public boolean activated()
	{
		return true;
	}
	
	
	/**
	 * Returns the structural feature of this test context overview. i.e. the test suites structural feature.
	 * @return the structural feature that corresponds to test suites
     * @see org.eclipse.hyades.test.ui.editor.form.util.EditorForm#getSelection()
     */
    public ISelection getSelection() {
        return new StructuredSelection(getTestContext());
    }
}
